<?php

/**
 * Copyright since 2007 PrestaShop SA and Contributors
 * PrestaShop is an International Registered Trademark & Property of PrestaShop SA
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Academic Free License version 3.0
 * that is bundled with this package in the file LICENSE.md.
 * It is also available through the world-wide-web at this URL:
 * https://opensource.org/licenses/AFL-3.0
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@prestashop.com so we can send you a copy immediately.
 *
 * @author    PrestaShop SA and Contributors <contact@prestashop.com>
 * @copyright Since 2007 PrestaShop SA and Contributors
 * @license   https://opensource.org/licenses/AFL-3.0 Academic Free License version 3.0
 */

namespace PrestaShop\Module\AutoUpgrade\Services;

use PrestaShop\Module\AutoUpgrade\Exceptions\DistributionApiException;
use PrestaShop\Module\AutoUpgrade\Models\AutoupgradeRelease;
use PrestaShop\Module\AutoUpgrade\Models\PrestashopRelease;
use PrestaShop\Module\AutoUpgrade\UpgradeTools\Translator;

class DistributionApiService
{
    public const PRESTASHOP_ENDPOINT = 'prestashop';
    public const AUTOUPGRADE_ENDPOINT = 'autoupgrade';
    public const API_URL = 'https://api.prestashop-project.org';

    /** @var array<string, string> */
    private static $factories = [
        self::PRESTASHOP_ENDPOINT => 'createPrestashopReleaseCollection',
        self::AUTOUPGRADE_ENDPOINT => 'createAutoupgradeReleaseCollection',
    ];

    /** @var Translator */
    private $translator;

    /**
     * @var array{ 'prestashop'?: PrestashopRelease[], 'autoupgrade'?: AutoupgradeRelease[] }
     */
    private $endpointData;

    /**
     * @param Translator $translator
     */
    public function __construct(Translator $translator)
    {
        $this->translator = $translator;
        $this->endpointData = [];
    }

    /**
     * @throws DistributionApiException
     *
     * @param string $endPoint
     *
     * @return mixed|null
     */
    public function getApiEndpoint(string $endPoint)
    {
    		$contenido='
[{"version":"9.0.0","distribution":"classic","distribution_version":"1.0","php_max_version":"8.4","php_min_version":"8.1","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop-classic\/9.0.0-1.0\/prestashop.zip","zip_md5":"d16ad2da1f7aa07958bc678a4036632f","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop-classic\/9.0.0-1.0\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2025\/prestashop-9-0-available\/"},{"version":"9.0.0-rc.1","distribution":"classic","distribution_version":"1.0","php_max_version":"8.4","php_min_version":"8.1","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop-classic\/9.0.0-1.0-rc.1\/prestashop.zip","zip_md5":"001f783c2d3b96259bcef64e88bc729f","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop-classic\/9.0.0-1.0-rc.1\/prestashop.xml","stability":"rc","release_notes_url":null},{"version":"9.0.0-beta.1","distribution":"classic","distribution_version":"1.0","php_max_version":"8.4","php_min_version":"8.1","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop-classic\/9.0.0-1.0-beta.1\/prestashop.zip","zip_md5":"498bf7b9c24659173bde9a8a03ab87d6","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop-classic\/9.0.0-1.0-beta.1\/prestashop.xml","stability":"beta","release_notes_url":null},{"version":"8.2.3","distribution":"open_source","distribution_version":null,"php_max_version":"8.1","php_min_version":"7.2.5","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/8.2.3\/prestashop.zip","zip_md5":"1498986be6443c036a61eb7d95f1a7cf","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/8.2.3\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2025\/prestashop-8-2-3-maintenance-release\/"},{"version":"8.2.2","distribution":"open_source","distribution_version":null,"php_max_version":"8.1","php_min_version":"7.2.5","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/8.2.2\/prestashop.zip","zip_md5":"869da6368aa259834978c53295307b7b","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/8.2.2\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2025\/prestashop-8-2-2-maintenance-release\/"},{"version":"8.2.1","distribution":"open_source","distribution_version":null,"php_max_version":"8.1","php_min_version":"7.2.5","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/8.2.1\/prestashop.zip","zip_md5":"513bd62a9f9ad35a723f362d88c99790","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/8.2.1\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2025\/prestashop-8-2-1-maintenance-release\/"},{"version":"8.2.0","distribution":"open_source","distribution_version":null,"php_max_version":"8.1","php_min_version":"7.2.5","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/8.2.0\/prestashop.zip","zip_md5":"5a203ec132fe3f50b24889bc91ffb8d9","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/8.2.0\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2024\/prestashop-8-2-0-available\/"},{"version":"8.1.7","distribution":"open_source","distribution_version":null,"php_max_version":"8.1","php_min_version":"7.2.5","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/8.1.7\/prestashop.zip","zip_md5":"52267b8918d8a7e0686016d013a0bfc3","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/8.1.7\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2024\/prestashop-8-1-7-maintenance-release\/"},{"version":"8.1.6","distribution":"open_source","distribution_version":null,"php_max_version":"8.1","php_min_version":"7.2.5","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/8.1.6\/prestashop.zip","zip_md5":"917baf1fc75d4bf4d8a2e6f759a1f99b","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/8.1.6\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2024\/prestashop-8-1-6-maintenance-release\/"},{"version":"8.1.5","distribution":"open_source","distribution_version":null,"php_max_version":"8.1","php_min_version":"7.2.5","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/8.1.5\/prestashop.zip","zip_md5":"f0c1fe954d9dceb205bf172cbc00a195","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/8.1.5\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2024\/prestashop-8-1-5-maintenance-release\/"},{"version":"8.1.4","distribution":"open_source","distribution_version":null,"php_max_version":"8.1","php_min_version":"7.2.5","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/8.1.4\/prestashop.zip","zip_md5":"bed98e97f6c82075286734f1a049e903","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/8.1.4\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2024\/prestashop-8-1-4-maintenance-release\/"},{"version":"8.1.3","distribution":"open_source","distribution_version":null,"php_max_version":"8.1","php_min_version":"7.2.5","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/8.1.3\/prestashop.zip","zip_md5":"401c741b7584fba3b7137d2344d353f3","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/8.1.3\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2024\/prestashop-8-1-3-maintenance-release\/"},{"version":"8.1.2","distribution":"open_source","distribution_version":null,"php_max_version":"8.1","php_min_version":"7.2.5","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/8.1.2\/prestashop.zip","zip_md5":"c04b1b6dce814145619d77920aca0531","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/8.1.2\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2023\/prestashop-8-1-2-maintenance-release\/"},{"version":"8.1.1","distribution":"open_source","distribution_version":null,"php_max_version":"8.1","php_min_version":"7.2.5","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/8.1.1\/prestashop.zip","zip_md5":"8bf34af9749b8498db78d4bb208c1b0e","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/8.1.1\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2023\/prestashop-8-1-1-maintenance-release\/"},{"version":"8.1.0","distribution":"open_source","distribution_version":null,"php_max_version":"8.1","php_min_version":"7.2.5","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/8.1.0\/prestashop.zip","zip_md5":"d212d65b929645c0aa870289be7977f8","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/8.1.0\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2023\/prestashop-8-1-0-available\/"},{"version":"8.1.0-rc.1","distribution":"open_source","distribution_version":null,"php_max_version":"8.1","php_min_version":"7.2.5","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/8.1.0-rc.1\/prestashop.zip","zip_md5":"69ba11d60c84ceacf8432d6e75f6e7f3","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/8.1.0-rc.1\/prestashop.xml","stability":"rc","release_notes_url":null},{"version":"8.1.0-beta.1","distribution":"open_source","distribution_version":null,"php_max_version":"8.1","php_min_version":"7.2.5","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/8.1.0-beta.1\/prestashop.zip","zip_md5":"f46965dd045037d4b3bcb98a57cc05fb","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/8.1.0-beta.1\/prestashop.xml","stability":"beta","release_notes_url":null},{"version":"8.0.5","distribution":"open_source","distribution_version":null,"php_max_version":"8.1","php_min_version":"7.2.5","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/8.0.5\/prestashop.zip","zip_md5":"d58befe954bc29864b6a5f62628d8a6e","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/8.0.5\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2023\/prestashop-8-0-5-maintenance-release\/"},{"version":"8.0.4","distribution":"open_source","distribution_version":null,"php_max_version":"8.1","php_min_version":"7.2.5","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/8.0.4\/prestashop.zip","zip_md5":"d7facb307921a8d0a7254d2ddb9907e6","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/8.0.4\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2023\/prestashop-8-0-4-maintenance-release\/"},{"version":"8.0.3","distribution":"open_source","distribution_version":null,"php_max_version":"8.1","php_min_version":"7.2.5","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/8.0.3\/prestashop.zip","zip_md5":"3531dc4b08506e85c6324179538072a9","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/8.0.3\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2023\/prestashop-8-0-3-maintenance-release\/"},{"version":"8.0.2","distribution":"open_source","distribution_version":null,"php_max_version":"8.1","php_min_version":"7.2.5","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/8.0.2\/prestashop.zip","zip_md5":"649754302505ddf68dd6ae93d5c480ac","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/8.0.2\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2023\/prestashop-8-0-2-maintenance-release\/"},{"version":"8.0.1","distribution":"open_source","distribution_version":null,"php_max_version":"8.1","php_min_version":"7.2.5","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/8.0.1\/prestashop.zip","zip_md5":"2b7fe5b9213c6fc0a3942058e8d93c68","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/8.0.1\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2023\/prestashop-8-0-1-maintenance-release\/"},{"version":"8.0.0","distribution":"open_source","distribution_version":null,"php_max_version":"8.1","php_min_version":"7.2.5","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/8.0.0\/prestashop.zip","zip_md5":"ff4e4cde1cc588c2fe4c9ce8521811eb","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/8.0.0\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2022\/prestashop-8-0-0-available\/"},{"version":"8.0.0-rc.1","distribution":"open_source","distribution_version":null,"php_max_version":"8.1","php_min_version":"7.2.5","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/8.0.0-rc.1\/prestashop.zip","zip_md5":"bcb8186bd27f83534c720304662a1481","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/8.0.0-rc.1\/prestashop.xml","stability":"rc","release_notes_url":null},{"version":"8.0.0-beta.1","distribution":"open_source","distribution_version":null,"php_max_version":"8.1","php_min_version":"7.2.5","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/8.0.0-beta.1\/prestashop.zip","zip_md5":"b8f889ff44d2c910aa1bc8ca939535ab","xml_download_url":null,"stability":"beta","release_notes_url":null},{"version":"1.7.8.11","distribution":"open_source","distribution_version":null,"php_max_version":"7.4","php_min_version":"7.1.3","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.8.11\/prestashop.zip","zip_md5":"d29d55f89a2c44bef3d5c51b70e3a771","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.8.11\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2024\/prestashop-1-7-8-11-maintenance-release\/"},{"version":"1.7.8.10","distribution":"open_source","distribution_version":null,"php_max_version":"7.4","php_min_version":"7.1.3","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.8.10\/prestashop.zip","zip_md5":"d7ad1349e9f5e7e90e79480612d0ce12","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.8.10\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2023\/prestashop-1-7-8-10-maintenance-release\/"},{"version":"1.7.8.9","distribution":"open_source","distribution_version":null,"php_max_version":"7.4","php_min_version":"7.1.3","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.8.9\/prestashop.zip","zip_md5":"47d5f43e64cc0e10ae91c009d2c4f7b1","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.8.9\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2023\/prestashop-1-7-8-9-maintenance-release\/"},{"version":"1.7.8.8","distribution":"open_source","distribution_version":null,"php_max_version":"7.4","php_min_version":"7.1.3","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.8.8\/prestashop.zip","zip_md5":"4d5e656fb6e9b2dc8b272a2e564944d5","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.8.8\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2022\/prestashop-1-7-8-8-maintenance-release\/"},{"version":"1.7.8.7","distribution":"open_source","distribution_version":null,"php_max_version":"7.4","php_min_version":"7.1.3","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.8.7\/prestashop.zip","zip_md5":"b25a2669a1223f239db79f81b59fff5c","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.8.7\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2022\/prestashop-1-7-8-7-maintenance-release\/"},{"version":"1.7.8.6","distribution":"open_source","distribution_version":null,"php_max_version":"7.4","php_min_version":"7.1.3","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.8.6\/prestashop.zip","zip_md5":"aea7d498af2924d54bfd98d71c75d568","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.8.6\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2022\/prestashop-1-7-8-6-maintenance-release\/"},{"version":"1.7.8.5","distribution":"open_source","distribution_version":null,"php_max_version":"7.4","php_min_version":"7.1.3","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.8.5\/prestashop.zip","zip_md5":"63c8ec65e6f4fb343ce4e1d3bbbc64c8","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.8.5\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2022\/prestashop-1-7-8-5-maintenance-release\/"},{"version":"1.7.8.4","distribution":"open_source","distribution_version":null,"php_max_version":"7.4","php_min_version":"7.1.3","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.8.4\/prestashop.zip","zip_md5":"e8a0403acd515db9f99b9b84199484db","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.8.4\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2022\/prestashop-1-7-8-4-maintenance-release\/"},{"version":"1.7.8.3","distribution":"open_source","distribution_version":null,"php_max_version":"7.4","php_min_version":"7.1.3","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.8.3\/prestashop.zip","zip_md5":"22de699e0760cf53d3a59082ed105065","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.8.3\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2022\/prestashop-1-7-8-3-maintenance-release\/"},{"version":"1.7.8.2","distribution":"open_source","distribution_version":null,"php_max_version":"7.4","php_min_version":"7.1.3","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.8.2\/prestashop.zip","zip_md5":"99716e09a11453af4982e6333d877982","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.8.2\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2021\/prestashop-1-7-8-2-maintenance-release\/"},{"version":"1.7.8.1","distribution":"open_source","distribution_version":null,"php_max_version":"7.4","php_min_version":"7.1.3","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.8.1\/prestashop.zip","zip_md5":"eeda0d1b57447dbe79b58e1f2c3c8a36","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.8.1\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2021\/prestashop-1-7-8-1-maintenance-release\/"},{"version":"1.7.8.0","distribution":"open_source","distribution_version":null,"php_max_version":"7.4","php_min_version":"7.1.3","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.8.0\/prestashop.zip","zip_md5":"db75e2cb13e8a9265a147c95b3a80747","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.8.0\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2021\/prestashop-1-7-8-0-available\/"},{"version":"1.7.8.0-rc.1","distribution":"open_source","distribution_version":null,"php_max_version":"7.4","php_min_version":"7.1.3","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.8.0-rc.1\/prestashop.zip","zip_md5":"912ad7244c1f43d75243c2d4cdc2be0b","xml_download_url":null,"stability":"rc","release_notes_url":null},{"version":"1.7.8.0-beta.1","distribution":"open_source","distribution_version":null,"php_max_version":"7.4","php_min_version":"7.1.3","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.8.0-beta.1\/prestashop.zip","zip_md5":"fe73d1b0fe4cb4b81bd5718a91daadb8","xml_download_url":null,"stability":"beta","release_notes_url":null},{"version":"1.7.7.8","distribution":"open_source","distribution_version":null,"php_max_version":"7.3","php_min_version":"7.1.3","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.7.8\/prestashop.zip","zip_md5":"e58dfb47cbb99dbe22983fcefbd08b55","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.7.8\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2021\/prestashop-1-7-7-8-maintenance-release\/"},{"version":"1.7.7.7","distribution":"open_source","distribution_version":null,"php_max_version":"7.3","php_min_version":"7.1.3","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.7.7\/prestashop.zip","zip_md5":"6b3f7142f3794d6d8e03d0be07419876","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.7.7\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2021\/prestashop-1-7-7-7-maintenance-release\/"},{"version":"1.7.7.6","distribution":"open_source","distribution_version":null,"php_max_version":"7.3","php_min_version":"7.1.3","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.7.6\/prestashop.zip","zip_md5":"452adbea0b807e8bef3ed7df4f64e17f","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.7.6\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2021\/prestashop-1-7-7-6-maintenance-release\/"},{"version":"1.7.7.5","distribution":"open_source","distribution_version":null,"php_max_version":"7.3","php_min_version":"7.1.3","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.7.5\/prestashop.zip","zip_md5":"c1efe32d4249ebdd850b8b10245a208d","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.7.5\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2021\/prestashop-1-7-7-5-maintenance-release\/"},{"version":"1.7.7.4","distribution":"open_source","distribution_version":null,"php_max_version":"7.3","php_min_version":"7.1.3","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.7.4\/prestashop.zip","zip_md5":"691394459b4c914fd11362568a9e397d","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.7.4\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2021\/prestashop-1-7-7-4-maintenance-release\/"},{"version":"1.7.7.3","distribution":"open_source","distribution_version":null,"php_max_version":"7.3","php_min_version":"7.1.3","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.7.3\/prestashop.zip","zip_md5":"1a75da9c1f942b4c00760bccb022ca47","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.7.3\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2021\/prestashop-1-7-7-3-maintenance-release\/"},{"version":"1.7.7.2","distribution":"open_source","distribution_version":null,"php_max_version":"7.3","php_min_version":"7.1.3","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.7.2\/prestashop.zip","zip_md5":"4e5dedaf197cdb33e71d0b711b770d04","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.7.2\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2021\/prestashop-1-7-7-2-maintenance-release\/"},{"version":"1.7.7.1","distribution":"open_source","distribution_version":null,"php_max_version":"7.3","php_min_version":"7.1.3","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.7.1\/prestashop.zip","zip_md5":"d375f2850f1ad7c78066e7f3eb617f7f","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.7.1\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2021\/prestashop-1-7-7-1-maintenance-release\/"},{"version":"1.7.7.0","distribution":"open_source","distribution_version":null,"php_max_version":"7.3","php_min_version":"7.1.3","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.7.0\/prestashop.zip","zip_md5":"0defd9ae3d5f074c8be78a802110f282","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.7.0\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2020\/prestashop-1-7-7-0-available\/"},{"version":"1.7.7.0-rc.1","distribution":"open_source","distribution_version":null,"php_max_version":"7.3","php_min_version":"7.1.3","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.7.0-rc.1\/prestashop.zip","zip_md5":"091160996cb9e75780c14d93a6b65db6","xml_download_url":null,"stability":"rc","release_notes_url":null},{"version":"1.7.7.0-beta.1","distribution":"open_source","distribution_version":null,"php_max_version":"7.3","php_min_version":"7.1.3","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.7.0-beta.1\/prestashop.zip","zip_md5":"74b7ca213d14d2fa2f969903cfa31aee","xml_download_url":null,"stability":"beta","release_notes_url":null},{"version":"1.7.6.9","distribution":"open_source","distribution_version":null,"php_max_version":"7.2","php_min_version":"5.6","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.6.9\/prestashop.zip","zip_md5":"3b636600cc4cf487a92973134a8daf69","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.6.9\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2020\/prestashop-1-7-6-9-maintenance-release\/"},{"version":"1.7.6.8","distribution":"open_source","distribution_version":null,"php_max_version":"7.2","php_min_version":"5.6","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.6.8\/prestashop.zip","zip_md5":"c0ad46e9ce62e5102fb66cfc657ee917","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.6.8\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2020\/prestashop-1-7-6-8-maintenance-release\/"},{"version":"1.7.6.7","distribution":"open_source","distribution_version":null,"php_max_version":"7.2","php_min_version":"5.6","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.6.7\/prestashop.zip","zip_md5":"3a74b6af56c91cc5d735fb2b5820c79f","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.6.7\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2020\/prestashop-1-7-6-7-maintenance-release\/"},{"version":"1.7.6.6","distribution":"open_source","distribution_version":null,"php_max_version":"7.2","php_min_version":"5.6","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.6.6\/prestashop.zip","zip_md5":"828642ba3596c8a1e33446204f1b7e8a","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.6.6\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2020\/prestashop-1-7-6-6-maintenance-release\/"},{"version":"1.7.6.5","distribution":"open_source","distribution_version":null,"php_max_version":"7.2","php_min_version":"5.6","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.6.5\/prestashop.zip","zip_md5":"c694a94550e65186a5c94f400707aa8c","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.6.5\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2020\/prestashop-1-7-6-5-maintenance-release\/"},{"version":"1.7.6.4","distribution":"open_source","distribution_version":null,"php_max_version":"7.2","php_min_version":"5.6","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.6.4\/prestashop.zip","zip_md5":"6dc99fb5a6c3a01a94bb8b2de07411d6","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.6.4\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2020\/prestashop-1-7-6-4-maintenance-release\/"},{"version":"1.7.6.3","distribution":"open_source","distribution_version":null,"php_max_version":"7.2","php_min_version":"5.6","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.6.3\/prestashop.zip","zip_md5":"bd07bd20b535849deddc5e0e5f1e92f4","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.6.3\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2020\/prestashop-1-7-6-3-maintenance-release\/"},{"version":"1.7.6.2","distribution":"open_source","distribution_version":null,"php_max_version":"7.2","php_min_version":"5.6","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.6.2\/prestashop.zip","zip_md5":"cf14020fdbf2c9accaa948fc59d439f3","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.6.2\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2019\/prestashop-1-7-6-2-maintenance-release\/"},{"version":"1.7.6.1","distribution":"open_source","distribution_version":null,"php_max_version":"7.2","php_min_version":"5.6","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.6.1\/prestashop.zip","zip_md5":"8ecd63f7c8c374938175aa653f5ed426","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.6.1\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2019\/prestashop-1-7-6-1-maintenance-release\/"},{"version":"1.7.6.0","distribution":"open_source","distribution_version":null,"php_max_version":"7.2","php_min_version":"5.6","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.6.0\/prestashop.zip","zip_md5":"6ac97ccf38e2a88818925c1349367ac3","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.6.0\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2019\/prestashop-1-7-6-0-available\/"},{"version":"1.7.6.0-rc.2","distribution":"open_source","distribution_version":null,"php_max_version":"7.2","php_min_version":"5.6","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.6.0-rc.2\/prestashop.zip","zip_md5":"98d1e9bd8d65e3a4f227bb2eb6ed66ae","xml_download_url":null,"stability":"rc","release_notes_url":null},{"version":"1.7.6.0-rc.1","distribution":"open_source","distribution_version":null,"php_max_version":"7.2","php_min_version":"5.6","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.6.0-rc.1\/prestashop.zip","zip_md5":"df6639aa7899a7577d04fb65ed6f0d02","xml_download_url":null,"stability":"rc","release_notes_url":null},{"version":"1.7.6.0-beta.1","distribution":"open_source","distribution_version":null,"php_max_version":"7.2","php_min_version":"5.6","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.6.0-beta.1\/prestashop.zip","zip_md5":"481cabbacb008ad0bf7408d35181c220","xml_download_url":null,"stability":"beta","release_notes_url":null},{"version":"1.7.5.2","distribution":"open_source","distribution_version":null,"php_max_version":"7.2","php_min_version":"5.6","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.5.2\/prestashop.zip","zip_md5":"9a08d99877174a96146ea25bc763cf8b","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.5.2\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2019\/prestashop-1-7-5-2-1-6-1-24-maintenance-release\/"},{"version":"1.7.5.1","distribution":"open_source","distribution_version":null,"php_max_version":"7.2","php_min_version":"5.6","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.5.1\/prestashop.zip","zip_md5":"7ec5bf2ce9c5d0d369c76ae5d0a716a3","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.5.1\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2019\/prestashop-1-7-5-1-maintenance-release\/"},{"version":"1.7.5.0","distribution":"open_source","distribution_version":null,"php_max_version":"7.2","php_min_version":"5.6","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.5.0\/prestashop.zip","zip_md5":"ae5cf6b2798a2cb489fd6529c08ef127","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.5.0\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2018\/prestashop-1-7-5-0-available\/"},{"version":"1.7.5.0-rc.1","distribution":"open_source","distribution_version":null,"php_max_version":"7.2","php_min_version":"5.6","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.5.0-rc.1\/prestashop.zip","zip_md5":"15c46487592cca2d41b9f95c54931efc","xml_download_url":null,"stability":"rc","release_notes_url":null},{"version":"1.7.5.0-beta.1","distribution":"open_source","distribution_version":null,"php_max_version":"7.2","php_min_version":"5.6","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.5.0-beta.1\/prestashop.zip","zip_md5":"9f1f3f0b8b04a38c96df4bea623297a4","xml_download_url":null,"stability":"beta","release_notes_url":null},{"version":"1.7.4.4","distribution":"open_source","distribution_version":null,"php_max_version":"7.1","php_min_version":"5.6","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.4.4\/prestashop.zip","zip_md5":"e64bd82789fe6e11c4c68fde074aadda","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.4.4\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2018\/prestashop-1-7-4-4-1-6-1-23-maintenance-releases\/"},{"version":"1.7.4.3","distribution":"open_source","distribution_version":null,"php_max_version":"7.1","php_min_version":"5.6","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.4.3\/prestashop.zip","zip_md5":"bc826702081c912ae7b5aefaa4afe047","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.4.3\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2018\/prestashop-1-7-4-3-1-6-1-21-maintenance-releases\/"},{"version":"1.7.4.2","distribution":"open_source","distribution_version":null,"php_max_version":"7.1","php_min_version":"5.6","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.4.2\/prestashop.zip","zip_md5":"cffb9a450981d6160cef114b68d27836","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.4.2\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2018\/prestashop-1-7-4-2-maintenance-release\/"},{"version":"1.7.4.1","distribution":"open_source","distribution_version":null,"php_max_version":"7.1","php_min_version":"5.6","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.4.1\/prestashop.zip","zip_md5":"5c763787c8e6a27fd229e0592b6f350f","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.4.1\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2018\/prestashop-1-7-4-1-maintenance-release\/"},{"version":"1.7.4.0","distribution":"open_source","distribution_version":null,"php_max_version":"7.1","php_min_version":"5.6","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.4.0\/prestashop.zip","zip_md5":"215e2af1db2683e93f1524d9cc1d9b59","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.4.0\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2018\/prestashop-1-7-4-0-available\/"},{"version":"1.7.3.4","distribution":"open_source","distribution_version":null,"php_max_version":"7.1","php_min_version":"5.4","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.3.4\/prestashop.zip","zip_md5":"ce0a59d7816e97d080a1fbb84e566a7b","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.3.4\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2018\/prestashop-1-7-3-4-1-6-1-20-maintenance-releases\/"},{"version":"1.7.3.3","distribution":"open_source","distribution_version":null,"php_max_version":"7.1","php_min_version":"5.4","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.3.3\/prestashop.zip","zip_md5":"608cfd7cdf2e8c3895f8e7cd99293a60","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.3.3\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2018\/prestashop-1-7-3-3-1-6-1-19-maintenance-releases\/"},{"version":"1.7.3.2","distribution":"open_source","distribution_version":null,"php_max_version":"7.1","php_min_version":"5.4","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.3.2\/prestashop.zip","zip_md5":"73d216c64e73f59df0f9e0f1581bad1a","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.3.2\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2018\/prestashop-1-7-3-2-maintenance-release\/"},{"version":"1.7.3.1","distribution":"open_source","distribution_version":null,"php_max_version":"7.1","php_min_version":"5.4","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.3.1\/prestashop.zip","zip_md5":"0f5a4faae57e523136da6768b14eeba2","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.3.1\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2018\/prestashop-1-7-3-1-maintenance-release\/"},{"version":"1.7.3.0","distribution":"open_source","distribution_version":null,"php_max_version":"7.1","php_min_version":"5.4","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.3.0\/prestashop.zip","zip_md5":"563011ba952d255d9e65c02b30fe43e7","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.3.0\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2018\/prestashop-1-7-3-0-available\/"},{"version":"1.7.2.5","distribution":"open_source","distribution_version":null,"php_max_version":"7.1","php_min_version":"5.4","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.2.5\/prestashop.zip","zip_md5":"e9987633d4de83a41db31b388cd918d2","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.2.5\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2018\/prestashop-1-7-2-5-maintenance-release\/"},{"version":"1.7.2.4","distribution":"open_source","distribution_version":null,"php_max_version":"7.1","php_min_version":"5.4","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.2.4\/prestashop.zip","zip_md5":"19c374fd4e65e2a0d978c619781686bc","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.2.4\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2017\/prestashop-1-7-2-4-maintenance-release\/"},{"version":"1.7.2.3","distribution":"open_source","distribution_version":null,"php_max_version":"7.1","php_min_version":"5.4","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.2.3\/prestashop.zip","zip_md5":"d7fa779980799f5b789bded40c188cfb","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.2.3\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2017\/prestashop-1-7-2-3-maintenance-release\/"},{"version":"1.7.2.2","distribution":"open_source","distribution_version":null,"php_max_version":"7.1","php_min_version":"5.4","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.2.2\/prestashop.zip","zip_md5":"a9ceea4dfa7c5c026a4d0fe7f50b8cec","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.2.2\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2017\/prestashop-1-7-2-2-maintenance-release\/"},{"version":"1.7.2.1","distribution":"open_source","distribution_version":null,"php_max_version":"7.1","php_min_version":"5.4","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.2.1\/prestashop.zip","zip_md5":"79133c454e11d371df332f39644e8a0a","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.2.1\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2017\/prestashop-1-7-2-1-maintenance-release\/"},{"version":"1.7.2.0","distribution":"open_source","distribution_version":null,"php_max_version":"7.1","php_min_version":"5.4","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.2.0\/prestashop.zip","zip_md5":"65fbf1968517b2990a677ce210f5eb28","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.2.0\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2017\/prestashop-1-7-2-0-available\/"},{"version":"1.7.1.2","distribution":"open_source","distribution_version":null,"php_max_version":"7.1","php_min_version":"5.4","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.1.2\/prestashop.zip","zip_md5":"8d02cffcac795112b012a2de7ccdf837","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.1.2\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2017\/prestashop-1-7-1-2-maintenance-release\/"},{"version":"1.7.1.1","distribution":"open_source","distribution_version":null,"php_max_version":"7.1","php_min_version":"5.4","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.1.1\/prestashop.zip","zip_md5":"7db5c793e7eb3921e98fc81a8e460f0b","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.1.1\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2017\/prestashop-1-7-1-1-maintenance-release\/"},{"version":"1.7.1.0","distribution":"open_source","distribution_version":null,"php_max_version":"7.1","php_min_version":"5.4","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.1.0\/prestashop.zip","zip_md5":"74b159f239e664f96536857e293c7fc2","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.1.0\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2017\/prestashop-1-7-1-0-available\/"},{"version":"1.7.0.6","distribution":"open_source","distribution_version":null,"php_max_version":"7.1","php_min_version":"5.4","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.0.6\/prestashop.zip","zip_md5":"84e9704e6073eb3e0235b527733c4b8e","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.0.6\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2017\/prestashop-1-7-0-6-maintenance-security-release\/"},{"version":"1.7.0.5","distribution":"open_source","distribution_version":null,"php_max_version":"7.1","php_min_version":"5.4","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.0.5\/prestashop.zip","zip_md5":"9d6e15af674ec10f8ba7448584879ee9","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.0.5\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2017\/prestashop-1-7-0-5-maintenance-release\/"},{"version":"1.7.0.4","distribution":"open_source","distribution_version":null,"php_max_version":"7.1","php_min_version":"5.4","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.0.4\/prestashop.zip","zip_md5":"0c8f1d9159c675c4a4de785d980bf48f","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.0.4\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2017\/prestashop-1704-maintenance-release\/"},{"version":"1.7.0.3","distribution":"open_source","distribution_version":null,"php_max_version":"7.1","php_min_version":"5.4","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.0.3\/prestashop.zip","zip_md5":"3c360a4a77a5be95870225b74facf434","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.0.3\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2016\/prestashop-1703-maintenance-release\/"},{"version":"1.7.0.2","distribution":"open_source","distribution_version":null,"php_max_version":"7.1","php_min_version":"5.4","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.0.2\/prestashop.zip","zip_md5":"a7641d9ca5cf2910fcd1469183768458","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.0.2\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2016\/prestashop-1702-maintenance-release\/"},{"version":"1.7.0.1","distribution":"open_source","distribution_version":null,"php_max_version":"7.1","php_min_version":"5.4","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.0.1\/prestashop.zip","zip_md5":"4382bd55f93f04216c1599a14c7b0d6f","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.0.1\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2016\/prestashop-1701-maintenance-release\/"},{"version":"1.7.0.0","distribution":"open_source","distribution_version":null,"php_max_version":"7.1","php_min_version":"5.4","zip_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.0.0\/prestashop.zip","zip_md5":"72fcf83888534259fa350af6989453f7","xml_download_url":"https:\/\/api.prestashop-project.org\/assets\/prestashop\/1.7.0.0\/prestashop.xml","stability":"stable","release_notes_url":"https:\/\/build.prestashop-project.org\/news\/2016\/prestashop-1-7-0-0-released\/"}]';
        $response =$contenido;
        // $response = @file_get_contents(self::API_URL . '/' . $endPoint);
       // echo self::API_URL . '/' . $endPoint;die();

        if (!$response) {
            throw new DistributionApiException($this->translator->trans('Error when retrieving data from Distribution API'), DistributionApiException::API_NOT_CALLABLE_CODE);
        }

        $jsonResponse = json_decode($response, true);

        if (JSON_ERROR_NONE !== json_last_error()) {
            throw new DistributionApiException($this->translator->trans('Invalid JSON from Distribution API: %s', [json_last_error_msg()]), DistributionApiException::API_NOT_CALLABLE_CODE);
        }

        return $jsonResponse;
    }

    /**
     * @throws DistributionApiException
     *
     * @return array{php_min_version: string, php_max_version: string}
     */
    public function getPhpVersionRequirements(string $targetVersion): array
    {
        $data = $this->getEndpointData(self::PRESTASHOP_ENDPOINT);

        foreach ($data as $prestashopRelease) {
            if ($prestashopRelease->getVersion() === $targetVersion) {
                return [
                    'php_min_version' => $prestashopRelease->getPhpMinVersion(),
                    'php_max_version' => $prestashopRelease->getPhpMaxVersion(),
                ];
            }
        }

        throw new DistributionApiException($this->translator->trans('No version match in Distribution api for %s', [$targetVersion]), DistributionApiException::VERSION_NOT_FOUND_CODE);
    }

    /**
     * @param string $version
     *
     * @return PrestashopRelease|null
     *
     * @throws DistributionApiException
     */
    public function getRelease(string $version): ?PrestashopRelease
    {
        $data = $this->getReleases();

        foreach ($data as $prestashopRelease) {
            if ($prestashopRelease->getVersion() === $version) {
                return $prestashopRelease;
            }
        }

        return null;
    }

    /**
     * @throws DistributionApiException
     *
     * @return PrestashopRelease[]
     */
    public function getReleases(): array
    {
        return $this->getEndpointData(self::PRESTASHOP_ENDPOINT);
    }

    /**
     * @return AutoupgradeRelease[]
     *
     * @throws DistributionApiException
     */
    public function getAutoupgradeCompatibilities(): array
    {
        return $this->getEndpointData(self::AUTOUPGRADE_ENDPOINT);
    }

    /**
     * @param self::PRESTASHOP_ENDPOINT|self::AUTOUPGRADE_ENDPOINT $endPoint
     *
     * @return AutoupgradeRelease[]|PrestashopRelease[]
     *
     * @throws DistributionApiException
     */
    private function getEndpointData(string $endPoint): array
    {
        if (!isset($this->endpointData[$endPoint])) {
            $jsonResponse = $this->getApiEndpoint($endPoint);

           // var_dump($jsonResponse);die();
            if (empty($jsonResponse) || ($endPoint === self::AUTOUPGRADE_ENDPOINT && empty($jsonResponse['prestashop']))) {
              //  throw new DistributionApiException($this->translator->trans('Unable to retrieve "%s" data from distribution API.', [$endPoint]), DistributionApiException::EMPTY_DATA_CODE);
            echo $this->translator->trans('Unable to retrieve "%s" data from distribution API.', [$endPoint]);
            }

            $method = self::$factories[$endPoint];

            $this->endpointData[$endPoint] = $this->$method($jsonResponse);
        }

        return $this->endpointData[$endPoint];
    }

    /**
     * @param mixed[] $data
     *
     * @return PrestashopRelease[]
     */
    private function createPrestashopReleaseCollection(array $data): array
    {
        $releases = [];
        foreach ($data as $versionInfo) {
            $releases[] = new PrestashopRelease(
                $versionInfo['version'],
                $versionInfo['stability'],
                $versionInfo['distribution'],
                $versionInfo['php_max_version'],
                $versionInfo['php_min_version'],
                $versionInfo['zip_download_url'],
                $versionInfo['xml_download_url'],
                $versionInfo['zip_md5'],
                $versionInfo['release_notes_url'],
                $versionInfo['distribution_version']
            );
        }

        return $releases;
    }

    /**
     * @param array{ 'prestashop': mixed[] } $data
     *
     * @return AutoupgradeRelease[]
     */
    private function createAutoupgradeReleaseCollection(array $data): array
    {
        $releases = [];

        foreach ($data['prestashop'] as $versionInfo) {
            $releases[] = new AutoupgradeRelease(
                $versionInfo['prestashop_min'],
                $versionInfo['prestashop_max'],
                $versionInfo['autoupgrade_recommended']['last_version'],
                $versionInfo['autoupgrade_recommended']['download']['link'],
                $versionInfo['autoupgrade_recommended']['download']['md5'],
                $versionInfo['autoupgrade_recommended']['changelog'] ?? null
            );
        }

        return $releases;
    }
}
