<?php
/**
 * Copyright since 2007 PrestaShop SA and Contributors
 * PrestaShop is an International Registered Trademark & Property of PrestaShop SA
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Open Software License (OSL 3.0)
 * that is bundled with this package in the file LICENSE.md.
 * It is also available through the world-wide-web at this URL:
 * https://opensource.org/licenses/OSL-3.0
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@prestashop.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade PrestaShop to newer
 * versions in the future. If you wish to customize PrestaShop for your
 * needs please refer to https://devdocs.prestashop.com/ for more information.
 *
 * @author    PrestaShop SA and Contributors <contact@prestashop.com>
 * @copyright Since 2007 PrestaShop SA and Contributors
 * @license   https://opensource.org/licenses/OSL-3.0 Open Software License (OSL 3.0)
 */
set_time_limit(0);

define('_PS_INSTALL_MINIMUM_PHP_VERSION_ID_', 70103);
define('_PS_INSTALL_MINIMUM_PHP_VERSION_', '7.1.3');
define('_PS_VERSION_', '1.7.7.3');

define('ZIP_NAME', 'prestashop.zip');
define('TARGET_FOLDER', __DIR__ . '/');
define('BATCH_SIZE', 500);

// bust cache, or else it won't load the installer after the extraction is done
header('Cache-Control: no-cache, no-store, must-revalidate');

if (!defined('PHP_VERSION_ID') || PHP_VERSION_ID < _PS_INSTALL_MINIMUM_PHP_VERSION_ID_) {
    die('You need at least PHP ' . _PS_INSTALL_MINIMUM_PHP_VERSION_ . ' to install PrestaShop. Your current PHP version is ' . PHP_VERSION);
}

// --------------------------------------------------------------------------------



/**
 * Copyright since 2007 PrestaShop SA and Contributors
 * PrestaShop is an International Registered Trademark & Property of PrestaShop SA
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Open Software License (OSL 3.0)
 * that is bundled with this package in the file LICENSE.md.
 * It is also available through the world-wide-web at this URL:
 * https://opensource.org/licenses/OSL-3.0
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@prestashop.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade PrestaShop to newer
 * versions in the future. If you wish to customize PrestaShop for your
 * needs please refer to https://devdocs.prestashop.com/ for more information.
 *
 * @author    PrestaShop SA and Contributors <contact@prestashop.com>
 * @copyright Since 2007 PrestaShop SA and Contributors
 * @license   https://opensource.org/licenses/OSL-3.0 Open Software License (OSL 3.0)
 */

/**
 * Very basic file cache.
 */
class BasicFileCache
{
    /**
     * @param string $filename
     *
     * @return string
     */
    public function get($filename)
    {
        if (false === $this->isCached($filename)) {
            throw new \Exception(sprintf('No cache entry for %s', $filename));
        }

        $filepath = $this->computeCacheFilepath($filename);

        return file_get_contents($filepath);
    }

    /**
     * @param string $filename
     *
     * @return bool
     */
    public function isCached($filename)
    {
        $filepath = $this->computeCacheFilepath($filename);

        return is_file($filepath) && is_readable($filepath);
    }

    /**
     * @param string $data
     * @param string $filename
     *
     * @return bool
     *
     * @throws Exception
     */
    public function save($data, $filename)
    {
        $filepath = $this->computeCacheFilepath($filename);

        if (is_file($filepath)) {
            throw new \Exception(sprintf('Could not cache file %s', $filepath));
        }

        return (bool) file_put_contents($filepath, $data);
    }

    /**
     * @param string $filename
     *
     * @return bool
     */
    public function delete($filename)
    {
        $filepath = $this->computeCacheFilepath($filename);

        return unlink($filepath);
    }

    /**
     * @param string $filename
     *
     * @return string
     */
    private function computeCacheFilepath($filename)
    {
        return __DIR__ . DIRECTORY_SEPARATOR . $filename . '.cache';
    }
}



/**
 * Copyright since 2007 PrestaShop SA and Contributors
 * PrestaShop is an International Registered Trademark & Property of PrestaShop SA
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Open Software License (OSL 3.0)
 * that is bundled with this package in the file LICENSE.md.
 * It is also available through the world-wide-web at this URL:
 * https://opensource.org/licenses/OSL-3.0
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@prestashop.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade PrestaShop to newer
 * versions in the future. If you wish to customize PrestaShop for your
 * needs please refer to https://devdocs.prestashop.com/ for more information.
 *
 * @author    PrestaShop SA and Contributors <contact@prestashop.com>
 * @copyright Since 2007 PrestaShop SA and Contributors
 * @license   https://opensource.org/licenses/OSL-3.0 Open Software License (OSL 3.0)
 */

/**
 * In charge of downloading the latest Prestashop Version.
 *
 * Most methods are copied from https://github.com/PrestaShop/autoupgrade/blob/master/classes/TaskRunner/Upgrade/Download.php
 * and https://github.com/PrestaShop/autoupgrade/blob/master/classes/Tools14.php
 */
class Download
{
    const PRESTASHOP_API_RELEASES_XML_FEED = 'https://api.prestashop.com/xml/channel.xml';
    const CACHED_FEED_FILENAME = 'XMLFeed';

    /**
     * @var BasicFileCache
     */
    private $cachingSystem;

    /**
     * @param BasicFileCache $cachingSystem optional FileCache
     */
    public function __construct(BasicFileCache $cachingSystem = null)
    {
        if (null === $cachingSystem) {
            $cachingSystem = new BasicFileCache();
        }

        $this->cachingSystem = $cachingSystem;
    }

    /**
     * @param string $source
     * @param string $destination
     *
     * @return bool|int
     */
    public static function copy($source, $destination)
    {
        return @file_put_contents($destination, self::fileGetContents($source));
    }

    /**
     * @param string $url
     *
     * @return bool|mixed|string
     */
    public static function fileGetContents($url)
    {
        $curl_timeout = 60;

        if (!extension_loaded('openssl') && strpos($url, 'https://') === true) {
            $url = str_replace('https', 'http', $url);
        }

        $stream_context = null;
        if (preg_match('/^https?:\/\//', $url)) {
            $stream_context = @stream_context_create([
                'http' => ['timeout' => $curl_timeout, 'header' => "User-Agent:MyAgent/1.0\r\n"],
            ]);
        }

        if (in_array(ini_get('allow_url_fopen'), ['On', 'on', '1']) || !preg_match('/^https?:\/\//', $url)) {
            $var = @file_get_contents($url, false, $stream_context);

            if ($var) {
                return $var;
            }
        } elseif (function_exists('curl_init')) {
            $curl = curl_init();
            curl_setopt($curl, CURLOPT_RETURNTRANSFER, 1);
            curl_setopt($curl, CURLOPT_URL, $url);
            curl_setopt($curl, CURLOPT_CONNECTTIMEOUT, 5);
            curl_setopt($curl, CURLOPT_TIMEOUT, $curl_timeout);
            curl_setopt($curl, CURLOPT_SSL_VERIFYPEER, 0);
            $opts = stream_context_get_options($stream_context);

            if (isset($opts['http']['method']) && strtolower($opts['http']['method']) == 'post') {
                curl_setopt($curl, CURLOPT_POST, true);
                if (isset($opts['http']['content'])) {
                    parse_str($opts['http']['content'], $datas);
                    curl_setopt($curl, CURLOPT_POSTFIELDS, $datas);
                }
            }
            $content = curl_exec($curl);
            curl_close($curl);

            return $content;
        } else {
            return false;
        }
    }

    /**
     * @return VersionNumber
     */
    public function getLatestStableAvailableVersion()
    {
        $feed = $this->getFeed();

        $branch = $this->getLatestStableBranchObjectFromFeed($feed);
        $versionNumberAsString = (string) $branch->num;

        return VersionNumber::fromString($versionNumberAsString);
    }

    /**
     * @return string
     */
    public function getLatestStableAvailableVersionLink()
    {
        $feed = $this->getFeed();

        $branch = $this->getLatestStableBranchObjectFromFeed($feed);

        return (string) $branch->download->link;
    }

    public function clearFileCache()
    {
        $this->cachingSystem->delete(self::CACHED_FEED_FILENAME);
    }

    /**
     * @return SimpleXMLElement
     *
     * @throws Exception
     */
    private function getFeed()
    {
        if (false === $this->isXmlFeedStoredInCache()) {
            $feed = @file_get_contents(self::PRESTASHOP_API_RELEASES_XML_FEED);

            if (false === $feed) {
                throw new PrestashopCouldNotDownloadLatestVersionException('Could not fetch API releases xml feed');
            }

            $this->storeFeedIntoFileCache($feed);
        }

        $feed = $this->getXmlFeedFromCache();
        $xml = simplexml_load_string($feed);

        if (false === $xml) {
            throw new PrestashopCouldNotDownloadLatestVersionException('Could not parse API releases xml feed');
        }

        return $xml;
    }

    /**
     * @param SimpleXMLElement $feed
     *
     * @return \StdClass
     *
     * @throws PrestashopCouldNotDownloadLatestVersionException
     */
    private function getLatestStableBranchObjectFromFeed($feed)
    {
        foreach ($feed->channel as $channel) {
            $channelName = (string) $channel['name'];

            if ('stable' !== $channelName) {
                continue;
            }

            $maxStableVersion = null;
            $maxStableBranch = null;
            foreach ($channel->branch as $branch) {
                $versionNumberAsString = (string) $branch->num;
                $versionNumber = VersionNumber::fromString($versionNumberAsString);

                if (null === $maxStableVersion) {
                    $maxStableVersion = $versionNumber;
                    $maxStableBranch = $branch;
                } elseif (1 === $versionNumber->compare($maxStableVersion)) {
                    $maxStableVersion = $versionNumber;
                    $maxStableBranch = $branch;
                }
            }

            return $maxStableBranch;
        }

        throw new PrestashopCouldNotDownloadLatestVersionException(
            'Could not find latest stable version from API releases xml feed'
        );
    }

    /**
     * @return bool
     */
    private function isXmlFeedStoredInCache()
    {
        return $this->cachingSystem->isCached(self::CACHED_FEED_FILENAME);
    }

    /**
     * @return string
     *
     * @throws Exception
     */
    private function getXmlFeedFromCache()
    {
        return $this->cachingSystem->get(self::CACHED_FEED_FILENAME);
    }

    /**
     * @param string $xml
     *
     * @return bool
     *
     * @throws Exception
     */
    private function storeFeedIntoFileCache($xml)
    {
        return $this->cachingSystem->save($xml, self::CACHED_FEED_FILENAME);
    }
}



/**
 * Copyright since 2007 PrestaShop SA and Contributors
 * PrestaShop is an International Registered Trademark & Property of PrestaShop SA
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Open Software License (OSL 3.0)
 * that is bundled with this package in the file LICENSE.md.
 * It is also available through the world-wide-web at this URL:
 * https://opensource.org/licenses/OSL-3.0
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@prestashop.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade PrestaShop to newer
 * versions in the future. If you wish to customize PrestaShop for your
 * needs please refer to https://devdocs.prestashop.com/ for more information.
 *
 * @author    PrestaShop SA and Contributors <contact@prestashop.com>
 * @copyright Since 2007 PrestaShop SA and Contributors
 * @license   https://opensource.org/licenses/OSL-3.0 Open Software License (OSL 3.0)
 */

/**
 * Gateway, manager and DIC for install util classes: download, unzip ...
 */
class InstallManager
{
    /**
     * @var ConfigurationValidator
     */
    private $validator;

    /**
     * @var Download
     */
    private $download;

    /**
     * @var Unzip
     */
    private $unzip;

    /**
     * @var string
     */
    private $downloadDirectoryPath;

    /**
     * @var string
     */
    private $extractDirectoryPath;

    public function __construct()
    {
        $this->validator = new ConfigurationValidator();
        $this->download = new Download();
        $this->unzip = new Unzip();

        // @todo: be able to use fallback util directories
        $this->downloadDirectoryPath = __DIR__ . DIRECTORY_SEPARATOR . 'download';
        $this->extractDirectoryPath = __DIR__ . DIRECTORY_SEPARATOR . 'extracted';
    }

    /**
     * Check whether download of latest PS version can be carried out.
     *
     * @return string[] array of issues. Empty if download can be performed.
     */
    public function testDownloadCapabilities()
    {
        return $this->validator->testSystemCanPerformDownloadUnzipAndReplace();
    }

    /**
     * @return VersionNumber
     *
     * @throws \RuntimeException
     */
    public function getLatestStableAvailableVersion()
    {
        return $this->download->getLatestStableAvailableVersion();
    }

    /**
     * @return bool
     *
     * @throws PrestashopCouldNotInstallLatestVersionException
     */
    public function downloadUnzipAndReplaceLatestPSVersion()
    {
        if (is_dir($this->downloadDirectoryPath) || file_exists($this->downloadDirectoryPath)) {
            throw new PrestashopCouldNotInstallLatestVersionException(sprintf(
                'Directory %s already exists.',
                $this->downloadDirectoryPath
            ));
        }

        // create 'download' directory
        $createDirectoryResult = @mkdir($this->downloadDirectoryPath);
        if (false === $createDirectoryResult) {
            throw new PrestashopCouldNotInstallLatestVersionException(sprintf(
                'Could not create directory %s',
                $this->downloadDirectoryPath
            ));
        }

        // download zip archive
        $destinationPath = realpath($this->downloadDirectoryPath) . DIRECTORY_SEPARATOR . 'prestashop-latest.zip';
        $link = $this->download->getLatestStableAvailableVersionLink();
        Download::copy($link, $destinationPath);

        if (false === is_file($destinationPath)) {
            throw new PrestashopCouldNotInstallLatestVersionException(
                'Failed to download latest Prestashop release zip archive'
            );
        }

        // @todo: validate checksum ?

        // unzip archive into 'extracted' directory
        $this->unzip->unzipArchive($destinationPath, $this->extractDirectoryPath);

        // test 3 extracted files are OK
        $this->verifyUnzipFile('Install_PrestaShop.html');
        $this->verifyUnzipFile('prestashop.zip');
        $this->verifyUnzipFile('index.php');

        // replace files
        $this->replaceInstallFile('Install_PrestaShop.html');
        $this->replaceInstallFile('prestashop.zip');
        $this->replaceInstallFile('index.php');

        // delete 2 util directories
        $this->deleteDirectoryWithItsContent($this->downloadDirectoryPath);
        $this->deleteDirectoryWithItsContent($this->extractDirectoryPath);

        $this->download->clearFileCache();

        return true;
    }

    /**
     * @param string $fileName
     *
     * @throws PrestashopCouldNotInstallLatestVersionException
     */
    private function verifyUnzipFile($fileName)
    {
        if (false === is_file($this->extractDirectoryPath . DIRECTORY_SEPARATOR . $fileName)) {
            throw new PrestashopCouldNotInstallLatestVersionException(sprintf(
                'After unzip, missing %s file',
                $fileName
            ));
        }
    }

    /**
     * @param string $fileName
     *
     * @throws PrestashopCouldNotInstallLatestVersionException
     */
    private function replaceInstallFile($fileName)
    {
        $replaceFileResult = rename(
            $this->extractDirectoryPath . DIRECTORY_SEPARATOR . $fileName,
            __DIR__ . DIRECTORY_SEPARATOR . $fileName
        );

        if (false === $replaceFileResult) {
            throw new PrestashopCouldNotInstallLatestVersionException(sprintf(
                'Could not replace %s file',
                $fileName
            ));
        }
    }

    /**
     * @param string $directoryPath
     *
     * @throws PrestashopCouldNotInstallLatestVersionException
     */
    private function deleteDirectoryWithItsContent($directoryPath)
    {
        $deleteDirectoryContentResult = array_map(
            'unlink',
            glob($directoryPath . DIRECTORY_SEPARATOR . '*.*')
        );

        $deleteDirectoryResult = @rmdir($directoryPath);

        if ((false === $deleteDirectoryContentResult) || (false === $deleteDirectoryResult)) {
            throw new PrestashopCouldNotInstallLatestVersionException(sprintf(
                'Cannot delete directory %s',
                $directoryPath
            ));
        }
    }
}



/**
 * Copyright since 2007 PrestaShop SA and Contributors
 * PrestaShop is an International Registered Trademark & Property of PrestaShop SA
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Open Software License (OSL 3.0)
 * that is bundled with this package in the file LICENSE.md.
 * It is also available through the world-wide-web at this URL:
 * https://opensource.org/licenses/OSL-3.0
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@prestashop.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade PrestaShop to newer
 * versions in the future. If you wish to customize PrestaShop for your
 * needs please refer to https://devdocs.prestashop.com/ for more information.
 *
 * @author    PrestaShop SA and Contributors <contact@prestashop.com>
 * @copyright Since 2007 PrestaShop SA and Contributors
 * @license   https://opensource.org/licenses/OSL-3.0 Open Software License (OSL 3.0)
 */
class ConfigurationValidator
{
    /**
     * Check whether download, unzip & replace of latest PS version can be carried out.
     *
     * @return string[] array of issues. Empty if download can be performed.
     */
    public function testSystemCanPerformDownloadUnzipAndReplace()
    {
        $errors = [];

        if (!$this->testCurl() && !$this->testFopen()) {
            $errors[] = 'You need allow_url_fopen or cURL enabled for automatic download to work.';
        }

        $systemErrors = $this->testFilesystemCapabilities();
        if (false === empty($systemErrors)) {
            $errors[] = sprintf('Cannot manipulate file on this system: %s', implode('; ', $systemErrors));
        }

        if (false === class_exists('ZipArchive')) {
            $errors[] = 'Cannot unzip files as php class ZipArchive is not available.';
        }

        // @todo: be able to use fallback util directories
        $downloadDirPath = __DIR__ . DIRECTORY_SEPARATOR . 'download';
        if (is_dir($downloadDirPath) || file_exists($downloadDirPath)) {
            $errors[] = "Directory 'download' already exists.";
        }
        $downloadDirPath = __DIR__ . DIRECTORY_SEPARATOR . 'extracted';
        if (is_dir($downloadDirPath) || file_exists($downloadDirPath)) {
            $errors[] = "Directory 'extracted' already exists.";
        }

        return $errors;
    }

    /**
     * @return bool
     */
    public function testFopen()
    {
        return in_array(ini_get('allow_url_fopen'), array('On', 'on', '1'));
    }

    /**
     * @return bool
     */
    public function testCurl()
    {
        return extension_loaded('curl');
    }

    /**
     * Test whether files and directories can be manipulated by php on given system
     * by performing multiple file operations.
     *
     * @return string[]
     */
    public function testFilesystemCapabilities()
    {
        $dirPath = $this->getRandomDirectoryPath();

        $this->checkRandomNameIsNotAlreadyUsed($dirPath);

        if (false === $this->createDirectoryTest($dirPath)) {
            return ['Cannot create directories'];
        }

        list($fileCreationTestPath, $createFileResult) = $this->createFileTest($dirPath);
        if (false === $createFileResult) {
            $this->deleteDirectoryTest($dirPath);

            return ['Cannot write files'];
        }

        if (false === $this->downloadFileTest($dirPath)) {
            $this->deleteDirectoryTest($dirPath);

            return ['Cannot download files from network'];
        }

        list($fileMoveTestPath, $moveResult) = $this->moveFileTest($fileCreationTestPath);
        if (false === $moveResult) {
            $this->deleteDirectoryTest($dirPath);

            return ['Cannot move files into prestashop root directory'];
        }

        if (false === $this->deleteFileTest($fileMoveTestPath)) {
            $this->deleteDirectoryTest($dirPath);

            return ['Cannot delete files in prestashop root directory'];
        }

        list($deleteDirectoryContentResult, $deleteDirectoryResult) = $this->deleteDirectoryTest($dirPath);
        if ((false === $deleteDirectoryContentResult) || (false === $deleteDirectoryResult)) {
            return ['Cannot delete directories in prestashop root directory'];
        }

        return [];
    }

    /**
     * Choose a random available directory name.
     *
     * @return string
     */
    private function getRandomDirectoryPath()
    {
        $randomDirectoryName = 'test-' . uniqid();

        return __DIR__ . DIRECTORY_SEPARATOR . $randomDirectoryName;
    }

    /**
     * @param string $dirPath
     *
     * @return bool
     */
    private function createDirectoryTest($dirPath)
    {
        return @mkdir($dirPath);
    }

    /**
     * @param string $dirPath
     *
     * @return array
     */
    private function createFileTest($dirPath)
    {
        $fileCreationTestPath = $dirPath . DIRECTORY_SEPARATOR . 'test-file.php';
        $createFileResult = @file_put_contents($fileCreationTestPath, " echo 'Hello world !';");

        return [$fileCreationTestPath, $createFileResult];
    }

    /**
     * @param string $dirPath
     *
     * @return bool
     */
    private function downloadFileTest($dirPath)
    {
        $downloadTestPath = $dirPath . DIRECTORY_SEPARATOR . 'test-download.txt';
        $target = 'https://www.google.com/robots.txt';

        return (bool) @file_put_contents($downloadTestPath, Download::fileGetContents($target));
    }

    /**
     * Move a file from test directory into root directory.
     *
     * @param string $fileCreationTestPath
     *
     * @return array
     */
    private function moveFileTest($fileCreationTestPath)
    {
        $fileMoveTestPath = __DIR__ . DIRECTORY_SEPARATOR . 'test-move.php';
        $moveResult = @rename($fileCreationTestPath, $fileMoveTestPath);

        return [$fileMoveTestPath, $moveResult];
    }

    /**
     * @param string $fileMoveTestPath
     *
     * @return bool
     */
    private function deleteFileTest($fileMoveTestPath)
    {
        return unlink($fileMoveTestPath);
    }

    /**
     * @param string $dirPath
     *
     * @return array
     */
    private function deleteDirectoryTest($dirPath)
    {
        $deleteDirectoryContentResult = array_map('unlink', glob($dirPath . DIRECTORY_SEPARATOR . '*.*'));
        $deleteDirectoryResult = @rmdir($dirPath);

        return [$deleteDirectoryContentResult, $deleteDirectoryResult];
    }

    /**
     * @param string $dirPath
     *
     * @return bool
     *
     * @throws \RuntimeException
     */
    private function checkRandomNameIsNotAlreadyUsed($dirPath)
    {
        if (is_dir($dirPath) || file_exists($dirPath)) {
            throw new \RuntimeException(sprintf('Test directory name is already used: %s', $dirPath));
        }

        return true;
    }
}



/**
 * Copyright since 2007 PrestaShop SA and Contributors
 * PrestaShop is an International Registered Trademark & Property of PrestaShop SA
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Open Software License (OSL 3.0)
 * that is bundled with this package in the file LICENSE.md.
 * It is also available through the world-wide-web at this URL:
 * https://opensource.org/licenses/OSL-3.0
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@prestashop.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade PrestaShop to newer
 * versions in the future. If you wish to customize PrestaShop for your
 * needs please refer to https://devdocs.prestashop.com/ for more information.
 *
 * @author    PrestaShop SA and Contributors <contact@prestashop.com>
 * @copyright Since 2007 PrestaShop SA and Contributors
 * @license   https://opensource.org/licenses/OSL-3.0 Open Software License (OSL 3.0)
 */

/**
 * In charge of unzipping the latest Prestashop Version.
 *
 * Most methods are copied from https://github.com/PrestaShop/autoupgrade/blob/master/classes/TaskRunner/Upgrade/Unzip.php
 */
class Unzip
{
    /**
     * @param string $zipFilepath
     * @param string $pathWhereToExtract
     *
     * @return bool
     *
     * @throws PrestashopCouldNotUnzipLatestVersionException
     */
    public function unzipArchive($zipFilepath, $pathWhereToExtract)
    {
        if ((is_dir($pathWhereToExtract) || is_file($pathWhereToExtract))) {
            throw new PrestashopCouldNotUnzipLatestVersionException(sprintf(
                'Destination folder %s already exists',
                $pathWhereToExtract
            ));
        }

        $this->extract($zipFilepath, $pathWhereToExtract);

        return @unlink($zipFilepath);
    }

    /**
     * @param string $fromFile
     * @param string $toDir
     *
     * @return bool
     *
     * @throws PrestashopCouldNotUnzipLatestVersionException
     */
    private function extract($fromFile, $toDir)
    {
        if (false === is_file($fromFile)) {
            throw new PrestashopCouldNotUnzipLatestVersionException('Given zip file is not a file');
        }

        if (false === file_exists($toDir)) {
            if (false === mkdir($toDir)) {
                throw new PrestashopCouldNotUnzipLatestVersionException('Unzip destination folder cannot be used');
            }
            chmod($toDir, 0775);
        }

        $this->extractWithZipArchive($fromFile, $toDir);

        return true;
    }

    /**
     * @param string $fromFile
     * @param string $toDir
     *
     * @return bool
     *
     * @throws PrestashopCouldNotUnzipLatestVersionException
     */
    private function extractWithZipArchive($fromFile, $toDir)
    {
        $zip = $this->openWithZipArchive($fromFile);
        if ($zip === false) {
            throw new PrestashopCouldNotUnzipLatestVersionException('Unzip operation failed');
        }

        if (false === $zip->extractTo($toDir)) {
            throw new PrestashopCouldNotUnzipLatestVersionException(sprintf(
                'zip->extractTo(): unable to use %s as extract destination.',
                $toDir
            ));
        }

        return $zip->close();
    }

    /**
     * @param string $zipFile
     *
     * @return ZipArchive
     *
     * @throws PrestashopCouldNotUnzipLatestVersionException
     */
    private function openWithZipArchive($zipFile)
    {
        $zip = new ZipArchive();

        if ($zip->open($zipFile) !== true || empty($zip->filename)) {
            throw new PrestashopCouldNotUnzipLatestVersionException('Failed to open zip archive');
        }

        return $zip;
    }
}



/**
 * Copyright since 2007 PrestaShop SA and Contributors
 * PrestaShop is an International Registered Trademark & Property of PrestaShop SA
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Open Software License (OSL 3.0)
 * that is bundled with this package in the file LICENSE.md.
 * It is also available through the world-wide-web at this URL:
 * https://opensource.org/licenses/OSL-3.0
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@prestashop.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade PrestaShop to newer
 * versions in the future. If you wish to customize PrestaShop for your
 * needs please refer to https://devdocs.prestashop.com/ for more information.
 *
 * @author    PrestaShop SA and Contributors <contact@prestashop.com>
 * @copyright Since 2007 PrestaShop SA and Contributors
 * @license   https://opensource.org/licenses/OSL-3.0 Open Software License (OSL 3.0)
 */
class PrestashopCouldNotInstallLatestVersionException extends \Exception
{
}



/**
 * Copyright since 2007 PrestaShop SA and Contributors
 * PrestaShop is an International Registered Trademark & Property of PrestaShop SA
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Open Software License (OSL 3.0)
 * that is bundled with this package in the file LICENSE.md.
 * It is also available through the world-wide-web at this URL:
 * https://opensource.org/licenses/OSL-3.0
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@prestashop.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade PrestaShop to newer
 * versions in the future. If you wish to customize PrestaShop for your
 * needs please refer to https://devdocs.prestashop.com/ for more information.
 *
 * @author    PrestaShop SA and Contributors <contact@prestashop.com>
 * @copyright Since 2007 PrestaShop SA and Contributors
 * @license   https://opensource.org/licenses/OSL-3.0 Open Software License (OSL 3.0)
 */
class PrestashopCouldNotDownloadLatestVersionException extends \Exception
{
}



/**
 * Copyright since 2007 PrestaShop SA and Contributors
 * PrestaShop is an International Registered Trademark & Property of PrestaShop SA
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Open Software License (OSL 3.0)
 * that is bundled with this package in the file LICENSE.md.
 * It is also available through the world-wide-web at this URL:
 * https://opensource.org/licenses/OSL-3.0
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@prestashop.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade PrestaShop to newer
 * versions in the future. If you wish to customize PrestaShop for your
 * needs please refer to https://devdocs.prestashop.com/ for more information.
 *
 * @author    PrestaShop SA and Contributors <contact@prestashop.com>
 * @copyright Since 2007 PrestaShop SA and Contributors
 * @license   https://opensource.org/licenses/OSL-3.0 Open Software License (OSL 3.0)
 */
class PrestashopCouldNotUnzipLatestVersionException extends \Exception
{
}



/**
 * Copyright since 2007 PrestaShop SA and Contributors
 * PrestaShop is an International Registered Trademark & Property of PrestaShop SA
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Open Software License (OSL 3.0)
 * that is bundled with this package in the file LICENSE.md.
 * It is also available through the world-wide-web at this URL:
 * https://opensource.org/licenses/OSL-3.0
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@prestashop.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade PrestaShop to newer
 * versions in the future. If you wish to customize PrestaShop for your
 * needs please refer to https://devdocs.prestashop.com/ for more information.
 *
 * @author    PrestaShop SA and Contributors <contact@prestashop.com>
 * @copyright Since 2007 PrestaShop SA and Contributors
 * @license   https://opensource.org/licenses/OSL-3.0 Open Software License (OSL 3.0)
 */
class VersionNumber
{
    /**
     * @var float
     */
    private $major;

    /**
     * @var int
     */
    private $minor;

    /**
     * @var int
     */
    private $patch;

    /**
     * @param float $major
     * @param int $minor
     * @param int $patch
     */
    public function __construct($major, $minor, $patch)
    {
        $this->major = $major;
        $this->minor = $minor;
        $this->patch = $patch;
    }

    /**
     * @param string $versionNumberAsString
     *
     * @return VersionNumber
     */
    public static function fromString($versionNumberAsString)
    {
        $regexp = '#^(\d+\.\d+|\d+)\.(\d+)\.(\d+)$#';
        $matches = [];

        $matchingResult = preg_match($regexp, $versionNumberAsString, $matches);

        if (1 !== $matchingResult) {
            throw new InvalidArgumentException(sprintf(
                'Failed to parse version number %s',
                $versionNumberAsString
            ));
        }

        return new static(
            $matches[1],
            $matches[2],
            $matches[3]
        );
    }

    /**
     * @return float
     */
    public function getMajor()
    {
        return $this->major;
    }

    /**
     * @return int
     */
    public function getMinor()
    {
        return $this->minor;
    }

    /**
     * @return int
     */
    public function getPatch()
    {
        return $this->patch;
    }

    /**
     * @return string
     */
    public function __toString()
    {
        return sprintf('%g.%d.%d', $this->major, $this->minor, $this->patch);
    }

    /**
     * @param VersionNumber $otherNumber
     *
     * @return int 1 if this version number is higher, -1 if lower, 0 if equal
     */
    public function compare(VersionNumber $otherNumber)
    {
        if ($this->major > $otherNumber->getMajor()) {
            return 1;
        }
        if ($this->major < $otherNumber->getMajor()) {
            return -1;
        }

        if ($this->minor > $otherNumber->getMinor()) {
            return 1;
        }
        if ($this->minor < $otherNumber->getMinor()) {
            return -1;
        }

        if ($this->patch > $otherNumber->getPatch()) {
            return 1;
        }
        if ($this->patch < $otherNumber->getPatch()) {
            return -1;
        }

        return 0;
    }
}


// --------------------------------------------------------------------------------

function getFileContent($fileOrContent, $debug)
{
    if ($debug) {
        return file_get_contents('content/' . $fileOrContent);
    }

    return base64_decode($fileOrContent);
}

function getZipErrorMessage($errorCode)
{
    $errors = [
        ZipArchive::ER_EXISTS => 'File already exists.',
        ZipArchive::ER_INCONS => 'Zip archive inconsistent or corrupted. Double check your uploaded files.',
        ZipArchive::ER_INVAL => 'Invalid argument.',
        ZipArchive::ER_MEMORY => 'Allocation error. Out of memory?',
        ZipArchive::ER_NOENT => 'Unable to find the release zip file. Make sure that the prestashop.zip file has been uploaded and is located in the same directory as this dezipper.',
        ZipArchive::ER_NOZIP => 'The release file is not a zip file or it is corrupted. Double check your uploaded files.',
        ZipArchive::ER_OPEN => "Can't open file. Make sure PHP has read access to the prestashop.zip file.",
        ZipArchive::ER_READ => 'Read error.',
        ZipArchive::ER_SEEK => 'Seek error.',
    ];

    if (isset($errors[$errorCode])) {
        return 'Unzipping error - ' . $errors[$errorCode];
    }

    return 'An unknown error was found while reading the zip file';
}

$selfUri = basename(__FILE__);

if (isset($_GET['run']) && ($_GET['run'] === 'check-version')) {
    try {
        $installManager = new InstallManager();

        $latestVersionAvailable = $installManager->getLatestStableAvailableVersion();

        $isThisTheLatestStableAvailableVersion = ($latestVersionAvailable->compare(VersionNumber::fromString(_PS_VERSION_)) < 1);
        if ($isThisTheLatestStableAvailableVersion) {
            die(json_encode([
                'thereIsAMoreRecentPSVersionAndItCanBeInstalled' => false,
            ]));
        }

        $possibleInstallIssues = $installManager->testDownloadCapabilities();
        if (false === empty($possibleInstallIssues)) {
            die(json_encode([
                'thereIsAMoreRecentPSVersionAndItCanBeInstalled' => false,
            ]));
        }

        die(json_encode([
            'thereIsAMoreRecentPSVersionAndItCanBeInstalled' => true,
        ]));
    } catch (\Exception $e) {
        die(json_encode([
            'thereIsAMoreRecentPSVersionAndItCanBeInstalled' => false,
        ]));
    }
}

if ((isset($_POST['downloadLatest'])) && ($_POST['downloadLatest'] === 'true')) {
    try {
        $installManager = new InstallManager();

        $installManager->downloadUnzipAndReplaceLatestPSVersion();
        die(json_encode([
            'success' => true,
        ]));
    } catch (\Exception $e) {
        die(json_encode([
            'error' => true,
            'message' => $e->getMessage(),
        ]));
    }
}

$startId = (isset($_POST['startId'])) ? (int) $_POST['startId'] : 0;

if (isset($_POST['extract'])) {
    if (!extension_loaded('zip')) {
        die(json_encode([
            'error' => true,
            'message' => 'You must install PHP zip extension first',
        ]));
    }

    $zip = new ZipArchive();
    if (true !== $error = $zip->open(__DIR__ . '/' . ZIP_NAME)) {
        die(json_encode([
            'error' => true,
            'message' => getZipErrorMessage($error),
        ]));
    }

    if (!is_writable(TARGET_FOLDER)) {
        die(json_encode([
            'error' => true,
            'message' => 'You need to grant write permissions for PHP on the following directory: '
                . realpath(TARGET_FOLDER),
        ]));
    }

    $numFiles = $zip->numFiles;
    $lastId = $startId + BATCH_SIZE;

    $fileList = array();
    for ($id = $startId; $id < min($numFiles, $lastId); ++$id) {
        $currentFile = $zip->getNameIndex($id);
        if (in_array($currentFile, ['/index.php', 'index.php'])) {
            $indexContent = $zip->getFromIndex($id);
            if (!file_put_contents(getcwd() . '/index.php.temp', $indexContent)) {
                die(json_encode([
                    'error' => true,
                    'message' => 'Unable to write to file ' . getcwd() . '/index.php.temp',
                ]));
            }
        } else {
            $fileList[] = $currentFile;
        }
    }

    foreach ($fileList as $currentFile) {
        if ($zip->extractTo(TARGET_FOLDER, $currentFile) === false) {
            die(json_encode([
                'error' => true,
                'message' => 'Extraction error - ' . $zip->getStatusString(),
                'file' => $currentFile,
                'numFiles' => $numFiles,
                'lastId' => $lastId,
                'files' => $fileList,
            ]));
        }
    }

    @chmod('install/index.php', 0644);
    @chmod('admin/index.php', 0644);
    @chmod('admin/ajax.php', 0644);
    @chmod('admin/ajax-tab.php', 0644);
    @chmod('index.php', 0644);

    $zip->close();

    if ($lastId >= $numFiles) {
        unlink(getcwd() . '/index.php');
        unlink(getcwd() . '/prestashop.zip');
        rename(getcwd() . '/index.php.temp', getcwd() . '/index.php');
    }

    die(json_encode([
        'error' => false,
        'numFiles' => $numFiles,
        'lastId' => $lastId,
    ]));
}

if (isset($_GET['element'])) {
    switch ($_GET['element']) {
        case 'font':
            header('Content-Type: application/font-sfnt');
            echo getFileContent('AAEAAAATAQAABAAwRFNJR54SRB0AAzucAAAVdEdERUYAJgOvAAM3fAAAAB5HUE9TCzcPNwADN5wAAAA4R1NVQg4rPbcAAzfUAAADxk9TLzKhPp7JAAABuAAAAGBjbWFwKasvaAAAELQAAAQaY3Z0IA9NGKQAAB2QAAAAomZwZ21+YbYRAAAU0AAAB7RnYXNwABUAIwADN2wAAAAQZ2x5ZnQ4mUsAACWMAAEvtGhlYWT3duKmAAABPAAAADZoaGVhDcwJcwAAAXQAAAAkaG10eOg1PN0AAAIYAAAOmmtlcm5UKwl+AAFVQAABtjZsb2NhKRTc8QAAHjQAAAdWbWF4cAVDAgoAAAGYAAAAIG5hbWVzsIiFAAMLeAAABcdwb3N0AkPvbAADEUAAACYrcHJlcEO3lqQAAByEAAABCQABAAAAARmaIcf1X18PPPUACQgAAAAAAMk1MYsAAAAAyehMTPua/dUJoghiAAAACQACAAAAAAAAAAEAAAiN/agAAAms+5r+ewmiAAEAAAAAAAAAAAAAAAAAAAOjAAEAAAOqAIoAFgBWAAUAAgAQAC8AXAAAAQ4A+AADAAEAAwS2AZAABQAIBZoFMwAAAR8FmgUzAAAD0QBmAfEIAgILBgYDBQQCAgTgAALvQAAgWwAAACgAAAAAMUFTQwBAACD//QYf/hQAhAiNAlggAAGfAAAAAARIBbYAAAAgAAMEzQDBAAAAAAQUAAACFAAAAiMAmAM1AIUFKwAzBJMAgwaWAGgF1wBxAcUAhQJeAFICXgA9BGoAVgSTAGgB9gA/ApMAVAIhAJgC8AAUBJMAZgSTALwEkwBkBJMAXgSTACsEkwCFBJMAdQSTAF4EkwBoBJMAagIhAJgCIQA/BJMAaASTAHcEkwBoA28AGwcxAHkFEAAABS8AyQUMAH0F1QDJBHMAyQQhAMkF0wB9BecAyQKqAFQCI/9gBOkAyQQnAMkHOQDJBggAyQY7AH0E0QDJBjsAfQTyAMkEZABqBG0AEgXTALoEwwAAB2gAGwSeAAgEewAABJEAUgKiAKYC8AAXAqIAMwRWADEDlv/8BJ4BiQRzAF4E5wCwA88AcwTnAHMEfQBzArYAHQRiACcE6QCwAgYAogIG/5EEMwCwAgYAsAdxALAE6QCwBNUAcwTnALAE5wBzA0QAsAPRAGoC0wAfBOkApAQCAAAGOQAXBDEAJwQIAAIDvgBSAwgAPQRoAe4DCABIBJMAaAIUAAACIwCYBJMAvgSTAD8EkwB7BJMAHwRoAe4EIQB7BJ4BNQaoAGQC1QBGA/oAUgSTAGgCkwBUBqgAZAQA//oDbQB/BJMAaALHADECxwAhBJ4BiQT0ALAFPQBxAiEAmAHRACUCxwBMAwAAQgP6AFAGPQBLBj0ALgY9ABoDbwAzBRAAAAUQAAAFEAAABRAAAAUQAAAFEAAABvz//gUMAH0EcwDJBHMAyQRzAMkEcwDJAqoAPAKqAFQCqv//AqoAPAXHAC8GCADJBjsAfQY7AH0GOwB9BjsAfQY7AH0EkwCFBjsAfQXTALoF0wC6BdMAugXTALoEewAABOMAyQT6ALAEcwBeBHMAXgRzAF4EcwBeBHMAXgRzAF4G3QBeA88AcwR9AHMEfQBzBH0AcwR9AHMCBv/aAgYAqQIG/7MCBv/sBMUAcQTpALAE1QBzBNUAcwTVAHME1QBzBNUAcwSTAGgE1QBzBOkApATpAKQE6QCkBOkApAQIAAIE5wCwBAgAAgUQAAAEcwBeBRAAAARzAF4FEAAABHMAXgUMAH0DzwBzBQwAfQPPAHMFDAB9A88AcwUMAH0DzwBzBdUAyQTnAHMFxwAvBOcAcwRzAMkEfQBzBHMAyQR9AHMEcwDJBH0AcwRzAMkEfQBzBHMAyQR9AHMF0wB9BGIAJwXTAH0EYgAnBdMAfQRiACcF0wB9BGIAJwXnAMkE6QCwBecAAATpABQCqv/iAgb/kAKqACoCBv/aAqoAHgIG/8wCqgBUAgYANQKqAFQCBgCwBM0AVAQMAKICI/9gAgb/kQTpAMkEMwCwBCUAsAQnAMkCBgCjBCcAyQIGAFkEJwDJAgYAsAQnAMkCgwCwBC8AHQIX//wGCADJBOkAsAYIAMkE6QCwBggAyQTpALAFcwABBggAyQTpALAGOwB9BNUAcwY7AH0E1QBzBjsAfQTVAHMHYgB9B4kAcQTyAMkDRACwBPIAyQNEAGAE8gDJA0QAggRkAGoD0QBqBGQAagPRAGoEZABqA9EAagRkAGoD0QBqBG0AEgLTAB8EbQASAtMAHwRtABIC0wAfBdMAugTpAKQF0wC6BOkApAXTALoE6QCkBdMAugTpAKQF0wC6BOkApAXTALoE6QCkB2gAGwY5ABcEewAABAgAAgR7AAAEkQBSA74AUgSRAFIDvgBSBJEAUgO+AFICjwCwBJ4AwwUUAAAEcwBeBvz//gbdAF4GOwB9BNUAcwRkAGoD0QBqBLwBDAS8AQwEsgEtBLwBJQIGAKIEngFvAZMAJQS8AQgEngDnBJ4B/ASeARsFEAAAAiEAmATy/9QGff/UA5j/5AaB/+QFhf/UBoH/5AK2/+kFEAAABS8AyQQpAMkEkwAnBHMAyQSRAFIF5wDJBjsAfQKqAFQE6QDJBNMAAAc5AMkGCADJBG0ASAY7AH0F1QDJBNEAyQSJAEoEbQASBHsAAAZiAGoEngAIBl4AbQZCAFACqgA8BHsAAATjAHMDzQBaBOkAsAK2AKgE3wCkBOMAcwUGALAEGQAKBKQAcQPNAFoD3QBzBOkAsAS8AHMCtgCoBCUAsARG//IE9ACwBFYAAAPNAHEE1QBzBTMAGQTVAKYD2wBzBOcAcwPJABIE3wCkBb4AcwRe/+wGBgCkBi8AcwK2AAkE3wCkBNUAcwTfAKQGLwBzBHMAyQXfABIEKQDJBR0AfQRkAGoCqgBUAqoAPAIj/2AHbwAAB6AAyQXfABIE5QDJBPgAGwXVAMkFEAAABOcAyQUvAMkEKQDJBXcADgRzAMkGwQACBKYASgYZAMsGGQDLBOUAyQWiAAAHOQDJBecAyQY7AH0F1QDJBNEAyQUMAH0EbQASBPgAGwZiAGoEngAIBeUAyQWPAKoIQgDJCEQAyQWBABIG0wDJBSUAyQUKAD0IZgDJBRcAMwRzAF4ExQB3BI0AsANtALAEkwApBH0AcwXjAAQD3QBEBRIAsAUSALAEJwCwBJEAEAXhALAFEgCwBNUAcwT4ALAE5wCwA88AcwO8ACkECAACBbgAcQQxACcFAgCwBN0AnAcfALAHLQCwBY8AKQYpALAEvACwA/AAOQamALAEcQAlBH0AcwTpABQDbQCwA/AAcwPRAGoCBgCiAgb/7AIG/5EGsgAQBxcAsATpABQEJwCwBAgAAgT4ALAENwDJA20AsAdoABsGOQAXB2gAGwY5ABcHaAAbBjkAFwR7AAAECAACBAAAUggAAFIIAABSA0r//AFcABkBXAAZAfYAPwFcABkCzQAZAs0AGQM9ABkEBAB7BBQAewMCAKQGRgCYCZ4AZAHFAIUDJQCFAm8AUgJvAFAD4wCYAQr+eQMnAG0EkwBiBJMARAYbAJoEuAA/BpgAjQQpAHcIJwDJBjUAJQZCAFAE9ABmBj0ARwY9ACAGPQBHBj0AagSmAGYEkwAnBekAyQUMAEwEkwBoBGQAJQWkAHcDEgAMBJMAYgSTAGgEkwBoBJMAaASqAG8EvAAdBLwAHQSeANsCBv+RBAABiQQAAXEEAAGBAscAJwLHABQCxwA7AscAKQLHADkCxwAzAscAIwQAAAAIAAAABAAAAAgAAAACqgAAAgAAAAFWAAAEeQAAAiEAAAGaAAAAzQAAAAAAAAAAAAAIAABUCAAAVAIG/5EBXAAZBPoACgSFAAAGuAASBzkAyQdxALAFEAAABHMAXgZS/t8CqgB1AzMAmAd1AB0HdQAdBj0AfQTfAHMGJQC6BVIApAAA/FMAAP0NAAD8GQAA/QgAAP07BHMAyQYZAMsEfQBzBRIAsAgXAIUGjQAABWYAFwUOABcHWgDJBeMAsAVtAAAEgwAKB14AyQYhALAFxQAUBSMADAfLAMkGxQCwBKgAPwPdABkGXgBtBgYApAY9AH0E1QBzBQIAAAQMAAAFAgAABAwAAAmsAH0IfQBzBo0AfQVCAHMH/gB9BncAcwffAF4GjQAABR0AfQPnAHME3wBqBHUAywSeAPgEngHfBJ4B4QfpACkHpgApBikAyQUlALAE5wAvBLwAFATjAMkE5wCwBDcALwNtABIFIwDJBDMAsAcfAAIGPQAEBKYASgPdAEQFSgDJBFwAsATpAMkERACwBOkALwQjABQFgwAQBOwAKQX4AMkFLwCwBoEAyQXjALAIiQDJBuwAsAY7AH0FHwBzBQwAfQPPAHMEbQAQA7wAKQR7AAAEAgAABHsAAAQCAAAE9AAIBFYAJwbXABAFvAApBYkAqgTfAJwFjwCqBM0AnAWPAMkErgCwBrQAPQVGADMGtAA9BUYAMwKqAFQGwQACBeMABAWDAMkEZACwBaYAAASTABAF0QDJBO4AsAX2AMkFOQCwBY8AqgTdAJwHOwDJBeMAsAKqAFQFEAAABHMAXgUQAAAEcwBeBvz//gbdAF4EcwDJBH0AcwXXAHUEeQBmBdcAdQR5AGYGwQACBeMABASmAEoD3QBEBKoASgPpABsGGQDLBRIAsAYZAMsFEgCwBjsAfQTVAHMGPQB9BNUAcwY9AH0E1QBzBQoAPQPwADkE+AAbBAgAAgT4ABsECAACBPgAGwQIAAIFjwCqBN0AnAQ3AMkDbQCwBtMAyQYpALAENwAvA20AEgT4AAgEUgAnBJ4ABgQxACcE5wCDBOcAcwcxAIMHKwBzBzsATgZqAFAFAABOBC8AUAfZAAAGzwAQCBkAyQdOALAGDAB9BR8AcwWuABAFLQApBKoAbwPNAFoFmgAABJEAEAUQAAAEcwBeBRAAAARzAF4FEAAABHMAXgUQAAAEcwAtBRAAAARzAF4FEAAABHMAXgUQAAAEcwBeBRAAAARzAF4FEAAABHMAXgUQAAAEcwBeBRAAAARzAF4FEAAABHMAXgRzAMkEfQBzBHMAyQR9AHMEcwDJBH0AcwRzAMkEfQBzBHMAXQR9AEoEcwDJBH0AcwRzAMkEfQBzBHMAyQR9AHMCqgBUAgYAewKqAFQCBgCdBjsAfQTVAHMGOwB9BNUAcwY7AH0E1QBzBjsAfQTVAGEGOwB9BNUAcwY7AH0E1QBzBjsAfQTVAHMGPQB9BN8AcwY9AH0E3wBzBj0AfQTfAHMGPQB9BN8AcwY9AH0E3wBzBdMAugTpAKQF0wC6BOkApAYlALoFUgCkBiUAugVSAKQGJQC6BVIApAYlALoFUgCkBiUAugVSAKQEewAABAgAAgR7AAAECAACBHsAAAQIAAIE5wBzAAD75QAA/HEAAPuaAAD8cQAA/GgAAPx5AAD8eQAA/HkAAPxoAaQAMQGkABkBpAAZAy0ANASJAHMC9AAtBBQAKQSTAF4EjwAXBJMAhQSTAHUEkwBeBJMAaASTAGoFbQAdBloAXARtABIC0wAfBOcAcQTnAHEE5wBxBOcAcQTnAHECOwDJAjsABQI7ALMCO//HAjsABQI7/6sCO//zAjv/5wI7AFYCOwC7BF4AyQLl/+QCOwDJAAUAyQAFAMkAyQCZALgAAAAAAAEAAwABAAAADAAEBA4AAACwAIAABgAwAEgASQB+AMsAzwEnATIBYQF/AZIBoQGwAfAB/wIbAjcCvALHAskC3QLzAwEDAwMJAw8DIwOKA4wDoQOqA84D0gPWBA0ETwRfBIYEkQS/BM8FEx4BHj8ehR7HHsoe8R75H00gCyAVIB4gIiAmIDAgMyA6IDwgRCBwIHkgfyCkIKcgrCEFIRMhFiEgISIhJiEuIV4iAiIGIg8iEiIaIh4iKyJIImAiZSXK+wT+///9//8AAAAgAEkASgCgAMwA0AEoATMBYgGSAaABrwHwAfoCGAI3ArwCxgLJAtgC8wMAAwMDCQMPAyMDhAOMA44DowOrA9ED1gQABA4EUARgBIgEkgTABNAeAB4+HoAeoB7IHsse8h9NIAAgEyAXICAgJiAwIDIgOSA8IEQgcCB0IH8goyCnIKshBSETIRYhICEiISYhLiFbIgIiBiIPIhEiGiIeIisiSCJgImQlyvsA/v///P///+MAAP/j/8IAAP/CAAD/wgAA/7AAvwCyAGH/SQAAAAD/lv6F/oT+dv9o/2P/Yv9dAGf/RAAA/c8AAAAA/c3+gv5/AAD9mgAA/gwAAP4JAAD+CeRY5BjjeuR9AADkfQAA4w3iQuHv4e7h7eHq4eHh4OHb4drh0+HL4cjhmeF24XQAAOEY4QvhCeJu4P7g++D04MjgJeAi4BrgGeAS4A/gA9/n39DfzdxpAAADTwJTAAEAAACuAAAAAACqAAAArgAAAMAAAAAAAAAAAAAAAPAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAOAAAADqARAAAAAAAAABGAAAATAAAAFMAAABXAAAAAAAAAAAAAABcAAAAXIAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAFgAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAABPAAAAAAAAAOWA5cDmAOZA5oDmwDrA5wA7QOdAO8DngDxA58A8wOgA48DkAEmAScBKAEpASoBKwEsAS0BLgEvATABMQEyATMBNAE1ATYBNwE4ATkBOgE7ATwBPQE+AT8BQAFBAUkBSgEkASUBVAFVAVYBVwFYAVkDoQFcAV0BXgFfAWABYQFiAWMBZAFlAWYDogFoAWkBagFrAWwBbQFuAW8BcAFxAXIBcwF0AXUBdgOjAmgBnAGdAZ4BnwGgA6QDpQGjAaQBpQGmAacCaQJqAeoB6wHsAe0B7gHvAfAB8QHyAfMB9AH1AmsB9gH3ApMClAKVApYClwKYApkCmgH4AfkDpgLKAssCzALNAs4CzwLQAtEC0gLTAtQC1QLWAtcDpwOoA0YDqQIAAgEDbwNwA3EDcgNzA3QDdQIcA40CNAI1Al0CXgAAQEdbWllYVVRTUlFQT05NTEtKSUhHRkVEQ0JBQD8+PTw7Ojk4NzY1MTAvLi0sKCcmJSQjIiEfGBQREA8ODQsKCQgHBgUEAwIBACwgsAFgRbADJSARRmEjRSNhSC0sIEUYaEQtLEUjRmCwIGEgsEZgsAQmI0hILSxFI0YjYbAgYCCwJmGwIGGwBCYjSEgtLEUjRmCwQGEgsGZgsAQmI0hILSxFI0YjYbBAYCCwJmGwQGGwBCYjSEgtLAEQIDwAPC0sIEUjILDNRCMguAFaUVgjILCNRCNZILDtUVgjILBNRCNZILAEJlFYIyCwDUQjWSEhLSwgIEUYaEQgsAFgIEWwRnZoikVgRC0sAbELCkMjQ2UKLSwAsQoLQyNDCy0sALAoI3CxASg+AbAoI3CxAihFOrECAAgNLSwgRbADJUVhZLBQUVhFRBshIVktLEmwDiNELSwgRbAAQ2BELSwBsAZDsAdDZQotLCBpsEBhsACLILEswIqMuBAAYmArDGQjZGFcWLADYVktLIoDRYqKh7ARK7ApI0SwKXrkGC0sRWWwLCNERbArI0QtLEtSWEVEGyEhWS0sS1FYRUQbISFZLSwBsAUlECMgivUAsAFgI+3sLSwBsAUlECMgivUAsAFhI+3sLSwBsAYlEPUA7ewtLLACQ7ABUlghISEhIRtGI0ZgiopGIyBGimCKYbj/gGIjIBAjirEMDIpwRWAgsABQWLABYbj/uosbsEaMWbAQYGgBOlktLCBFsAMlRlJLsBNRW1iwAiVGIGhhsAMlsAMlPyMhOBshEVktLCBFsAMlRlBYsAIlRiBoYbADJbADJT8jITgbIRFZLSwAsAdDsAZDCy0sISEMZCNki7hAAGItLCGwgFFYDGQjZIu4IABiG7IAQC8rWbACYC0sIbDAUVgMZCNki7gVVWIbsgCALytZsAJgLSwMZCNki7hAAGJgIyEtLEtTWIqwBCVJZCNFabBAi2GwgGKwIGFqsA4jRCMQsA72GyEjihIRIDkvWS0sS1NYILADJUlkaSCwBSawBiVJZCNhsIBisCBharAOI0SwBCYQsA72ihCwDiNEsA72sA4jRLAO7RuKsAQmERIgOSMgOS8vWS0sRSNFYCNFYCNFYCN2aBiwgGIgLSywSCstLCBFsABUWLBARCBFsEBhRBshIVktLEWxMC9FI0VhYLABYGlELSxLUViwLyNwsBQjQhshIVktLEtRWCCwAyVFaVNYRBshIVkbISFZLSxFsBRDsABgY7ABYGlELSywL0VELSxFIyBFimBELSxFI0VgRC0sSyNRWLkAM//gsTQgG7MzADQAWURELSywFkNYsAMmRYpYZGawH2AbZLAgYGYgWBshsEBZsAFhWSNYZVmwKSNEIxCwKeAbISEhISFZLSywAkNUWEtTI0tRWlg4GyEhWRshISEhWS0ssBZDWLAEJUVksCBgZiBYGyGwQFmwAWEjWBtlWbApI0SwBSWwCCUIIFgCGwNZsAQlELAFJSBGsAQlI0I8sAQlsAclCLAHJRCwBiUgRrAEJbABYCNCPCBYARsAWbAEJRCwBSWwKeCwKSBFZUSwByUQsAYlsCngsAUlsAglCCBYAhsDWbAFJbADJUNIsAQlsAclCLAGJbADJbABYENIGyFZISEhISEhIS0sArAEJSAgRrAEJSNCsAUlCLADJUVIISEhIS0sArADJSCwBCUIsAIlQ0ghISEtLEUjIEUYILAAUCBYI2UjWSNoILBAUFghsEBZI1hlWYpgRC0sS1MjS1FaWCBFimBEGyEhWS0sS1RYIEWKYEQbISFZLSxLUyNLUVpYOBshIVktLLAAIUtUWDgbISFZLSywAkNUWLBGKxshISEhWS0ssAJDVFiwRysbISEhWS0ssAJDVFiwSCsbISEhIVktLLACQ1RYsEkrGyEhIVktLCCKCCNLU4pLUVpYIzgbISFZLSwAsAIlSbAAU1ggsEA4ERshWS0sAUYjRmAjRmEjIBAgRophuP+AYoqxQECKcEVgaDotLCCKI0lkiiNTWDwbIVktLEtSWH0belktLLASAEsBS1RCLSyxAgBCsSMBiFGxQAGIU1pYuRAAACCIVFiyAgECQ2BCWbEkAYhRWLkgAABAiFRYsgICAkNgQrEkAYhUWLICIAJDYEIASwFLUliyAggCQ2BCWRu5QAAAgIhUWLICBAJDYEJZuUAAAIBjuAEAiFRYsgIIAkNgQlm5QAABAGO4AgCIVFiyAhACQ2BCWbEmAYhRWLlAAAIAY7gEAIhUWLICQAJDYEJZuUAABABjuAgAiFRYsgKAAkNgQllZWVlZWbEAAkNUWEAKBUAIQAlADAINAhuxAQJDVFiyBUAIugEAAAkBALMMAQ0BG7GAAkNSWLIFQAi4AYCxCUAbsgVACLoBgAAJAUBZuUAAAICIVblAAAIAY7gEAIhVWlizDAANARuzDAANAVlZWUJCQkJCLSxFGGgjS1FYIyBFIGSwQFBYfFloimBZRC0ssAAWsAIlsAIlAbABIz4AsAIjPrEBAgYMsAojZUKwCyNCAbABIz8AsAIjP7EBAgYMsAYjZUKwByNCsAEWAS0ssICwAkNQsAGwAkNUW1ghIxCwIBrJG4oQ7VktLLBZKy0sihDlLUCZCSFIIFUgAR5VH0gDVR8eAQ8ePx6vHgNNSyYfTEszH0tGJR8mNBBVJTMkVRkT/x8HBP8fBgP/H0pJMx9JRiUfEzMSVQUBA1UEMwNVHwMBDwM/A68DA0dGGR/rRgEjMyJVHDMbVRYzFVURAQ9VEDMPVQ8PTw8CHw/PDwIPD/8PAgYCAQBVATMAVW8AfwCvAO8ABBAAAYAWAQUBuAGQsVRTKytLuAf/UkuwCVBbsAGIsCVTsAGIsEBRWrAGiLAAVVpbWLEBAY5ZhY2NAEIdS7AyU1iwIB1ZS7BkU1iwEB2xFgBCWXNzKytec3R1KysrKyt0K3N0KysrKysrKysrKysrK3N0KysrGF4AAAAGFAAXAE4FtgAXAHUFtgXNAAAAAAAAAAAAAAAAAAAESAAUAJEAAP/sAAAAAP/sAAAAAP/sAAD+FP/sAAAFtgAT/JT/7f6F/+r+qf/sABj+vAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAgAAAAAAACLAIEA3QCYAI8AjgCZAIgAgQEPAIoAAAAAAB8AHwAfAB8AUQB3AP8BewHsAmoCgwKuAtkDFQNBA18DdAOWA68D8QQaBFsEuQT7BUYFowXFBjQGkQbHBvsHGwdEB2QHuwhBCIAI2wkZCVUJigm4CggKOQpsCpQKwwrhCx8LVgucC9kMLAx5DMwM8A0kDUsNjw2/DeYOEg42Dk8Ocg6TDqkOyA8kD3kPtBAHEFQQlBEoEWYRlBHSEhASJxJ/ErkS+hNPE6MT1hQoFGgUpRTMFRcVRxWAFawV7hYGFksWhRaFFrYXARdTF6EX9RgaGJUYyxlHGZQZzxntGfUafxqVGs0a2RsTG2MbghvBG/EcExxFHGwcpRzdHPMdCB0eHXsdjB2dHa4dvx3RHd0eKx43HkgeWR5qHnwejR6eHq8ewR8ZHyofOx9MH10fbh+AH64gGSAqIDsgTCBeIG8gsSEYISghOCFIIVghaSF6IgUiESIhIjEiQSJSImMidCKFIpci/yMPIx8jLyM/I08jYCOmJAwkHCQsJDwkTSRdJLQkxSTWJOYk9yUHJRMlHyUwJUAlUSVhJXIlgyWUJaQltSXGJc4mOiZLJlsmbCZ8Jo0mniaqJrYmxybXJugm+CcJJxknKic7J0cnVydoJ3knySgiKDMoRChVKGYodyiIKJMoniivKMYo0ijeKO8pACkMKRcpTCldKW4peSmFKZYppimyKb4p+CotKj4qTipaKmUqdiqGKpcq3isnKzgrSCtZK2kreyuMK+8saSx6LIoslSyhLLIswyzULOQs9S0FLREtHS0uLT4tSS1ULWUtdS2yLgQuFS4lLjYuRi5XLmcueS6KLpwurS65LsUu1i7nLvgvCC8aLysvOy9ML10vbi9+L6Uv+DB3MRYxJzE4MUkxWTFkMW8xmDHBMdcx/zIfMlQyezK0MuYzBTNOM18zZzN4M4oznDOtM78z0DPjM+sz8zQSNBo0IjQqNDI0izSTNJs0wTTJNNE1BjUONTI1OjVxNXk1gTXoNfA2PDaQNqI2tDbENtQ25Db1Nwc3azfQOAY4ZzjFORI5TDmmOdI52josOjQ6XzrKOtI7EDtcO6g77TwlPF08uj0QPV89uT3LPdw97D38Pg0+Hz5vPoA+yj7SPto+7D70P1M/pj/lP/ZAB0A3QD9AhkCOQJZA30DnQSxBiUHBQdJCAUI8QkRCTEJUQlxCZEJsQnRCs0K7QsNC9EMrQ1tDlUPbRCNEYUSvRQ9FVkVeRbpGFUY0RnxGhEbKRyNHW0drR5tH0UgUSElIUUh1SH1IhUiqSLJJE0kbSUxJg0m0Se9KNEp9SrhLCEtlS6lLukwlTDVMg0yLTJNMpUytTQZNWE1gTXBNgE2xTdZN/U4OTh5OL05ATlJOZE51ToZOm06wTrhO2k73TxVPHU86T2lPmk+0T/JQWlB6UIpRJFEsUTRRV1F7UYdRoFHTUhhShlL4U25T1FQsVKBU9FT8VUtVYlV5VZBVp1YKVj5WY1aXVq5W0lcyV2JX41gsWD5YUFh9WIlYlVi8WONZAlkhWUBZdVm3WfxaTVpuWtNbJ1snWydbJ1snWydbJ1snWydbJ1snWydbJ1snXHFczFzdXOVdbF2nXgteHF4tXjleRV5XXoxew17TXuNfQF+XX+BgMWA6YENgTGB6YJlgqmC7YMtg22FOYZlh7WI7Ypti/mM/Y4Bj1mQsZI9k9GVpZeBmjGcwZzhnQGedZ/ZoL2hnaHloi2kBaQ1pgGnzap1rO2vRbDpsfWy/bQNtM21gbYZtrG6QbxtvgW/fcDFwgnDXcUNxe3G0cgZyVXKocvtzB3MTc1BzjHPNdBB0WHSsdOZ1HnVddaJ13XYddnN2xndCd7l3xXfReAJ4NHg8eG94rXjxeTB5cXmueex6MHpzer97C3tDe3p76HxLfMF9LX01fUZ9V32sffx+RH6Hfsx/FX9Vf5Z/2oAegG+AvYDFgNaA5oD4gQmBEYEZgSqBOoGLgdqB7IH9gg+CIYIzgkSCkILaguuC+4MNgx6DMINBg0mDUYNjg3SDhoOXg6iDuIPKg9uD7YP+hBCEIYRMhHeEiYSbhKeEsoS+hMqFEIVWhZSFnIX2hmSGyYcnh4GH1IgriHmIxIkTiWaJsInvii2KioqSip6Kqoq2isKK04rkivaLCIsaiyyLPotQi2KLdIuJi52Lr4vBi9OL5Yv3jAmMG4wtjEKMVoxijG6Mf4yQjKGMsYzDjNWM54z5jQuNHY0vjUGNVo1qjXuNjI2YjaSNsI28jc2N3o3wjgKOFI4mjjiOSo5cjm6Og46XjqiOuI7JjtmO6o77jwyPHI8ojzSPQI9Mj12Pbo9/j4+PoI+wj8GP0o/jj/OP/5ALkBeQI5A0kEWQVpBmkHKQppDhkR2RapHCkfqSMpJ7ks2S9ZMYkzuTRJODk62T7pROlJOU3pTmlQmVEZVulXqV9pYClg6WcZaBlpGWopaylseW2JbplvqXDJcdly6XP5dKl1uXZ5d5l4GXk5ebl62XtZe9l86X2gAAAAIAwQAABAoFtgADAAcAFbcEAwUCBAMHAAAvMi8zAS8zLzMxMBMhESE3IREhwQNJ/LdoAnn9hwW2+kpoBOYAAgCY/+MBiQW2AAMADgArQBQDCQkCBAQPEAEBDAIMBk9ZDBYCAwA/PysREgA5GC8REgE5ETMzETMxMAEjAzMDNDMyFhUUBiMiJgFGaTPP4Xg6P0A5NEQBkwQj+rSIRkJARz8AAAIAhQOmArAFtgADAAcAH0ANAAMHBAMECAkGAgcDAwA/M80yERIBOTkRMxEzMTABAyMDIQMjAwE/KGkpAispaCkFtv3wAhD98AIQAAACADMAAAT2BbYAGwAfAJlAVQgfHBUEFAkRDAwJEg8OCwQKExMUFh0eBwQGFwQBABkEGAUFBhQGCiEDGhcDGAoYICEIBAwNDE5ZHAENHwAQERBOWRkVEU8NAU8RAQ0RDREFFxMDCgUALzM/MxI5OS8vXV0RMzMrEQAzMxEzMysRADMzERIBOTkRFzMREjk5ETMREhc5ERIXOREzERIXOTIyETMREhc5MTABAyEVIQMjEyEDIxMhNSETITUhEzMDIRMzAyEVASETIQPVQgEb/s1UiVT+0VKIUP76AR9E/usBK1KLUgExVIZUAQj85QEvQv7RA4P+rIH+UgGu/lIBroEBVH8BtP5MAbT+TH/+rAFUAAMAg/+JBAwGEgAgACYALQBmQDUnESUdFwQEKhQNBSEAABkFEQkFLi8lDQYNTVkDBiQOKg5MWR0qKxwUHE1ZFyoUBhQGFAUWBQAvLxI5OS8vEjkyKxEAMxEzKxEAMxEzKxEAMxESARc5ETMRMzMzMxEzMzMRMzEwARQGBxUjNSImJzUWFjMRJiY1NDY3NTMVFhcHJicRHgIHNCYnETYBFBYXEQYGBAzMt4Fw0kNT2VnNpcungbirNJWanZxKqlmA2f3dWm9jZgHBiLEX6N8jH5wlLwG4QayIg6gStrQFRYM7C/5OMl97ZUhZLP57HgMHTFwpAYMQXQAABQBo/+wGLQXLAAkAFQAhAC0AMQBFQCQAEAUKFigcIiIuKAowEAYyMwMNHysNKw0rMDEGMBgZJRkHEwcAPzM/Mz8/Ejk5Ly8RMxEzERIBFzkRMxEzETMRMzEwExQWMzIRECMiBgUUBiMiJjU0NjMyFgEUFjMyNjU0JiMiBgUUBiMiJjU0NjMyFgEBIwHySlOkpFNKAcqZlIyblZKRnAGmSlRUUFBUVEoBy5mUjpmVko6f/v781ZMDKwQCqqoBVAFSqKrk6e7f4+bu/Nurqaetq6Wlq+Pp7t7j5usDIPpKBbYAAAMAcf/sBdMFzQALABUANQBRQDATFgAdBiMqKy4rLSMOJhkdFgk2NzMMSVkzEw8nLQ4wBS8DGSYDKiogLxIgCUpZIAQAPysAGD8SOS8XORIXOT8rERIBFzkRMxEzETMRMzEwARQWFzY2NTQmIyIGEzI3AQ4CFRQWJTQ2Ny4CNTQ2MzIWFRQGBwE2NjczAgcBIycGBiMiJgGeSFeBZWdWWW+b8Z/+S29cLJv+uYu0VT0kxK+iuoidAZc4QxeoRIkBK+W5dvSW1+0Ek0V9WEt/U01hYPudmgGoRFlmQXWJ+oLIZl9iajmWqKeVa7Vd/nk+p2P+4pT+3bJqXNQAAAEAhQOmAT8FtgADABS3AAMDBAUCAwMAP80REgE5ETMxMAEDIwMBPyhpKQW2/fACEAAAAQBS/rwCIQW2AA0AHEAMBwAKBAAEDg8LJwMDAD8/ERIBOTkRMxEzMTATEBI3MwYCFRQSFyMmAlKbkqKQkZSLoJOaAjEBCQHOrsH+MvTw/ja9qgHGAAABAD3+vAIMBbYADQAcQAwECgcACgAODwoDBCcAPz8REgE5OREzETMxMAEQAgcjNhI1NAInMxYSAgybkqCLlJGQopOaAjH++f46qLwBy/D0Ac7Br/4xAAEAVgJ/BA4GFAAOADBAGwMFBAEHDQoJCwkPEAQKAQ0CDAwNCgcEBggOAAA/xDIXOREzETMRMxESARc5MTABAyUXBRMHAwMnEyU3BQMCkSsBjhr+g/issKCw8v6HHQGHKwYU/nVvth/+ul4Bav6WXgFGH7ZvAYsAAAEAaADjBCkEwwALAChAEwAEBAkFBQwNAwcIB1BZAA8IAQgAL10zKxEAMxESATkRMzMRMzEwASEVIREjESE1IREzAo0BnP5ki/5mAZqLAxeK/lYBqooBrAABAD/++AFtAO4ACAARtQUACQoFAAAvzRESATk5MTAlFwYCByM2EjcBXg8aYjV9G0EN7hdk/vdyaAEyXAABAFQB2QI/AnEAAwARtQIABQQAAQAvMxESATk5MTATNSEVVAHrAdmYmAABAJj/4wGJAPIACwAYQAsGAAAMDQkDT1kJFgA/KxESATkRMzEwNzQ2MzIWFRQGIyImmD05OkFCOTNDakNFRUNBRj8AAAEAFAAAAtsFtgADABO3AgAEBQMDAhIAPz8REgE5OTEwAQEjAQLb/d+mAiEFtvpKBbYAAgBm/+wELQXNAAsAFwAoQBQSAAwGAAYZGAkVS1kJBwMPS1kDGQA/KwAYPysREgE5OREzETMxMAEQAiMiAhEQEjMyEgEQEjMyEhEQAiMiAgQt7/bs9u707vf84ZakppWVpqSWAt3+hf6KAX8BcgF+AXL+fv6S/sH+3QEnATsBOwEl/t8AAQC8AAACywW2AAoAJEAQCQABCAELDAQJBwcBCQYBGAA/PxI5LxI5ERIBOTkRMzMxMCEjETQ3BgYHJwEzAsuiCBU01FgBg4wEEoJ0FS6scgErAAEAZAAABCUFywAZACtAFxgBBxMAEw4BBBobEApLWRAHARhMWQEYAD8rABg/KxESARc5ETMRMzEwISE1AT4CNTQmIyIGByc2MzIWFRQCBwEVIQQl/D8BgbBwOI5+W6NkWMruzuqc1v7AAvCPAYOymJBTdYk8T3Go07KL/vDQ/scIAAABAF7/7AQbBcsAJwBDQCQbABMHBwADFiINBigpAxcWFxZLWRcXCiUlHktZJQcKEUtZChkAPysAGD8rERIAORgvKxESADkREgEXOREzETMxMAEUBgcVFhYVFAQhIiYnNRYWMyARECEjNTMyNjU0JiMiBgcnNjYzMhYD7p2QsKr+3v71dMFbX9dgAXv+XpCSq8iTfmCqbVRa64LV7ARejLIeCBa0ktHhIyyeLzEBKQEKj5eGa3o0RnBHUcMAAAIAKwAABGoFvgAKABIAPEAeEgUJAgILBwMAAwUDExQBBRIFTFkJDwcSEgMHBgMYAD8/EjkvEjkzKxEAMxESARc5ETMzMxEzETMxMAEjESMRITUBMxEzIRE0NyMGBwEEatmf/TkCtrDZ/ogKCDAq/jcBUP6wAVCRA938KQHmj7RgP/12AAEAhf/sBB0FtgAaADpAHw8DGRQIFBcDBBwbABFLWQAABhUVGExZFQYGDEtZBhkAPysAGD8rERIAORgvKxESARc5ETMRMzEwATIEFRQAIyInNRYWMzI2NRAhIgcnEyEVIQM2Ai3nAQn+3/73gkbQZbDD/olfn1Y3Atf9tyVzA33lx+P+/k+gLTOmnQEyHTcCrJn+SRcAAAIAdf/sBC8FywAWACQAREAjGhELISEAAAYRAyYlDAsOHU1ZCw4OFAMUF0tZFBkDCE1ZAwcAPysAGD8rERIAORgvOSsRADMREgEXOREzETMRMzEwExAAITIXFSYjIgIDMzYzMhYVFAIjIgAFMjY1NCYjIgYGFRQWFnUBTwFIcUFNY+v4DAxu7sXj+dTj/vYB646dkpFalllQkwJxAa8BqxOPGf7b/sas7szk/vsBVcizqZGmSoJGZ7JoAAEAXgAABCsFtgAGAB9AEAEFBQACAwcIAwJMWQMGABgAPz8rERIBFzkRMzEwIQEhNSEVAQEdAl784wPN/aoFHZmF+s8AAwBo/+wEKQXLABYAIgAuAE1AKRcPJhQsAx0JCQMGERQPBi8wBhEpICkgS1kpKQwADBpNWQwZACNNWQAHAD8rABg/KxESADkYLysREgA5ORESARc5ETMRMxEzETMxMAEyFhUUBgcWFhUUBiMiJjU0JSYmNTQ2AxQWMzI2NTQmJwYGASIGFRQWFzY2NTQmAkjI6oaTspb+3er8ATKKeOt3p5eVppzClYYBOn2Odp+Pd5EFy7qkbLJJVbt7ttnNvPuMTrVwn737pniGjHphl0dAmwNneGRchEI8ilxldwAAAgBq/+wEJQXLABcAJQBBQCIbESIKCgAABBEDJicOHk1ZCxQODgIUFBhLWRQHAgdNWQIZAD8rABg/KxESADkYLxI5KxESARc5ETMRMxEzMTABECEiJzUWMzISEyMGBiMiJjU0EjMyFhIBIgYVFBYzMjY2NTQmJgQl/Wh0RFBm8PULDDe2csLk/9CV33j+FI+ckJNbmVhSkwNG/KYUjxoBKQEzU1fo0OQBCJn+2wEwuKSQpUqARmmyZgAAAgCY/+MBiQRkAAsAFQAoQBQQBgYMAAAWFw4TT1kOEAkDT1kJFgA/KwAYPysREgE5ETMzETMxMDc0NjMyFhUUBiMiJhE0MzIVFAYjIiaYPTk6QUI5M0N2e0I5M0NqQ0VFQ0FGPwO7h4dBRj8AAgA//vgBhQRkAAgAEgAiQBABDQ0FCQkUEwsQT1kLEAUAAC/NPysREgE5ETMzETMxMCUXBgIHIzYSNwM0MzIVFAYjIiYBXg8aYjV9G0ENFXd7Qjk6Pe4XZP73cmgBMlwC74eHQUZGAAABAGgA8gQpBNkABgAVQAkEAAUBBAcIAwAALy8REgEXOTEwJQE1ARUBAQQp/D8DwfzyAw7yAaZiAd+V/o3+uAAAAgB3AcEEGQPjAAMABwAqQBUHAgQAAgAJCAQFUFkEAQBQWQ8BAQEAL10rABgvKxESATk5ETMRMzEwEzUhFQE1IRV3A6L8XgOiA1qJif5niYkAAAEAaADyBCkE2QAGABVACQUBAgAEBwgGAwAvLxESARc5MTATAQE1ARUBaAMP/PEDwfw/AYkBRgF1lf4hYv5aAAACABv/4wM5BcsAGwAmADlAHSEcGwAHExMAHA4EJygAACQQJB5PWSQWEApJWRAEAD8rABg/KxESADkYLxESARc5ETMRMxEzMTABNTQ2NzY2NTQmIyIGByc2MzIWFRQGBgcGBhUVAzQzMhYVFAYjIiYBIUhiiEeDe0+WYTu9zr/UJ0x+ZUGyeDo/QDk0RAGTNnWXVHN0UmZvJTGHY7yrSW9jblZyXyH+14hGQkBHPwACAHn/Rga4BbQANQA/AEVAIiMuNg47BxQbAAApFA4uBUBBGDg4BD0IEQsRCxErHzIDJisALzM/MxI5OS8vEjkyMzMRMxESARc5ETMRMzMRMxEzMTABFAYGIyImJyMGBiMiJjU0NjMyFhcDFRQzMjY1NAIkIyIEAhUQACEyNxUGIyAAERASJCEyBBIBFDMyExMmIyIGBrhYoGhWdgsIKJVmlqnswESsRRmFW3KU/u+x3/62rgFCAS/S4sD0/pX+b9YBjAEA1wFPt/v2w88SDkhVgpMC2Y7sgmhRV2LNsMz/GRb+Khay16y1ARCTuf6p4f7P/rhWhVQBjwFmAQQBlt+1/rP+pP4BOQEFFLQAAgAAAAAFEAW8AAcADgA5QB4CDgsIAQUAAwAHAwQHBBAPDgJJWQsFDg4EBQMABBIAPzM/EjkvEjkrERIBOTkRMxEzERIXOTEwIQMhAyMBMwEBAyYnBgcDBGC2/ba0rAJCjwI//mWqISMWKawB0f4vBbz6RAJqAcVWfWBz/jsAAwDJAAAEvgW2AA4AFwAgAElAJhMEHQoPGRkOCgQHDgQhIggPGA8YSlkPDw4ADhlKWQ4SABdKWQADAD8rABg/KxESADkYLysREgA5ERIBFzkRMxEzETMRMzEwEyEgBBUUBgcVBBEUBCMhEyEyNjU0JiMjEREhMjY1NCYjyQGdASMBBJGLAU3+9+7+AqoBGLSesMD6ATGxs7e7BbauvIKpGQo5/tvE3ANEcYZ7bf2R/d2JkoiAAAABAH3/7ATPBcsAFgAmQBQDDhQJDgMXGBIASVkSBAsGSVkLEwA/KwAYPysREgEXOREzMTABIgAREAAzMjcVBiMgABE0EiQzMhcHJgM78f7pAQ35mcSY3/69/qGpAT/Y5qxIpgUz/r/+6f7h/sc3lTkBiAFp4gFUuFSSTgAAAgDJAAAFWAW2AAgAEQAoQBQOBAkABAASEwUNSlkFAwQOSlkEEgA/KwAYPysREgE5OREzETMxMAEQACEhESEgAAMQACEjETMgAAVY/nf+j/5rAcABVQF6tP7h/uX3zwEwATIC6f6W/oEFtv6G/qcBHgEi+3ABKwAAAQDJAAAD+AW2AAsAOkAfBgoKAQQACAEEDA0GCUlZBgYBAgIFSVkCAwEKSVkBEgA/KwAYPysREgA5GC8rERIBFzkRMxEzMTAhIREhFSERIRUhESED+PzRAy/9ewJe/aIChQW2l/4plv3mAAEAyQAAA/gFtgAJADJAGgYAAAEDCAEDCgsGCUlZBgYBAgIFSVkCAwESAD8/KxESADkYLysREgEXOREzETMxMCEjESEVIREhFSEBc6oDL/17Al79ogW2l/3plwAAAQB9/+wFPQXLABsAOkAfFAgZAgIOGwgEHB0AG0lZAAAFDAwRSVkMBAUXSVkFEwA/KwAYPysREgA5GC8rERIBFzkRMxEzMTABIREGBiMgABE0EiQzMhcHJiMgABEQACEyNxEhA0wB8XTwnv60/o63AVjn6spCxrf+9f7UASEBGJiR/rkC/v05JSYBiwFk5AFXtVaWVP7C/ub+2P7OIwHCAAEAyQAABR8FtgALADNAGQkBAQAIBAQFAAUNDAgDSVkICAUKBgMBBRIAPzM/MxI5LysREgE5OREzETMRMxEzMTAhIxEhESMRMxEhETMFH6r8/qqqAwKqArD9UAW2/ZICbgAAAQBUAAACVgW2AAsAN0AcBQEKAwgAAAMBAwwNCQQGBEpZBgMKAwEDSlkBEgA/KxEAMxg/KxEAMxESARc5ETMRMxEzMTAhITU3ESc1IRUHERcCVv3+rKwCAqysYiMEqiViYiX7ViMAAf9g/n8BaAW2AA0AHUANCwgIDg8JAwAFSVkAIgA/KwAYPxESATkRMzEwAyInNRYzMjY1ETMRFAYMXjZHTWNnqsD+fxuRFHhxBbb6WL7RAAABAMkAAATpBbYACwAqQBUIBAQFBQILCgAFDQwCCAUJBgMBBRIAPzM/MxI5ORESARc5ETMRMzEwISMBBxEjETMRATMBBOnI/euZqqoCl8n9tALFiP3DBbb9KwLV/YUAAQDJAAAD+AW2AAUAH0AOAwAABAYHAQMAA0lZABIAPysAGD8REgE5OREzMTAzETMRIRXJqgKFBbb65JoAAQDJAAAGcQW2ABMAMkAYCAUFBgsODg0GDRQVAQoRAwYLBwMOAAYSAD8zMz8zEhc5ERIBOTkRMxEzETMRMzEwIQEjFhURIxEhATMBMxEjETQ3IwEDUP4QCA6dAQABzwgB0/6qDgj+DAUQmtT8XgW2+0oEtvpKA66ivvryAAEAyQAABT8FtgAQAC5AFQkGBgcBDw8ABwAREgsDBw8IAwEHEgA/Mz8zEjk5ERIBOTkRMxEzETMRMzEwISMBIxYVESMRMwEzJgI3ETMFP8L84QgQncADHQgCDgKfBMvYtPzBBbb7OhsBJT8DRwAAAgB9/+wFvgXNAAsAFwAoQBQSAAwGAAYZGAkVSVkJBAMPSVkDEwA/KwAYPysREgE5OREzETMxMAEQACEgABEQACEgAAEQEjMyEhEQAiMiAgW+/p3+xP69/qEBYAFEATsBYvtz/fHz+Pfy8/0C3f6h/m4BiwFoAWUBif5w/qD+1/7NATIBKgEnATH+zQACAMkAAARoBbYACQASADRAGgoFBQYOAAYAExQKBEpZCgoGBwcSSlkHAwYSAD8/KxESADkYLysREgE5OREzETMRMzEwARQEISMRIxEhIAEzMjY1NCYjIwRo/tH+5qyqAXsCJP0LmeLKvsm+BAze7/3BBbb9G5KhkY4AAAIAff6kBb4FzQAPABsANEAbEAoWAAAEAwoEHB0DDQcNGUlZDQQHE0lZBQcTAD/GKwAYPysREgA5ERIBFzkRMxEzMTABEAIHASMBByAAERAAISAAARASMzISERACIyICBb7izgFc9/7jN/69/qEBYAFEATsBYvtz/fHz+Pfy8/0C3f7n/oxC/pYBSgIBiwFoAWUBif5w/qD+1/7NATIBKgEnATH+zQACAMkAAATPBbYADAAVAEhAJQ0BAQIMCREHCwoKBwkCBBYXCQ0ADQBKWQ0NAgMDFUlZAwMLAhIAPzM/KxESADkYLysREgA5ERIBFzkRMxEzETMRMxEzMTABESMRISAEFRAFASMBJTMyNjU0JiMjAXOqAZEBDQEB/toBjcn+nv7P6bSoq73dAmD9oAW2zs/+3mb9bwJgko+PkYAAAQBq/+wEAgXLACQANEAbHhMMAAAYEwUEJSYMHgMWFhtJWRYEAwlJWQMTAD8rABg/KxESADk5ERIBFzkRMxEzMTABFAQjICc1FhYzMjY1NCYmJyYmNTQ2MzIXByYjIgYVFBYWFxYWBAL+6PD+/Ixa1GiqrD2Pksyv/tHatzW1q4eYOIWJ5q0BhcHYQ6QmLIFzTGFSNEnIoanIUJRMdGdMYVExUrwAAAEAEgAABFoFtgAHACRAEgABBQEDAwgJBwMEA0lZBAMBEgA/PysRADMREgEXOREzMTAhIxEhNSEVIQKLqv4xBEj+MQUfl5cAAAEAuv/sBRkFtgARACVAERABCgcBBxMSEQgDBA1JWQQTAD8rABg/MxESATk5ETMRMzEwAREUACEgADURMxEUFjMyNjURBRn+0v74/vj+36rIwrnIBbb8Tvr+4gEg/AOu/Ea3xMW4A7gAAQAAAAAEwwW2AAoAGkALAQQMCwgDAAQDAxIAPz8zEjkREgE5OTEwATMBIwEzARYXNjcEDLf98aj99LQBUDoiJDoFtvpKBbb8TqOaoqEAAQAbAAAHTAW2ABkAJEAQGQobGhUODgUJGBEKAwEJEgA/Mz8zMxI5OREzERIBOTkxMCEjASYmJwYHASMBMxMWFzY3ATMBFhc2NxMzBcWo/tkVNAEWMP7iqP57tOcwFhs1AQa0ARMwIRM15rQD00HGFISd/DMFtvx5vpq3rwN5/H+bw47MA4UAAAEACAAABJYFtgALACNAEgQGBQsKAAYNDAIIBAkGAwEEEgA/Mz8zEjk5ERIBFzkxMCEjAQEjAQEzAQEzAQSWwf53/nC0Aeb+O7wBawFutf47AoP9fQL8Arr9vQJD/UwAAAEAAAAABHsFtgAIACBADwQFAgUHAwkKAAUBBwMFEgA/PzMSORESARc5ETMxMAEBMwERIxEBMwI9AYa4/his/hm6AtsC2/yB/ckCLwOHAAEAUgAABD8FtgAJACtAFwgBAwcABwQBBAoLBQRJWQUDAQhJWQESAD8rABg/KxESARc5ETMRMzEwISE1ASE1IRUBIQQ//BMDCP0QA7/8+AMehQSYmYX7aQABAKb+vAJvBbYABwAgQA4GAQQAAQAICQUCAwYBJwA/Mz8zERIBOTkRMxEzMTABIREhFSERIQJv/jcByf7fASH+vAb6jfohAAABABcAAALdBbYAAwATtwMBBAUDAwISAD8/ERIBOTkxMBMBIwG6AiOm/eAFtvpKBbYAAAEAM/68AfwFtgAHACBADgMAAQYABggJAAcnAwQDAD8zPzMREgE5OREzETMxMBchESE1IREhMwEh/t8Byf43tgXfjfkGAAABADECJwQjBcEABgAYQAkAAwcIBQIABAIALy8zEjkREgE5OTEwEwEzASMBATEBsmMB3Zj+jP6yAicDmvxmAun9FwAB//z+xQOa/0gAAwARtQAFAQQBAgAvMxEBMxEzMTABITUhA5r8YgOe/sWDAAEBiQTZAxIGIQAJABO2AAQLCgaAAQAvGs0REgE5OTEwASMmJic1MxYWFwMSbkGyKMsgciwE2TTAPxVFtTUAAgBe/+wDzQRaABkAJABHQCUiCAseHhkZEggDJSYBAgseR1kCCwsAFRUPRlkVEAUaRlkFFgAVAD8/KwAYPysREgA5GC85KxEAMxESARc5ETMRMxEzMTAhJyMGBiMiJjUQJTc1NCYjIgcnNjYzMhYVESUyNjU1BwYGFRQWA1IhCFKjeqO5AhO6b3qJrTNRwWHEvf4Om7Gmxq9tnGdJqJsBTBAGRIF7VH8sMq7A/RR1qpljBwdtc1peAAIAsP/sBHUGFAATAB8AREAiChcXDw8MHQMMAyAhDQAMFRIRChEGAAYaRlkGFgAURlkAEAA/KwAYPysREgA5OREzGD8/ERIBOTkRMxEzETMRMzEwATISERACIyImJyMHIxEzERQHMzYXIgYVFBYzMjY1NCYCrtjv8dZrsTwMI3emCAh0zKqWmqqZlpYEWv7Z/vL+8v7VT1KNBhT+hn9lpIvD5+fH39HW0gAAAQBz/+wDiwRcABYAJkAUDwMDFQkDGBcGDUZZBhAAEkZZABYAPysAGD8rERIBFzkRMzEwBSIAERAAMzIWFwcmJiMgERQWMzI3FQYCZu7++wEJ9U+eLTM3gjL+sqOgiZBuFAElAQwBEwEsIheNFh3+VsrYO5M5AAIAc//sBDcGFAASAB8AQkAhHQYXAA4OEQYRICESFQ8AAAEBDAMJCRpGWQkQAxNGWQMWAD8rABg/KxESADk5ETMYPz8REgE5OREzETMzETMxMCUjBiMiAhEQEjMyFzMnJxEzESMlMjY1NTQmIyIGFRQWA5oJc+XX7/DW33cNBwSmh/6eqpmbqpKbmpOnASYBDwEPASyiT00Bvvnsd7nOI+nH48/S1gACAHP/7AQSBFwAEwAaADtAHxgKFwsDAxEKAxwbFwtGWRcXAAYGFEZZBhAADkZZABYAPysAGD8rERIAORgvKxESARc5ETMzETMxMAUiABEQADMyEhUVIRYWMzI3FQYGAyIGByE0JgJ/8/7nAQXczvD9DQW5qLGtWJ2chJ0OAj2MFAEoAQcBCQE4/vHeacHISpQmIQPlrJidpwAAAQAdAAADDgYfABQAOUAdFAwMEwICBwMFAxUWCg9GWQoAAQUHBUZZEwcPAxUAPz8zKxEAMxg/KxESATk5ETMzETMzEjkxMAEhESMRIzU3NRAhMhcHJiMiBhUVIQKe/ummxMQBYVd1K2BEXloBFwPH/DkDx0s8PQGUI4UffYpHAAADACf+FAQxBFwAKgA3AEEAbkA+Kxk4JQwfPQUxEwETBQIqIhwfJRkKQkMcDzUPNUZZCDtHWQoiCCoPCA8IFioqAkdZKg8oP0dZKBAWLkdZFhsAPysAGD8rABg/KxESADk5GC8vERI5OSsrERIAORESARc5ETMRMxEzETMRMzEwARUHFhYVFAYjIicGFRQWMzMyFhUUBCEiJjU0NjcmJjU0NjcmJjU0NjMyFwEUFjMyNjU0JiMjIgYTFBYzMjU0IyIGBDHLHCzcwDErakpawrK//tz+6NfpgHQqOUBFVWvYxlZF/hGWjNHJbpjHcX5agnTz9nV+BEhpGCNxR6HACDhVLSuWj7a/oJJkkhoTUDU8WiojqGy0wxT7AFlcfWtZRWwDPHN27Pd+AAEAsAAABEQGFAAWADNAGQ4MCAgJABYJFhcYDgkSEgRGWRIQCgAACRUAPzM/PysREgA5ERIBOTkRMxEzETMzMTAhETQmIyIGFREjETMRFAczNjYzMhYVEQOeeoKtn6amCAoxtXTJyQLFhoS81v3DBhT+KVU4T1u/0P01AAACAKIAAAFmBd8AAwAPACNAEQoAAAQBARARDQdIWQ0CDwEVAD8/zisREgE5ETMzETMxMCEjETMDNDYzMhYVFAYjIiYBVqamtDgqKDo6KCo4BEgBKTk1Njg4NzcAAAL/kf4UAWYF3wAMABgALEAWEwsLDQgIGRoWEEhZFkAJDwAFRlkAGwA/KwAYPxrOKxESATkRMzMRMzEwEyInNRYzMjY1ETMREAM0NjMyFhUUBiMiJitfO0VDTkmmtDgqKDo6KCo4/hQZhxRVVwT8+xD+vAddOTU2ODg3NwABALAAAAQdBhQAEAA2QBsQDgoKCwsIBgQFCAQREgwAABAQCAgDBwsVAw8APz8zEjkvOREzPxESARc5ETkRMxEzMzEwATY3ATMBASMBBxEjETMRFAcBVCtYAWLF/kQB28n+fX2kpAgCMT1jAXf+Lf2LAgZs/mYGFPzHN3MAAQCwAAABVgYUAAMAFkAJAAEBBAUCAAEVAD8/ERIBOREzMTAhIxEzAVampgYUAAEAsAAABssEXAAjAEZAIxURERIICQAjCRIjAyQlHBYVFRIZBA0ZDUZZHxkQEw8JABIVAD8zMz8/MysRADMREjkYLzMzERIBFzkRMxEzETMRMzEwIRE0JiMiBhURIxE0JiMiBhURIxEzFzM2NjMgFzM2NjMyFhURBiVwdpuUpnB3nJGmhxsIL6tqAQFPCDG6d7q5AsmDg7K5/ZwCyYODu9X9wQRIllBaulZkv9L9NQAAAQCwAAAERARcABQAMUAYABQMCAgJFAkWFQwJEBAERlkQEAoPAAkVAD8zPz8rERIAORESATk5ETMRMxEzMTAhETQmIyIGFREjETMXMzY2MzIWFREDnnqCrKCmhxsIM7hxxsgCxYaEutb9wQRIllFZv9L9NQACAHP/7ARiBFwADAAYAChAFBMADQcABxoZChZGWQoQAxBGWQMWAD8rABg/KxESATk5ETMRMzEwARAAIyImAjUQADMyAAEUFjMyNjU0JiMiBgRi/vLuk+R8AQzu5gEP/L2oo6OpqaWjpgIl/vT+04oBAq0BDAEr/s7++9Lc29PR2dYAAgCw/hQEdQRcABQAIQA/QCAZCwQHBwgfEggSIiMECwAPDxVGWQ8QCQ8IGwAcRlkAFgA/KwAYPz8/KxESADk5ERIBOTkRMxEzETMzMzEwBSImJyMWFREjETMXMzY2MzISERACAyIGBxUUFjMyNjU0JgKua7E8DAymhxcIQKpu2u3x7qiWApqqjqGhFE9SYFb+PQY0llpQ/tb+8/7y/tUD47rLJefH5srN2wACAHP+FAQ3BFwADAAfAERAIgoQHRYDGhoZEBkgIRobFw8dHh4WDRMTB0ZZExANAEZZDRYAPysAGD8rERIAOTkRMxg/PxESATk5ETMRMzMzETMxMCUyNjc1NCYjIgYVFBYXIgIREBIzMhczNzMRIxE0NyMGAk6mmAWcqZKbmX3U7vDW4XkJGIOmCw1zd7LTJebK48/P2YsBKgELAQ0BLqqW+cwB1WRGpwABALAAAAMnBFwAEAAqQBQNCQkKCgIREgsPDQAKFQAFRlkAEAA/KwAYPxI5PxESATk5ETMRMzEwATIXByYjIgYVESMRMxczNjYCpEk6F0Q0hb2miRMIPawEXAyaD9ih/bQESMtrdAABAGr/7ANzBFwAJAA2QBweEwwAABgFEwQlJgweAxYWG0ZZFhAGAwlGWQMWAD8rABgvPysREgA5ORESARc5ETMRMzEwARQGIyInNRYWMzI2NTQmJy4CNTQ2MzIXByYjIgYVFBYWFxYWA3Pkztp6T7VUgoxvoZmBP9q+sak7pYZ2eC1kjsOJASuZpkWaKC5TVUBbPjlVbEuGm0iHREpBLD44NUeQAAEAH//sAqgFRgAWADRAGxAUFAkLCRIDBBgXChMQE0dZDkAQDwcARlkHFgA/KwAYPxrNKxEAMxESARc5ETMRMzEwJTI2NxUGBiMgEREjNTc3MxUhFSERFBYCEixSGBtpKv7CnZ1GYAE+/sJedQ0Hfw0RAU8CjFBF6v6B/XtjagAAAQCk/+wEOQRIABQANEAZARMHDAwKEwoVFgwNDRAIFA8QBEZZEBYLFQA/PysAGD8zEjkRMxESATk5ETMRMxEzMTABERQWMzI2NREzESMnIwYGIyImNREBTHqCrJ+miRgJM7V0yMcESP05hoS81QJA+7iTUVa+0QLNAAABAAAAAAQCBEgACwAYQAoBCgwNBQkBDwAVAD8/MzkREgE5OTEwIQEzExYXMzYSEzMBAaD+YLLsUA4IC3XMsv5gBEj9duRENQFNAjD7uAABABcAAAYjBEgAHAAsQBQJGx0eFxYODQMEDQQIGhIJDwAIFQA/Mz8zMxI5OREzETMzMxESATk5MTAhAyYnIwYHAyMBMxISFzM2NjcTMxMWFzM2NhMzAQQvyRM0CCgez8D+1a5qbwgICzESybTEOBQIBCO/rP7RAoM70a9f/X8ESP5j/lBLObU1AnX9i6x1JJYC3Pu4AAABACcAAAQIBEgACwAiQBEHBQYAAQUMDQkDAQgLFQQBDwA/Mz8zEjk5ERIBFzkxMAEBMwEBMwEBIwEBIwG4/oO9ASEBILv+gwGRvP7N/sq8AjECF/5cAaT96f3PAbz+RAABAAL+FAQGBEgAFQAkQBIJDwADFhcEDQANEkZZDRsIAA8APzI/KxESADkREgEXOTEwEzMTFhczNjYTMwEGBiMiJzUWMzI3NwKy8E8TCA1T5rL+KUa7iExKN0SrST0ESP2P1l8z9wJ8+yC5mxGFDMCcAAABAFIAAANtBEgACQArQBcIAQMHAAcEAQQKCwUER1kFDwEIR1kBFQA/KwAYPysREgEXOREzETMxMCEhNQEhNSEVASEDbfzlAlb9zwLn/bICXXEDVoGB/LoAAQA9/rwCwQW2ABwALEAVGRoaCxcAAA8HFAMDBwsDHR4TAwQnAD8/ERIBFzkRMxEzMxEzETMRMzEwJRQWFxUmJjURNCYjNTY2NRE0NjMVBhURFAcVFhUB23VxvtB+eIJ02Lbm398MZlwCjAKqmgEvaFmNAlxgATKbrIsGwf7Z1ycMJ9cAAQHu/hACewYUAAMAFkAJAgMDBAUDGwAAAD8/ERIBOREzMTABMxEjAe6NjQYU9/wAAQBI/rwCywW2AB0ALEAVFQUKEhICGQAdHQ4OGQUDHh8VJwYDAD8/ERIBFzkRMxEzETMzETMRMzEwASY1ETQnNTIWFREUFhcVIgYVERQGBzU2NjURNDY3Agrf47jTdoJ6fs2+b3RucQI/J9cBJ8EGi66Z/s5hWwKNWWj+0ZmrAowCXGYBKXJ4FAAAAQBoAlAEKQNUABcAJEARAw8YGRIMUFkDEg8GBgBQWQYALysAEBjEL8QrERIBOTkxMAEiBgc1NjMyFhcWFjMyNjcVBiMiJicmJgFSNX82ZJBEcVlCYi82gDZmjkh+SEtaAslDNpdtHCYcG0A5lm4hICAYAAACAJj+iwGJBF4AAwAOACtAFAIEBAMJCQ8QAAADDAwGT1kMEAMiAD8/KxESADkYLxESATkRMzMRMzEwEzMTIxMUIyImNTQ2MzIW22kzz+F5PDw/OTNGAqz73wVMh0dAP0hAAAEAvv/sA9sFywAbAD5AHhYIDQMDCgQAEBAECAMcHRkFAhMKDQINAg0ECwcEGQA/PxI5OS8vETMzETMzERIBFzkRMxEzMxEzETMxMCUGBxUjNSYCNRAlNTMVFhYXByYjIgYVFBYzMjcDy2mThcvBAYyHS44xMYVtrKKfp42O8DYGyM4gARH6Afw+rKQDIReMM9PZ1Ms7AAEAPwAABEQFyQAdAEhAJhgTCQ0NGhYRAgsWEwUeHwwYGRhOWQkZGRMAExBMWRMYAAVLWQAHAD8rABg/KxESADkYLzMrEQAzERIBFzkRMzMRMxEzMTABMhcHJiMiBhURIRUhFRQGByEVITU2NTUjNTMRNDYCqr6qPZqPe30Bpv5aQUoDG/v7zcbG4AXJVIVNfIz+2X/dZIgsmo0v9N9/ATyyzQAAAgB7AQYEFwSgABsAJwAgQA0cACIOAA4oKR8VFSUHAC8zMy8zERIBOTkRMxEzMTATNDcnNxc2MzIXNxcHFhUUBxcHJwYjIicHJzcmNxQWMzI2NTQmIyIGuEqHXodogn9miV+GSkqDXIlmf4Zkh1yFSoGddHSeoHJ0nQLTemuMXIVJSYVcinF2g2eHXIVHSYVciGt8cKCfcXKipAAAAQAfAAAEcQW2ABYAVkAuEg4HCwsQDAUJAgkDDBQOFQcXGAoODgcPBhISAwATFQ8THxMCDxMPEwwBFQYMGAA/PzMSOTkvL10REjkyMhEzETMzETMREgEXOREzETMzETMRMzEwAQEzASEVIRUhFSERIxEhNSE1ITUhATMCSAF7rv5gAQb+wwE9/sOk/sQBPP7EAQD+ZbIC3wLX/P5/qn/+9AEMf6p/AwIAAgHu/hACewYUAAMABwAkQBACBgYDBwcICQQDBAMHGwAAAD8/OTkvLxESATkRMzMRMzEwATMRIxEzESMB7o2NjY0GFPz4/g389wACAHv/+AOWBh0AMQA9AENAJjIAEwYqHjgZGR4MBgAjBj4/FQM7NhwtBiEJISdHWSEVCRBHWQkAAD8rABg/KxESABc5ERIBFzkRMxEzETMRMzEwEzQ2NyYmNTQ2MzIWFwcmJiMiBhUUFhcWFhUUBgcWFRQGIyInNRYWMzI2NTQmJicuAjcUFhcXNjU0JicGBotWTkpUz8Ven2E1YodMdHR7mrqWUkqZ6tTagE7CUoaNMGxzjoZCkoSnMYmTuURVAylWiSUob1V5ix0ngycbO0A8VDdEl2tajSlRkoyZQZQlLUxHLjo6KzRacmJNaT0TUG9TcDkTZAACATUFDgNoBdMACwAXAB5ADAYADBIAEhgZDwMVCQAvM80yERIBOTkRMxEzMTABNDYzMhYVFAYjIiYlNDYzMhYVFAYjIiYBNTUlJjc3JiU1AX01JSU3NyUlNQVxNC4uNDIxMTI0Li40MjExAAADAGT/7AZEBcsAFgAmADYARkAnJxcDDy8fHxQJDxcFNzgGDAASDwwfDAIAEhASAgwSDBIbKyMTMxsEAD8zPzMSOTkvL11dETMRMxESARc5ETMRMxEzMTABIgYVFBYzMjcVBgYjIiY1NDYzMhcHJgE0EiQzMgQSFRQCBCMiJAI3FBIEMzIkEjU0AiQjIgQCA319h3+DVn0wZUbC0N2/gHY6bPyXyAFeysgBXsrC/qLQz/6iw2muAS2srgEqr67+17Cu/tavBCOumqiiLXwUHPHY0fY8djP+uMgBXsrI/qLKxf6m0M8BWsat/tOtrgEpsK4BKq+u/tcAAAIARgMUAnEFxwAWAB8AN0AcFwYbCgEBFhYQBgMgIRwKChIZFgADEAMCAw0SHwA/M9RdxDMSOS8zERIBFzkRMxEzMxEzMTABJwYjIiY1NDY3NzU0IyIHJzYzMhYVESUUMzI1NQcGBgIUGFyMX2+apXWUZGgrcoWCif5QcMlicGcDIVRhY2ZmaQYEJ4UzYDhpef48vGS0MQQEOQACAFIAdQOqA74ABgANAClAEwMGCg0CBAsJCQQNBgQODwwFCAEALzMvMxESARc5ETMRMxEzETMxMBMBFwEBBwElARcBAQcBUgFWd/7fASF3/qoBiwFYdf7hAR91/qgCJwGXRf6i/qFHAZcbAZdF/qL+oUcBlwAAAQBoAQgEKQMXAAUAG0AMAgEEAQYHBQRQWQUCAC8vKxESATk5ETMxMAERIxEhNQQpifzIAxf98QGFigD//wBUAdkCPwJxAgYAEAAAAAQAZP/sBkQFywAIABYAJgA2AF1AMycXABEREgQJLx8fDQkMEhcGNzgMEBAAAA4TDhIIEw8SHxICABMQEwISExITGysjEzMbBAA/Mz8zEjk5Ly9dXREzETMREjkvMxEzERIBFzkRMxEzETMRMxEzMTABMzI2NTQmIyMFFAYHEyMDIxEjESEyFgE0EiQzMgQSFRQCBCMiJAI3FBIEMzIkEjU0AiQjIgQCAtNsUGFWXWoBslVN7qjPh5QBBaab+9/IAV7KyAFeysL+otDP/qLDaa4BLayuASqvrv7XsK7+1q8C+lNAS0GIUHse/nUBYv6eA3uC/sXIAV7KyP6iysX+ptDPAVrGrf7Tra4BKbCuASqvrv7XAAH/+gYUBAYGkwADABG1AAUBBAECAC8zEQEzETMxMAEhNSEEBvv0BAwGFH8AAgB/A1wC7gXLAAwAGAAhQA4NABMGAAYZGhAKwBYDBAA/MxrMMhESATk5ETMRMzEwEzQ2MzIWFRQGBiMiJjcUFjMyNjU0JiMiBn+1goK2UpJUgrVzdVFQc3FSU3MEk4K2tYNUj1S0g1JycVNUcXIA//8AaAABBCkEwwImAA4AAAAHAisAAP10AAEAMQJKAo0FyQAYACNAEQcTFwEBDhMABBoZChAfFwEgAD8zPzMREgEXOREzETMxMAEhNTc+AjU0JiMiBgcnNjMyFhUUBgcHIQKN/aTsWVIhUD80YkVCg5iEk1mTrgG4Akpo5lZhTDZERSYyWG+CcFCXiqUAAQAhAjkCjQXJACMAOUAiDwUFAAMSHgoGJCUSXRNtEwJMEwELExsTAhMTCBohHw0IIQA/Mz8zEjkvXV1dMxESARc5ETMxMAEUBgcWFRQGIyInNRYzMjU0IyM1MzI2NTQmIyIGByc2NjMyFgJzUkSwuKiYdJN70+d1d2djUENCcDhFP4xeiJ0E51BnFy+igI84e0SikWtPRD1EKyNaLTZ3AAEBiQTZAxIGIQAJABO2CQQKCwSACQAvGs0REgE5OTEwATY2NzMVBgYHIwGJMG8gyiyuQG8E8j6wQRVBvjQAAQCw/hQERARIABYANUAaBQoKCBAAExMUCBQYFwYVDxQbDQJGWQ0WCRUAPz8rABg/PzMREgE5OREzETMzETMRMzEwARAzMjY1ETMRIycjBiMiJyMWFREjETMBVv6rn6aIGgpv5ZZYCgqmpgF9/vq91AJA+7iTp1xUoP7ABjQAAQBx/vwEYAYUAA8AJ0ASBAUBAAAFCwMQEQgIBQMPBQEFAC8zPzMSOS8REgEXOREzETMxMAEjESMRIxEGIyImNRA2MyEEYHLVcz5U2Mva6AIt/vwGsPlQAzMS+vsBBP4AAQCYAkwBiQNaAAsAF0AKBgAADQwDCU9ZAwAvKxESATkRMzEwEzQ2MzIWFRQGIyImmD44OkFCOTNDAtNCRUVCQUY/AAABACX+FAG0AAAAEgAkQBARDgsAAA4FAxMUDhERCAMQAC/MMjkvMxESARc5ETMRMzEwARQGIyInNRYzMjY1NCYnNzMHFgG0mZYzLS07T1FPbVhuN7T+32FqCWoIKDYrNRGycycAAQBMAkoB4QW2AAoAIEAOAgADAwoMCwkJAyAGAB4APzI/OS8REgE5OREzMzEwATMRIxE0NwYGBycBUo+FBhY2h0MFtvyUAkNbWhYtX2AAAgBCAxQCvgXHAAsAFwAlQBIMBhIABgAYGQ8AAxADAgMVCR8APzPEXTIREgE5OREzETMxMAEUBiMiJjU0NjMyFgUUFjMyNjU0JiMiBgK+q5aSqaiXmKX9/ltoaVxcaWdcBG+kt7qho7W2onp6enp7dnYAAgBQAHUDqAO+AAYADQAjQBELCQQCAAMHAgoJBg4PDAUIAQAvMy8zERIBFzkRMxEzMTABAScBATcBBQEnAQE3AQOo/qh1AR/+4XUBWP51/qh1AR/+4XUBWAIM/mlHAV8BXkX+aRv+aUcBXwFeRf5p//8ASwAABdEFtgAnAhcCgwAAACYAe/8AAQcCPAMd/bcACbMDAhIYAD81NQD//wAuAAAF2wW2ACcCFwI/AAAAJgB74gABBwB0A079twAHsgIQGAA/NQD//wAaAAAGIQXJACYAdfkAACcCFwLfAAABBwI8A239twAJswMCKxgAPzU1AAACADP+dwNUBF4AHQAoAEFAIggUHiMBHA8cIxQEKSoAHQEMAx0dESYmIE9ZJhARC0lZESMAPysAGD8rERIAORgvX15dERIBFzkRMxEzETMxMAEVFAYHDgIVFBYzMjY3FwYjIiY1ND4CNzY2NTUTFCMiJjU0NjMyFgJOS2F5PRmEelCWYjvFxr7YI0BZNmVBtHk7PkI3M0YCrDN6lFRqS004ZHEmMIdguqpGaVlSL1h0XR8BK4dFQkBHQP//AAAAAAUQB3MCJgAkAAABBwBD/8IBUgAIswIQBSYAKzX//wAAAAAFEAdzAiYAJAAAAQcAdgCFAVIACLMCGAUmACs1//8AAAAABRAHcwImACQAAAEHAUsAIwFSAAizAh0FJgArNf//AAAAAAUQBy8CJgAkAAABBwFSAAQBUgAIswIYBSYAKzX//wAAAAAFEAclAiYAJAAAAQcAagA3AVIACrQDAiQFJgArNTX//wAAAAAFEAcGAiYAJAAAAAcBUAA5AIEAAv/+AAAGgQW2AA8AEwBOQCwKDg4RAQAIDAEQBQUVBRQJEwYTSVkQA0lZCg1JWRAKEAoBBgMFEgEOSVkBEgA/KwAYPz8SOTkvLysrKxEAMxEBMxESFzkRMzMRMzEwISERIQMjASEVIREhFSERIQEhESMGgf0S/f7jsAK6A8n9vAId/eMCRPtUAb52AdH+LwW2l/4plv3mAdICtQD//wB9/hQEzwXLAiYAJgAAAAcAegICAAD//wDJAAAD+AdzAiYAKAAAAQcAQ/+3AVIACLMBDQUmACs1//8AyQAAA/gHcwImACgAAAEHAHYAPwFSAAizARUFJgArNf//AMkAAAP4B3MCJgAoAAABBwFL//sBUgAIswEaBSYAKzX//wDJAAAD+AclAiYAKAAAAQcAagASAVIACrQCASEFJgArNTX//wA8AAACVgdzAiYALAAAAQcAQ/6zAVIACLMBDQUmACs1//8AVAAAAnMHcwImACwAAAEHAHb/YQFSAAizARUFJgArNf////8AAAKhB3MCJgAsAAABBwFL/vMBUgAIswEaBSYAKzX//wA8AAACbwclAiYALAAAAQcAav8HAVIACrQCASEFJgArNTUAAgAvAAAFSAW2AAwAFwBXQDIRFRUIBA0AABMEBgQYGRQGBwZJWREPBz8HrwfPB98HBQsDBwcECQkQSlkJAwQVSlkEEgA/KwAYPysREgA5GC9fXl0zKxEAMxESARc5ETMRMzMRMzEwARAAISERIzUzESEgAAMQISMRIRUhETMgBUj+d/6P/nuamgGyAVEBfLX9x+cBe/6FvgJiAun+lv6BAomWApf+if6kAkD9/Jb+Cv//AMkAAAU/By8CJgAxAAABBwFSAJMBUgAIswEaBSYAKzX//wB9/+wFvgdzAiYAMgAAAQcAQwB5AVIACLMCGQUmACs1//8Aff/sBb4HcwImADIAAAEHAHYBCgFSAAizAiEFJgArNf//AH3/7AW+B3MCJgAyAAABBwFLALQBUgAIswImBSYAKzX//wB9/+wFvgcvAiYAMgAAAQcBUgCaAVIACLMCIQUmACs1//8Aff/sBb4HJQImADIAAAEHAGoA1QFSAAq0AwItBSYAKzU1AAEAhQEQBAwEmAALABlACQcJAwEJAQwNCAAZLxESATk5ETMRMzEwARcBAQcBAScBATcBA6xg/qABXmD+nv6kZQFe/qBkAWEEmGP+nv6gYwFf/qFjAWABYGX+nQAAAwB9/8MFvgX2ABMAGwAjAE5ALBYfFx4EHBQcChQAABIPBQgKBiQlFh4hGQ0hSVkPEggFBAMQDQQDGUlZBgMTAD/GKwAYP8YSFzkrERIAOTkREgEXOREzETMREhc5MTABEAAhIicHJzcmERAAITIXNxcHFgMQJwEWMzISARAXASYjIgIFvv6d/sTrlGV4bLIBYAFE0Z1heGrAtG79YHOw8/j8J2UCnWqo8/0C3f6h/m5kjU+axgFtAWUBiV6HUJTK/pUBEJr8TFIBMgEq/vqaA69J/s0A//8Auv/sBRkHcwImADgAAAEHAEMARgFSAAizARMFJgArNf//ALr/7AUZB3MCJgA4AAABBwB2AM8BUgAIswEbBSYAKzX//wC6/+wFGQdzAiYAOAAAAQcBSwB9AVIACLMBIAUmACs1//8Auv/sBRkHJQImADgAAAEHAGoAmAFSAAq0AgEnBSYAKzU1//8AAAAABHsHcwImADwAAAEHAHYAMQFSAAizARIFJgArNQACAMkAAAR5BbYADAAVADZAHA0JBQUGEQAGABYXDQRKWQkVSlkNCQ0JBgcDBhIAPz8SOTkvLysrERIBOTkRMxEzETMzMTABFAQhIxEjETMRMyAEATMyNjU0JiMjBHn+0f7huKqq1wEZARb8+qjiyr7KzAMQ4+7+wQW2/wDP/eqPpJWKAAABALD/7AScBh8AMABBQCIpKgUdIwAXDAwAHREqBTEyEhIqLi4mRlkuACoVDxVGWQ8WAD8rABg/PysREgA5GC8REgEXOREzETMRMxEzMTABFAcGBhUUFhYXFhYVFAYjIic1FhYzMjU0JicmJjU0Njc2NjU0JiMgFREjETQ2MzIWBBmPWDgbR06MZsKzvGs/nEjXU25/YEVHS0CIf/7sptzezuEE8odzRkMhICo5M1+dZaCrRZonL7ZLa0ZSe1Q/ajU5WjVQVd/7TASysrud//8AXv/sA80GIQImAEQAAAEGAEOOAAAIswImESYAKzX//wBe/+wDzQYhAiYARAAAAQYAdisAAAizAi4RJgArNf//AF7/7APNBiECJgBEAAABBgFL2AAACLMCMxEmACs1//8AXv/sA80F3QImAEQAAAEGAVK9AAAIswIuESYAKzX//wBe/+wDzQXTAiYARAAAAQYAauIAAAq0AwI6ESYAKzU1//8AXv/sA80GhQImAEQAAAEGAVD3AAAKtAMCKBEmACs1NQADAF7/7AZzBFwAKQA0ADsAYUAzKgAkETA4GRkEMDkYGB8wCwAFPD0bLSctRlkZMQQxR1k4JCcRBAQOIicWNQgOCEZZFA4QAD8zKxEAMxg/MxI5LzkSOTMrEQAzKxEAMxESARc5ETMRMzMRMxI5OREzMTATNDY3NzU0JiMiByc2NjMyFhc2NjMyEhUVIRIhMjY3FQYGIyAnBgYjIiY3FBYzMjY1NQcGBgEiBgchNCZe+P64dHeQozRKx2KCpSk1q27A6P1DCAE6W51UVpVl/t99UcWGo7mua1iRqJ66pAO9eYsLAgeAAS+hswgGRIF7VH8pNVdfWGD+9d5r/nUjJ5QmIel/aqqXX1mpmmMHCG0CMqaenKgA//8Ac/4UA4sEXAImAEYAAAAHAHoBRgAA//8Ac//sBBIGIQImAEgAAAEGAEO1AAAIswIcESYAKzX//wBz/+wEEgYhAiYASAAAAQYAdk4AAAizAiQRJgArNf//AHP/7AQSBiECJgBIAAABBgFL9wAACLMCKREmACs1//8Ac//sBBIF0wImAEgAAAEGAGoKAAAKtAMCMBEmACs1Nf///9oAAAFjBiECJgDzAAABBwBD/lEAAAAIswEFESYAKzX//wCpAAACMgYhAiYA8wAAAQcAdv8gAAAACLMBDREmACs1////swAAAlUGIQImAPMAAAEHAUv+pwAAAAizARIRJgArNf///+wAAAIfBdMCJgDzAAABBwBq/rcAAAAKtAIBGREmACs1NQACAHH/7ARiBiEAGwAmAEpAKyEGDBwcAAAYGRYOERMQBgknKAkfRlkLAxYRGQ4PBRQJCQMXFAEDJEZZAxYAPysAGD8zEjkvEhc5EjkrERIBFzkRMxEzETMxMAEQACMiADU0ADMyFzcmJwUnNyYnNxYXNxcHFhIDNCYjIBEUFjMyNgRi/vv33v7pAQfc4mQIOc3+8UnpXF5FnGbuTM+Ypai0nP6vr6KvoQIz/uf+0gEN4uYBBnkE1r+bbIU+MXVJS4prd4/+cv7ok6r+mKe3yQD//wCwAAAERAXdAiYAUQAAAQYBUg4AAAizAR4RJgArNf//AHP/7ARiBiECJgBSAAABBgBD1AAACLMCGhEmACs1//8Ac//sBGIGIQImAFIAAAEGAHZWAAAIswIiESYAKzX//wBz/+wEYgYhAiYAUgAAAQYBSw4AAAizAicRJgArNf//AHP/7ARiBd0CJgBSAAABBgFS8QAACLMCIhEmACs1//8Ac//sBGIF0wImAFIAAAEGAGobAAAKtAMCLhEmACs1NQADAGgA/AQpBKgAAwAPABsAM0AYFgoKEAQCBAEDHB0ZExMBBw0NAQEAUFkBAC8rEQAzGC8zETMvMxESARc5ETMzETMxMBM1IRUBNDYzMhYVFAYjIiYRNDYzMhYVFAYjIiZoA8H9rjs2NDo7MzQ9OzY0OjszND0CjYqK/ug8PT86OUA/AvQ8PT86OUA/AAMAc/+8BGIEhwATABsAIwBLQCkXHxwUFAocAAASDwUICgYkJRYeIRkNGUZZDxIIBQQDEA0QAyFGWQYDFgA/xisAGD/GEhc5KxESADk5ERIBFzkRMxEzERI5OTEwARAAIyInByc3JhEQADMyFzcXBxYFFBcBJiMiBgU0JwEWMzI2BGL+8u6acFRyXoEBDO6adFR1YX/8vTUB0Utyo6YClzP+L0dxo6kCJf70/tNFdU6DmAEAAQwBK0x3TIWY+atmAoY11tSkZP19M9sA//8ApP/sBDkGIQImAFgAAAEGAEPEAAAIswEWESYAKzX//wCk/+wEOQYhAiYAWAAAAQYAdnEAAAizAR4RJgArNf//AKT/7AQ5BiECJgBYAAABBgFLEgAACLMBIxEmACs1//8ApP/sBDkF0wImAFgAAAEGAGohAAAKtAIBKhEmACs1Nf//AAL+FAQGBiECJgBcAAABBgB2EgAACLMBHxEmACs1AAIAsP4UBHUGFAAWACIAPkAfIAYbFBAQEQYRJCMSABEbDBYJAwkeRlkJFgMXRlkDEAA/KwAYPysREgA5ORg/PxESATk5ETMRMzMRMzEwATY2MzISERACIyInIxcWFREjETMRFAclIgYHFRQWMyARNCYBWEKqatfw8dbeegwECKamBgFIqJgCmqoBL5QDtFlP/tT+9f70/tOhIk0//jUIAP4uNFobuMkp58cBsNfR//8AAv4UBAYF0wImAFwAAAEGAGq1AAAKtAIBKxEmACs1Nf//AAAAAAUQBrQCJgAkAAABBwFNAD8BUgAIswISBSYAKzX//wBe/+wDzQViAiYARAAAAQYBTfUAAAizAigRJgArNf//AAAAAAUQBzcCJgAkAAABBwFOACsBUgAIswIPBSYAKzX//wBe/+wDzQXlAiYARAAAAQYBTuQAAAizAiURJgArNf//AAD+QgURBbwCJgAkAAAABwFRA6AAAP//AF7+QgQABFoCJgBEAAAABwFRAo8AAP//AH3/7ATPB3MCJgAmAAABBwB2AQgBUgAIswEgBSYAKzX//wBz/+wDiwYhAiYARgAAAQYAdkQAAAizASARJgArNf//AH3/7ATPB3MCJgAmAAABBwFLAKwBUgAIswElBSYAKzX//wBz/+wDiwYhAiYARgAAAQYBS9QAAAizASURJgArNf//AH3/7ATPBzECJgAmAAABBwFPAhsBUgAIswEgBSYAKzX//wBz/+wDiwXfAiYARgAAAQcBTwFQAAAACLMBIBEmACs1//8Aff/sBM8HcwImACYAAAEHAUwAwQFSAAizASIFJgArNf//AHP/7AOhBiECJgBGAAABBgFM8wAACLMBIhEmACs1//8AyQAABVgHcwImACcAAAEHAUwAWAFSAAizAh0FJgArNf//AHP/7AWBBhQCJgBHAAABBwI4AwwAAAAHsgIjAAA/NQD//wAvAAAFSAW2AgYAkgAAAAIAc//sBNMGFAAaACcAZEA3JQYSDgAeHhUZFhkQBgQoKRoVGBAREEdZFQ8RHxEvEQMJAxERCRMAAQwDCQkiRlkJEAMbRlkDFgA/KwAYPysREgA5ORg/EjkvX15dMysRADMYPxESARc5ETMzETMzMxEzMTAlIwYjIgIREBIzMhczJjU1ITUhNTMVMxUjESMlMjY1NTQmIyIGFRQWA5oJc+XX7/DW33cNC/5AAcCmnJyH/p6qmZuqkpuak6cBJgEPAQ8BLKJTSYWBuLiB+yV3uc4j6cfjz9LW//8AyQAAA/gGtAImACgAAAEHAU0AEgFSAAizAQ8FJgArNf//AHP/7AQSBWICJgBIAAABBgFNCgAACLMCHhEmACs1//8AyQAAA/gHNwImACgAAAEHAU4AEAFSAAizAQwFJgArNf//AHP/7AQSBeUCJgBIAAABBgFO+wAACLMCGxEmACs1//8AyQAAA/gHFAImACgAAAEHAU8BbwE1AAizARUFJgArNf//AHP/7AQSBd8CJgBIAAABBwFPAVQAAAAIswIkESYAKzX//wDJ/kID+AW2AiYAKAAAAAcBUQJzAAD//wBz/mEEEgRcAiYASAAAAAcBUQJmAB///wDJAAAD+AdzAiYAKAAAAQcBTAAQAVIACLMBFwUmACs1//8Ac//sBBIGIQImAEgAAAEGAUz7AAAIswImESYAKzX//wB9/+wFPQdzAiYAKgAAAQcBSwDpAVIACLMBKgUmACs1//8AJ/4UBDEGIQImAEoAAAEGAUvKAAAIswNQESYAKzX//wB9/+wFPQc3AiYAKgAAAQcBTgEAAVIACLMBHAUmACs1//8AJ/4UBDEF5QImAEoAAAEGAU7OAAAIswNCESYAKzX//wB9/+wFPQcxAiYAKgAAAQcBTwJkAVIACLMBJQUmACs1//8AJ/4UBDEF3wImAEoAAAEHAU8BHwAAAAizA0sRJgArNf//AH3+OwU9BcsCJgAqAAAABwI5AScAAP//ACf+FAQxBiECJgBKAAABBgI6RAAACLMDRhEmACs1//8AyQAABR8HcwImACsAAAEHAUsAlgFSAAizARoFJgArNf//ALAAAAREB6oCJgBLAAABBwFLAB8BiQAIswElAiYAKzUAAgAAAAAF5wW2ABMAFwBUQCwXAw8PABAUBAwMBwsICxASBBgZFw5JWRYKEhMSSlkHAxMXExcTAQwQEgUBAwA/Mz8zEjk5Ly8RMzMrEQAzMysREgEXOREzMxEzMxEzMxEzMzEwEzUzFSE1MxUzFSMRIxEhESMRIzUBNSEVyaoDAqrIyKr8/qrJBHX8/gS++Pj4+I37zwKw/VAEMY3+iunpAAEAFAAABEQGFAAeAFlAMhYUEAgIDQkAHh4SCQsEHyAXFhoERlkTCwwLR1kQDA8MHwwvDAMWGgwMGhYDCQ4AAAkVAD8zPxIXOS8vL10RMysRADMrEQAzERIBFzkRMxEzMxEzMzMxMCERNCYjIgYVESMRIzUzNTMVIRUhFRQHMzY2MzIWFREDnnqCrp6mnJymAcH+PwgKMbV0yckCnoaEutX95wTbf7q6f8RUOE9bv9L9XP///+IAAALKBy8CJgAsAAABBwFS/toBUgAIswEVBSYAKzX///+QAAACeAXdAiYA8wAAAQcBUv6IAAAACLMBDREmACs1//8AKgAAAoIGtAImACwAAAEHAU3+/QFSAAizAQ8FJgArNf///9oAAAIyBWICJgDzAAABBwFN/q0AAAAIswEHESYAKzX//wAeAAACigc3AiYALAAAAQcBTv75AVIACLMBDAUmACs1////zAAAAjgF5QImAPMAAAEHAU7+pwAAAAizAQQRJgArNf//AFT+QgJWBbYCJgAsAAAABgFRaAD//wA1/kIBgQXfAiYATAAAAAYBURAA//8AVAAAAlYHMQImACwAAAEHAU8AUAFSAAizARUFJgArNQABALAAAAFWBEgAAwAWQAkAAQEFBAIPARUAPz8REgE5ETMxMCEjETMBVqamBEj//wBU/n8EEAW2ACYALAAAAAcALQKoAAD//wCi/hQDbAXfACYATAAAAAcATQIGAAD///9g/n8CZQdzAiYALQAAAQcBS/63AVIACLMBHAUmACs1////kf4UAk8GIQImAjcAAAEHAUv+oQAAAAizARsRJgArNf//AMn+OwTpBbYCJgAuAAAABwI5AIkAAP//ALD+OwQdBhQCJgBOAAAABgI5KwAAAQCwAAAEGwRGAA0AL0AZDQsHBwgDAQIFCAUODwINBQYECAAJDwQIFQA/Mz8zEhc5ERIBFzkRMxEzMzEwATMBASMBBxEjETMRFAcDL8/+YgG7yf6Xh7KyDARG/h79nAH4cf55BEb+5aZx//8AyQAAA/gHcwImAC8AAAEHAHb/YwFSAAizAQ8FJgArNf//AKMAAAIsB6wCJgBPAAABBwB2/xoBiwAIswENAiYAKzX//wDJ/jsD+AW2AiYALwAAAAYCOTEA//8AWf47AVcGFAImAE8AAAAHAjn+6AAA//8AyQAAA/gFtwImAC8AAAEHAjgBHf+jAAeyAQkDAD81AP//ALAAAAKgBhQCJgBPAAABBgI4KwAAB7IBBwAAPzUA//8AyQAAA/gFtgImAC8AAAAHAU8CBP1n//8AsAAAAqgGFAAmAE8AAAAHAU8BQv04AAEAHQAAA/gFtgANAD1AIQcLCwQADAkAAwQPDgkHBAoDAQYIAggCCAAFAwALSVkAEgA/KwAYPxI5OS8vEhc5ERIBFzkRMzMRMzEwMxEHJzcRMxElFwURIRXJaUOsqgEpQ/6UAoUB/DtyZQMe/UauedP+PJoAAf/8AAACJwYUAAsAN0AcAAQECQUFDAINCAwAAgkDCAYGAQcBBwEFCgAFFQA/PxI5OS8vEhc5EQEzETMSOREzMxEzMTABNxcHESMRByc3ETMBVolI0aZuRrSmA2BecI39PwJUSHF3AyAA//8AyQAABT8HcwImADEAAAEHAHYBAgFSAAizARoFJgArNf//ALAAAAREBiECJgBRAAABBgB2eQAACLMBHhEmACs1//8Ayf47BT8FtgImADEAAAAHAjkAzQAA//8AsP47BEQEXAImAFEAAAAGAjlWAP//AMkAAAU/B3MCJgAxAAABBwFMAKYBUgAIswEcBSYAKzX//wCwAAAERAYhAiYAUQAAAQYBTB8AAAizASARJgArNf//AAEAAATLBbYAJwBRAIcAAAEGAgfoAAAHsgEcAwA/NQAAAQDJ/n8FPwW2ABkAOEAcEA0NDggUFBcXAg4DGhsSCg4VDwMOEgAFSVkAIgA/KwAYPz8zEjk5ERIBFzkRMxEzETMRMzEwASInNRYzMjY1ASMSFREjETMBMyY1ETMRFAYDyWI2R1NpavzACBCdwAMdCA6fwf5/G5EUem8Ey/74nvzbBbb7TpXgAz36WMPMAAEAsP4UBEQEXAAdADhAHhMPDxAHGxsCEAMeHxcLRlkXEBMQEQ8QFQAFRlkAGwA/KwAYPz8SOT8rERIBFzkRMxEzETMxMAEiJzUWMzI1ETQmIyIGFREjETMXMzY2MzIWFREUBgMlVjc8Pox6gqygpocbCjS0bsvHjP4UGYcUrAN5hoS61v3BBEiWUli/0vyNmqr//wB9/+wFvga0AiYAMgAAAQcBTQDHAVIACLMCGwUmACs1//8Ac//sBGIFYgImAFIAAAEGAU0SAAAIswIcESYAKzX//wB9/+wFvgc3AiYAMgAAAQcBTgDBAVIACLMCGAUmACs1//8Ac//sBGIF5QImAFIAAAEGAU4OAAAIswIZESYAKzX//wB9/+wFvgdzAiYAMgAAAQcBUwEUAVIACrQDAisFJgArNTX//wBz/+wEYgYhAiYAUgAAAQYBU1oAAAq0AwIsESYAKzU1AAIAff/sBucFzQAUAB8AU0AuGAYPExMdAA0RHQYFICEPEklZDw8ACwsOSVkLAwkVSVkJBAMbSVkDEgATSVkAEgA/KwAYPysAGD8rABg/KxESADkYLysREgEXOREzETMRMzEwISEGIyAAERAAITIXIRUhESEVIREhASIAERAAMzI3ESYG5/0AZlz+uf6fAVwBQGZaAw79swIn/dkCTfxE+f7/AQH3cFdXFAGJAWoBaAGGF5f+KZb95gSd/s/+2f7X/s0hBHUeAAMAcf/sBx8EWgAeACoAMQBVQC0fCA4CFhYlLxUVHCUIBDIzKygLKEZZLhZGWQIFDgsuLgURCxAYIgUiRlkABRYAPzMrEQAzGD8zEjkvEjkSOSsrEQAzERIBFzkRMxEzEjk5ETMxMAUgJwYGIyIAERAAMzIWFzY2MzISFRUhEiEyNjcVBgYBFBYzMjY1NCYjIgYlIgYHITQmBZb+230+0Ynf/vQBBuuDzT46wH7J7v0nCAFKXqFXWJj7IZino5mbpaaVBEd/kQwCIIQU63R3ATEBCAEJASx3cnB5/vfiaf53IyeUJyACOdPb1dHd1djYpJ6epP//AMkAAATPB3MCJgA1AAABBwB2AHkBUgAIswIfBSYAKzX//wCwAAADJwYhAiYAVQAAAQYAdtwAAAizARoRJgArNf//AMn+OwTPBbYCJgA1AAAABgI5fQD//wBg/jsDJwRcAiYAVQAAAAcCOf7vAAD//wDJAAAEzwdzAiYANQAAAQcBTAAbAVIACLMCIQUmACs1//8AggAAAycGIQImAFUAAAEHAUz/dgAAAAizARwRJgArNf//AGr/7AQCB3MCJgA2AAABBwB2AFABUgAIswEuBSYAKzX//wBq/+wDcwYhAiYAVgAAAQYAduoAAAizAS4RJgArNf//AGr/7AQCB3MCJgA2AAABBwFL/+oBUgAIswEzBSYAKzX//wBq/+wDcwYhAiYAVgAAAQYBS5cAAAizATMRJgArNf//AGr+FAQCBcsCJgA2AAAABwB6AScAAP//AGr+FANzBFwCJgBWAAAABwB6ANUAAP//AGr/7AQCB3MCJgA2AAABBwFM/+QBUgAIswEwBSYAKzX//wBq/+wDcwYhAiYAVgAAAQYBTJkAAAizATARJgArNf//ABL+OwRaBbYCJgA3AAAABgI5GQD//wAf/jsCqAVGAiYAVwAAAAYCOYIA//8AEgAABFoHcwImADcAAAEHAUz/3AFSAAizARMFJgArNf//AB//7ALXBhQCJgBXAAABBgI4YgAAB7IBGgAAPzUAAAEAEgAABFoFtgAPAD9AIQcLCwAMBAkMDgIFEBEKDg8OSlkHDw8DDBIGAgMCSVkDAwA/KxEAMxg/EjkvMysRADMREgEXOREzMxEzMTABESE1IRUhESEVIREjESE1AeH+MQRI/jEBNv7Kqv7HAy8B8JeX/hCN/V4Coo0AAQAf/+wCqAVGABwATEApFxMbGwwIAhUZCAoOBh0eDhYTFkdZGgoLCkdZFwsLBhFAEw8GAEZZBhYAPysAGD8azRI5LzMrEQAzKxEAMxESARc5ETMzETMzMTAlMjcVBgYjIBE1IzUzESM1NzczFSEVIREhFSEVFAIXVTwgair+yI2NnZ1GYAE+/sIBLf7TdRR/DhABXP6BAQBQRer+gf8AgfTdAP//ALr/7AUZBy8CJgA4AAABBwFSAG8BUgAIswEbBSYAKzX//wCk/+wEOQXdAiYAWAAAAQYBUvcAAAizAR4RJgArNf//ALr/7AUZBrQCJgA4AAABBwFNAJEBUgAIswEVBSYAKzX//wCk/+wEOQViAiYAWAAAAQYBTRkAAAizARgRJgArNf//ALr/7AUZBzcCJgA4AAABBwFOAIsBUgAIswESBSYAKzX//wCk/+wEOQXlAiYAWAAAAQYBThIAAAizARURJgArNf//ALr/7AUZB9cCJgA4AAABBwFQAJwBUgAKtAIBFQUmACs1Nf//AKT/7AQ5BoUCJgBYAAABBgFQIwAACrQCARgRJgArNTX//wC6/+wFGQdzAiYAOAAAAQcBUwDhAVIACrQCASUFJgArNTX//wCk/+wEOQYhAiYAWAAAAQYBU2gAAAq0AgEoESYAKzU1//8Auv5CBRkFtgImADgAAAAHAVECIQAA//8ApP5CBGUESAImAFgAAAAHAVEC9AAA//8AGwAAB0wHcwImADoAAAEHAUsBVAFSAAizASgFJgArNf//ABcAAAYjBiECJgBaAAABBwFLAMEAAAAIswErESYAKzX//wAAAAAEewdzAiYAPAAAAQcBS//gAVIACLMBFwUmACs1//8AAv4UBAYGIQImAFwAAAEGAUutAAAIswEkESYAKzX//wAAAAAEewclAiYAPAAAAQcAav/xAVIACrQCAR4FJgArNTX//wBSAAAEPwdzAiYAPQAAAQcAdgBCAVIACLMBEwUmACs1//8AUgAAA20GIQImAF0AAAEGAHboAAAIswETESYAKzX//wBSAAAEPwcxAiYAPQAAAQcBTwFEAVIACLMBEwUmACs1//8AUgAAA20F3wImAF0AAAEHAU8A3wAAAAizARMRJgArNf//AFIAAAQ/B3MCJgA9AAABBwFM/+0BUgAIswEVBSYAKzX//wBSAAADbQYhAiYAXQAAAQYBTIYAAAizARURJgArNQABALAAAALbBh8ADAAdQA4AAQENBg4ECUZZBAABFQA/PysRATMSOREzMTAhIxEQITIXByYjIgYVAVamAWdgZCtXSWFZBJwBgyWFHnt6AAABAMP+FAQXBcsAIABEQCQaHh4MCBIcCAoCBSEiHQoMCkZZGgwMEAAQFkZZEAQABUZZABsAPysAGD8rERIAORgvMysRADMREgEXOREzMxEzMTABIic1FjMyNjURIzU3NTQ2MzIXBwcmIyIGFRUhFSERFAYBSEVARj1fTd7eorZVeBYVZjxiUAEa/uqe/hQTixJmcQPNSzyLw7IrQEEgaXyVgfw3uK8ABAAAAAAFFAeqABAAGAAiAC4AYUA0EQUEGAYUBwQDBwgjACkLCAsJIhQCAB0DCTAvJg4sAgkYBklZCRQOGCIOGBgOIgMIHAQIEgA/My8SFzkvLy8REjk5KxEAMzMRMxESARc5ETMRMxEzETMREjk5ETk5MTABFAcBIwMhAyMBJjU0NjMyFhMDJicGBgcDEzY2NzMVBgYHIxM0JiMiBhUUFjMyNgNoaAIUrrD9nqauAhRqemNkfRuyGS8OMAmxmDFmF8sgqEJv00IzM0I8OTVABZaFOPsnAZH+bwTXNIhlcnX8NgGwOpEwhxj+VASFO5UqEC6hLf71OTw8OTc9PQAFAF7/7APNB6oACQAkAC8AOwBHAGdANy0SQjY8MCkVFQskJAYwADYdEgdISQkJBD85RTMRCwwVKUdZDBUVDyAgGUZZIBAPJUZZDxYKFQQALz8/KwAYPysREgA5GC85KxEAMxg/M8QyETkvERIBFzkRMzMRMxEzETMRMzEwATU2NjchFQYGBwEnIwYGIyImNRAlNzU0JiMiBgcnNjYzMhYVESUyNjU1BwYGFRQWARQGIyImNTQ2MzIWBzQmIyIGFRQWMzI2AdcuahYBBBWkgAECIQhSo3qjuQIZtHeFYKdHN1TQZdHJ/g6bsabGr20BqntmZXl5ZWV8bUEzM0I8OTRABtkQKngfDBhpRPknnGdJqJsBTBAGRIJ6NCB/KzOuwP0UdaqZYwcHbXNaXgU9Ynd0Y2Jzd144PT04OD09AP////4AAAaBB3MCJgCIAAABBwB2AkwBUgAIswIdBSYAKzX//wBe/+wGcwYhAiYAqAAAAQcAdgGFAAAACLMDRREmACs1//8Aff/DBb4HcwImAJoAAAEHAHYBGQFSAAizAy0FJgArNf//AHP/vARiBiECJgC6AAABBgB2VgAACLMDLREmACs1//8Aav47BAIFywImADYAAAAGAjkGAP//AGr+OwNzBFwCJgBWAAAABgI5uQAAAQEMBNkDrgYhAA4AGEAJBwAQDwsEgA4JAC8zGs0yERIBOTkxMAE2NjczFhYXFSMmJwYHIwEMf2YXphZtfXdYhYhTcwTwiIApKoWCFzeDhjQAAAEBDATZA64GIQAOABhACQYAEA8FAYADCwAvMxrNMhESATk5MTABMxYXNjczFQcGByMmJicBDHNyaYJbd0KQLqYXZn8GIUpzgjsZRJRXKX6IAAABAS0E2QOFBWIAAwARtQABBAUAAwAvMxESATk5MTABIRUhAS0CWP2oBWKJAAEBJQTZA5EF5QAOABhACQwDEA8LBIAIAAAvMhrMMhESATk5MTABIiYnMx4CMzI2NzMGBgJWjJwJaAYpSVVlYApoCqcE2YmDMTgaQEN+jgAAAQCiBQIBZgXfAAsAE7YGAAAMDQMJAC/NERIBOREzMTATNDYzMhYVFAYjIiaiOCooOjooKjgFcTk1Njg4NzcAAAIBbwTZAy0GhQALABcAHkAMEgYMAAYAGBkPCRUDAC8zzDIREgE5OREzETMxMAEUBiMiJjU0NjMyFgc0JiMiBhUUFjMyNgMte2ZleHlkZXxsQjMzQjw5NEEFsmJ3dWJic3deOD09ODg9PQABACX+QgFxAAAADwAYQAoACQQNCQMQEQIHAC8zERIBFzkRMzEwFxQzMjcVBiMiNTQ2NzMGBrJeKjdBPM9WSHhERe5eDW0SvEaHNUJtAAABAQgE2QPwBd0AFwAkQA8JFRgZEQAFDAAMAAwVgAkALxrMOTkvLxEzETMREgE5OTEwASIuAiMiBgcjNjYzMh4CMzI2NzMGBgMUK1JPSSIyMw5iDXNbLlZOSCAxMA9jDXEE2yUtJTw9eYklLSU7PnmJAAACAOcE2QO2BiEACQATABtADA4FEwkEFBUNBIATCQAvMxrNMhESARc5MTATNjY3MxUGBgcjJTY2NzMVBgYHI+ckbh+6Jas6YQFlMWUauiWrOmAE8jC6RRU/xDAZRLE6FT/EMAAAAQH8BNkDEAZzAAkAE7YEAAsKBIAJAC8azRESATk5MTABNjY3MxUGBgcjAfwbNQy4Em0xZAT2SONSF0rtTAADARsFDgODBrQACAAUACAAK0AUDwkVGxsDCAkEISIYDAgMCAwDHhIALzPMOTkvLxEzERIBFzkRMxEzMTABNjczFQYGByMnNDYzMhYVFAYjIiYlNDYzMhYVFAYjIiYCAEEfvSF5M1DlNCYpMTcjJjQBtDQmKTE3IyY0BYWphhRDsz0ENC40LjIxMTI0LjQuMjEx//8AAAAABRAGCgImACQAAAEHAVT+IP+XAAeyAhIAAD81AP//AJgCTAGJA1oCBgB5AAD////UAAAEdQYKACYAKH0AAQcBVP3Y/5cAB7IBEAAAPzUA////1AAABbUGCgAnACsAlgAAAQcBVP3Y/5cAB7IBEAAAPzUA////5AAAA0QGCgAnACwA7gAAAQcBVP3o/5cAB7IBEAAAPzUA////5P/sBgIGCgAmADJEAAEHAVT96P+XAAeyAhwAAD81AP///9QAAAWFBgoAJwA8AQoAAAEHAVT92P+XAAeyAQ0AAD81AP///+QAAAYzBgoAJgF2PwABBwFU/ej/lwAHsgEjAAA/NQD////p/+wCkwa0AiYBhgAAAQcBVf7OAAAADLUDAgEuESYAKzU1Nf//AAAAAAUQBbwCBgAkAAD//wDJAAAEvgW2AgYAJQAAAAEAyQAAA/gFtgAFAB1ADgMEBAAGBwUCSVkFAwQSAD8/KxESATk5ETMxMAEVIREjEQP4/XuqBbaZ+uMFtgD//wAnAAAEbQW2AgYCKAAA//8AyQAAA/gFtgIGACgAAP//AFIAAAQ/BbYCBgA9AAD//wDJAAAFHwW2AgYAKwAAAAMAff/sBb4FzQADAA8AGwA/QCACAxAWEAoWBAoEHB0AA0lZAAAHDQ0ZSVkNBAcTSVkHEwA/KwAYPysREgA5GC8rERIBOTkRMxEzERI5OTEwASEVISUQACEgABEQACEgAAEQEjMyEhEQAiMiAgHjAnX9iwPb/p3+xP69/qEBYAFEATsBYvtz+vTz+Pfy9fsDM5U//qH+bgGLAWgBZQGJ/nD+oP7Y/swBMAEsASoBLv7OAP//AFQAAAJWBbYCBgAsAAD//wDJAAAE6QW2AgYALgAAAAEAAAAABNMFtgAKABpACwgADAsECAkDAQgSAD8zPxI5ERIBOTkxMCEjASYnBgcBIwEzBNO2/rZXFiFH/ri2AhCxA6D8WovJ/F4Ftv//AMkAAAZxBbYCBgAwAAD//wDJAAAFPwW2AgYAMQAAAAMASAAABCUFtgADAAcACwA0QB0KBwMCBggGDQwAA0lZAAAKBAoLSVkKEgQHSVkEAwA/KwAYPysREgA5GC8rERIBFzkxMBMhFSEDIRUhARUhNcMC5/0ZUgOL/HUDtPwjA0iWAwSX+3mYmP//AH3/7AW+Bc0CBgAyAAAAAQDJAAAFDAW2AAcAI0ARAQAEBQAFCQgGA0lZBgMBBRIAPzM/KxESATk5ETMRMzEwISMRIREjESEFDKr9EaoEQwUf+uEFtgD//wDJAAAEaAW2AgYAMwAAAAEASgAABFwFtgAMADVAHAgKCgAJAgsGAwIABQ0OBwgECElZBAMACklZABIAPysAGD8rEQAzERIBFzkRMxEzETMxMDM1AQE1IRUhJwEBIRVKAeH+KwPL/VxgAcz+HwNUjQJvAiuPmQL93/2amAD//wASAAAEWgW2AgYANwAA//8AAAAABHsFtgIGADwAAAADAGr/7AX4BcsAGQAiACsAUEApJxQaAg0NKxkOHgcHDhQDLC0MEBoqECpKWSIkGCRKWQIYEBgQGA4TAAQAPz85OS8vETMrEQAzKxEAMxEzERIBFzkRMxEzMzMRMzMRMzEwATMVMzIWFhUUAgQjIxUjNSMiJAI1NDY2MzMTMzI2NTQmKwMiBhUUFjMzAtusRqv7hZX+/bAprC2w/v6Sh/yrQ6wZyd/OuTqsObbR3soYBcu0iPifpv79guHhhAEEoZ74i/xF28O50tS3xdkA//8ACAAABJYFtgIGADsAAAABAG0AAAXyBbYAHQA+QB8KBxEAAA4BFRgYAQcDHh8dAw0DSVkRDQ0BFg8IAwESAD8/MzMSOS8zKxEAMxESARc5ETMRMzMRMxEzMTAhIxEjIiYmNREzERQWMzMRMxEzMjY1ETMRFAYEIyMDg6otsP+Qrs/UG6od08+wkP79ry0Bvnr3pAHj/iG8yQNk/JzGuwHj/h+l93sAAAEAUAAABfQFzQAfADlAIAMNHRMYExYZBwoNCAggIRAASVkQBBoWBgkICUlZGQgSAD8zKxEAMzMzGD8rERIBFzkRMxEzMTABIgIVFBIXFSE1ISYCNRAAISAAERQCByEVITU2EjU0AgMh7vqttP22AWyXoAFiAToBOwFinpcBa/22t6n5BTX+//3h/rOEhZh2AV7LATYBYP6l/sfP/qZ4mIWGAU7e/AEC//8APAAAAm8HJQImACwAAAEHAGr/BwFSAAq0AgEhBSYAKzU1//8AAAAABHsHJQImADwAAAEHAGr/7wFSAAq0AgEeBSYAKzU1//8Ac//sBMcGcwImAX4AAAEGAVQdAAAIswI0ESYAKzX//wBa/+wDhwZzAiYBggAAAQYBVMgAAAizAS8RJgArNf//ALD+FAREBnMCJgGEAAABBgFUOwAACLMBHhEmACs1//8AqP/sApMGcwImAYYAAAEHAVT+xAAAAAizARkRJgArNf//AKT/7ARxBrQCJgGSAAABBgFVOwAADLUDAgE0ESYAKzU1NQACAHP/7ATHBFwACwAqAEdAJAkPJxUEBB0iHQ8DKywYDycoKBYMEhIHRlkSEB8ADABGWSQMFgA/MysRADMYPysREgA5OREzGD8REgEXOREzETMzETMxMCUyNjU1NCYjIBEUFhciAhEQEjMyFhczNjczBgYVERQzMjcVBiMiJicjBgYCUKmWmKn+0ZOF1u704XmhNgwYKYEVHFQdIS5BUVkSDTund8PaD+XH/lDU1IsBKQEMARIBKVRUXDhC9nT+SXIKdxpRVlZRAAIAsP4UBKgGHwATACkATEAoGA8PECcDHggIAwUiEAUqKxAbIyJGWQ4jDiMLAAsbRlkLFgAURlkAAAA/KwAYPysREgA5ORgvLysAGD8REgEXOREzETMRMxEzMTABMhYVEAUVBBEUBCMiJicRIxE0NhciBhURFhYzMjY1NCYjIzUzMjY1NCYCk9z5/scBef747m2gT6b95J6dXaFWq62+sXBcm6KcBh/Qt/7aMwgq/pHR4R8m/eMGNOH2jKyl/IkxJZadnaSOk4l7hQABAAr+FAQOBEgAEgAhQBAPBAEFBBMUCgkJAQ4FDwEbAD8/MxI5LzMREgEXOTEwASM0EjcBMxMWFzM+AhMzAQYCAhS0QCv+P6zwXhMIBSkr6qz+azA1/hRgASZyBDz9uOtnHo6BAm3703z+3AACAHH/7ARgBhIAHgAqADtAICUcEAMfFhYJAAMcBSssEAAiAxkGGShGWRkWBg1GWQYAAD8rABg/KxESABc5ERIBFzkRMxEzETMxMAEmJjU0NjMyFhcHJiYjIgYVFBYXFhYVFAAjIiQ1NBIBNCYnBgYVFBYzMjYCIYx0wqRnvX5IcJ9RVWFrp9Kx/vDs4/7w4gJhe43Ov7KToq4DqE6fY4KYLT+HPixPQkdvW3PxpOv++PjSsQEF/nOAt0o12aCQq7oAAAEAWv/sA4cEXAAlAE1AKwQQIxcdCwETFxAGJicUJQIlAkZZDyUfJQILAyUlDRoaIUZZGhANB0ZZDRYAPysAGD8rERIAORgvX15dKxESADkREgEXOREzETMxMAEVIyAVFBYzMjY3FQYjIiY1NDY3NSYmNTQ2MzIWFwcmJiMiFRQhAsuU/smTklSmZInd0vFugmJr4MBhpWQ/XoJP+gE9AoGNw1piJy+US6mUYoMpCxx/XIWeIS2FKhyirAAAAQBz/m8DoAYUACAAMEAYBxkeExMODgMAGQQhIhEjHgMAAQBGWQEAAD8rEQAzMxg/ERIBFzkRMxEzETMxMBM1IRUGAAIVFBYWFxYWFRQHIzY1NCYnJiY1ND4CNwYhsALw1/7gijt9rJWIf6Z9b4/LvDtwyfIo/vEFh42BtP69/t+mYnZJJR9tW5WkoWs4PRok28Jy0MPl2ggAAAEAsP4UBEQEXAAUAC9AGAAUDAgICRQJFhUQBEZZEBAMCQoPCRUAGwA/Pz8SOT8rERIBOTkRMxEzETMxMAERNCYjIgYVESMRMxczNjYzMhYVEQOeeoKsoKaHGwgzuHHGyP4UBLGGhLrW/cEESJZRWb/S+0kAAwBz/+wESgYrAAsAEgAZAElAJxYQEAYXDw8ABgAaGxYQRlkPFr8WAgsDFhYDCQkTRlkJAQMMRlkDFgA/KwAYPysREgA5GC9fXl0rERIBOTkRMxEzETMRMzEwARACIyICERASMzISATISEyESEhMiAgMhAgIESvT68Pn19PT6/hKknAb9eQSWp6GWCgKFC5gDDP5q/nYBkwGNAZcBiP5r++EBMQEz/tD+zAUp/uH+5wEZAR8AAQCo/+wCkwRIAA8AH0AOAQ4HDhEQDw8LBEZZCxYAPysAGD8REgE5OREzMTABERQWMzI2NxUGBiMiJjURAU5JVyVlGx9pMqCRBEj8+mhlDQd/DRGoqQML//8AsAAABBsERgIGAPoAAAAB//L/7ARGBiEAIgAzQBsIARUDJAAAIxgTRlkYFh4fHwALCwZGWQsBABUAPz8rERIAOREzGD8rEQEzERIXOTEwIwEnLgIjIgc1NjMyFhYXARYWMzI3FQYjIiYnAyYnIwYHAw4B2ToeMkMxOjlEP1t5WDYBaxMqIxshMD1KUx2cVBYJHFj+BDeiVUYkDYURPIKY/AwxMwp5GExTAbTwYHTR/bYA//8AsP4UBEQESAIGAHcAAAABAAAAAAQCBEgADgAcQAwJCgoAEA8FDhUJAA8APzI/ORESATk5ETMxMBEzExYWFzM2EhEzEAIHI6zbGlMQCLGfps/hugRI/bJD7j6vAb0BUf6V/gThAAEAcf5vA6AGFAAxAElAJwQZLR8dHBMMDCgAHB8lGQcyMxwwATABR1kwMBAmKSUmJUZZJgAQIwA/PysRADMREjkYLysREgA5ERIBFzkRMxEzETMRMzEwASMiBhUUHgIXFhYVFAYHIzY2NTQmJyYmNTQ2NzUmNTQ2NwYjIzUhFSMiBgYVFBYzMwNWsrDVMl+HVI6HNkOcNUJzj8jHnoDZi6aAc0QCujOC4H+nr6oC8rKOUGI9JBIdblpBlWNHkzQ3PRkiyLCM0icMQNl1njIMjYNQkF9zbP//AHP/7ARiBFwCBgBSAAAAAQAZ/+wE9ARIABUANkAdCgsHExADEwsNBRYXEgkNDw1GWQ8PCxUFAEZZBRYAPysAGD8/KxEAMzMREgEXOREzETMxMCUyNxUGIyI1ESERIxEjNTchFSMRFBYEfSYwK1Tb/iOm3Y8ETNUzdRKDGP0C0fxGA7pKRI79PEo3AAIApv4UBGIEXAAQABwANkAbFQkJChoACgAdHgYDDg4RRlkOEAobAxdGWQMWAD8rABg/PysREgA5ERIBOTkRMxEzETMxMAEQACMiJyMWFREjERASMzISJSIGFREWMzI2NTQmBGL/AOmzeAgIqPvq2/z+IZ6XerefmJACJf7x/tZePdT+2wQfAQoBH/7Ros/R/q5m0N7W1AAAAQBz/m8DogRcACAALkAXDgcAFRUHGwMiIQQSEhgLGB5GWRgQCyMAPz8rERIAOREzERIBFzkRMxEzMTABFBYWFxYWFRQGByM2NjU0JiYnJiY1EAAzMhYXByYjIgYBHzuPoJSDNkOcNkMzbmHMwwEU+E+eNjWCcrCqAgqHhFAiIGtaQphfRpQyKC8mEiX+2wEeATYhGI0z2gACAHP/7AS2BEgADQAZADBAGRQADgcHDAALBBsaDBcJF0ZZCQ8EEUZZBBYAPysAGD8rEQAzERIBFzkRMxEzMTABFAYGIyIANRAhIRUhFgEUFjMyNjUQJyMiBgRge+Wa6/74AlAB8/74svy/qqGfq65B3sgB/J3xggEg/gI+jqf+98LRxbYBDrrQAAABABL/5wOTBEgAEwAsQBcDDwAJDxEEFBUCERMRRlkTDwwFRlkMFgA/KwAYPysRADMREgEXOREzMTABFSERFDMyNjcVBgYjIiY1ESE1NwOT/lDNL2IbI28wtar+15QESI79lt8NB30PEqqqAn9KRAAAAQCk/+wEcQRIABUAJUARDBMGAxMDFxYPBA8ACUZZABYAPysAGD8zERIBOTkRMxEzMTAFIiYRETMRFBYzMjY1NCYnMxYWFRAAAnPn6KaemaehHCKmJBz+/hT6AQoCWP2wwMPu+4LgiJDWjP7C/tQAAAIAc/4UBUwEXAAYACIAQUAjCgQgGBgMABkTEwAHBAQjJBAcRlkQEAYPIAwBDEZZFwEWABsAPz8zKxEAMxg/PysREgEXOREzETMzETMRMzEwAREkABEQNxcGBhUQBRE0NjMyEhUUAgYHEQE0JiMiBhURNjYCg/78/vTPg1lRAWimlbTaiPilAXl8ZklOs8b+FAHaCwEjAQ8BKP1adeB8/nUjAmy7vv7b+rL++5AI/iYEJ7nbeHL9khDsAAH/7P4UBFAETgAgADlAIQ4HCAUVGB4HIhchBRgIFQQGFxsRDEZZERsGDwAcRlkADwA/KwAYPz8rABg/Ehc5EQEzEhc5MTATMhYWFxMBMwETFhYzMjcVBiMiJicDASMBAyYmIyIHNTayNk4+LJEBPrT+VL4wUj8tLTw7c407lv6WsgHQrCZGKyUbMQROK1tw/o8CYfz8/hx6SgiBD3afAYP9aANEAbxjUAuBEQABAKT+FAWHBhIAGgA9QB8WEwEODhkPBAoKDxMDGxwaAAcUDwEZEBlGWQ0QFg8bAD8/MysRADMYPzM/ERIBFzkRMxEzMxEzETMxMAERNjY1NCYnMxIVEAAFESMRJAARETMRFBYXEQNavMsaJaY//uP+8KT++P72prS4BhL6aQ/nzHjrqP7w9P7s/s4Q/iYB2gkBIgEQAh/928PaDQWZAAEAc//sBbwESAAnAD1AHgoDJhMTEBkgIBADAygpJhERABwGDxYNAA1GWSMAFgA/MisRADMYPzMSOS85ERIBFzkRMxEzEjkRMzEwBSICNTQSNzMGBhUUFjMyNjURMxEUFjMyNjU0AiczFhIVFAIjIicjBgH0tss3RKxEOXhrXmmhal1reDdFrEE5y7bcRAlBFAEo/pwBAZmc/53B2I99ATf+yYCM2MGXAQSdkv75nfz+1ra2//8ACf/sApMF0wImAYYAAAEHAGr+1AAAAAq0AgElESYAKzU1//8ApP/sBHEF0wImAZIAAAEGAGo5AAAKtAIBKxEmACs1Nf//AHP/7ARiBnMCJgBSAAABBgFUIQAACLMCIhEmACs1//8ApP/sBHEGcwImAZIAAAEGAVQnAAAIswEfESYAKzX//wBz/+wFvAZzAiYBlgAAAQcBVADJAAAACLMBMREmACs1//8AyQAAA/gHJQImACgAAAEHAGoAJwFSAAq0AgEhBSYAKzU1AAEAEv/sBUIFtgAdAEZAJhYODg8IGxsUAg8RBR4fFg1JWRYWDxIVERIRSVkSAw8SAAVJWQATAD8rABg/PysRADMREjkYLysREgEXOREzETMRMzEwBSInNRYzMjY1NTQmIyERIxEhNSEVIREhMhYVFRQGA89gNjdbZWiDjP6Dqv6wA7f+QwGMzd3EFBaWE3xwg4Bx/RsFH5eX/l6/so++0///AMkAAAP4B3MCJgFhAAABBwB2AFoBUgAIswEPBSYAKzUAAQB9/+wE4wXNABgAOEAeBgMRFgwFEQQZGgMGSVkDAw4UFABJWRQEDglJWQ4TAD8rABg/KxESADkYLysREgEXOREzMzEwASIEByEVIRIAMzI3FQYjIAAREAAhMhcHJgNC4v7zHgLT/SkKAQv5osmh4v60/qIBeQFO7bJHqQUz+vGW/u7+4zeVOQGEAW0BXwGRWJRS//8Aav/sBAIFywIGADYAAP//AFQAAAJWBbYCBgAsAAD//wA8AAACbwclAiYALAAAAQcAav8HAVIACrQCASEFJgArNTX///9g/n8BaAW2AgYALQAAAAIAAP/pByMFtgAaACMAR0AmGBsbBB8AAAQNAyQlGCNJWRgYCxYWBklZFgMLEEpZCxIEG0pZBBIAPysAGD8rABg/KxESADkYLysREgEXOREzETMRMzEwARQEISERIQICBgYjIic1FjMyPgISEyERMyABMzI2NTQmIyMHI/7t/vz+uf6TOVRQi2tFQDI/MEErN0RBAqZ6Ajr9TIXGt8DcZgGqztwFH/5I/fb7eRmPGj5n+gG+AeL9kP1Ni4yKfAACAMkAAAdUBbYAEQAaAEpAJgsHBwgPEhIMBBYAAAQIAxscGgYLBklZDwsLBA0JAwgSBBJKWQQSAD8rABg/PzMSOS8zKxEAMxESARc5ETMRMzMRMxEzETMxMAEUBCEhESERIxEzESERMxEzIAEzMjY1NCYjIwdU/vD++/63/X2qqgKDrHkCOf1OhcS5wdtmAarO3AKw/VAFtv2SAm79kP1Ni4yJfQAAAQASAAAFQgW2ABMAOkAfAAwMDQYFBRINDwQUFRMPEA9JWQALSVkAAA0QAwYNEgA/Mz8SOS8rKxEAMxESARc5ETMRMxEzMTABITIWFREjETQmIyERIxEhNSEVIQIMAZDN2ap9jP59qv6wA/b+BAN9vLX99AH2fnH9GwUfl5f//wDJAAAE5QdzAiYBtAAAAQcAdgCiAVIACLMBFAUmACs1//8AG//sBPgHXgImAb0AAAEHAjYARAFSAAizARcFJgArNQABAMn+gwUMBbYACwAwQBgIBQIDCQAAAwUDDA0KBgMFCElZAQUSAyIAPz8zKwAYPzMREgEXOREzETMRMzEwISERIxEhETMRIREzBQz+L7D+PqoC76r+gwF9Bbb65AUcAP//AAAAAAUQBbwCBgAkAAAAAgDJAAAEfQW2AA0AFgA9QCASAAkODgQEBwADGBcJFklZCQkEBQUISVkFAwQOSlkEEgA/KwAYPysREgA5GC8rERIBFzkRMxEzETMxMAEUBCEhESEVIREzMhYWATMyNjU0JiMjBH3+/f77/lQDXv1M48HydPz2776tsNvPAara0AW2l/4nWa7+VIKVjngA//8AyQAABL4FtgIGACUAAP//AMkAAAP4BbYCBgFhAAAAAgAO/oMFSgW2AA0AEwBDQCQEBRMHEAoODAEAAAwKBwUFFBUKEElZCgMBBSITDAYDBklZAxIAPysRADMzGD8zPysREgEXOREzETMRMxEzETMxMAEjESERIxEzEhITIREzIREhBgIHBUqi/AiicZrbDAKRuf6d/rMSzon+gwF9/oMCFwEDAuYBM/rkBIPy/VnqAP//AMkAAAP4BbYCBgAoAAAAAQACAAAGvAW2ABEAPEAfBg0NAw4KCQgBDgARBxITDwwJBgMAAAEOCxESBwQBAwA/MzM/MzMSOREzMzMzMxESARc5ETMzETMxMAEBMwERMxEBMwEBIwERIxEBIwJW/cG+AjmkAjq+/cACUsT9uqT9u8cC8ALG/TwCxP08AsT9PP0OAuX9GwLl/RsAAQBK/+wENQXLACgAQ0AkHAATBwcAAxcjDAYpKgMYFxgXSlkYGAomJh9KWSYEChBKWQoTAD8rABg/KxESADkYLysREgA5ERIBFzkRMxEzMTABFAYHFRYWFRQEISInNRYWMzI2NTQmIyM1MzI2NTQmIyIGByc2NjMyFgQZt6G3vf7O/un/o2DfZ8bL4d/a0c3hoolusnVUZfuH4f8EYJC0GAgZtJHN5U+eLjKWjYaKj5OEa4AySnJLTcUAAQDLAAAFUgW2AA8ANEAYDgICDwYJCQgPCBARBQQMDQQNCQ8SBgADAD8yPzM5OREzETMREgE5OREzETMRMxEzMTATMxEUBzMBMxEjETQ3IwEjy58OCAM0uqARCfzLugW2/NPhtgTE+koDJcnd+zUA//8AywAABVIHXgImAbIAAAEHAjYA4QFSAAizARAFJgArNQABAMkAAATlBbYACgAtQBYHAwMEAAkKBAQLDAoHAgcECAUDAQQSAD8zPzMSOTkRMxESARc5ETMRMzEwISMBESMRMxEBMwEE5c79XKqqApPD/XkC5f0bBbb9PALE/ToAAQAA/+cE2QW2ABMALUAYAxIBAAASCgMUFRIDSVkSAwgNSlkIEwESAD8/KwAYPysREgEXOREzETMxMCEjESEHAgIGJyInNRYzMjY2EhMhBNmq/iUfPV2Yfko7Njs1Tz1dOAMSBR/w/iH+Ra4CGY8aV9cCWQG4//8AyQAABnEFtgIGADAAAP//AMkAAAUfBbYCBgArAAD//wB9/+wFvgXNAgYAMgAA//8AyQAABQwFtgIGAW4AAP//AMkAAARoBbYCBgAzAAD//wB9/+wEzwXLAgYAJgAA//8AEgAABFoFtgIGADcAAAABABv/7AT4BbYAFgAqQBUSCAIJBBcYDg0IDQARCQMABUlZABMAPysAGD8zEjk5ETMREgEXOTEwBSInNRYzMjY3ATMBFhczNjcBMwEOAgElb1RdYG6FQv3HvAGwGQ4IHAsBZ7T+LVSHqRQepitliwRB/MExL1QWAzX76ruqT///AGr/7AX4BcsCBgFzAAD//wAIAAAElgW2AgYAOwAAAAEAyf6DBbgFtgALADJAGQgFCQADAgIABQMMDQoGAwAIBQhJWQUSAyIAPz8rEQAzGD8zERIBFzkRMxEzETMxMCUzESMRIREzESERMwUMrKH7sqoC76qa/ekBfQW2+uQFHAAAAQCqAAAExwW2ABMALUAWCwgRAQEACAAUFQUOSVkFBQESCQMBEgA/PzMSOS8rERIBOTkRMxEzETMxMCEjEQYGIyImNREzERQWMzI2NxEzBMeqlcZqz9+qf49hsamqAlw1J76zAkX9z3l0HTcCygABAMkAAAd5BbYACwAxQBgEAQgFCQAABQEDDA0KBgIDCAQBBElZARIAPysRADMYPzMzERIBFzkRMxEzETMxMCEhETMRIREzESERMwd5+VCqAliqAlisBbb65AUc+uQFHAABAMn+gwgEBbYADwA7QB4DAAcECAsODQ0LBAAEEBEOIgkFAQMLBwMAA0lZABIAPysRADMzGD8zMz8REgEXOREzETMRMxEzMTAzETMRIREzESERMxEzESMRyaoCR6wCSKqsogW2+uQFHPrkBRz65P3pAX0AAAIAEgAABRcFtgAMABUAPUAgCQ0NBBEAAAQGAxYXCRVJWQkJBAcHBklZBwMEDUpZBBIAPysAGD8rERIAORgvKxESARc5ETMRMxEzMTABFAQjIREhNSERMyAEATMyNjU0JiMjBRf+/fn+R/6wAfr0AQUBEvz1/LWpr8vgAarO3AUfl/2Qzf4ai4yIfgAAAwDJAAAGCgW2AAoAEwAXAD9AIAMLCwAPBxUUFAcAAxgZFRIDE0lZAwMAFgEDAAtKWQASAD8rABg/MxI5LysAGD8REgEXOREzETMRMxEzMTAzETMRMyAEFRQEIyUzMjY1NCYjIwEjETPJqu8BBQES/v35/vb3taqzyNsEl6qqBbb9kM3PztyRjYyJe/1SBbYAAgDJAAAEugW2AAoAEgAyQBkHCwsEDgAEABMUBxJJWQcHBAUDBAtKWQQSAD8rABg/EjkvKxESATk5ETMRMxEzMTABFAQjIREzESEgBAEhIBE0JiMhBLr+8fv+GaoBIwELARn8uQErAWy7zv7yAarL3wW2/ZDT/iABF4d/AAEAPf/sBIkFywAaADpAHxgVFQkJFg8DBBscFxZJWRcXDAUMEklZDBMFAElZBQQAPysAGD8rERIAORgvKxESARc5ETMRMzEwASIHJzYzMgQSFRAAISInNRYWMyAAEyE1ISYAAdOsokis7NkBOaL+lP6q45xTrGMBDwEUCP0xAs0W/vEFM0yQVLD+ut3+iP5sOZUVIgEhARCY5QECAAIAyf/sB+cFzQASAB4AR0AmDAgICRMNBhkAAAYJAx8gEBxJWRAEDAdJWQwMCQoDCRIDFklZAxMAPysAGD8/EjkvKwAYPysREgEXOREzETMzETMRMzEwARAAISAAAyERIxEzESESACEgAAEQEjMyEhEQAiMiAgfn/qv+0P7T/qsL/p6qqgFkFwFRAR8BMwFW+6Du5+rt6+jp8ALd/p7+cQFvAVX9UAW2/ZIBNwFO/m/+of7Y/swBMgEqASoBLv7PAAIAMwAABE4FtgANABUAPUAgFQwMCxIGAgYDCwQXFgAUSlkDCQAAAgkJD0pZCQMMAhIAPzM/KxESADkYLxI5KxESARc5ETMRMxEzMTABASMBJiY1NCQhIREjEREjIgYVECEzAnv+gckBmqGSAQ8BEwGSquO3vgF73QJi/Z4CfzPPnsTT+koCYgLBfo7+3f//AF7/7APNBFoCBgBEAAAAAgB3/+wEVAYhABcAIgA7QB4aEiALAAAGEgMkIwwLDxxGWQsPDxUFFRhGWRUWBQEAPz8rERIAORgvOSsRADMREgEXOREzMxEzMTATEBI3JDcXBAcGBgczNjYzMhIVEAAjIgAFIBEQISIGBgcQEnfU5gEe2h/+pZWRkQcMPsRryuL++urn/voB/AEx/utMjXUgpgKRAWgBkzI9JpI6IiH21FRg/vro/v/+3wFi1wGFAXM/aDf++f7tAAMAsAAABEwESAAOABYAHwBJQCYcFBQLFwAPBwcAAwsEICEEHBMcE0ZZHBwLDAwbRlkMDwsURlkLFQA/KwAYPysREgA5GC8rERIAORESARc5ETMRMxEzETMxMAEUBgcVFhYVFAYjIREhIAM0JiMhESEgAzQmIyERITI2BCl7b4yB4dj+HQHhAZiDh5z+0wExAR8fe33+xwEZmn4DNWtvEwkTfm+ZpgRI/QJZUf6XAppQQ/7LTAAAAQCwAAADRARIAAUAHUAOAgMAAwcGBAFGWQQPAxUAPz8rERIBOTkRMzEwASERIxEhA0T+EqYClAO6/EYESAACACn+hQRoBEgADQATAENAJAQFEwcQCg4MAQAADAoHBQUUFQoQR1kKDwEFIhMMBgMGRlkDFQA/KxEAMzMYPzM/KxESARc5ETMRMxEzETMRMzEwASMRIREjETM2EhMhETMhESMGAgcEaKH9AqBWhpgDAiud/sP2DZFs/oUBe/6FAgq2AeoBGfxHAzbe/jmRAP//AHP/7AQSBFwCBgBIAAAAAQAEAAAF3wRGABEAPEAfAgkJEQoGBAUKDg8NBxMSEQsIBQIODg0DAA8PCgcNFQA/MzM/MzMSOREzMzMzMxESARc5ETMzETMxMAEzEQEzAQEjAREjEQEjAQEzAQKkmQHFtv42AfHA/h6Z/h+/AfD+N7YBwwRG/e0CE/3t/c0CK/3VAiv91QIzAhP97QABAET/7AN/BFwAIgBNQCsCDR4TEw0PIQgYBiMkECIhIiFGWQ8iHyICCwMiIhYKFhtGWRYWCgRGWQoQAD8rABg/KxESADkYL19eXSsREgA5ERIBFzkRMxEzMTABIDU0IyIGByc2MzIWFRQHFRYWFRQGIyInNRYzMjY1NCEjNQGBATf8TX5mO6rJvdrNfnT12O2Bt7uQk/7JmAKBrKIcKodMm4a4OQgliWeYqUeYVmNdv40AAQCwAAAEYgRIAA0ANEAZCAQHBwYLAwMMBgwPDgMKDAQNDwwVBxUEDwA/Pz8/ERI5ORESATk5ETMRMxEzETMzMTABEQcHATMRIxE3NwEjEQFMBwMCUc+bAwX9sM8ESP1JtjkDpvu4Ap6EgvxcBEgA//8AsAAABGIGDAImAdIAAAEGAjY9AAAIswEOESYAKzUAAQCwAAAEDARIAAoALUAWCgYGBwMBAgcEDAsCCgUKBwAIDwQHFQA/Mz8zEjk5ETMREgEXOREzETMxMAEzAQEjAREjETMRAy+2/icCAML+DKamBEj97/3JAiv91QRI/esAAQAQ//ID4QRIABAALUAYAQADDwoPAAMSEQ8DRlkPDwcMR1kHFgEVAD8/KwAYPysREgEXOREzETMxMCEjESECAgYjIic1FjMyEhMhA+Go/rcbYJl2NiAWHHOIIwKBA7r+nP5ewgx7BgHmAe8AAQCwAAAFLwRGABQANUAZAwYGBRIPDxAFEBYVBw4ADgsDEQ8GEBULFQA/PzM/MxI5OREzERIBOTkRMxEzETMRMzEwJTc3ATMRIxEHBwEjASYnESMRMwEWAukfKwEp05MUOv7li/7lNRSUywEfK6BddgLT+7oDiTqZ/UoCuIZL/HcERv1JbgABALAAAARiBEgACwA5QB4CBgYFAQkJCgUKDQwBCEZZLwE/AQIBAQoDCw8GChUAPzM/MxI5L10rERIBOTkRMxEzETMRMzEwAREhETMRIxEhESMRAVYCZqam/ZqmBEj+NQHL+7gB7v4SBEj//wBz/+wEYgRcAgYAUgAAAAEAsAAABEgESAAHACNAEQABBQQBBAgJAgdGWQIPBQEVAD8zPysREgE5OREzETMxMCEjESERIxEhAVamA5io/bYESPu4A7gA//8AsP4UBHUEXAIGAFMAAP//AHP/7AOLBFwCBgBGAAAAAQApAAADkwRIAAcAJEASAgMAAwUDCAkBBQYFRlkGDwMVAD8/KxEAMxESARc5ETMxMAEhESMRITUhA5P+nKb+oANqA7r8RgO6jv//AAL+FAQGBEgCBgBcAAAAAwBx/hQFRgYUABEAGAAeAExAJxIJHA8EBBUMBRkAAAUJAx8gDQAbFgwWRlkPDBAcFQYVRlkDBhYFGwA/PzMrEQAzGD8zKxEAMxg/ERIBFzkRMxEzMzMRMzMRMzEwARQABxEjESYANTQANxEzERYABRQWFxEGBgUQJRE2NgVG/uX+pPj+4AEf/577AR772bDAubcDe/6Tvq8CJfn+2RX+JAHcEwEu9PkBJhQBvP5EF/7U8MDaEgNUEc/IAX8n/K4T2v//ACcAAAQIBEgCBgBbAAAAAQCw/oUE3QRIAAsAMkAZBgMHCgEAAAoDAwwNCAQPCgYDBkZZAxUBIgA/PysRADMYPzMREgEXOREzETMRMzEwASMRIREzESERMxEzBN2m/HmmAkamm/6FAXsESPxHA7n8RwABAJwAAAQtBEgAEgAtQBYGCgoJAREJERQTAw5GWQMDCgcSDwoVAD8/MxI5LysREgE5OREzETMRMzEwAREUMzI2NxEzESMRBgYjIiY1EQFC21umaaamabNxpLoESP5wwDhDAdX7uAHwSDuskwGcAAEAsAAABm8ESAALADFAGAgFAAkBBAQJBQMMDQoCBg8ACAUIRlkFFQA/KxEAMxg/MzMREgEXOREzETMRMzEwJSERMxEhETMRIREzA+EB5qj6QaYB5aaPA7n7uARI/EcDuQAAAQCw/ocHCgRGAA8AO0AeDAkADQEEBwYGBA0JBBARDgIKDwQADAkMRlkJFQciAD8/KxEAMzMYPzMzERIBFzkRMxEzETMRMzEwJSERMxEzESMRIREzESERMwPhAeamnaj6TqYB5aaPA7f8Sf34AXkERvxJA7cAAAIAKQAABR0ESAAMABQAPUAgABISCA0EBAgKAxUWABFGWQAACAsLCkZZCw8IEkZZCBUAPysAGD8rERIAORgvKxESARc5ETMRMxEzMTABITIWFRQGIyERITUhATQmIyERISACLQE54Nff3P4l/qICBAJMfJ3+zQE5ARMCg5qbpqgDuo78/F1T/pcAAAMAsAAABXkESAAKAA4AFgA/QCAAEBAIBBMMCwsTCAMXGAwVAA9GWQAACA0JDwgQRlkIFQA/KwAYPzMSOS8rABg/ERIBFzkRMxEzETMRMzEwASEyFhUUBiMhETMBIxEzAREhIDU0JiMBVgEr0cnVz/45pgQjpqb73QEZAQh6kwKDm5qlqQRI+7gESP2s/pe5XFQAAgCwAAAETARIAAkAEgAyQBkPAwALCwcDBxQTAApGWQAABwgPBwtGWQcVAD8rABg/EjkvKxESATk5ETMRMxEzMTABISARFAYjIREzEREhMjY1NCYjAVYBUgGk29P+EqYBQISMgZQCg/7LoqwESP2s/pdcXVtVAAEAOf/sA30EXAAaAERAJgwJCRgYChICBBscCwpGWQ8LHwsCCwMLCwAVFQ9GWRUQAAZGWQAWAD8rABg/KxESADkYL19eXSsREgEXOREzETMxMAUiJzUWFjMyNjchNSEmJiMiByc2NjMgABEQAAFWp3Y8jFuuvQr91QIpEKmhZ5cvN6RQAQABCv7fFDmTFyS6uY2soDaMGiP+2/7s/vP+1gACALD/7AYzBFwAEgAeAFFALQwICAkTDQYZAAAGCQMfIBAcRlkQEAwHRlkPDB8MAgsDDAwJCg8JFQMWRlkDFgA/KwAYPz8SOS9fXl0rABg/KxESARc5ETMRMzMRMxEzMTABEAAjIgInIREjETMRITY2MzIAARQWMzI2NTQmIyIGBjP+/+DV+g7+4aamASEU/M/cAQH87pKhnpWSoaGSAiX+8/7UAQv3/hIESP415Pv+z/7609vV2dLY2AACACUAAAPBBEgADQAUAD1AIBELCwoOBQEFAgoEFhUNEEZZAggNDQEICBNGWQgPCwEVAD8zPysREgA5GC8SOSsREgEXOREzETMRMzEwMyMBJiY1NDYzIREjESEBFCEhESEi58IBO3+HyrUB6Kb+6/72ARQBC/7T8gHPHKF6lqz7uAG2AU6+AXL//wBz/+wEEgXTAiYASAAAAQYAaggAAAq0AwIwESYAKzU1AAEAFP4UBEQGFAAnAGZAOh0bFw8PFBAHJSUZAhASBSgpHh0hC0ZZGhITEkdZFxMPEx8TLxMDCQMdIRMTIR0DEBUAEBUABUZZABsAPysAGD8/Ehc5Ly8vX15dETMrEQAzKxEAMxESARc5ETMRMzMRMzMzMTABIic1FjMyNRE0JiMiBhURIxEjNTM1MxUhFSEVFAczNjYzMhYVERQGAy9PNDo3gXqCrZ2onJymAZH+bwgKMbV0ycmJ/hQZiRSqA1KGhLzT/ecE23+6un/EVDhPW7/S/Lacqv//ALAAAANEBiECJgHNAAABBgB28QAACLMBDxEmACs1AAEAc//sA6oEXAAZAERAJg8SEgMJGBEDBBobDxJGWQ8PHw8CCwMPDwAGBgxGWQYQABVGWQAWAD8rABg/KxESADkYL19eXSsREgEXOREzETMxMAUiABEQADMyFhcHJiMiBgchFSEWFjMyNxUGAnn4/vIBE/tSnjkxj22kqhACKf3VCaqnjJd0FAEjARABEwEqIBmNM6Opjb61O5M5//8Aav/sA3MEXAIGAFYAAP//AKIAAAFmBd8CBgBMAAD////sAAACHwXTAiYA8wAAAQcAav63AAAACrQCARkRJgArNTX///+R/hQBZgXfAgYATQAAAAIAEP/yBkIESAAVAB0ATEApCRQAGxsHFgQEBxQOBB4fABpGWQAADBQUCUZZFA8MEUdZDBUHG0ZZBxUAPysAGD8rABg/KxESADkYLysREgEXOREzETMRMxEzMTABMzIWFRAhIREhAgIjIic1FjMyEhMhATQmIyMRMyADsPTTy/5L/mX+/ii1qzggFhxziCMCUAHsfZ7n7QEVAoObmv6yA7r9+v4+DHsGAeYB7/z8W1X+lwACALAAAAakBEYAEQAZAEpAJg8LCwwBExMQCBYFBQgMAxobEgoPCkZZAQ8PCBENDwwVCBNGWQgVAD8rABg/PzMSOS8zKxEAMxESARc5ETMRMzMRMxEzETMxMAERITIWFRAhIREhESMRMxEhERMRMyA1NCYjBAABANnL/k7+YP4KrKwB+qbwARSAmQRG/juZmv6yAe7+EgRG/jcByf2u/pe5XFQA//8AFAAABEQGFAIGAOkAAP//ALAAAAQMBiECJgHUAAABBgB2MwAACLMBFBEmACs1//8AAv4UBAYGDAImAFwAAAEGAja3AAAIswEWESYAKzUAAQCw/ocERgRGAAsAMkAZBAEKCwUICAsBAwwNCyIGAg8JAQEERlkBFQA/KxEAMxg/Mz8REgEXOREzETMRMzEwISERMxEhETMRIREjAi/+gaYCSqb+j6YERvxJA7f7uv6HAAABAMkAAAQIBuMABwAjQBEAAwUGAwYJCAcESVkBBwMGEgA/P8YrERIBOTkRMxEzMTABETMRIREjEQNmov1rqgW2AS3+OvrjBbYAAAEAsAAAA0QFiQAHACdAEgUAAgMAAwkIBgQEAUdZBA8DFQA/PysAGBDGERIBOTkRMxEzMTABIREjESERMwNE/hKmAe6mA8f8OQRIAUEA//8AGwAAB0wHcwImADoAAAEHAEMBFwFSAAizARsFJgArNf//ABcAAAYjBiECJgBaAAABBgBDcwAACLMBHhEmACs1//8AGwAAB0wHcwImADoAAAEHAHYBsAFSAAizASMFJgArNf//ABcAAAYjBiECJgBaAAABBwB2ARsAAAAIswEmESYAKzX//wAbAAAHTAclAiYAOgAAAQcAagFkAVIACrQCAS8FJgArNTX//wAXAAAGIwXTAiYAWgAAAQcAagDPAAAACrQCATIRJgArNTX//wAAAAAEewdzAiYAPAAAAQcAQ/+UAVIACLMBCgUmACs1//8AAv4UBAYGIQImAFwAAAEHAEP/YQAAAAizARcRJgArNQABAFIB2QOuAnEAAwARtQACBAUAAQAvMxESATk5MTATNSEVUgNcAdmYmAABAFIB2QeuAnEAAwARtQACBAUAAQAvMxESATk5MTATNSEVUgdcAdmYmP//AFIB2QeuAnECBgIDAAAAAv/8/jEDTv/TAAMABwAcQAsEAAkFAQEIBQYCAQAvMy8zEQEzETMRMzIxMAEhNSE1ITUhA078rgNS/K4DUv4xi4yLAAABABkDwQFEBbYABwAStgEFCAkABAMAP80REgE5OTEwEyc2EjczBgclDBZiOHtCJQPBFloBDHn+9wAAAQAZA8EBRAW2AAcAErYFAQgJBQcDAD/GERIBOTkxMAEXBgIHIxI3ATUPGmI1ekYgBbYWZP73cgEd2AD//wA//vgBbQDuAgYADwAAAAEAGQPBAUYFtgAHABK2AgYJCAMHAwA/zRESATk5MTATFhcjJgInN98lQnstbRgOBbb7+l4BHGUWAAACABkDwQK0BbYABwAPABpADAQBDQkEEBEACAMMAwA/M80yERIBFzkxMAEnNhMzBgIHISc2EjczBgcBlg84enseOw391wwWYjh7QiUDwRbXAQhz/t9hFloBDHn+9wAAAgAZA8ECtAW2AAcAEAAaQAwJDQEFBBESDQUQBwMAPzPGMhESARc5MTABFwYCByMSNyEXBgIHIzYSNwE1DxpiNXpGIAInDhhgOH0aQg0FthZk/vdyAR3YFlv+9npkATRdAP//ABn++QK0AO4BBwILAAD7OAAgtwEAB0ANDUgHuP/AswwMSAe4/8CzCQlIBwARKysrNTUAAQB7AAADiQYUAAsAQ0AhCQICCAMKAQEHBAAEAwUEDA0ABQULBgYHCAABBAQKBwMSAD8uMzMRMz8SOS8zMxEzERIBFzkRMzMRMxEzMxEzMTABJRMjEwU1BQMzAyUDif6gMcQx/rQBTDHEMQFgA+cf+/oEBh+qHgGh/l8eAAEAewAAA5oGFAAVAHVAOgwHFRAEBA8KBRQRAAMDDgsJBhMBAQYFBwQWFwEICAIHAwYGAAkUCwsRDhMMDBIJDg0HDQcNBQ8ABRIAPz8SOTkvLxI5OTIyETMRMzMRMxEzMxEzETMzETMREgEXOREzETMzMzMRMzMzETMzMxEzMxEzMTABJRUlEyMTBTUFAxMFNQUDMwMlFSUTAjkBYf6fMcYx/qYBWisr/qYBWjHGMQFh/p8rAecfqB3+hQF7HagfASsBGx+oHgF8/oQeqB/+5QABAKQB9AJeA+MACwATtgYAAAwNCQMAL80REgE5ETMxMBM0NjMyFhUUBiMiJqRxbGl0c2prcgLseX58e3eBgwD//wCY/+MFrgDyACYAEQAAACcAEQISAAAABwARBCUAAAAHAGT/7Ak7BcsACQAUABgAJAAvADsARgBbQDAAEAUKMEI2PBkrHyUlKzwVQgoXEAhHSBwzMyg/GQMNIjk5LUQNRA1EFxgGFxgHEgcAPzM/PxI5OS8vETMzETMRMz8zMxEzERIBFzkRMxEzETMRMxEzETMxMBMUFjMyERAjIgYFFAYjIiY1ECEyFiUBIwEBFBYzMjY1NCYjIgYFFAYjIiY1ECEyFgUUFjMyNjU0JiMiBgUUBiMiJjUQITIW7FNdtLRdUwHtoZyVowE4mKUCafzVlAMrAqBTXVtZWVtdUwHtopuUowE3lqf7OFFdW1lZW11RAeuim5WjATiWpwQCqqoBVAFSqKrm5+7fAcnw2/pKBbb8Aqupp62rpaWr5ubv3QHJ7N2rqaetq6Wlq+bm7t4ByewA//8AhQOmAT8FtgIGAAoAAP//AIUDpgKwBbYABgAFAAAAAQBSAHUCHwO+AAYAGkAKBAIDBgIGCAcFAQAvLxESATk5ETMRMzEwEwEXAQEHAVIBVnf+3wEhd/6qAicBl0X+ov6hRwGXAAEAUAB1Ah0DvgAGABpACgMABAIAAggHBQEALy8REgE5OREzETMxMAEBJwEBNwECHf6odQEf/uF1AVgCDP5pRwFfAV5F/mkA//8AmP/jA0oFtgAmAAQAAAAHAAQBwQAAAAH+eQAAAo8FtgADABO3AAUCBAMDAhIAPz8RATMRMzEwAQEjAQKP/HmPA4cFtvpKBbYAAQBtAyECwwXHABIAJkARABIMCAgJEgkUEwQPHwAJCh8AP80yPzMREgE5OREzETMRMzEwARE0JiMiBhURIxEzFzM2MyAVEQJMTlByW3RgDgpLkQECAyEBpFRHaXr+pAKZWGX6/lQAAQBiAAAEIwW2ABEAS0AoDgAEBAkFCxACBQcFEhMDBwgHTlkACA4RTFkIDggOBQoKDUxZCgYFGAA/PysREgA5ORgvLysRADMrEQAzERIBFzkRMzMRMzMxMAEhFSERIxEjNTMRIRUhESEVIQG4ATT+zKawsAMR/ZUCRP28AYuB/vYBCoEEK5f96ZcAAQBEAAAESAXJACUAcEBADQkRESIeGgsPFQIPGhwgFwcmJxAcHRxOWQ0dDCAhIE5ZCSEPIR8hPyFPIQQJAx0hHSEXABcUTFkXGAAFS1kABwA/KwAYPysREgA5ORgvL19eXREzKxEAMxEzKxEAMxESARc5ETMRMzMzETMzMTABMhcHJiMiBhUVIRUhFSEVIRUUBgchFSE1NjU1IzUzNSM1MzU0NgKwyZ48mJN6fgGk/lwBpP5cQUoDG/v8zsjIyMjgBclQg0eHgbqBpoEhZIgsmo0w8yOBpoHPss0AAAMAmv/sBdEFtgAWACEAKgBgQDciHBwdJhcQFBQNCQISCRcLHQYrLBsiS1kQE05ZAxsLEA4OEAsbAwUdHh4qS1keBh0YBgBNWQYZAD8rABg/PysREgAXORgvLy8vLysrERIBFzkRMzMRMxEzETMRMzEwJTI2NxUGIyImNREjNTc3MxUzFSMRFBYBFAQhIxEjESEgFgEzMjY1NCYjIwVOIlYLPG5tgZ2dPmLd3TT+kf7r/vZApQEGAQD+/aE0yLmst1J1DgR9HoiKAc9QRb/Tgf5HTVIDl+Pq/cEFttP97pGikY4AAAEAP//sBIkFywAmAHFAPx0XHxYWGgsCBwcaJBEEChoXBicoCxcYF05ZCBgFHR4dTlkCHg8eHx4vHgMJAxgeGB4TIiIATFkiBxMOTFkTGQA/KwAYPysREgA5ORgvL19eXREzKxEAMxEzKxEAMxESARc5ETMRMzMRMxEzETMxMAEgAyEVIQcVFyEVIRYWMzI3FQYjIgADIzUzJzU3IzUzEgAzMhcHJgMb/sFPAf799AICAc/+QSXLqpyZkqvt/t8uppgCApikJwEk7cmlR6YFNf5tgTlALYG0xUKWQQENAQGBKixQgQEFASRhi1YABACN//gGCgXBAAMADwAXACsARUAkJRsgKhAKFAQEAAoqAhsGLC0jHgYSBxgWDScYDRgNGAIDBgIYAD8/Ejk5Ly8RMxEzPzM/MxESARc5ETMRMxEzETMxMAEBIwEBFAYjIiY1NDYzMhYFFDMyNTQjIiUiJjU0NjMyFwcmIyIVFDMyNxUGBR/81ZQDKwF/qZSLqqeUjar+FbKwsLL9yqa2vKtoWCFRUODcYlpOBbb6SgW2+5ift7mdnri6nO7u69uxoaizI2cf7ushZSUAAgB3/+wDnAXLABwAJAA9QB8jGhoPCR0WAxYJDAQlJiMPDRkKBQwTAgwCDAYfEwAGAC8zLzMSOTkvLxESFzkREgEXOREzETMzETMxMCUyNzMGBiMiJjU1Bgc1NjcRNDYzMhYVFAIHERQWEzQjIgYVESQCfa4SXwiZjpagYGBOcpaHdYfOr1Kuf0M+AQBv1aaytanzIxZxFSYB8oqfoYq5/tBK/uVoewQrwlZs/kuJAAAEAMkAAAfDBbYADwAbACcAKwBfQDEJBgYHAQ0NABwWIhAQKygWAAcGLC0fEyUZCygTAxkIExkTGSgIKClKWSgSDggDAQcSAD8zPzM/KxESADk5GC8vERI5ERI5ETMRMxESARc5ETMRMxEzETMRMxEzMTAhIwEjEhURIxEzATMmNREzARQGIyImNTQ2MzIWBRQWMzI2NTQmIyIGAzUhFQTHu/1MCBCXwgKqCA6YAvyhk4uioZOLov4iUV1bT09bXFJWAgAEy/7gbPzBBbb7OvWKA0f8t6O4u6CjtbudcnZ1c3NwcP0gh4cAAgAlAuUFhQW2AAcAGABPQCcAAQ8MDA0RFBQTEw0GAQMFGRoXFgkKChEODgQHAwMEEAgIFA0BBAMAP8QyMjkvMxEzETMRMxEzMxEzMzMREgEXOREzETMRMxEzETMxMAEjESM1IRUjAQMjFxEjETMTEzMRIxE3IwMBcXvRAh/TAljJCAZ3u8TLtH8GCNMC5QJnamr9mQIvgf5SAtH90QIv/S8BpIn90wD//wBQAAAF9AXNAgYBdgAAAAIAZv/dBIsESAAXAB8ANEAaHw4OBBgMDBUEAyAhDRQvHz8fAh8fERwIEQAALzIvMxI5L105MxESARc5ETMRMxEzMTAFIiYCNTQ2NjMyFhIVIREWFjMyNjcXBgYTESYmIyIHEQJ5nfGFivSVmPOH/MUxplKDt1FIYtmTMqNYrXojkwEFnav/jI7+/aX+nDVGaYEpm3wCiwEVNUJ1/un//wBH/+wF8wW2ACcCFwJcAAAAJgB7+wABBwJAA2D9swALtAQDAhkZAD81NTUA//8AIP/sBggFyQAnAhcCogAAACcCQAN1/bMBBgB1/wAAC7QBAwIOGQA/NTU1AP//AEf/7AYEBbYAJwIXApwAAAAmAj0MAAEHAkADcf2zAAu0BAMCLBkAPzU1NQD//wBq/+wGAAW2ACcCFwJGAAAAJwJAA239swEGAj8xAAALtAEDAg4ZAD81NTUAAAIAZv/sBDUFxwAaACgAQUAiJgcfDw8AABQHAykqCyJHWQ4ECwsYBBgRRlkYAwQbRlkEFgA/KwAYPysREgA5GC8SOSsREgEXOREzETMRMzEwARACBCMiJjU0EjYzMhYXNxAhIgYHNTY2MzISATI2EjcmJiMiBgYVFBYENaf+7K2su4jol2GSKwT+5j6QMC+bStLY/aJfpngWGYBQZaVlZQOm/vr+NenJwKkBM6FdS1oBlSwhnxcl/uz7xpABA5ZhbIT6gHaCAAIAJwAABG0FtgAFAAwAKEATCQUKBAUEDg0GBQEFCUlZBRIBAwA/PysREgA5ERIBOTkRMxEzMTA3ATMBFSEBBgcBIQEmJwHPpgHR+7oCIT0o/vwC0f7+RGgFTvqwZgT04Xn8/gL5ygAAAQDJ/hAFIQW2AAcAI0ARAAcDBAcECQgFAklZBQMABBsAPzM/KxESATk5ETMRMzEwAREhESMRIREEd/z8qgRY/hAHDfjzB6b4WgABAEz+EATdBbYACwAxQBoHCQkDAAgCCgYCAAQMDQQHSVkEAwAJSVkAGwA/KwAYPysREgEXOREzETMzETMxMBM1AQE1IRUhAQEhFUwCd/2ZBED8sAJD/aQDqv4QawOcAzNsl/z8/I2YAAEAaAKNBCkDFwADABVACQIABQQBAFBZAQAvKxESATk5MTATNSEVaAPBAo2KigABACX/8gS8BpgACAAcQAsICgMJAwYEBAEIAQAvLxI5LzkzEQEzETMxMAUjASM1IRMBMwJvf/7ptAEh6wICiQ4DDof9VAW9AAADAHcBkwUtBAwAFQAhAC0AM0AYHwwrAAAlGQwELi8iHBwRBgkTDygWFgMJAC8zMxEzLzMSOTkzETMREgEXOREzETMxMAEUBiMiJicGBiMiJjU0NjMyFzYzMhYBMjY3JiYjIgYVFBYBIgYHFhYzMjY1NCYFLaeAXZlBPJlYg6iog7V6fLmFovx9Qm02Mm1ITGRhAqFCbTczbkdMZGUCz4O5anRoca2OhrPb16/+u1tkYV1pV1NqAXlcYmFea1RVaQABAAz+FAL4BhQAFAAcQAwIEgISDQMVFhALBQAALzIvMxESARc5ETMxMAEyFxUmIyIVERQGIyInNRYzMjUREAJ9TywxPrClo0o7PTq2BhQQiRbz+uGwuxOHFvMFHwFqAAACAGIBhwQtBB8AFwAvAHBAQCgPGwMPAzEwJx4eGFBZDx4fHi8eAwkDHipAKiRQWRsqQA8GBgBQWQ8GHwYvBgMJAwYSQBIMUFkDABIQEiASAxIAL13EKwAaGBDNX15dKwAQGMQa3sQrABoYEM1fXl0rABAYxBESATk5ETMRMzEwASIGBzU2MzIWFxYWMzI2NxUGIyImJyYmAyIGBzU2MzIWFxYWMzI2NxUGIyImJyYmAVA2fzlslENwWE1bLTWANmWZQ29YSVsxOYA1apZFdFJFXzE3gTNkmkV2T1RVAgBAOZZuHCUhGUI5l20dJR4ZAZZENZVtICIdGkI3lm4gISIYAAABAGgApgQpBQIAEwBGQCYFARALCwkKDgQAEwEIFBUNBQYFUFkKCA8GAQkDBg4CAQJQWRIRAQAvM8QrEQAzGC9fXl3GMysRADMREgEXOREzETMxMAEhNSETITUhExcHIRUhAyEVIQMnAX3+6wFUf/4tAhOHfW0BF/6qgQHX/emDfQHBiQEQiQEfOeaJ/vCJ/uU3//8AaAABBCkE2QImAB8AAAEHAisAAP10AAmzAQAHEgA/NTUA//8AaAABBCkE2QImACEAAAEHAisAAP10AAmzAQAHEgA/NTUAAAIAbwAABD0FwwAFAAkAIEANCAAGAwADCgsJBwIFAgAvLxI5ORESATk5ETMRMzEwEwEzAQEjCQNvAcJIAcT+PEgBYv7D/sMBPQLfAuT9HP0hAuECE/3t/ewA//8AHQAABBwGHwAmAEkAAAAHAEwCtgAA//8AHQAABAwGHwAmAEkAAAAHAE8CtgAAAAEA2wTZA74GDAANABhACQsDDw4KBIAHAAAvMhrMMhESATk5MTABIiYnMxYWMzI2NzMGBgJIuaoKnAlbcWdjC50MsgTZj6RoUlhinpUAAAH/kf4UAVYESAAMAB1ADQsICA4NCQ8ABUZZABsAPysAGD8REgE5ETMxMBMiJzUWMzI2NREzERArXztFQ05Jpv4UGYcUVVcE/PsQ/rwAAAEBiQTNAnUGFAAJABO2CQQKCwSACQAvGs0REgE5OTEwATY2NzMVBgYHIwGJEycKqAtYL1oE5TenURIzvEYAAQFx/jsCb/+DAAkAE7YJBAoLCYAEAC8azRESATk5MTABNjY3MxUGBgcjAXEcMweoC2I3Wv5UQLo1EjPBQgABAYEE2QJ/BiEACQATtgkECgsJgAQALxrNERIBOTkxMAEGBgcjNTY2NzMCfx01BqYOYzFcBgg9wTETPb85AAIAJwI5Ap4FxwALABUAIEAOBgwAEQwRFxYJEx8DDiEAPzM/MxESATk5ETMRMzEwExQWMzI2NTQmIyIGBRAhIiY1ECEyFrBSXl5WVl5eUgHu/sSenQE7np4EAKimpauqpKWp/jfs3QHF6AACABQCSgK0BbwACgAUADxAHxQFCwcDAwkCAAIFAxUWAQUFCQ8UHxQCFBQDDgcfAyAAPz8zEjkvXTMzETMREgEXOREzMxEzMxEzMTABIxUjNSE1ATMRMyE1NDcOAwcHArR9kf5uAZiLff7yBgUYHh4LqAMUysplAkP9zcOGSwwnLS0R9gABADsCNwKJBaoAHQArQBUQAxwXCRcaAwQfHhMAAAYbGB4NBiEAPzM/MxI5LzMREgEXOREzETMxMAEyFhUUBiMiJic1FhYzMjY1NCYjIgYHJxMhFSEHNgFIkbCqpkqLKTiMNl9ubWY5TB87IQHv/oMUPgRoj3uMmx8XgyImU1lOWBEIKQGgaOYMAAACACkCOQKiBccAFwAjADZAHBsSIQsAAAYSAyUkHgsVAA8QDwIPDwMYFSEIAx8APzM/MxI5L10SOTMREgEXOREzMxEzMTATEDYzMhcVJiMiBgczNjYzMhYVFAYjIiYFMjY1NCYjIgYVFBYp29tKMTRTjZYKCB1xVX2Upo2ZrQFEUWNYVlVwagPDAQX/D3ISmaYrO5R+kKTSY11jT1taO1l8AAABADkCSgKPBbYABgAcQA0BBQUAAgMHCAIDHgAgAD8/MxESARc5ETMxMBMBITUhFQGiAV7+OQJW/qACSgL4dF788gADADMCOQKTBccAFQAiAC0AP0AiFg0mEysDHAcHAwUQEw0GLi8FECAgCykbKQIpKRkKISMAHwA/Mj8zOS9dMxI5ORESARc5ETMRMxEzETMxMAEyFhUUBxYVFAYjIiY1NDY3JiY1NDYDFBYzMjY1NCYnJwYGEyIGFRQWFzY1NCYBZHyXlLClipKfSVVKOZ01VFZaVF1RHEhGrERLRFGMTgXHdmiCTEqecYmAdEV0Li5dRGZ+/WY8SUk8P08cCiJUAe88OS9HITZhOTwAAgAjAjkCnAXJABYAIgA8QB8aESAKAAAFEQMjJB0OCgsLFA8OHw4CDg4DFxQfCAMhAD8zPzMSOS9dEjkRMxEzERIBFzkRMzMRMzEwARAGIyInNRYzIBMjBgYjIiY1NDYzMhYlIgYVFBYzMjY1NCYCnNrUUzExXQEUFQojdEGDmamImLD+uFFfVVdUc2cERv7y/w90FAFGMzSSg4ilyltfV1FfVT5hcgAAFgBU/oEHwQXuAAUACwARABcAGwAfACMAJwArAC8AMwA3ADsAPwBDAEcAUwBbAGsAdAB8AIkA+ECHQUA9PDEwDwUADFROWEh2a3BgemeFhkVEKSglJBQKCRcXhgYSOxt/Z2A4GDcvazQsSCMfIBwDEU4MGYqLCgAqQlpRhlx0XClBRj5kdXVsRT2CfVZLa3ZrJjIlMRUNAEIBQT5cPWwNMTIDawxcbGtrbFwDAS0sHRwZGBMSDww5ODU0ISAHBgQBAC8zMzMzMzMzMzMvMzMzMzMzMzMzEhc5Ly8vERIXORE5Ejk5ETk5ETMRMxEzETMQxDLEMhEzETMSOREzETMRMxDExDIRMxEzERIBFzkRMzMzMzMzMzMzETMRMxEzETMRMxEzETMzMzMzMzMzMzEwExEhFSMVJTUhESM1AREzFTMVITUzNTMRITUhFSE1IRUBNSEVASMRMxEjETMBNSEVASMRMwE1IRUzNSEVASMRMzUjETMBIxEzBRQGIyImNTQ2MzIWBRQzMjU0IyIlMzIWFRQGBxUWFhUUBiMjEzMyNjU0JiMjFRUzMjY1NCMBIic1FjMyNREzERQGVAEvwAXOATBt+QBvwAUOw239SQER++EBDv7yAQ4Et21tbW37wgEQ/DBvbwLAARB3ARH6qG9vb28G/m1t+5+Hf3+Hh39+iP5zh4eHhwHhrG1wLiw9Lm1ez3tCLiQqLztKMSVaAV40HCsZVn1pBL4BMG/BwW/+0MH5AgEvwm1twv7RbW1tbQb+b2/6qAEOAgIBD/o7bW0BpgEOBEpvb29v/C8BEHkBD/1oARBJkZyckZKbmpPFxcRhQ1MxQggIDkQ1UVkBYiIgIh3jmislSv76CmYIVgGS/nJfYwAAAwBU/sEHqgYUAAMAHgAqAC5AGQELFyUEHh8RAwkrLCgeFA4iHg4OHiIDAgAALy8XOS8vLxEzETMREgEXOTEwCQMFNTQ2NzY2NTQmIyIGBxc2MzIWFRQGBwYGFRUDFBYzMjY1NCYjIgYD/gOs/FT8VgPrLEFnSbulT7pHUqBaPz4xSFQ7G0dGQklIQ0hFBhT8VvxXA6n7LzJBMVJ+WIeaOCqyUDovNUs2RHBKO/7tP0hJPkBJSP///5H+FAJXBiECJgI3AAABBwFM/qkAAAAIswEYESYAKzX//wAZA8EBRAW2AgYCBwAAAAIACv/sBN8GKwAtADYAZkA5GwcXCzQlLh8fKwItAiULBxIGNzgUDkdZACEuIUdZKy4PLh8uAgkDFC4ULgUoKDFGWSgBBR1GWQUWAD8rABg/KxESADk5GC8vX15dETMrEQAzKxESARc5ETMzETMRMxEzETMxMAEWFRAAISARNDc2NTQmIyIGByc2MzIWFRQHBhUUMyARNCcmJCY1NDYzMgATMxUlJgIjIgYVFAQEVgT+4P79/ncQDyQgGTYPIVNfWF0PEOkBdwTf/smgtqjQAQAqj/7HHLd7XWEBEwNOLkH+n/5uAVg5e3oXLyMPCXYnXV0jg4Q6zwJwPywCabyDkKP+zf7XgYHTAQBfS42aAAEAAAAABHsFwwAVAChAFBESBxIUAxYXABIUAxISBQpKWQUEAD8rABg/PxI5ERIBFzkRMzEwARISNjYzMhcVJiMiDgMHESMRATMCOXqNTVw6MCgaHyg7VnxlH6z+I7oCzQEjATdsMA+HBjih/OxV/eMCLwOHAAACABL/7AZ3BEgAFAApAExAJxgDEiEhHicNCg0eAwYFKisTHx8ACBULBggGRlkIDyQbABtGWRAAFgA/MisRADMYPysRADMzERI5GC85ERIBFzkRMxEzEjkRMzEwBSImNTQTITU3IRUjFhUUBiMiJyMGAQYCFRQWMzI2NTUzFRQWMzI2NTQnAim6x4f+444F1/p1yLndRAhE/s8/Qmx1XWyia111bW8U5/DwAQdKRI78+/DntrYDzoT+/meuqI99vLx6kqmt/u8A//8AyQAABnEHdQImADAAAAEHAHYBnAFUAAizAR0FJgArNf//ALAAAAbLBiECJgBQAAABBwB2Ac0AAAAIswEtESYAKzX//wAA/dUFEAW8AiYAJAAAAAcCWwE1AAD//wBe/dUDzQRaAiYARAAAAAcCWwDHAAD///7f/+wF0gXNACYAMhQAAQcCXP5HAAAACbMDAhoDAD81NQAAAgB1/dUCNf+DAAsAFwAeQAwSBgwABgAYGRUDDwkALzPMMhESATk5ETMRMzEwARQGIyImNTQ2MzIWBzQmIyIGFRQWMzI2AjV9ZmV4eGVlfm5CMzNCPDk1QP6uYXh1YmJ1dmE5PDw5OD09AAIAmARoAs8FxQAIABcAHkAODgkDCAwTCQUYGQILCBUAL8TcxhESARc5ETMxMAE2NzMVBgYHIyU0NxUGFRQeAhUUIyImAbBGHL0pdzFO/ujteR8lH103QwSHtXoUTqw5dqM9SCk1FBMQGhxKRAD//wAdAAAG0wYfACcASQKwAAAAJgBJAAAABwBMBW0AAP//AB0AAAbDBh8AJwBJArAAAAAmAEkAAAAHAE8FbQAAAAIAff/sBmQGFAAVACEAPEAfFgYPEREcAAAUCwYEIiMUCwMJCR9JWQ8JBAMZSVkDEwA/KwAYP8YrERIAOTkREgEXOREzMxEzETMxMAEQACEgABEQACEgFz4CNTMXBgYHFgEQEjMyEhEQAiMiAgW8/p3+xv69/qEBYQFDAUWzMjobtg4dg2hg+3X69PP29fLz/QLd/p7+cQGJAWoBaAGG1wxDZmkWm60nsP7+/tb+zgExASsBJwEx/tEAAAIAc//sBRkE8AAWACIAPEAfFwcQEhIdAAAVDAcEIyQVDAMKCiBGWRAKEAMaRlkDFgA/KwAYP8YrERIAOTkREgEXOREzMxEzETMxMAEQACMiJgI1EAAzMhc+AjUzFwYGBxYFFBYzMjY1NCYjIgYEYv7y7pPkfAEM7tmJMzoatA8feWZH/L2era+dn6+tnAIl/vT+04oBAq0BDAErjQ9BY24XnK8mirnT29vT0tjYAAEAuv/sBnsGFAAbADNAGAUHBwELFBELER0cCgEOGwUSAw4XSVkOEwA/KwAYP8YzEjk5ERIBOTkRMxEzMxEzMTABFT4CNTMXBgYHERAAISAANREzERQWMzI2NREFGTpGH7UOIayV/uH++P70/tSqzMa4wQW2xgg+cG4WtrgZ/Y3+/v7qAR/9A678RrfEwbwDuAAAAQCk/+wFlgTyAB0AREAiARwNDw8TFAcHChMcEx4fFRYKEhYDFA0IHQ8ZBEZZGRYUFQA/PysAGD8zxhIXOREzERIBOTkRMzMRMxEzETMRMzEwAREUFjMyNjURMxU2NjUzFwYGBxEjJyMGBiMiJjURAUx6gqyfplJKsg8gsI2JGAk0tW/LyARG/TuGhLzVAj55C4CaF7q/Dvysk1JVvtECywD///xTBNn93AYhAAcAQ/rKAAD///0NBNn+lgYhAAcAdvuEAAD///wZBNn/AQXdAAcBUvsRAAAAAf0IBLj+cwaPABEAHkAMAgUFDQ0IAAATCxAEAC/MMhEBMxEzMxI5ETMxMAEUBwcjJzY2NTQmIyIHNTYzIP5zpgppDFZOQ0k+ICZFAQAF14wicbAOMisrKQZkCgAB/Tv+oP4C/30ACwARtQYAAA0JAwAvzREBMxEzMTAFNDYzMhYVFAYjIib9OzsqKDo6KCo78jk2Njk3NzcA//8AyQAAA/gHcwImACgAAAEHAEP/2AFSAAizAQ0FJgArNf//AMsAAAVSB3MCJgGyAAABBwBDAGgBUgAIswERBSYAKzX//wBz/+wEEgYhAiYASAAAAQYAQ7cAAAizAhwRJgArNf//ALAAAARiBiECJgHSAAABBgBD3AAACLMBDxEmACs1AAEAhf/sB5EFyQAxAEVAJCIWKicvCQkEJxsWBTIzAB8ZH0lZECgoEwYZBCwlEyVJWQwTEwA/MysRADMYPzMSOS85KxEAMxESARc5ETMRMxEzMTABIgYHJzYzMgAREAAjIiYnIwYGIyAAERASMzIXByYmIyICERASMzI3ETMRFjMyEhEQAgWkPF4tRX6W5AEB/uX/bKxTCFCpa/8A/uX/5Jl8Ri1dPJOlz7uLZqpmjrvOpQUvKR+SUP6I/q3+jf5hLTMyLgGbAXcBUwF4UJIfKf7X/vb+0/6yTAHJ/jdMAUsBMAELASgAAQAAAAAGHQRIAB0AKEAWFwANDgUFHh8bFQ0AEgoEBBYOBQ8EFQA/PzMzEhc5PxESARc5MTABBgYDIwEzExYXMzY2EwMzABYXMzYSETMQAgcjAyYDJwoUs9X+f6z2IC4IE0qOrLIBCS0KCK2ZpsPbtn0hAckaM/6EBEj9SV29NaMBJAHV/P+QLLgBswFS/pb+B+UBWlwAAgAXAAAE/AYUABEAGgBMQCgIBBISAQ8WCwsGDwAEGxwHEQARSVkEAAgaSVkACAAIDwIADxJKWQ8SAD8rABg/Ejk5Ly8rEQAzKxEAMxESARc5ETMRMzMRMzMxMBMhETMRIRUhETMgERQEISERIQEzMjY1NCYjIxcBP6wBov5eyQIx/vf++/5o/sEB69XAtbratgT6ARr+5pT+4P5k0NoEZvwriZCKegAAAgAXAAAEnAUnABEAGQBHQCYEABMTDwsWBwcCCw0EGhsDDQ4NRlkEEkZZBAQLEAAODwsTRlkLFQA/KwAYPzPGEjkvKysRADMREgEXOREzETMzETMzMTABIRUhESEgERQGIyERIzUzNTMRESEgNTQmIwGoAVj+qAE/AbXf3P4h6+umATEBH4ecBEiM/sX+zaaoA7yM3/zN/pe5XFQAAQDJ/+wHIQXLACAASkApFxMTFAYYHQwFGBEUBiEiGwBJWRsEBhIXEklZAxcXFBUDFBIOCUlZDhMAPysAGD8/EjkvMysRADMYPysREgEXOREzETMRMzEwASIEByEVIRIAMzI3FQYjIAADIREjETMRIRIAJTIXByYmBY/j/vwfAr/9PQgBCfeawpje/sH+pQj+oqqqAWQeAXEBMNW2SGSdBTP68Zb+7/7iN5U5AXABVP1QBbb9kgEzAU4CXJIwJgAAAQCw/+wFnARcACEAWUAyFhkZCgMJBQUGECAYAwYFIiMNE0ZZDRAZBAkERlkWDwkfCQILAwkJBgcPBhUAHEZZABYAPysAGD8/EjkvX15dMysRADMYPysREgEXOREzETMRMzMRMzEwBSIAJyERIxEzESE2JDMyFhcHJiMiBgchFSEWFjMyNjcVBgR36/70C/7hpqYBIRgBDd9RmjYyimWjpxACGP3mCamkPXdibhQBCvj+EgRI/jPr9iAZjTOkqo28tRYlkzkAAgAAAAAFbQW2AAsAEgA0QBsCAwcMAw0KBRQTAQUMBUlZEAgMDAcIAwsDBxIAPzMzPxI5LxI5KxEAMxESARc5ETMxMAEjESMRIwEjATMBIwEhJyYnBgcDmJSclf7fsgJongJnt/1cAUxSOB4YQAKq/VYCqv1WBbb6SgM/z5BkYqQAAAIACgAABHkESAALABIANUAcBQYKDAYNAwEGFBMECAwIRlkRCwwMCgsPBgIKFQA/MzM/EjkvEjkrEQAzERIBFzkRMzEwAQEjAyMRIxEjAyMBAyEmJicjBgKoAdGsz3GXc82sAdEhAQ8rOCIJHARI+7gB6f4XAen+FwRI/i1simpcAAACAMkAAAdeBbYAEwAaAEZAJQ4KCgsCAxIVAxQIBwsHGxwFAQkOCUlZFBgMDg4LEAwDEwcDCxIAPzMzMz8zEjkvEjkzKxEAMzMREgEXOREzETMRMzEwASMRIxEjASMBIREjETMRIQEzASMBIQImJwYGBYWPmpP+47oBIv5fqqoB4QEGngJmvP1mAT52HAwTIwKw/VACsP1QArD9UAW2/ZICbvpKA0gBNVYvQ2gAAgCwAAAGFARIABMAGQBNQCsRDQ0OBQYBGQYYCwoOBxobCAQMEQxGWRgVEy8RPxECEREOEw8PDwoGAg4VAD8zMzM/PxI5L10SOTMrEQAzMxESARc5ETMRMxEzMTABASMDIxEjESMDIxMhESMRMxEhExcjBgYHIQRGAc6q0HGYbtGs0f7fpqYBXsVoCAogWQEMBEj7uAHu/hIB7v4SAe7+EgRI/jMBzXMiX9kAAAIAFAAABa4FtgAfACIAS0AoIAEPECEeHh0QAgEHBiQjHgEhHx8hSVkOEh0SSlkiAh0dGB8DEAgYEgA/MzM/EjkvMzMrEQAzKxESADk5ERIBFzkRMxEzETMxMAEVAR4CFxMjAy4CIyMRIxEjIgYGBwMjEz4CNwE1BSEBBSn+WnaaZDKFrokjRGVZG6oaW2NBIIe5iC9jlXb+ZQO+/QoBewW2hf4RBkiLpP47AclvYCb9QgK+J19v/jcBxZ+OSQcB74WZ/jkAAAIADAAABRQESAAgACMATkAqIQEPECIfGB8eEAIBBwclJB8BIiAgIkZZEQ4SHhJHWSMCHh4YIA8QCBgVAD8zMz8SOS8zMysRADMzKxESADk5ERIBFzkRMxEzETMxMAEVAR4DEyMDLgIjIxEjESMiBgYHAyMTPgM3ATUFIQEEi/6uV29JMZushSI6VEwKmQtLUjgnh6qDGDBJblf+sQMg/bQBJQRIaf6gBzBQaf5xAVBXRxz99gIKGkBe/q4BUD1pTzIIAWBpjP7BAAACAMkAAAfFBbYAJAAnAGFANSEdHR4mIw8QAiclAQcBJxAiGyMYHgkpKCMBJCYkJklZEg4cIRxJWScCISEeJAMfAxgQCB4SAD8zMzM/PxI5LzMzKxEAMzMrERIAOTkREgEXOREzETMRMxEzETMRMzEwARUBHgIXEyMDLgIjIxEjESMiBgYHAyMTNjchESMRMxEhATUFIQEHPf5deJllLYioih9GaV8YrBleZEIhh7KHNzj+UqqqAtf+aAPB/QoBewW2hf4OBkiQnP47AcloYyj9RAK8KF9s/jcBvrg6/VAFtv2SAemFmf43AAACALAAAAa6BEgAJAAnAGdAOiEdHR4mIw8QAiclAQcBJxAiGyMYHgkpKCMBJCYkJkZZEg4cIRxGWScCLyE/IQIhIR4kDx8PGBAIHhUAPzMzMz8/EjkvXTMzKxEAMzMrERIAOTkREgEXOREzETMRMxEzETMRMzEwARUBHgMTIwMuAiMjESMRIyIGBgcDIxM2NyERIxEzESEBNQUhAQYx/q5Yb0kwm6yFIjpWSgqaCktUNyaHqoMvJf7NpqYCNf6wAyH9tAElBEhp/p4HMU5p/nIBUFZGHP34AggbP1z+rgFQeCj+EARI/jUBYmmM/scAAQA//k4ENQbRAEsAhEBNABMhPxlGRgo/N0M8KhwtKBMLTE1JFkpZSRM5NDEPLh8uLy4DCQMuKkBDHRwdHEpZHR0QPCoqJEpZKgQKCUlZChAQA0lZECMMB0lZDCIAPysAGD8rABgQxisAGD8rEQAzEjkYLysREgA5GhgQ3V9eXTnEMj8rERIBFzkRMxEzETMxMBcUFjMyNzYzMhcVJiMiBwYjIiY1NDY3NjY1ECEjNTMyNjU0JiMiBgcnNjcmJyc1MxYXNjYzMhcVJiMiBgcWFhUUBgcVFhYVFAQFBgbwV1lheHhGm0dQoERpaWmzuNnozLX+QNrRzeGiiWq7blaovjl1MXtcg1yDQDIwGCssbzCywb+qusv+5f7mioaJNzIHBiemMwUFfYV+gQkIio0BDI+ThGuAN0VychxCeTQbO4hzVg5xClJHF72PjLgaCBiykNDVCQU3AAABABn+ewN/BU4ARgCDQE4XKTYLLhAQIAsDDgg+MkA8KQtHSEQ+QQAFR1kAD0EfQS9BAwkDQT4mGkZZIx1GWQ4zMjMyRlkmIzMzIyYDID4+OEZZCD4QICITLEdZExYAPysAGD8/MysREgAXORgvLy8rERIAOSsrABgQ1F9eXcQrERIAORESARc5ETMRMxEzMTABMhcVJiMiBgcWFhUUBxUWFRQGBw4CFRQWMzI3NzIXFSYmIwcGIyImNTQ2NyQ1NCYjIzUzIDU0IyIGByc2NyYnNTMWFzY2AvgzLRgpL2cteozT+PLhXW0wS1lWeq99JxVUN7OCXJCfvrQBTpyflHcBN/xKj1g7fH5cZ3tLjFiGBU4PcApPPhyKa7g5CEfKlKgDAhcqLDErBQUnjxMYBQV3cHR9AwS+YVqNrKIiJIc3D3ViGzSJblX//wBtAAAF8gW2AgYBdQAA//8ApP4UBYcGEgIGAZUAAAADAH3/7AW+Bc0ACwASABkAR0AlFhAQBhcPDwAGABobFhBJWQ8WAQsDFhYDCQkTSVkJBAMMSVkDEwA/KwAYPysREgA5GC9fXl0rERIBOTkRMxEzETMRMzEwARAAISAAERAAISAAATISEyESEhMiAgMhJgIFvv6d/sT+vf6hAWABRAE7AWL9YeX3DfwrDfno4PsTA9MR9ALd/qH+bgGLAWgBZQGJ/nD8RAERAQz+9f7uBLT+/v8A/gEEAAADAHP/7ARiBFwADAATABoASUAnFxERBxgQEAAHABscFxFGWQ8XHxcCCwMXFwMKChRGWQoQAw1GWQMWAD8rABg/KxESADkYL19eXSsREgE5OREzETMRMxEzMTABEAAjIiYCNRAAMzIAATI2NyEWFhMiBgchJiYEYv7y7pPkfAEM7uYBD/4InqQK/WkJoKCcng0Ckw+hAiX+9P7TigECrQEMASv+zv1NuL+6vQNYraeorAAAAQAAAAAFSAXDABUAIEAQBhYTFxEASlkRBAoFBgMFEgA/PxI5PysRATMSOTEwASIGBwEjATMBFhc2NxM+AjMyFxUmBOE7Tjn+uMX97rQBUkgjIEaiO1RuWSpPOAU3Z7X75QW2/FbHj5DfAga/mEETjRQAAQAAAAAEPQRSABYAHkAPARcPGA0SR1kNEAUBDwAVAD8/OT8rEQEzEjkxMCEBMxMSFzM2ExM+AjMyFxUmIyIGBwMBlv5qruFkEwgXUmAlR1tULR4dJi86HPgESP2b/vRkdgELATV6ezQKfwhUXPzf//8AAAAABUgHcwImAoAAAAEHA3YE1wFSAAq0AgEhBSYAKzU1//8AAAAABD0GIQImAoEAAAEHA3YEZAAAAAq0AgEiESYAKzU1AAMAff4UCaIFzQALABcALgBEQCYMBhIAIS4nGAAGBi8wJSpKWSUbHRwcAyAYDwkVSVkJBAMPSVkDEwA/KwAYPysAGD8zEjkRMz8rERIBFzkRMxEzMTABEAAhIAAREAAhIAABEBIzMhIREAIjIgIlMxMWFzM2NhMzAQYGIyInNRYzMjY3NwVU/rn+3P7X/r0BQwEsASMBRfvd39na3dzY2uEEb7D2ThQIC1PksP4rRbyITEo3Ql51Iz0C3f6g/m8BiwFoAWYBiP5w/qD+1/7NATEBKwEpAS/+0kH9i89mLPsCg/sgtp4RhQxnWZz//wBz/hQIewRcACYAUgAAAAcAXAR1AAAAAgB9/4cGEAYtABMAKABRQCoUCiYNBxEiIgMcHwAAHAcXCgUpKiQiJg0mSVkRDw0DHBoXBxdJWQUDBxIAPzMzKxEAMzMYPzMzKxEAMzMREgEXOREzETMzETMRMzMRMzEwARAABQYjIickABEQACU2MzIXBAABFBIXNjYzMhc2EjU0AicGIyInBgIGEP7R/vgad3wU/vT+0QErARAUfHkWAQwBLfshyr0RSTZuH73Kyr0fbnEfvcoC3f7S/nMsb28pAYoBNgExAYUsbGws/nP+1fT+zykwJlYpATH09AEvJ1hWJ/7TAAACAHP/kwTPBLQAFwAtAFBAKhgMDwkrGyUVAyMAAAMgGwkMBi4vKCUrDytGWRUSDxAgHhsJG0ZZBgMJFQA/MzMrEQAzMxg/MzMrEQAzMxESARc5ETMRMzMRMxEzETMxMAEUAgcGBiMiJicmAjU0Ejc2NjMyFhcWEgUUFhc2NjMyFzY2NRAlBgYjIiYnBgYEz+DMCUA4OT0Jy+Xg0Ag+OThACcri/FB9iQw8NWcYhnz+/A09MzU8DIl9AiXp/t8lNi0rOCQBJuXpASAkOCorOSb+3OGx0h8qIkof0q8BYD4qICAsH9EAAAMAff/sB38IOwAVAEUAVABVQC5DNx8rKwEmRktQSDwMNwpVVhUCAgcHEAxSQEg6IkA6QElZKDoEHBY0FklZLjQTAD8zKxEAMxg/MysRADMYENYa3NTNMhI5LzMREgEXOREzETMxMAEVIyIuAiMiBhUVIzU0NjMyHgIzATI2NxYWMzISERACIyIGByc2MzIAERAAISImJwYGIyAAERAAMzIXByYmIyICERASARQHNTY1NC4CNTQzMhYFohFUjnhmKy88fXRwOnB3hU79KFirPTerXbzSpZM8XytGeZrkAQH+4P79aKpMS6du/vz+4wEB5Jp5RitePJSl0gKA7XgfJB9cOEMHx3kkKyQ0MxAcZ24kLCT4ukI/OUgBTgEtAQsBKCsfklL+iP6t/oz+YigwLSsBnQF1AVUBdlKSHyv+2f70/tH+tAZooj1IKTUUEhEaHElEAAADAHP/7AYEBwYAKgA/AE4AXEAzEwccKCgsIkBFDUpCNgcKT1AyOj8tLTZMQgpAHxAKEEZZAhdGWQIEJQoQGhUEFUZZAAQWAD8zKxEAMxg/MxI5KysRADMaGBDe3NQyETPNMhESARc5ETMRMzEwBSInBiMiAhEQEjMyFhcHJiMiBhUQITI3FhYzIBE0JiMiByc2NjMyEhEQAgMVIyIuAiMiFRUjNTQ2MzIeAjMFFAc1NjU0LgI1NDMyFgQrlF5cj+H6z7o+dyg5WUd0bQExe3A+b0MBLW5zR1k5KHc+u873URBUj3hlK2t9c3A6cXaDTv7w7nceJB5cOEMUQUEBIwEOARcBKCAZizPW1v5eUComAaLW1jOLGSD+1/7q/vX+2galeCQqJGYRH2RvJSsl3aE+SCg4FBERGRtKRAAAAgBe/+wHfwcEAA0AQABfQDQwJDk2PhcXARI2KQwkB0FCDi0nLUlZHjc3IScFCQkNQAkPSA0HAwtAFCcEOzMhM0lZGiETAD8zKxEAMxg/MxreMjLNKzIRMxESOS85KxEAMxESARc5ETMRMxEzMTABFQcjJyMHIycjByMnNQEiBgcnNjMyEhEQACEiJicjBgYjIAAREAAzMhcHJiYjIgIREBIzMjY3ETMRFjMyEhEQAgWLUCAyujEhMbwvIVADQzxdLUZ8meT//uL+/XSsTAlOrHD+/P7jAQHlln5GLV08k6XSvkGCM6pmkbzUpQcEG6xnZ2dnrBv+KykfklD+iP6t/ov+YzAwMS8BoAFyAVUBdlCSHyn+1/72/tH+tCYmAcn+N0wBSgExAQsBKAAAAgAAAAAGHQWkAA0AKgA/QCQkAQ4aGwwSByssKBUOHxYDERIFCQkNQAkPSA0HAwsjGxIPERUAPz8zM94yMs0rMhEzERIXOT8REgEXOTEwARUHIycjByMnIwcjJzUBBwMjATMTFhczNjYTAzMAFhczNhIRMxACByMDJgS2Uh4yvDEfMbwyHlABrCeq1f5/rPYnKQgMI7qssgEJLQoIrZmmw9u2fSEFpBusZ2dnZ6wb/CVf/pYESP1Jb6sjUQGIAdX8/5AsuAGzAVL+lv4H5QFaXAAAAQB9/hQE4wXLABcALUAYAw8JChUKDwMYGRMASVkTBAwGSVkMEwobAD8/KwAYPysREgEXOREzETMxMAEiABEQACEyNxEjESMgABE0EiQzMhcHJgNI9f7gAQoBAm85qhT+tf6frwFI2O2qR6sFM/7A/uj+2v7UF/10AdgBhAFt4AFWuFSSTgABAHP+FAOiBFwAGAAvQBgPAxcWCRYDAxkaFxsGDEZZBhAAEkZZABYAPysAGD8rABg/ERIBFzkRMxEzMTAFIgAREAAzMhYXByYjIgYVFBYzMjY3ESMRAnX+/vwBEftPpDAxjmixq6urNVA5phQBHwESARQBKyIXjTPN3dzIERr9bgHYAAABAGr//AR1BQYAEwAvQCEEAggDBgARBwoQDRIMDg4VFBMAAxEGDwUQBw0KCQwLARIAP80XORESARc5MTABAycTJTcFEyU3BRMXAwUHJQMFBwICtnm2/uFCASHN/t9DASG5drgBIUT+4cwBHkEBOf7DQwFCpnOoAWSmdagBPUP+wKZzpv6eqHMAAQDLBJEDrAW0ABMAHkAMAAYKEAYQFBUDAA0JAC8zMzIREgE5OREzETMxMAEGBiMiJjU0NjMhNjYzMhYVFAYjAYcGKjAzKSo2AcEGKy8zLSw2BPAtMjI1NSkuMDEzOCgAAQD4BOUD2wXXABMAHEALBxIVFAASEgwEgAkALxrMMjMRMxESATk5MTABMjc2MzIWFRUjNTQjIg4CIyM1AQR4lpVRb3R9aitmeY5UEAViOzpvZB8RZiQrJHkAAQHfBNcCzQY1AA4AGEAKCgAMBQADDxADDQAvzBESARc5ETMxMAE0NjMyFRQOAhUUFxUmAd9DOFweJB537gW4OEVMGxkQEhQ2KEpAAAEB4QTXAs8GNQAOABhACgUAAAoCAw8QDAIAL8wREgEXOREzMTABFAc1NjU0LgI1NDMyFgLP7nceJB5cOEMFuKFASig2FBIQGRtMRQAIACn+wQfBBZEADAAaACgANgBEAFIAXwBtAIBASV8oRFoiPgwaBxRSNm1MMGcQbm8ABzpISEFPRUQ+TFZjY1xqZl9abR4sLCUzLyIoAzYQFwdPTGptMzYXFzYzbWpMTwcICQ0UAwkALzMvMxIXOS8vLy8vLy8vETMRFzMRMzMRMxEzMzMRMzMRMxEzMzMRMzMRMxEzERIBFzkxMAEmJiMiBgcjNjMyFhcDJiYjIgYHIzY2MzIWFwEmJiMiBgcjNjYzMhYXISYmIyIGByM2NjMyFhcBJiYjIgYHIzY2MzIWFyEmJiMiBgcjNjYzMhYXASYmIyIGByM2MzIWFyEmJiMiBgcjNjYzMhYXBG8FPEVOMgVLC8VdcQdPBTxFTjIFSwVkZ1xzBgH0BTxETjIFTAVlZ1xzBvsvBTxETjIFTAVlZ1xzBgQxBTxETjIFTAVlZ1xzBvsvBTxETjIFTAVlZ1xzBgTwBTxETjMFSwvGXHMG+b4FPEROMgVMBWVnXHMGBM8sLCkvwmVd+fIsLCkvWWlmXAEWLSsnMVppZl0tKycxWmlmXQPbLSsnMVppZl0tKycxWmlmXf4ZLCwoMMJoWi0rJzFaaGZcAAAIACn+fwd9BdMABwAPABcAHwAnAC4ANQA+ADRAJRUXJSA+OgUBKSwfHDI1CQ0QP0A7KwcuNhkVHREvJw8kMw4FDAUALy8SFzkREgEXOTEwBRcGBgcjNjcDJzY2NzMGBwE3FhYXFSYnBQcmJic1FhcBNzY2NxcGBwEHBgcnNjcDJyYnNxYXARcWFhcHJiYnBDcLEUYkYTUROwsTSR9hNBICIw5HyEHdgftoDkK/T92BA6YCQ75DRbF4/OoCm6lFsXgrEVJFQ3tMA2oRJ1oWQx+CJiMOQr9P3YEEmA5HyEHcgv4WCxNJH2E1ETsLEUYkYTURAaoQJ1gZRG5Y/JUQWT9EblgC3gKMt0bGY/zpAkXCPEYywzQAAAIAyf6DBggHXgAUACIAWUAvDQoMBw4OCRMCAhQUGCAJCgUkIxQSBgUREgUSDgAOCUlZDhIMIh8PGAEYHBUHAAMAPzLeMs1dMj8/KxESADk5ETMRMxg/ERIBFzkRMxEzETMRMzMRMzEwEzMRFAcHMwEzETMDIxMjETQ3IwEjASImJzMWFjMyNjczBgbJoQoECAM0uLiPxZygEwn8yboCQ7qoCpsKXW5pYwmeDLUFtvzRds5TBMb64v3rAX0DJa/3+zUGK4+kbE5dXZ+UAAIAsP6HBRIGDAARAB8AT0AqCgcJBAsLBg8BARAQFR0GBwUhIAMOEBEPCwZGWQsQFQkiHA8VARUZEgQPAD/eMs1dMj8/MysAGD8SOTkREgEXOREzETMRMxEzMxEzMTABERQHATMRMwMjEyMRNDcBIxElIiYnMxYWMzI2NzMGBgFMCgJRz7CBrH2bCP2uzQHsuaoKnAdadGdkCp0MsgRI/WqIiAOm/Ef9+AF5AqCeaPxaBEiRj6RmVFpgnpUAAgAvAAAEfQW2ABEAGQBNQCkIBBISAQ8VCwsGDxEEGhsIGUlZBxEAEUlZBAAIAAgADwIPEkpZDxICAwA/PysREgA5ORgvLxEzKxEAMysREgEXOREzETMzETMzMTATMzUzFSEVIREzIBEUBCEhESMBMyARNCYjIy+aqgFW/qrAAkr+7P7x/m+aAUTdAXu4ydcE/Lq6lv7g/mTS2ARm/CsBGYSAAAACABQAAARMBhQAEgAaAEtAKAQAFBQQDBcICAIMDgQbHAQTRlkDDg8OR1kADwQPBA8MEQAMFEZZDBUAPysAGD8SOTkvLxEzKxEAMysREgEXOREzETMzETMzMTABIRUhESEyFhUUBiMhESM1MzUzEREhIDU0JiMBVgEn/tkBQN/X4N3+IZycpgExAR+EnwUfgf3lmpukqgSegfX74P6XuVxUAAACAMkAAAR5BbYADwAcAEhAKRAKCgsYAAAEBQMWBhUTFAsKHR4WExwQDBxKWQkQSlkGAwwJCQsMAwsSAD8/EjkvEjk5KysREgA5ORESARc5ETMRMxEzMTABFAYHFwcnBiMjESMRISAEATMyNyc3FzY1NCYjIwR5c2x4ZJVmiLiqAYkBEgEV/PqmV0xsbIx/wsrIBAx/yTmdVMAb/cEFttf98gqNUrBIspGOAAIAsP4UBHUEXAAYACkAVUAxHQsEBwcIJxISFRYUJRciJCMICiorJSIZIA8ZRlkMCwsEFBcEAA8QCQ8IGwAgRlkAFgA/KwAYPz8/Ehc5ETMrERIAOTkREgEXOREzETMRMzMzMTAFIiYnIxYVESMRMxczNjYzMhIREAcXBycGAyIGBxUUFjMyNyc3FzY1NCYCrmuxPAwMpocZCECpbdrtt3Nkg0dtqJYCmqovKXlqgWWWFE9SlCL+PQY0llpQ/tb+8/6ukZxQrhgD47rLJefHDJ5Qqmf519EAAAEALwAABAgFtgANADxAHwMHBwwIAAUICgQODwYKCwpJWQMLCwgNDQJJWQ0DCBIAPz8rERIAORgvMysRADMREgEXOREzMxEzMTABFSERIRUhESMRIzUzEQQI/WsBqP5YqpqaBbaZ/gKW/XcCiZYClwABABIAAANCBEgADQA8QB8CBgYLBwAEBwkEDg8FCQoJR1kCCgoHDAwBRlkMDwcVAD8/KxESADkYLzMrEQAzERIBFzkRMzMRMzEwASERIRUhESMRIzUzESEDQv4UAVr+pqaengKSA7z+qH/+GwHlfwHkAAABAMn+AATbBbYAGwBBQCMJAwMEGQ4OBxQEBBwdERdJWREcCwBJWQsLBAUFCElZBQMEEgA/PysREgA5GC8rABg/KxESARc5ETMRMxEzMTABIgcRIxEhFSERNjMgABEQACEiJic1FjMgETQAAjFkWqoDSf1hWnkBQAFV/uL+/VN9RnuJAX//AAKPDP19BbaZ/fwK/q3+xv7F/qUVHJgxAf71AQQAAAEAsP4KA/oESAAbAEFAIwgZFA4ODw8CEhkEHRwWC0ZZFhYPEBATRlkQDw8VAAVGWQAbAD8rABg/PysREgA5GC8rERIBFzkRMxEzETMxMAEiJzUWMzI2NTQmIyIHESMRIRUhETYzIAAREAICRpFldHuFiLK1RUqmApr+DFI7ARABB+T+CjyVP8rX39AR/iUESI7+twz+5f7Z/vX+2gAAAQAC/oMG+AW2ABUATUApBhERAxINDAwICRIAARUHFhcSFRITEAkGAwAADwEPCklZDxINIgcEAQMAPzMzPz8rERIAOREzMzMzMxg/MxESARc5ETMRMzMRMzEwAQEzAREzEQEzAQEzESMRIwERIxEBIwJW/cG+AjmkAjq+/cAB2rSiXv26pP27xwLwAsb9PALE/TwCxP08/aj96QF9AuX9GwLl/RsAAAEABP6HBh8ESAAVAEtAKAINDRUOCQgIBAUOEhMRBxYXFQ8MBQISEgsDABMPDhEVCwZGWQsVCSIAPz8rABg/Mz8zMxI5ETMzMzMzERIBFzkRMxEzMxEzMTABMxEBMwEBMxEjESMBESMRASMBATMBAqSZAcW2/jYBcMGiXv4emf4fvwHw/je2AcMESP3tAhP97f5a/fgBeQIt/dMCLf3TAjUCE/3tAP//AEr+QgQ1BcsCJgGxAAAABwN/AVgAAP//AET+QgN/BFwCJgHRAAAABwN/AQgAAAABAMn+gwUrBbYADwA7QCAMCAgJAwICDg8GCQUQEQ8MBgMFDQoDCRIFAElZBRIDIgA/PysAGD8/MxIXORESARc5ETMRMxEzMTAlMxEjESMBBxEjETMRATMBBH+somb96ZmqqgKXyf20mv3pAX0CxYj9wwW2/SsC1f2FAAEAsP6FBD0ESAAOADpAHw4KCgsGBQUBAgsEDxACDgkDCAAMDwsVCANGWQgVBiIAPz8rABg/PzMSFzkREgEXOREzETMRMzEwATMBATMRIxEjAREjETMRAy+2/icBf7KfVP4MpqYESP3v/lj99gF7Aiv91QRI/esAAAEAyQAABOkFtgASADhAHgYCAgMKEREHEg4MEgMEExQICgYAEBIGAwsEAw8DEgA/Mz8zEhc5ERIBFzkRMzMRMxEzETMxMAEHESMRMxE3ETMVATMBASMBESMB8H2qqn19AZvL/bQCYsj+TH0CqGv9wwW2/SWLAV3TAcb9hfzFAlz+zwABALAAAAQ7BEgAEwA6QB8GAgIDDgoSEgcTDwwTAwQUFQgKBgEREwYDCwQPEAMVAD8zPzMSFzkREgEXOREzMxEzMxEzETMxMAEnESMRMxE3ETMVATMBFQEjARUjAc13pqZ3gwEOtv48AevC/tWBAbJ5/dUESP3reQFKzQEf/iVr/f4BO90AAAEALwAABOkFtgATAEdAJggEEBABEQsODAoGDhETBhQVBxMAE0lZBAsIDgMRAAACDRESCQIDAD8zPzMSOS8SFzkzKxEAMxESARc5ETMRMzMRMzMxMBMzNTMVMxUjEQEzAQEjAQcRIxEjL5qq3d0Clcv9tAJizv3xmaqaBQSyspf+bgLb/YX8xQLFhv3BBG0AAAEAFAAABBsGFAAZAE1AKwoIBBYWARcSEAYRFxkGGhsUCg8TFxUHGQAZR1kEDwAfAC8AAwAAAg8PAgAAPz8SOS9dMysRADMYPzMSOTkREgEXOREzMxEzMzMxMBMzNTMVIRUhEQcHMzc2NgEzAQEjAQcRIxEjFJykAX3+gwMDCBI3KAFwx/5EAdnH/n19pJwFWrq6f/3oWzcYSjABhf4t/YsCBGr+ZgTbAAEAEAAABYMFtgANADVAGwIKCgsFCAYECAsEDg8IAgAHCxIDAwANSVkAAwA/KwAYPz8zEjk5ERIBFzkRMxEzETMxMBMhEQEzAQEjAQcRIxEhEAH8ApbL/bQCYsn97Jqq/q4Ftv0lAtv9hfzFAsWI/cMFHQAAAQApAAAE4wRIAAwANUAbBQEBCQkKDAoEBgQODQgCAAcKFQMPAAxGWQAPAD8rABg/PzMSOTkREgEXOREzETMRMzEwEyERATMBASMBESMRISkCAgHbtv4nAgDC/gqk/qIESP3rAhX97f3LAiv91QO8AAEAyf6DBcEFtgAPAERAJAwICAkNBQUAAwICAAkDEBEMB0lZDAwFDgoDCRIFAElZBRIDIgA/PysAGD8/MxI5LysREgEXOREzETMRMxEzETMxMCUzESMRIxEhESMRMxEhETMFH6Kiqvz+qqoDAqqa/ekBfQKw/VAFtv2SAm4AAAEAsP6HBPgESAAPAE5AKwENDQ4CCgoFCAcHBQ4DEBEBDEZZDwEfAQILAwEBCgMPDw4VCgVGWQoVCCIAPz8rABg/PzMSOS9fXl0rERIBFzkRMxEzETMRMxEzMTABESERMxEzESMRIxEhESMRAVYCZqaWppb9mqYESP41Acv8R/34AXkB7v4SBEgAAAEAyQAABm8FtgANAD9AIQoGBgcLAwMCAAIHAw4PCgVJWQoKBwwMAUlZDAMIAwMHEgA/Mz8/KxESADkYLysREgEXOREzETMRMxEzMTABIREjESERIxEzESERIQZv/rCs/QCqqgMAAfwFHfrjArD9UAW2/ZICbgABALAAAAXBBEgADQBJQCcBCwsMAggIBwQHDAMODw0PAQpGWQ8BHwECCwMBAQMIDBUDBkZZAw8APysAGD8zEjkvX15dKwAYPxESARc5ETMRMxEzETMxMAERIREhFSERIxEhESMRAVYCZgIF/qGm/ZqmBEj+NQHLjPxEAe7+EgRIAAEAyf4ACB0FtgAdAEdAJgQFCAAAARcNDRIBBQQeHxAVSVkQHAoaSVkKCgUGBgNJWQYDAQUSAD8zPysREgA5GC8rABg/KxESARc5ETMRMxEzETMxMCEjESERIxEhETYzIAAREAAhIic1FjMgETQCIyIGBwTZqv1EqgQQRH0BMgFR/uX+/px7hn8BeuboKn8YBR364wW2/WEM/qj+yP7H/qYxmDEB/vIBBQcFAAABALD+CgaoBEgAHABHQCYREhUNDQ4HGhoCDhIEHR4XCkZZFxcSExMQRlkTDw4SFQAFRlkAGwA/KwAYPzM/KxESADkYLysREgEXOREzETMRMxEzMTABIic1FjMyETQmIyIHESMRIREjESERNjMyABEQAgUXg2FtbPCmrENIqP3fpgNvS0L2AQbR/go8lT8Bod/QFf4pA7j8SARI/icO/tf+5/70/tsAAgB9/6wF4QXNACgANABQQCwbES8jKQAIAAMWICMRBzU2JixKWQwyJiYOFBQZSVkUBAoFSVkKDg4eSVkOEwA/KwAYEMQrABg/KxESADkYLzk5KxESARc5ETMRMxEzMTABFAIHFjMyNxUGIyInBiMgABEQACEyFwcmIyAREBIzMjcmAjU0EjMyEgM0JiMiBhUUFhc2NgW4inRCWk49OFuylGaQ/sr+oQFJATp/XC9UWv4z/+s2LlZcxq+1wbBnXV5nXVNmcwKmtf7LVh4WmRlkJAGJAVYBeAGKI5Ec/Z7+4P7OCmcBHKD0AQr+9v7+sczJsIz+VUP/AAACAHP/xwTTBFwACgA1AFBALB4TACYGLDQsLxgkJhMHNjcpCEdZDQMpKQ8WFhtGWRYQCzFGWQsPDyFGWQ8WAD8rABgQxCsAGD8rERIAORgvOTkrERIBFzkRMxEzETMxMAEUFhc2NjU0IyIGASInBiMiJiY1EBIzMhcHJiMiBhUUFjMyNjcmNTQ2MzIWFRQGBxYzMjcVBgLuRD9EU4dISwFmk4Jge5XievjjW00lNk+ckaqkJTUGi6iXlJ1rXjRDQjEnAfJeoTUsnm7rff1jTSiL/qQBEwEwFooT0efO0gkDlOGtwb2xfdFAGg6JDgD//wB9/kIEzwXLAiYAJgAAAAcDfwIlAAD//wBz/kIDiwRcAiYARgAAAAcDfwGDAAAAAQAQ/oMEWgW2AAsAMkAbBgsICQMJCwEEDA0LBklZCxIJIgUBAgFJWQIDAD8rEQAzGD8/KxESARc5ETMRMzEwASE1IRUhETMRIxEjAd/+MQRK/jGioqwFHZmZ+3396QF9AAABACn+hwORBEgACwA0QBsGCwgJAwkLAQQMDQkiBQECAUZZAg8LBkZZCxUAPysAGD8rEQAzGD8REgEXOREzETMxMAEhNSEVIREzESMRIwGJ/qADaP6elqaWA7yMjPzT/fgBeQD//wAAAAAEewW2AgYAPAAAAAEAAP4UBAIESAANAClAFAABDAEDAw4PCAcNBwILAw8CFQEbAD8/PzMSOTkRMxESARc5ETMxMAEjEQEzExYXMzY3EzMBAlSm/lKs7FMTCCFG6az+Uv4UAegETP2b3mGKtQJl+7QAAAEAAAAABHsFtgAQADpAHgQICA0JAgYJCw8FERIHCwwLSVkEAA8MDAkBDwMJEgA/PzMSOS8SOTMrEQAzERIBFzkRMzMRMzEwAQEzARUhFSERIxEhNSE1ATMCPQGGuP4YASv+1az+0wEt/hm6AtsC2/yBO5j+nAFkmDMDhwABAAD+FAQCBEgAEwA8QB8RAQEGAhATAgQHBRQVDAsLBQ8HDwAEBQRHWREFFQIbAD8/MysRADMYPzMSOREzERIBFzkRMzMRMzEwBREjESE1IQEzExYXMzY3EzMBIRUCVKb+6gEU/lSs7FMTCCFG6az+VAESgf6VAWuBBEj9m95hirUCZfu4gQAAAQAI/oME1QW2AA8AN0AgAwICDg8MBgkKCAgQEQwPCQYEBQ0KAwgSBQBJWQUSAyIAPz8rABg/PzMSFzkREgEXOREzMTAlMxEjESMBASMBATMBATMBBDOiol7+d/5wtAHm/ju8AWsBbrX+O5r96QF9AoP9fQL8Arr9vQJD/UwAAQAn/oUENwRIAA8AOUAhCgkJBQYDDQABDwgQEQ8VAwYADQQMAQwHRlkMFQoiBAEPAD8zPz8rERIAFzkYPxESARc5ETMxMAEBMwEBMwEBMxEjESMBASMBuP6DvQEhASC7/oMBK5WmRf7N/sq8AjECF/5cAaT96f5e/fYBewG8/kQAAAEAEP6DBqgFtgAPAEBAIgwFAA0DAgINCgUHBRARDgMLBwgHSVkIAwAMBQxJWQUSAyIAPz8rEQAzGD8rEQAzGD8REgEXOREzETMRMzEwJTMRIxEhESE1IRUhESERMwX+qqL7tP5WBC/+JQLwqpr96QF9BR2Zmft9BRwAAQAp/ocFmARGAA8AP0AiAgsGAwkICAMACw0FEBEBDQ4NRlkODwYCCwJGWQsVCSIEDwA/Pz8rEQAzGD8rEQAzERIBFzkRMxEzETMxMAEhESERMxEzESMRIREhNSEDef6XAkamnKb8eP6/A1ADuvzVA7f8Sf34AXkDuowAAAEAqv6DBWgFtgAXADtAHxUABQMCDwwCBQwDGBkSCUlZEhIFFg0DBQBJWQUSAyIAPz8rABg/MxI5LysREgEXOREzETMRMzMxMCUzESMRIxEGBiMiJjURMxEUFjMyNjcRMwTHoaGqlcZqz9+qf49hsamqmv3pAX0CXDUnvrMCRf3PeXQdNwLKAAABAJz+hQTDBEgAFgA7QB8BFQkGDgwLCw4VAxcYAxJGWQMDDgcWDw4JRlkOFQwiAD8/KwAYPzMSOS8rERIBFzkRMxEzMxEzMTABERQzMjY3ETMRMxEjESMRBgYjIiY1EQFC21umaaaWppZps3GkugRI/nDAOEMB1fxH/fYBewHwSDuskwGcAAEAqgAABMcFtgAWAEpAJgUCCxUVCBYNEREQEBYCAxcYFAAIAElZCwgWCAkJCBYDAxESDgMDAD8zPxIXOS8vLxEzKxEAMxESARc5ETMRMxEzMxEzETMxMAEgEREzERQWMxEzETY3ETMRIxEGBxEjAnX+NaqHmn2Go6ysqIF9AgABcQJF/c93dgFc/qoNPALP+koCWEER/s8AAQCcAAAEHQRIABcASkAmARYGEBADEQgMDAsLERYDGBkPEwMTRlkGAxEDBAQDEQMMCRcPDBUAPz8zEhc5Ly8vETMrEQAzERIBFzkRMxEzETMzETMRMzEwAREUFxEzETY3ETMRIxEGBxUjNSMiJjURAULId3GFpqaAdncWoLgESP5wugYBLf7dGFkB1fu4AfBbGvjqqpUBnAABAMkAAATlBbYAEgAvQBcCERESCQgIEhQTBA1JWQISBAQJEhIAAwA/PzM5LxI5KxESATk5ETMRMxEzMTATMxEkMzIWFREjETQmIyIGBxEjyaoBAMTP36p/j2u6laoFtv2kXL+x/boCMXh2IjL9NQAAAQCwAAAEQgRIABIAL0AXABILBwcIEggUEw4DRlkLDg4ICQ8ACBUAPzM/EjkvOSsREgE5OREzETMRMzEwIRE0IyIGBxEjETMRNjYzMhYVEQOa2Vicd6amX7pyo74BjcExSv4tBEj+DkU+qJf+ZgACAD3/7AY/Bc0AIAAnAFFAKgUDACQREQgeJRAQGB4ABCgpER4HHklZJAcCBwIbDBsUSVkbEwwhSVkMBAA/KwAYPysREgA5ORgvLzMrEQAzERIBFzkRMxEzMxEzETMzMTATNDczBhUUMzM3EgAhIAARFSESADMyNjcVBgYjIAADIiYBIgIHIRAmPRuRFHEiBR0BTQEXASkBKPvcDgEF92XKjXLdgv7G/qMTjpsDr9HwEANuywOHSTYyPGcrASoBR/6F/o9F/vj+7x8rnCceAWQBTHYCI/71+QEJ+wAAAgAz/+wE3QRaAB8AJgBMQCgKCAUWDSQVFR0NAwUFJygWAwwDRlkjDAcMBwARESBGWREQABlGWQAWAD8rABg/KxESADk5GC8vMysRADMREgEXOREzETMRMzMxMAUiACckNTQ3MwYVFDMzNzY2MzISFRUhFhYzMjY3FQYGAyIGByE0JgNK8/7sBv72GY0UahUGIvq3z/H9DAasrWWfYlidoIaXDgI9jBQBHvwE3UUyLztnI8rg/vfiacbDICqUJiED46SenaUAAgA9/oMGPwXNACIAKQBdQDELCQYmFxcOAyEiJxYWHiIDBgUqKyIiIBMXAw0DSVkmDQgNCAASEiNJWRIEABpKWQATAD8rABg/KxESADk5GC8vMysRADMYPz8REgEXOREzETMRMzMRMxEzMzEwBSQAAyImNTQ3MwYVFDMzNxIAISAAERUhEgAzMjY3FQYHESMTIgIHIRAmA6D+/v7bE46bG5EUcSIFHQFNARcBKQEo+9wOAQX3ZcqNsOumTNHwEANuywwdAVoBMXZ1STYyPGcrASoBR/6F/o9F/vj+7x8rnD4F/pUGsv71+QEJ+wACADP+hwTdBFoAIQAoAFhALwoIBRYNICEmFRUdIQ0DBQYpKiEiHxYWAwwDRlklDAcMBwARESJGWREQABlGWQAVAD8rABg/KxESADk5GC8vMysRADMYPz8REgEXOREzETMRMxEzMzEwBSYCJyQ1NDczBhUUMzM3NjYzMhIVFSEWFjMyNjcVBgcRIxMiBgchNCYC1b/TBv72GY0UahUGIvq3z/H9DAasrWWfYo6lpkSGlw4CPYwKHwER4ATdRTIvO2cjyuD+9+JpxsMgKpRBBP6ZBUiknp2lAP//AFQAAAJWBbYCBgAsAAD//wACAAAGvAdgAiYBsAAAAQcCNgEQAVQACLMBEgUmACs1//8ABAAABd8GDAImAdAAAAEHAjYApAAAAAizARIRJgArNQABAMn+AAUZBbYAHABCQCUHAwMEGg4OCQoUBAUdHhEXSVkRHAcCSVkLAEpZBwsLBAgFAwQSAD8/MxI5LzkrKwAYPysREgEXOREzETMRMzEwASIHESMRMxEBMwE3IAAREAAhIiYnNRYzMhI1NCQCXoxfqqoCic39hRoBTwFi/tn+9VJ8RnqYu8j+6wJ7H/2kBbb9PALE/VQC/rv+z/7G/qQUHZgxAQ3x6P0AAAEAsP4KBCEESAAcAEJAJQQAAAEXChAKBgcBBR0eDhRGWQ4bBBxHWQcaRlkEBwcBBQIPARUAPz8zEjkvOSsrABg/KxESARc5ETMRMxEzMTAhIxEzEQEzAQQSERQGBiMiJzUWFjMyNjU0JiMiBwFUpKQB47f+NwEA/G7MhYhfLmxHh5i7vlJcBEj9+gIG/h4E/uT+9bH8hDyRGSbZyNPPGAABAAD+gwWRBbYAFwA5QB8DAAUEAQEFDgMYGRYHSVkWAwwRSlkMEgUASVkFEgMiAD8/KwAYPysAGD8rERIBFzkRMxEzMzEwJTMDIxMjESEHAgIGJyInNRYzMjY2EhMhBNm4j8Wcqv4lHz1dmH5KOzY7NU89XTgDEpr96QF9BR/w/iH+Ra4CGY8aV9cCWQG4AAABABD+hwSPBEYAFAA5QB8DAAUEAQEFDQMVFhMHRlkTDwsQR1kLFQUARlkFFQMiAD8/KwAYPysAGD8rERIBFzkRMxEzMzEwJTMDIxMjESECAgYjIic1FjMyEhMhA9+wgax9pv61HF6YdjocFhxxiSICgY/9+AF5A7j+mP5kwAp/BgHZAfYAAAEAyf4ABR8FtgAVAD1AIBIODg8TCwsAAAYPAxYXEg1JWRISDxQQAw8SAwlJWQMcAD8rABg/PzMSOS8rERIBFzkRMxEzETMRMzEwJRAAISImJzUWMyARESERIxEzESERMwUf/ub++1J6TXuHAYz8/qqqAwKqlv7C/qgTHpYxAfcCI/1QBbb9kgJuAAEAsP4KBGIESAAVAEdAJw8LCwwQCAgTEwIMAxYXDwpGWQ8PHw8CCwMPDwwRDQ8MFQAFRlkAGwA/KwAYPz8zEjkvX15dKxESARc5ETMRMxEzETMxMAEiJzUWMzI2NREhESMRMxEhETMREAIC04Rdb2Z9dv2cpqYCZKjP/go6lT3GzwG9/hIESP41Acv76/70/uMAAQDJ/oMF1wW2AA8AREAkDAgICQ0DAAUEAQEFCQMQEQwHSVkMDAUOCgMJEgUASVkFEgMiAD8/KwAYPz8zEjkvKxESARc5ETMRMzMzETMRMzEwJTMDIxMjESERIxEzESERMwUfuJHFnqr8/qqqAwKqmv3pAX0CsP1QBbb9kgJuAAABALD+hwUSBEYADwBEQCQBDQ0OCAUCCgkGBgoOAxARAQxGWQEBCgMPDw4VCgVGWQoVCCIAPz8rABg/PzMSOS8rERIBFzkRMxEzMzMRMxEzMTABESERMxEzAyMTIxEhESMRAVYCZqawgax9pv2apgRG/jcByfxJ/fgBeQHu/hIERgAAAQCq/oMExwW2ABcAPUAgDwwCAxUFBQAAAwwDGBkSCUlZEhIBFg0DAyIBBElZARIAPysAGD8/MxI5LysREgEXOREzETMRMxEzMTAhIxEjETMRBgYjIiY1ETMRFBYzMjY3ETMEx6qiopXGas/fqn+PYbGpqv6DAhcBwjUnvrMCRf3PeXQdNwLKAAEAnP6FBC0ESAAWAD1AIAEVCwwGDg4JCQwVAxcYAxJGWQMDCgcWDwwiCg1GWQoVAD8rABg/PzMSOS8rERIBFzkRMxEzETMRMzEwAREUMzI2NxEzESMRIxEzEQYGIyImNREBQttbpmmmlaaVabNxpLoESP5wwDhDAdX7uP6FAgoBYUg7rJMBnAABAMn+gwcpBbYAGABIQCUJBgYHEQ4MExIPDxMHAxkaFxYCCwITCBMOSVkTEhEiDAgDAAcSAD8zPzM/PysREgA5OREzMxESARc5ETMRMzMzETMRMzEwIQEjFxYVESMRIQEzATMRMwMjEyMRNDcjAQNQ/hAIBwedAQAB0QgB0f64j8eeqg4I/gwFEH/AL/xeBbb7SgS2+uT96QF9A66E3PryAAABALD+hwXfBEYAGAA/QCATFAgFCgkGBgoUAxkaCxIAEg8DFQ8UFQoFRlkKDxUIIgA/PzMrABg/PzMSOTkRMxESARc5ETMRMzMRMzEwJTc3ATMRMwMjEyMRBwcBIwEmJxEjETMBFgLpHysBKdOwgax9kxQ6/uWL/uU1FJTLASktoF12AtP8Sf34AXkDiTqZ/UoCuIZL/HcERv0tbv//AFQAAAJWBbYCBgAsAAD//wAAAAAFEAdeAiYAJAAAAQcCNgA5AVIACLMCDwUmACs1//8AXv/sA80GDAImAEQAAAEGAjboAAAIswIlESYAKzX//wAAAAAFEAclAiYAJAAAAQcAagA9AVIACrQDAiQFJgArNTX//wBe/+wDzQXTAiYARAAAAQYAavMAAAq0AwI6ESYAKzU1/////gAABoEFtgIGAIgAAP//AF7/7AZzBFwCBgCoAAD//wDJAAAD+AdeAiYAKAAAAQcCNgAQAVIACLMBDAUmACs1//8Ac//sBBIGDAImAEgAAAEGAjYMAAAIswIbESYAKzUAAgB1/+wFWAXNABIAGQA9QCAXDhAWFgkJAg4DGhsPF0lZDw8MBgwTSVkMEwYASVkGBAA/KwAYPysREgA5GC8rERIBFzkRMxEzETMxMAEiBzU2NjMgABEQACEgETUhAgADMhI3IRAWApjj4nPShgFLAW/+pv7L/awELxH++cPS+RD8h8wFNUyeJiD+cf6b/qL+cQLrRgEKAQ77TgEN9/74/AAAAgBm/+wEBgRcABQAGwA7QB8ZCRgLAwMRCQMcHQoZRlkKCgYABhVGWQYWAA5GWQAQAD8rABg/KxESADkYLysREgEXOREzMxEzMTABMgAREAAjIgI1NSEmJiMiBgc1NjYTMjY3IRQWAfr1ARf+/drQ8wL0BbOmYqVfWaKahZoM/cONBFz+1P77/vj+yQEM4WnMuyEpkygi/BulnJ2kAP//AHX/7AVYByUCJgLhAAABBwBqAJMBUgAKtAMCLwUmACs1Nf//AGb/7AQGBdMCJgLiAAABBgBq6gAACrQDAjERJgArNTX//wACAAAGvAclAiYBsAAAAQcAagEQAVIACrQCAScFJgArNTX//wAEAAAF3wXTAiYB0AAAAQcAagCiAAAACrQCAScRJgArNTX//wBK/+wENQclAiYBsQAAAQcAav/zAVIACrQCAT4FJgArNTX//wBE/+wDfwXTAiYB0QAAAQYAapQAAAq0AgE4ESYAKzU1AAEASv/sBDcFtgAZAEBAIwATFRkPAwMZExYIBRobGRYXFklZABJKWQAABhcDBgxKWQYTAD8rABg/EjkvKysRADMREgEXOREzETMRMzEwAQQEFRQEISAnNRYWMzI2NTQmIyM1ASE1IRUB/AEXAST+zf7q/v+jYN5qx8rh34wB7v1OA4cDPwnTwc7oT54uMpmQhoqNAd6ZiwAAAQAb/hQDpgRIABkAQEAjABMVGQ8EBBkTFgkFGhsZFhcWRlkAEkdZAAAHFw8HDEZZBxsAPysAGD8SOS8rKxEAMxESARc5ETMRMxEzMTABHgIVFAAjIic1FjMyNjU0JiMjNQEhNSEVAayV5n/+2O/qirfIocXWynkBxf2JAzgBzwdyyoje/u5Gmla+oKSqcgH+jnsA//8AywAABVIGtAImAbIAAAEHAU0AtAFSAAizARMFJgArNf//ALAAAARiBWICJgHSAAABBgFNMQAACLMBEREmACs1//8AywAABVIHJQImAbIAAAEHAGoAvgFSAAq0AgElBSYAKzU1//8AsAAABGIF0wImAdIAAAEGAGo9AAAKtAIBIxEmACs1Nf//AH3/7AW+ByUCJgAyAAABBwBqANEBUgAKtAMCLQUmACs1Nf//AHP/7ARiBdMCJgBSAAABBgBqHQAACrQDAi4RJgArNTX//wB9/+wFvgXNAgYCfgAA//8Ac//sBGIEXAIGAn8AAP//AH3/7AW+ByUCJgJ+AAABBwBqANEBUgAKtAQDLwUmACs1Nf//AHP/7ARiBdMCJgJ/AAABBgBqGwAACrQEAzARJgArNTX//wA9/+wEiQclAiYBxwAAAQcAav/tAVIACrQCATAFJgArNTX//wA5/+wDfQXTAiYB5wAAAQYAao4AAAq0AgEwESYAKzU1//8AG//sBPgGtAImAb0AAAEHAU0ALwFSAAizARoFJgArNf//AAL+FAQGBWICJgBcAAABBgFNrQAACLMBGREmACs1//8AG//sBPgHJQImAb0AAAEHAGoAOwFSAAq0AgEsBSYAKzU1//8AAv4UBAYF0wImAFwAAAEGAGq3AAAKtAIBKxEmACs1Nf//ABv/7AT4B3MCJgG9AAABBwFTAI0BUgAKtAIBKgUmACs1Nf//AAL+FAQGBiECJgBcAAABBgFTBAAACrQCASkRJgArNTX//wCqAAAExwclAiYBwQAAAQcAagBqAVIACrQCASkFJgArNTX//wCcAAAELQXTAiYB4QAAAQYAahcAAAq0AgEoESYAKzU1AAEAyf6DBAgFtgAJAC1AGAQJBgcBBwkDCgsJBElZCRIHIgADSVkAAwA/KwAYPz8rERIBFzkRMxEzMTATIRUhETMRIxEjyQM//WuhoaoFtpn7ff3pAX0AAQCw/ocDQgRGAAkALUAYBAkGBwEHCQMKCwkERlkJFQciAANGWQAPAD8rABg/PysREgEXOREzETMxMBMhFSERMxEjESOwApL+FJamlgRGjPzV/fgBef//AMkAAAYKByUCJgHFAAABBwBqARsBUgAKtAQDLQUmACs1Nf//ALAAAAV5BdMCJgHlAAABBwBqAMUAAAAKtAQDLBEmACs1Nf//AC/+dQQIBbYCJgKbAAAABwOAAJMAAP//ABL+dQNCBEgCJgKcAAAABgOBdQD//wAI/nUEyQW2ACYAOwAAAAcDgANYAAD//wAn/nUENARIACYAWwAAAAcDgQLDAAAAAQAGAAAElgW2ABEAO0AiDwIRARANBAoHCQYLDBMSChEAEUlZBw0PBAAAAgwPEgUCAwA/Mz8zEjkvORI5MysRADMREgEXOTEwEyEBMwEBMwEhFSEBIwEBIwEhfwEz/ne8AWsBbLf+cAE8/roBvcH+d/5wtgG//roDVAJi/bsCRf2emP1EAoP9fQK8AAABACcAAAQIBEgAEQA7QCIPAhEBEA0ECgcJBgsMExIKEQARR1kHDQ8EAAACDA8VBQIPAD8zPzMSOS85EjkzKxEAMxESARc5MTATIQEzAQEzASEVIQEjAQEjASF1ARL+tL0BIQEgu/6yARj+4gFovP7N/sq8AWb+6AJ3AdH+XAGk/i+B/goBvP5EAfYAAAIAgwAABDcFtgAKABMANEAaBBMTBw8ABwAVFAMMSVkDAwgFCBJKWQgSBQMAPz8rERIAORgvKxESATk5ETMRMxEzMTATNCQhMxEzESEgJAEjIgYVFBYzM4MBJAEgxqr+Y/71/vQDCrrewrbL2QGk1M4CcPpK1QHbfI6PhP//AHP/7AQ3BhQCBgBHAAAAAgCD/+wGdwW2ABkAIwBGQCQeAxgKCgcjDxISIwMDJCUGG0lZGAYQBhAACAMMIAAgSlkVABMAPzIrEQAzGD8SOTkvLzkrERIBFzkRMxEzMxI5ETMxMAUiJjU0JCEzETMRFDMyNjURMxEUBiMiJicGEyMiBhUQITI2NQJO4ukBKgEikarmZHmqz7h2nzNxKZfUwgEhf40S0dDZ3gJw+7fse24B5v4Yrs5SWqoCwIuW/vR3cAAAAgBz/+wGhwYUACIALgBRQCksEwwgIB0aJgMGBiYTAy8wHgANEBoWBAQQFhYqRlkWEAAjECNGWQkQFgA/MysRADMYPysREgA5GC8SORI5PxESARc5ETMRMzMzEjkRMzEwJTI2NREzERQGIyImJyMGBiMiAhEQEjMyFhczJiY1ETMRFBYhMjY1NTQmIyARFBYE/nZrqMi9gZ4rCEu5gdDo589qnz8MAgimbf25opKUov7ii3eEiAE5/r3IxVtxcVsBKQEMAQwBL01VEXAbAb77jKCJuc4j58n+TtbSAAEATv/sBoEFywAqAEtAKAYTKBkfIiIWGRMBDQYrLBcCAQIBSlkCIAIgJRAlHElZJRMQCUpZEAQAPysAGD8rERIAOTkYLy8rERIAORESARc5ETMRMxEzMTABIzUzMjY1NCYjIgYHJzY2MzIWFRQGBxUEExYWMzI2NREzERQGIyImJyYmAa7JwcDVmoBnsWdUXfaC1vWynAFiBgJsfHdwqNK9ytACAs0CrI+ThGx/N0VySFDEp423Gggz/tGWf3mHAc3+KcbH0ciWkQABAFD/7AXFBFwAJQBLQCgSHgokAgUFJB4gDhgGJichDw4PDkZZDwMPAwgbGxRGWRsQCABGWQgWAD8rABg/KxESADk5GC8vKxESADkREgEXOREzETMRMzEwJTIRETMRFAYjIAMmJiMjNTMgNTQjIgYHJzY2MzIWFRQHFRYWFxYEQt2mu8T+hhAFjZSMbwEh8kuHTTlVo2i408BjewUJdwEMATn+vcrDAU1jWI2soiQihygkm4a4OQgUemrTAAEATv6DBNEFywAjAEpAKBkaHiMhICAWGiMEEAYkJRoFBAUESlkFBSMTIx5JWSMSISITDEpZEwQAPysAGD8/KxESADkYLysREgA5ERIBFzkRMxEzETMxMAE0JiMjNTMyNjU0JiMiBgcnNjYzMhYVFAYHFRYWFREzESMRIwOD5eLZ0c3hpIdpw2lUYf6E3P29o7jDrKKsAZyFi4+ThGuAOkJySk7Ep4y3GQgZs5T+/v3pAX0AAAEAUP6HBBAEWgAeAEpAKAcSGR4cGxsVHhIDDQYgHxUEAwQDRlkEBB4PHhlGWR4VHCIPCkZZDxAAPysAGD8/KxESADkYLysREgA5ERIBFzkRMxEzETMxMAE0ISM1MyA1NCYjIgcnNjMyFhUUBxUWFhUVMxEjESMC1f7LlnUBOYV3mZY9ocu/1ct+cJ2mlQEtx42sUlBGh0qah7Y5CyWJZpz9+AF5AAABAAD/6QchBbYAIwA6QB0UIxodHSMJAyQlGxsHEhIBSVkSAxcMBwxKWSAHEwA/MysRADMYPysREgA5GC8REgEXOREzETMxMAEhBwICBgYjIic1FjMyNjYSEhMhERQWMzI2NREzERQGIyImNQQM/kgfK0xTgmRFQDI/MUAsOEo3Au9vc3BxqM28xMgFH/D+rv5E0mYZjxo+aAECAekBrvvPiXl5hwHN/inBzMzFAAABABD/7AYpBEYAHQA6QB0ADgUICA4WAx8eBgYUHBwQRlkcDwMZFBlHWQsUFgA/MysRADMYPysREgA5GC8REgEXOREzETMxMAEUFjMyEREzERQGIyImNREhAgIGIyInNRYzMhITIQPPaHfVpru+vMv+xRxemHY6HBYccYkiAnEBg4mDAQoBO/69ysPEywI9/pj+ZMAKfwYB2QH2AAABAMn/7AdeBbYAGQBDQCMXAA8GCRYSEhMJDxMDGhsWEUlZFgcWBxMYFAMTEgwDSVkMEwA/KwAYPz8zEjk5Ly8rERIBFzkRMxEzETMRMzMxMAEUFjMyNjURMxEUBiMiJjURIREjETMRIREzBPZuc3Bxpsi/w8j9J6qqAtmqAYWJeXmHAc3+Kb/Oy8YBM/1QBbb9kgJuAAABALD/7AaoBEgAGABNQCoFAhMKDQEWFhcNExcDGRoBFUZZDwEfAQILAwELAQsXAxgPFxUQCEZZEBYAPysAGD8/MxI5OS8vX15dKxESARc5ETMRMxEzETMzMTABESERMxEUFjMyEREzERQGIyImNTUhESMRAVYCUKZqd9Wmu8C6zf2wpgRI/jUBy/09iYUBDAE5/r3Kw8bJc/4SBEgAAAEAff/sBZoFywAcADpAHxYIGwICDxwIBB0eABxJWQAABQwME0lZDAQFGUlZBRMAPysAGD8rERIAORgvKxESARc5ETMRMzEwASEVEAAhIAARNBIkMzIWFwcmJiMgABEQADMgESEDZgI0/sz+yf67/pOzAVXqeO1TQlrWV/71/t4BC/cBtP5/AvBW/qH+sQGRAWDlAVS1MSeUJi7+xf7j/uP+wwHXAAABAHP/7ASwBFwAGQA6QB8SBxgCAgwZBwQaGwAZRlkAAAQKCg9GWQoQBBVGWQQWAD8rABg/KxESADkYLysREgEXOREzETMxMAEhFRAhIAAREAAhMhcHJiMiBhUUFjMyNjUhArIB/v3+/u7+1wFDASHUrzuops3lzMWpr/6qAj9D/fABJwEQAQ4BK1CDSt7Sz9+gnQAAAQAQ/+wE9AW2ABQAOUAdBRMKDQ0DEwAEFRYLCxABEAhJWRATBAABAElZAQMAPysRADMYPysREgA5GC8REgEXOREzETMxMBM1IRUhERQWMzIRETMRFAYjIiY1ERAEPP4vd3LoqNO9xs0FHZmZ/GiJewEAAc/+KcDNzsMDoAAAAQAp/+wEhwRGABQANkAcAhAHCgoAEBIEFRYBEhMSRlkICA0TDw0FRlkNFgA/KwAYPxI5LysRADMREgEXOREzETMxMAEhERQWMzIRETMRFAYjIiY1ESE1IQOB/qZtdtemvcDAyf6oA1gDuv3JiYMBBAFB/r3Kw8vEAj+MAAEAb//sBFgFywAmAEdAJhUgDAAkIwUbESMAIAYnKCMPEg8SSlkPDx0DHRhKWR0TAwlKWQMEAD8rABg/KxESADkYLysREgA5ERIBFzkRMxEzETMxMBM0JDMgFwcmJiMiBhUUFjMzFSMiBhUUFjMyNxUGISAkNTQ2NzUmJpwBCOEBAtFeabVljJ/RyNnV3ujKt+nHr/77/vT+28+8qrQEXKnGkHhENHtygJONjoqOjVyeTdzFl8AWCBmy//8AWv/sA4cEXAIGAYIAAP//AAD+dQVrBbYAJgG1AAAABwOAA/oAAP//ABD+dQRzBEgCJgHVAAAABwOBAwIAAP//AAD+oAUQBbwCJgAkAAAABwJnBOkAAP//AF7+oAPNBFoCJgBEAAAABwJnBHkAAP//AAAAAAUQB+ECJgAkAAABBwJmBPwBUgAIswITBSYAKzX//wBe/+wDzQaPAiYARAAAAQcCZgSmAAAACLMCKREmACs1//8AAAAABRAH0QImACQAAAEHA3cE5QFSAAq0AwIVBSYAKzU1//8AXv/sBEEGfwImAEQAAAEHA3cEkwAAAAq0AwIrESYAKzU1//8AAAAABRAH0QImACQAAAEHA3gE3QFSAAq0AwIVBSYAKzU1//8ALf/sA80GfwImAEQAAAEHA3gEkwAAAAq0AwIrESYAKzU1//8AAAAABRAISgImACQAAAEHA3kE2QFSAAq0AwIVBSYAKzU1//8AXv/sBBcG+AImAEQAAAEHA3kEnAAAAAq0AwIrESYAKzU1//8AAAAABRAIYgImACQAAAEHA3oE5QFSAAq0AwItBSYAKzU1//8AXv/sA80HEAImAEQAAAEHA3oEkQAAAAq0AwJDESYAKzU1//8AAP6gBRAHcwImACQAAAAnAmcE6QAAAQcBSwArAVIACLMDKQUmACs1//8AXv6gA80GIQImAEQAAAAnAmcEeQAAAQYBS9QAAAizAz4RJgArNf//AAAAAAUQCBMCJgAkAAABBwN7BOwBUgAKtAMCFwUmACs1Nf//AF7/7APNBsECJgBEAAABBwN7BJoAAAAKtAMCLREmACs1Nf//AAAAAAUQCBMCJgAkAAABBwN8BOkBUgAKtAMCFwUmACs1Nf//AF7/7APNBsECJgBEAAABBwN8BJgAAAAKtAMCLREmACs1Nf//AAAAAAUQCFgCJgAkAAABBwN9BOkBUgAKtAMCIQUmACs1Nf//AF7/7APNBwYCJgBEAAABBwN9BKAAAAAKtAMCNxEmACs1Nf//AAAAAAUQCF4CJgAkAAABBwN+BOMBUgAKtAMCJwUmACs1Nf//AF7/7APNBwwCJgBEAAABBwN+BJgAAAAKtAMCPREmACs1Nf//AAD+oAUQB0kCJgAkAAAAJwFOAC0BZAEHAmcE6QAAAAizAg8FJgArNf//AF7+oAPNBeUCJgBEAAAAJgFO2AABBwJnBHkAAAAIswIlESYAKzX//wDJ/qAD+AW2AiYAKAAAAAcCZwTBAAD//wBz/qAEEgRcAiYASAAAAAcCZwS4AAD//wDJAAAD+AfhAiYAKAAAAQcCZgTRAVIACLMBEAUmACs1//8Ac//sBBIGjwImAEgAAAEHAmYEyQAAAAizAh8RJgArNf//AMkAAAP4By8CJgAoAAABBwFS/+QBUgAIswEVBSYAKzX//wBz/+wEEgXdAiYASAAAAQYBUtAAAAizAiQRJgArNf//AMkAAARvB9ECJgAoAAABBwN3BMEBUgAKtAIBEgUmACs1Nf//AHP/7ARcBn8CJgBIAAABBwN3BK4AAAAKtAMCIREmACs1Nf//AF0AAAP4B9ECJgAoAAABBwN4BMMBUgAKtAIBEgUmACs1Nf//AEr/7AQSBn8CJgBIAAABBwN4BLAAAAAKtAMCIREmACs1Nf//AMkAAAQ5CEoCJgAoAAABBwN5BL4BUgAKtAIBEgUmACs1Nf//AHP/7AQdBvgCJgBIAAABBwN5BKIAAAAKtAMCIREmACs1Nf//AMkAAAP4CGICJgAoAAABBwN6BLgBUgAKtAIBKgUmACs1Nf//AHP/7AQSBxACJgBIAAABBwN6BKIAAAAKtAMCOREmACs1Nf//AMn+oAP4B3MCJgAoAAAAJwJnBL4AAAEHAUsAAgFSAAizAiUFJgArNf//AHP+oAQSBiECJgBIAAAAJwJnBLAAAAEGAUvxAAAIswM0ESYAKzX//wBUAAACVgfhAiYALAAAAQcCZgPJAVIACLMBEAUmACs1//8AewAAAeYGjwImAPMAAAEHAmYDcwAAAAizAQgRJgArNf//AFT+oAJWBbYCJgAsAAAABwJnA7QAAP//AJ3+oAFmBd8CJgBMAAAABwJnA2IAAP//AH3+oAW+Bc0CJgAyAAAABwJnBX8AAP//AHP+oARiBFwCJgBSAAAABwJnBMkAAP//AH3/7AW+B+ECJgAyAAABBwJmBY8BUgAIswIcBSYAKzX//wBz/+wEYgaPAiYAUgAAAQcCZgTZAAAACLMCHREmACs1//8Aff/sBb4H0QImADIAAAEHA3cFfQFSAAq0AwIeBSYAKzU1//8Ac//sBHUGfwImAFIAAAEHA3cExwAAAAq0AwIfESYAKzU1//8Aff/sBb4H0QImADIAAAEHA3gFfQFSAAq0AwIeBSYAKzU1//8AYf/sBGIGfwImAFIAAAEHA3gExwAAAAq0AwIfESYAKzU1//8Aff/sBb4ISgImADIAAAEHA3kFewFSAAq0AwIeBSYAKzU1//8Ac//sBGIG+AImAFIAAAEHA3kExwAAAAq0AwIfESYAKzU1//8Aff/sBb4IYgImADIAAAEHA3oFeQFSAAq0AwI2BSYAKzU1//8Ac//sBGIHEAImAFIAAAEHA3oExQAAAAq0AwI3ESYAKzU1//8Aff6gBb4HcwImADIAAAAnAmcFfwAAAQcBSwDBAVIACLMDMQUmACs1//8Ac/6gBGIGIQImAFIAAAAnAmcEzQAAAQYBSw4AAAizAzIRJgArNf//AH3/7AZkB3MCJgJfAAABBwB2ASsBUgAIswIrBSYAKzX//wBz/+wFGQYhAiYCYAAAAQYAdm0AAAizAisRJgArNf//AH3/7AZkB3MCJgJfAAABBwBDAIcBUgAIswIjBSYAKzX//wBz/+wFGQYhAiYCYAAAAQYAQ9QAAAizAiQRJgArNf//AH3/7AZkB+ECJgJfAAABBwJmBY8BUgAIswImBSYAKzX//wBz/+wFGQaPAiYCYAAAAQcCZgTZAAAACLMCJxEmACs1//8Aff/sBmQHLwImAl8AAAEHAVIAoAFSAAizAisFJgArNf//AHP/7AUZBd0CJgJgAAABBgFS9QAACLMCIxEmACs1//8Aff6gBmQGFAImAl8AAAAHAmcFewAA//8Ac/6gBRkE8AImAmAAAAAHAmcEyQAA//8Auv6gBRkFtgImADgAAAAHAmcFSgAA//8ApP6gBDkESAImAFgAAAAHAmcEuAAA//8Auv/sBRkH4QImADgAAAEHAmYFVAFSAAizARYFJgArNf//AKT/7AQ5Bo8CJgBYAAABBwJmBNUAAAAIswEZESYAKzX//wC6/+wGewdzAiYCYQAAAQcAdgDuAVIACLMBJQUmACs1//8ApP/sBZYGIQImAmIAAAEGAHZ5AAAIswEmESYAKzX//wC6/+wGewdzAiYCYQAAAQcAQwBaAVIACLMBHQUmACs1//8ApP/sBZYGIQImAmIAAAEGAEO7AAAIswEfESYAKzX//wC6/+wGewfhAiYCYQAAAQcCZgVgAVIACLMBIAUmACs1//8ApP/sBZYGjwImAmIAAAEHAmYE2wAAAAizASIRJgArNf//ALr/7AZ7By8CJgJhAAABBwFSAH8BUgAIswElBSYAKzX//wCk/+wFlgXdAiYCYgAAAQYBUv8AAAizAR4RJgArNf//ALr+oAZ7BhQCJgJhAAAABwJnBUwAAP//AKT+oAWWBPICJgJiAAAABwJnBLIAAP//AAD+oAR7BbYCJgA8AAAABwJnBJwAAP//AAL+FAQGBEgCJgBcAAAABwJnBZ7//f//AAAAAAR7B+ECJgA8AAABBwJmBKoBUgAIswENBSYAKzX//wAC/hQEBgaPAiYAXAAAAQcCZgRqAAAACLMBGhEmACs1//8AAAAABHsHLwImADwAAAEHAVL/wgFSAAizARIFJgArNf//AAL+FAQGBd0CJgBcAAABBgFSigAACLMBHxEmACs1//8Ac/7FBNMGFAImANMAAAAHAEIAtAAAAAL75QTZ/rQGIQAJABMAHkAMBAoODgAAFQ8GgAsBAC8zGs0yEQEzETMSOTkxMAEjJiYnNTMWFhcFIyYmJzUzFhYX/rRgNLEluhxjMf6cYDiuJbscYzEE2SrKPxU9rkQZLMg/FT2uRAAAAvxxBNn/rgZ/AA0AFQAoQBEVAAYRERcDBgoVChUKEcAGAQAvMxrMOTkvLxESOREBMxEzOTkxMAEjJicGByM1NzY3MxYXJzY3MxUGByP+015wY3JhXjVwNLBCl1BJNqxTeGAE2UtbZUEZPHtNXqbCW3AVbmAAAAL7mgTZ/tcGfwANABUAKkASBg4REQAAFwMGCg8KDwoTwAYBAC8zGsw5OS8vERI5EQEzETMSOTkxMAEjJicGByM1NzY3MxYXJSMmJzUzFhf+115hcmppXjVwNLBCl/3uX3hUrDRLBNlBZWBGFzx7TV6mrF5wFWxhAAL8cQTZ/3sG+AANAB8ANEAYEBMAExsDBgYWDg4hAwoGEgoSChkewAYBAC8zGswyOTkvLxESOREBMxEzMxIXOREzMTABIyYnBgcjNTc2NzMWFxMUBwcjJzY2NTQmIyIHNTYzMv7TXnBjcmFeNXA0sEKXqH8GUAo5PzkrLhoZN8ME2UtbZUEZPHtNXqYBe2cdUYMJICYlGQZQBgAC/GgE2f7nBxAAFwAlADpAGxgeCQkVFScbHiIeGREJAAUMIgAMDAAiAxXAGQAvGswXOS8vLxEzEMQzETMREjkRATMRMxI5OTEwASIuAiMiBgcjNjYzMh4CMzI2NzMGBhMjJicGByM1NzY3MxYX/i0lR0M/HCgqDlsNZUslSUM+GygqDFoLY15eYXJqaV41cDSwQpcGNR4lHjEyanEeJB4xMWhz/qRBZWBGFzx7TV6mAAL8eQTZ/scGwQAHABQAJEAPBwQKChISFgNABxEKgA4IAC8zGt0y1BrNEQEzETMSOTkxMAE2NzMVBgcjEyADMxYWMzI2NzMGBv1eUDGsVndgPv7sD2YJTGpiVghpC5UF9GhlFXJd/vwBBEg5QUB4jAAC/HkE2f7HBsEABwAUACRADwcECgoSEhYEQAERCoAOCAAvMxrdMtQazREBMxEzEjk5MTABIyYnNTMWFwMgAzMWFjMyNjczBgb90V53Vqw0SzX+7A9mCUxqYlYIaQuVBd1dchVsYf7lAQRIOUFAeIwAAvx5BNn+xwcGABEAHgAuQBUIAAAFDQMUFBwcIAsQBAQYGBsUgBIALxrNMjMROS/EMhEBMxEzEhc5ETMxMAEUBwcjJzY2NTQmIyIHNTYzMgMgAzMWFjMyNjczBgb+MX8GUgo5QjksJSQWPsCV/uwPZglMamJWCGkLlQZ5ZB0pWgkgJSUaBk4I/dMBBEg5QUB4jAAC/GgE2f7nBwwAFwAkADBAFRoiCQkVJgUMDB4eGBVAEQkAIRqAGAAvGt0y1sQzGs0RMxE5LzMRATMyETk5MTABIi4CIyIGByM2NjMyHgIzMjY3MwYGAyADMxYWMzI2NzMGBv4tJUdDPxwoKg5bDWRMJUlDPhsoKgxaC2Pd/uwPZglMamJWCGkLlQYzHiQeMDJocR4kHjExZ3L+pgEESDlBQHiMAAEAMf5CAW0AAAAPABpACwAFBQIKAxARDQgDAC/MMhESARc5ETMxMBc0JzMWFRQGIyInNRYzMjbfi3ueZmNBMiA2JTPuZ4d4hFtnEGwKMAAAAQAZ/nUBcQCaAAsAGEAJCgAGAAwNCAMAAC/MMhESATk5ETMxMCURECMiJzUWMzI1EQFx5Dg8KT1emv7f/vwYjBNkATAAAAEAGf51AXEAjwALABhACQoABgAMDQgDAAAvzDIREgE5OREzMTAlERAjIic1FjMyNREBceQ4PCk9Xo/+6v78GIwTZAElAP//ADQAAAJDBbYABwAU/3gAAAACAHP/7AQXBHMACwAXAChAFAwGEgAGABgZCRVLWQkmAw9NWQMZAD8rABg/KxESATk5ETMRMzEwARACIyICERASMzISARQWMzI2NTQmIyIGBBf33tn2+drY+f0Em46Nnp6PjZoCL/71/sgBNQEOAQ8BNf7L/vHQ6OrOzOzpAAABAC0AAAI3BF4ACgAmQBEJAQEACAALDAcEBwQBCRABGAA/PxI5OS8vERIBOTkRMxEzMTAhIxE0NwYHBycBMwI3oQhDPpZaAX+LAjHvjEMwcHIBIwABACkAAAPXBHMAGQAsQBgHEwATFw4BBRobEApLWRAmGBcBF0xZARgAPysRADMYPysREgEXOREzMTAhITUBPgI1NCYjIgYHJzYzMhYVFAYHBRchA9f8UgGRnXEsi3dYnFxawPLG2oK6/rkCAr6FAS93aFNBV2c9Sm2oqJZzu4DnBgAAAQBe/pUEGwR0ACcAR0AmAwQbABMHBwAEFiINBigpBBcWFxZLWRcXCiUlHktZJSYKEUtZCiUAPysAGD8rERIAORgvKxESADkREgEXOREzETMRMzEwARQGBxUWFhUUBCEiJic1FhYzIBEQISM1MzI2NTQmIyIGByc2NjMyFgPunZCwqv7e/vV0wVtf12ABe/5ekJKryJN+YKptVFrrgtXsAweMsh4IFrSS0eEjLJ4vMQEpAQqPl4ZrejRGcEdRwwAAAgAX/qgEZgReAAoAEgBCQCESBQkCAgsHAwADBQMTFAEFEgVNWQkSDg8PBxISAwcQAyQAPz8SOS8SOREzETMrEQAzERIBFzkRMzMzETMRMzEwJSMRIxEhNQEzETMhETQ3IwYHAQRm2aj9MgK+uNn+hgwKKUT+ORv+jQFzfQPG/EQBXNreVlz9ngAAAQCF/pUEHQRfABoAOkAfDwMZFAgUFwMEHBsAEUtZAAAGFRUYTFkVEAYMS1kGJQA/KwAYPysREgA5GC8rERIBFzkRMxEzMTABMgQVFAAjIic1FhYzMjY1ECEiBycTIRUhAzYCLecBCf7f/veCRtBlsMP+iV6gVjcC1/23JXMCJuXH4/7+T6AtM6adATIdNwKsmf5JFwD//wB1/+wELwXLAgYAGQAAAAEAXv6pBCsEXwAGAB9AEAEFBQACAwcIAwJMWQMQACQAPz8rERIBFzkRMzEwAQEhNSEVAQEdAl784wPN/ar+qQUdmYX6z///AGj/7AQpBcsCBgAbAAAAAgBq/pUEJQR0ABcAJQBBQCIbESIKCgAABBEDJicOHk1ZChQODgIUFBhLWRQmAgdNWQIlAD8rABg/KxESADkYLxI5KxESARc5ETMRMxEzMTABECEiJzUWMzISEyMGBiMiJjU0EjMyFhIBIgYVFBYzMjY2NTQmJgQl/Wh0RFBm8PULDDe2csLk/9CV33j+FI+ckJNbmVhSkwHv/KYUjxoBKQEzU1fo0OQBCJn+2wEwuKSQpUqARmmyZgD//wAdAAAFxAYfACcASQK2AAAABgBJAAAAAgBcAt0FqgXBACIAMwBaQC4sMDAuKiYmKAoAHBEFERYAKC4GNTQrMSQDLS8tKS8jIygcChQIAwMoKRkUFCkDAD8zLzMQzTIvMxI5OREzETMRMxESFzkREgEXOREzETMRMxEzETMRMzEwARQGIyInNRYzMjU0JiYnJiY1NDYzMhcHJiMiBhUUFhYXFhYBAyMXESMRMxMTMxEjETcjAwJIlXyRSmp3lBc2VXhRjm59XCJkUzxLEitfgVABpskIBne8w8u0fwYI0wOsYm0hbChkISghHyxbTFZpJ2MlLigdJBwkMlr+7AIvgf5SAtH90QIv/S8BpIn90///ABL+FARaBbYCJgA3AAAABwB6AT8AAP//AB/+FAKoBUYCJgBXAAAABwB6AMUAAAACAHH+FAQ3BFwADAAqAEdAJgoVGgMqKh4eJBUDKywhJ0ZZJCEbHA8aDxgSGAdGWRgQEgBGWRIWAD8rABg/KxESADk5GD8/MysREgEXOREzETMzETMxMCUyNjc1NCYjIgYVFBYFNDcjBiMiAhEQEjMyFzM3MxEUBiMiJzUWFjMyNjUCTKqXBJ6rkJmXAdsJC3Dm2e/z0997CxiD7PnylUvSdo6ld7fKK+LM4NDR2WskY6cBLQEKAQgBMaaS+6Ts7EaeKi6pkv//AHH+FAQ3BiECJgORAAABBgFLBgAACLMCOREmACs1//8Acf4UBDcF5QImA5EAAAEGAU4MAAAIswIrESYAKzX//wBx/hQENwXfAiYDkQAAAQcBTwFWAAAACLMCNBEmACs1//8Acf4UBDcGIQImA5EAAAEGAjp3AAAIswIvESYAKzUAAQDJAAABcwW2AAMAEbYABAUBAwASAD8/ERIBOTEwMxEzEcmqBbb6SgD//wAFAAABjgdzAiYDlgAAAQcAQ/58AVIACLMBBQUmACs1//8AswAAAjwHcwImA5YAAAEHAHb/KgFSAAizAQ0FJgArNf///8cAAAJpB3MCJgOWAAABBwFL/rsBUgAIswESBSYAKzX//wAFAAACOAclAiYDlgAAAQcAav7QAVIACrQCARkFJgArNTX///+rAAACkwcvAiYDlgAAAQcBUv6jAVIACLMBDQUmACs1////8wAAAksGtAImA5YAAAEHAU3+xgFSAAizAQcFJgArNf///+cAAAJTBzcCJgOWAAABBwFO/sIBUgAIswEEBSYAKzX//wBW/kIBogW2AiYDlgAAAAYBUTEA//8AuwAAAX8HMQImA5YAAAEHAU8AGQFSAAizAQ0FJgArNf//AMn+fwOjBbYAJgOWAAAABwAtAjsAAP///+QAAAIdBgoAJwOWAKoAAAEHAVT96P+XAAeyAQgAAD81AP//AMkAAAFzBbYCBgOWAAD//wAFAAACOAclAiYDlgAAAQcAav7QAVIACrQCARkFJgArNTX//wDJAAABcwW2AgYDlgAA//8ABQAAAjgHJQImA5YAAAEHAGr+0AFSAAq0AgEZBSYAKzU1//8AyQAAAXMFtgIGA5YAAP//AMkAAAFzBbYCBgOWAAD//wCZAAACBAfhAiYDlgAAAQcCZgORAVIACLMBCAUmACs1//8AuP6gAX8FtgImA5YAAAAHAmcDfQAAAAAAAQAAtjIAAUkGgAAADjYkAAUAJP9xAAUANwApAAUAOQApAAUAOgApAAUAPAAUAAUARP+uAAUARv+FAAUAR/+FAAUASP+FAAUASv/DAAUAUP/DAAUAUf/DAAUAUv+FAAUAU//DAAUAVP+FAAUAVf/DAAUAVv/DAAUAWP/DAAUAgv9xAAUAg/9xAAUAhP9xAAUAhf9xAAUAhv9xAAUAh/9xAAUAnwAUAAUAov+FAAUAo/+uAAUApP+uAAUApf+uAAUApv+uAAUAp/+uAAUAqP+uAAUAqf+FAAUAqv+FAAUAq/+FAAUArP+FAAUArf+FAAUAtP+FAAUAtf+FAAUAtv+FAAUAt/+FAAUAuP+FAAUAuv+FAAUAu//DAAUAvP/DAAUAvf/DAAUAvv/DAAUAwv9xAAUAw/+uAAUAxP9xAAUAxf+uAAUAxv9xAAUAx/+uAAUAyf+FAAUAy/+FAAUAzf+FAAUAz/+FAAUA0f+FAAUA0/+FAAUA1f+FAAUA1/+FAAUA2f+FAAUA2/+FAAUA3f+FAAUA3//DAAUA4f/DAAUA4//DAAUA5f/DAAUA+v/DAAUBBv/DAAUBCP/DAAUBDf/DAAUBD/+FAAUBEf+FAAUBE/+FAAUBFf+FAAUBF//DAAUBGf/DAAUBHf/DAAUBIf/DAAUBJAApAAUBJgApAAUBK//DAAUBLf/DAAUBL//DAAUBMf/DAAUBM//DAAUBNf/DAAUBNgApAAUBOAAUAAUBOgAUAAUBQ/9xAAUBRP+uAAUBRv+uAAUBSP+FAAUBSv/DAAUBVv9xAAUBX/9xAAUBYv9xAAUBaf9xAAUBef+uAAUBev/XAAUBe//XAAUBfv+uAAUBgf/DAAUBgv/XAAUBg//XAAUBhP/XAAUBh//XAAUBif/XAAUBjP+uAAUBjv/DAAUBj/+uAAUBkP+uAAUBk/+uAAUBmf+uAAUBpP+FAAUBqv9xAAUBrv+FAAUBtf+FAAUByv/XAAUBzv9xAAUBz/+FAAUB1f9xAAUB2P+FAAUB2/+FAAUB3v+FAAUB6v+FAAUB7f+FAAUB7v/DAAUB8v9xAAUB+gApAAUB/AApAAUB/gApAAUCAAAUAAUCV//DAAUCWP9xAAUCWf+uAAUCYP+FAAUCYv/DAAUCav+FAAUCcv9xAAUCc/9xAAUCff/sAAUCf/+FAAUChf+FAAUCh/+FAAUCif+FAAUCjf+FAAUCsv+FAAUCtP+FAAUCzv+FAAUCz/9xAAUC2f9xAAUC2v/XAAUC2/9xAAUC3P/XAAUC3f9xAAUC3v/XAAUC4P+FAAUC4v/XAAUC5P/XAAUC8P+FAAUC8v+FAAUC9P+FAAUDCf9xAAUDCv+FAAUDC/9xAAUDDP+FAAUDEf+FAAUDEv9xAAUDFv+FAAUDGv+FAAUDG/+FAAUDHP9xAAUDHf9xAAUDHv+uAAUDH/9xAAUDIP+uAAUDIf9xAAUDIv+uAAUDI/9xAAUDJf9xAAUDJv+uAAUDJ/9xAAUDKP+uAAUDKf9xAAUDKv+uAAUDK/9xAAUDLP+uAAUDLf9xAAUDLv+uAAUDL/9xAAUDMP+uAAUDMf9xAAUDMv+uAAUDM/9xAAUDNP+uAAUDNv+FAAUDOP+FAAUDOv+FAAUDPP+FAAUDQP+FAAUDQv+FAAUDRP+FAAUDSv+FAAUDTP+FAAUDTv+FAAUDUv+FAAUDVP+FAAUDVv+FAAUDWP+FAAUDWv+FAAUDXP+FAAUDXv+FAAUDYP+FAAUDYv/DAAUDZP/DAAUDZv/DAAUDaP/DAAUDav/DAAUDbP/DAAUDbv/DAAUDbwAUAAUDcQAUAAUDcwAUAAUDjwApAAoAJP9xAAoANwApAAoAOQApAAoAOgApAAoAPAAUAAoARP+uAAoARv+FAAoAR/+FAAoASP+FAAoASv/DAAoAUP/DAAoAUf/DAAoAUv+FAAoAU//DAAoAVP+FAAoAVf/DAAoAVv/DAAoAWP/DAAoAgv9xAAoAg/9xAAoAhP9xAAoAhf9xAAoAhv9xAAoAh/9xAAoAnwAUAAoAov+FAAoAo/+uAAoApP+uAAoApf+uAAoApv+uAAoAp/+uAAoAqP+uAAoAqf+FAAoAqv+FAAoAq/+FAAoArP+FAAoArf+FAAoAtP+FAAoAtf+FAAoAtv+FAAoAt/+FAAoAuP+FAAoAuv+FAAoAu//DAAoAvP/DAAoAvf/DAAoAvv/DAAoAwv9xAAoAw/+uAAoAxP9xAAoAxf+uAAoAxv9xAAoAx/+uAAoAyf+FAAoAy/+FAAoAzf+FAAoAz/+FAAoA0f+FAAoA0/+FAAoA1f+FAAoA1/+FAAoA2f+FAAoA2/+FAAoA3f+FAAoA3//DAAoA4f/DAAoA4//DAAoA5f/DAAoA+v/DAAoBBv/DAAoBCP/DAAoBDf/DAAoBD/+FAAoBEf+FAAoBE/+FAAoBFf+FAAoBF//DAAoBGf/DAAoBHf/DAAoBIf/DAAoBJAApAAoBJgApAAoBK//DAAoBLf/DAAoBL//DAAoBMf/DAAoBM//DAAoBNf/DAAoBNgApAAoBOAAUAAoBOgAUAAoBQ/9xAAoBRP+uAAoBRv+uAAoBSP+FAAoBSv/DAAoBVv9xAAoBX/9xAAoBYv9xAAoBaf9xAAoBef+uAAoBev/XAAoBe//XAAoBfv+uAAoBgf/DAAoBgv/XAAoBg//XAAoBhP/XAAoBh//XAAoBif/XAAoBjP+uAAoBjv/DAAoBj/+uAAoBkP+uAAoBk/+uAAoBmf+uAAoBpP+FAAoBqv9xAAoBrv+FAAoBtf+FAAoByv/XAAoBzv9xAAoBz/+FAAoB1f9xAAoB2P+FAAoB2/+FAAoB3v+FAAoB6v+FAAoB7f+FAAoB7v/DAAoB8v9xAAoB+gApAAoB/AApAAoB/gApAAoCAAAUAAoCV//DAAoCWP9xAAoCWf+uAAoCYP+FAAoCYv/DAAoCav+FAAoCcv9xAAoCc/9xAAoCff/sAAoCf/+FAAoChf+FAAoCh/+FAAoCif+FAAoCjf+FAAoCsv+FAAoCtP+FAAoCzv+FAAoCz/9xAAoC2f9xAAoC2v/XAAoC2/9xAAoC3P/XAAoC3f9xAAoC3v/XAAoC4P+FAAoC4v/XAAoC5P/XAAoC8P+FAAoC8v+FAAoC9P+FAAoDCf9xAAoDCv+FAAoDC/9xAAoDDP+FAAoDEf+FAAoDEv9xAAoDFv+FAAoDGv+FAAoDG/+FAAoDHP9xAAoDHf9xAAoDHv+uAAoDH/9xAAoDIP+uAAoDIf9xAAoDIv+uAAoDI/9xAAoDJf9xAAoDJv+uAAoDJ/9xAAoDKP+uAAoDKf9xAAoDKv+uAAoDK/9xAAoDLP+uAAoDLf9xAAoDLv+uAAoDL/9xAAoDMP+uAAoDMf9xAAoDMv+uAAoDM/9xAAoDNP+uAAoDNv+FAAoDOP+FAAoDOv+FAAoDPP+FAAoDQP+FAAoDQv+FAAoDRP+FAAoDSv+FAAoDTP+FAAoDTv+FAAoDUv+FAAoDVP+FAAoDVv+FAAoDWP+FAAoDWv+FAAoDXP+FAAoDXv+FAAoDYP+FAAoDYv/DAAoDZP/DAAoDZv/DAAoDaP/DAAoDav/DAAoDbP/DAAoDbv/DAAoDbwAUAAoDcQAUAAoDcwAUAAoDjwApAAsALQC4AA8AJv+aAA8AKv+aAA8AMv+aAA8ANP+aAA8AN/9xAA8AOP/XAA8AOf+FAA8AOv+FAA8APP+FAA8Aif+aAA8AlP+aAA8Alf+aAA8Alv+aAA8Al/+aAA8AmP+aAA8Amv+aAA8Am//XAA8AnP/XAA8Anf/XAA8Anv/XAA8An/+FAA8AyP+aAA8Ayv+aAA8AzP+aAA8Azv+aAA8A3v+aAA8A4P+aAA8A4v+aAA8A5P+aAA8BDv+aAA8BEP+aAA8BEv+aAA8BFP+aAA8BJP9xAA8BJv9xAA8BKv/XAA8BLP/XAA8BLv/XAA8BMP/XAA8BMv/XAA8BNP/XAA8BNv+FAA8BOP+FAA8BOv+FAA8BR/+aAA8BZv+uAA8Bbf+uAA8Bcf9xAA8Bcv+FAA8Bc/+aAA8Bdf+FAA8BeP+FAA8Bhf/XAA8Bnf9xAA8Bn/+aAA8Bpv9xAA8BuP+aAA8Bu/+aAA8BvP9xAA8Bvv+uAA8Bwf9cAA8BxP9xAA8B3P+aAA8B4f+FAA8B5P+aAA8B+v+FAA8B/P+FAA8B/v+FAA8CAP+FAA8CVP+FAA8CX/+aAA8CYf/XAA8CbP+aAA8CfP9cAA8Cfv+aAA8CgP+FAA8Cgv+FAA8ChP+aAA8Chv+aAA8CiP+aAA8Civ+aAA8CjP+aAA8Cqf9xAA8Cqv+aAA8Csf+aAA8Cs/+aAA8Ctf9xAA8Ctv+aAA8Ct/+FAA8Cuf+FAA8Cvf9xAA8Cvv+aAA8Cv/9cAA8CwP+FAA8Cwf9cAA8Cwv+FAA8Cxf+FAA8Cx/+FAA8C1P9cAA8C1f+FAA8C7/+aAA8C8f+aAA8C8/+aAA8C/f9cAA8C/v+FAA8DDf+FAA8DDv+aAA8DD/+FAA8DEP+aAA8DFf+aAA8DF/9xAA8DGP+aAA8DSf+aAA8DS/+aAA8DTf+aAA8DT/+aAA8DUf+aAA8DU/+aAA8DVf+aAA8DV/+aAA8DWf+aAA8DW/+aAA8DXf+aAA8DX/+aAA8DYf/XAA8DY//XAA8DZf/XAA8DZ//XAA8Daf/XAA8Da//XAA8Dbf/XAA8Db/+FAA8Dcf+FAA8Dc/+FAA8Dj/9xABAAN/+uABABJP+uABABJv+uABABcf+uABABnf+uABABpv+uABABvP+uABABxP+uABAB3P/XABAB5P/XABACqf+uABACqv/XABACtf+uABACtv/XABACvf+uABACvv/XABADF/+uABADGP/XABADj/+uABEAJv+aABEAKv+aABEAMv+aABEANP+aABEAN/9xABEAOP/XABEAOf+FABEAOv+FABEAPP+FABEAif+aABEAlP+aABEAlf+aABEAlv+aABEAl/+aABEAmP+aABEAmv+aABEAm//XABEAnP/XABEAnf/XABEAnv/XABEAn/+FABEAyP+aABEAyv+aABEAzP+aABEAzv+aABEA3v+aABEA4P+aABEA4v+aABEA5P+aABEBDv+aABEBEP+aABEBEv+aABEBFP+aABEBJP9xABEBJv9xABEBKv/XABEBLP/XABEBLv/XABEBMP/XABEBMv/XABEBNP/XABEBNv+FABEBOP+FABEBOv+FABEBR/+aABEBZv+uABEBbf+uABEBcf9xABEBcv+FABEBc/+aABEBdf+FABEBeP+FABEBhf/XABEBnf9xABEBn/+aABEBpv9xABEBuP+aABEBu/+aABEBvP9xABEBvv+uABEBwf9cABEBxP9xABEB3P+aABEB4f+FABEB5P+aABEB+v+FABEB/P+FABEB/v+FABECAP+FABECVP+FABECX/+aABECYf/XABECbP+aABECfP9cABECfv+aABECgP+FABECgv+FABEChP+aABEChv+aABECiP+aABECiv+aABECjP+aABECqf9xABECqv+aABECsf+aABECs/+aABECtf9xABECtv+aABECt/+FABECuf+FABECvf9xABECvv+aABECv/9cABECwP+FABECwf9cABECwv+FABECxf+FABECx/+FABEC1P9cABEC1f+FABEC7/+aABEC8f+aABEC8/+aABEC/f9cABEC/v+FABEDDf+FABEDDv+aABEDD/+FABEDEP+aABEDFf+aABEDF/9xABEDGP+aABEDSf+aABEDS/+aABEDTf+aABEDT/+aABEDUf+aABEDU/+aABEDVf+aABEDV/+aABEDWf+aABEDW/+aABEDXf+aABEDX/+aABEDYf/XABEDY//XABEDZf/XABEDZ//XABEDaf/XABEDa//XABEDbf/XABEDb/+FABEDcf+FABEDc/+FABEDj/9xACQABf9xACQACv9xACQAJv/XACQAKv/XACQALQEKACQAMv/XACQANP/XACQAN/9xACQAOf+uACQAOv+uACQAPP+FACQAif/XACQAlP/XACQAlf/XACQAlv/XACQAl//XACQAmP/XACQAmv/XACQAn/+FACQAyP/XACQAyv/XACQAzP/XACQAzv/XACQA3v/XACQA4P/XACQA4v/XACQA5P/XACQBDv/XACQBEP/XACQBEv/XACQBFP/XACQBJP9xACQBJv9xACQBNv+uACQBOP+FACQBOv+FACQBR//XACQB+v+uACQB/P+uACQB/v+uACQCAP+FACQCB/9xACQCC/9xACQCX//XACQDSf/XACQDS//XACQDTf/XACQDT//XACQDUf/XACQDU//XACQDVf/XACQDV//XACQDWf/XACQDW//XACQDXf/XACQDX//XACQDb/+FACQDcf+FACQDc/+FACQDj/9xACUAD/+uACUAEf+uACUAJP/XACUAN//DACUAOf/sACUAOv/sACUAO//XACUAPP/sACUAPf/sACUAgv/XACUAg//XACUAhP/XACUAhf/XACUAhv/XACUAh//XACUAn//sACUAwv/XACUAxP/XACUAxv/XACUBJP/DACUBJv/DACUBNv/sACUBOP/sACUBOv/sACUBO//sACUBPf/sACUBP//sACUBQ//XACUBoP/sACUB+v/sACUB/P/sACUB/v/sACUCAP/sACUCCP+uACUCDP+uACUCWP/XACUDHf/XACUDH//XACUDIf/XACUDI//XACUDJf/XACUDJ//XACUDKf/XACUDK//XACUDLf/XACUDL//XACUDMf/XACUDM//XACUDb//sACUDcf/sACUDc//sACUDj//DACYAJv/XACYAKv/XACYAMv/XACYANP/XACYAif/XACYAlP/XACYAlf/XACYAlv/XACYAl//XACYAmP/XACYAmv/XACYAyP/XACYAyv/XACYAzP/XACYAzv/XACYA3v/XACYA4P/XACYA4v/XACYA5P/XACYBDv/XACYBEP/XACYBEv/XACYBFP/XACYBR//XACYCX//XACYDSf/XACYDS//XACYDTf/XACYDT//XACYDUf/XACYDU//XACYDVf/XACYDV//XACYDWf/XACYDW//XACYDXf/XACYDX//XACcAD/+uACcAEf+uACcAJP/XACcAN//DACcAOf/sACcAOv/sACcAO//XACcAPP/sACcAPf/sACcAgv/XACcAg//XACcAhP/XACcAhf/XACcAhv/XACcAh//XACcAn//sACcAwv/XACcAxP/XACcAxv/XACcBJP/DACcBJv/DACcBNv/sACcBOP/sACcBOv/sACcBO//sACcBPf/sACcBP//sACcBQ//XACcBoP/sACcB+v/sACcB/P/sACcB/v/sACcCAP/sACcCCP+uACcCDP+uACcCWP/XACcDHf/XACcDH//XACcDIf/XACcDI//XACcDJf/XACcDJ//XACcDKf/XACcDK//XACcDLf/XACcDL//XACcDMf/XACcDM//XACcDb//sACcDcf/sACcDc//sACcDj//DACgALQB7ACkAD/+FACkAEf+FACkAIgApACkAJP/XACkAgv/XACkAg//XACkAhP/XACkAhf/XACkAhv/XACkAh//XACkAwv/XACkAxP/XACkAxv/XACkBQ//XACkCCP+FACkCDP+FACkCWP/XACkDHf/XACkDH//XACkDIf/XACkDI//XACkDJf/XACkDJ//XACkDKf/XACkDK//XACkDLf/XACkDL//XACkDMf/XACkDM//XAC4AJv/XAC4AKv/XAC4AMv/XAC4ANP/XAC4Aif/XAC4AlP/XAC4Alf/XAC4Alv/XAC4Al//XAC4AmP/XAC4Amv/XAC4AyP/XAC4Ayv/XAC4AzP/XAC4Azv/XAC4A3v/XAC4A4P/XAC4A4v/XAC4A5P/XAC4BDv/XAC4BEP/XAC4BEv/XAC4BFP/XAC4BR//XAC4CX//XAC4DSf/XAC4DS//XAC4DTf/XAC4DT//XAC4DUf/XAC4DU//XAC4DVf/XAC4DV//XAC4DWf/XAC4DW//XAC4DXf/XAC4DX//XAC8ABf9cAC8ACv9cAC8AJv/XAC8AKv/XAC8AMv/XAC8ANP/XAC8AN//XAC8AOP/sAC8AOf/XAC8AOv/XAC8APP/DAC8Aif/XAC8AlP/XAC8Alf/XAC8Alv/XAC8Al//XAC8AmP/XAC8Amv/XAC8Am//sAC8AnP/sAC8Anf/sAC8Anv/sAC8An//DAC8AyP/XAC8Ayv/XAC8AzP/XAC8Azv/XAC8A3v/XAC8A4P/XAC8A4v/XAC8A5P/XAC8BDv/XAC8BEP/XAC8BEv/XAC8BFP/XAC8BJP/XAC8BJv/XAC8BKv/sAC8BLP/sAC8BLv/sAC8BMP/sAC8BMv/sAC8BNP/sAC8BNv/XAC8BOP/DAC8BOv/DAC8BR//XAC8B+v/XAC8B/P/XAC8B/v/XAC8CAP/DAC8CB/9cAC8CC/9cAC8CX//XAC8CYf/sAC8DSf/XAC8DS//XAC8DTf/XAC8DT//XAC8DUf/XAC8DU//XAC8DVf/XAC8DV//XAC8DWf/XAC8DW//XAC8DXf/XAC8DX//XAC8DYf/sAC8DY//sAC8DZf/sAC8DZ//sAC8Daf/sAC8Da//sAC8Dbf/sAC8Db//DAC8Dcf/DAC8Dc//DAC8Dj//XADIAD/+uADIAEf+uADIAJP/XADIAN//DADIAOf/sADIAOv/sADIAO//XADIAPP/sADIAPf/sADIAgv/XADIAg//XADIAhP/XADIAhf/XADIAhv/XADIAh//XADIAn//sADIAwv/XADIAxP/XADIAxv/XADIBJP/DADIBJv/DADIBNv/sADIBOP/sADIBOv/sADIBO//sADIBPf/sADIBP//sADIBQ//XADIBoP/sADIB+v/sADIB/P/sADIB/v/sADICAP/sADICCP+uADICDP+uADICWP/XADIDHf/XADIDH//XADIDIf/XADIDI//XADIDJf/XADIDJ//XADIDKf/XADIDK//XADIDLf/XADIDL//XADIDMf/XADIDM//XADIDb//sADIDcf/sADIDc//sADIDj//DADMAD/72ADMAEf72ADMAJP+aADMAO//XADMAPf/sADMAgv+aADMAg/+aADMAhP+aADMAhf+aADMAhv+aADMAh/+aADMAwv+aADMAxP+aADMAxv+aADMBO//sADMBPf/sADMBP//sADMBQ/+aADMCCP72ADMCDP72ADMCWP+aADMDHf+aADMDH/+aADMDIf+aADMDI/+aADMDJf+aADMDJ/+aADMDKf+aADMDK/+aADMDLf+aADMDL/+aADMDMf+aADMDM/+aADQAD/+uADQAEf+uADQAJP/XADQAN//DADQAOf/sADQAOv/sADQAO//XADQAPP/sADQAPf/sADQAgv/XADQAg//XADQAhP/XADQAhf/XADQAhv/XADQAh//XADQAn//sADQAwv/XADQAxP/XADQAxv/XADQBJP/DADQBJv/DADQBNv/sADQBOP/sADQBOv/sADQBO//sADQBPf/sADQBP//sADQBQ//XADQBoP/sADQB+v/sADQB/P/sADQB/v/sADQCAP/sADQCCP+uADQCDP+uADQCWP/XADQDHf/XADQDH//XADQDIf/XADQDI//XADQDJf/XADQDJ//XADQDKf/XADQDK//XADQDLf/XADQDL//XADQDMf/XADQDM//XADQDb//sADQDcf/sADQDc//sADQDj//DADcAD/+FADcAEP+uADcAEf+FADcAIgApADcAJP9xADcAJv/XADcAKv/XADcAMv/XADcANP/XADcANwApADcARP9cADcARv9xADcAR/9xADcASP9xADcASv9xADcAUP+aADcAUf+aADcAUv9xADcAU/+aADcAVP9xADcAVf+aADcAVv+FADcAWP+aADcAWf/XADcAWv/XADcAW//XADcAXP/XADcAXf+uADcAgv9xADcAg/9xADcAhP9xADcAhf9xADcAhv9xADcAh/9xADcAif/XADcAlP/XADcAlf/XADcAlv/XADcAl//XADcAmP/XADcAmv/XADcAov9xADcAo/9cADcApP9cADcApf9cADcApv9cADcAp/9cADcAqP9cADcAqf9xADcAqv9xADcAq/9xADcArP9xADcArf9xADcAtP9xADcAtf9xADcAtv9xADcAt/9xADcAuP9xADcAuv9xADcAu/+aADcAvP+aADcAvf+aADcAvv+aADcAv//XADcAwv9xADcAw/9cADcAxP9xADcAxf9cADcAxv9xADcAx/9cADcAyP/XADcAyf9xADcAyv/XADcAy/9xADcAzP/XADcAzf9xADcAzv/XADcAz/9xADcA0f9xADcA0/9xADcA1f9xADcA1/9xADcA2f9xADcA2/9xADcA3f9xADcA3v/XADcA3/9xADcA4P/XADcA4f9xADcA4v/XADcA4/9xADcA5P/XADcA5f9xADcA+v+aADcBBv+aADcBCP+aADcBDf+aADcBDv/XADcBD/9xADcBEP/XADcBEf9xADcBEv/XADcBE/9xADcBFP/XADcBFf9xADcBF/+aADcBGf+aADcBHf+FADcBIf+FADcBJAApADcBJgApADcBK/+aADcBLf+aADcBL/+aADcBMf+aADcBM/+aADcBNf+aADcBN//XADcBPP+uADcBPv+uADcBQP+uADcBQ/9xADcBRP9cADcBRv9cADcBR//XADcBSP9xADcBSv+FADcB+//XADcB/f/XADcCAv+uADcCA/+uADcCBP+uADcCCP+FADcCDP+FADcCV/+aADcCWP9xADcCWf9cADcCX//XADcCYP9xADcCYv+aADcDHf9xADcDHv9cADcDH/9xADcDIP9cADcDIf9xADcDIv9cADcDI/9xADcDJf9xADcDJv9cADcDJ/9xADcDKP9cADcDKf9xADcDKv9cADcDK/9xADcDLP9cADcDLf9xADcDLv9cADcDL/9xADcDMP9cADcDMf9xADcDMv9cADcDM/9xADcDNP9cADcDNv9xADcDOP9xADcDOv9xADcDPP9xADcDQP9xADcDQv9xADcDRP9xADcDSf/XADcDSv9xADcDS//XADcDTP9xADcDTf/XADcDTv9xADcDT//XADcDUf/XADcDUv9xADcDU//XADcDVP9xADcDVf/XADcDVv9xADcDV//XADcDWP9xADcDWf/XADcDWv9xADcDW//XADcDXP9xADcDXf/XADcDXv9xADcDX//XADcDYP9xADcDYv+aADcDZP+aADcDZv+aADcDaP+aADcDav+aADcDbP+aADcDbv+aADcDcP/XADcDjwApADgAD//XADgAEf/XADgAJP/sADgAgv/sADgAg//sADgAhP/sADgAhf/sADgAhv/sADgAh//sADgAwv/sADgAxP/sADgAxv/sADgBQ//sADgCCP/XADgCDP/XADgCWP/sADgDHf/sADgDH//sADgDIf/sADgDI//sADgDJf/sADgDJ//sADgDKf/sADgDK//sADgDLf/sADgDL//sADgDMf/sADgDM//sADkAD/+aADkAEf+aADkAIgApADkAJP+uADkAJv/sADkAKv/sADkAMv/sADkANP/sADkARP/XADkARv/XADkAR//XADkASP/XADkASv/sADkAUP/sADkAUf/sADkAUv/XADkAU//sADkAVP/XADkAVf/sADkAVv/sADkAWP/sADkAgv+uADkAg/+uADkAhP+uADkAhf+uADkAhv+uADkAh/+uADkAif/sADkAlP/sADkAlf/sADkAlv/sADkAl//sADkAmP/sADkAmv/sADkAov/XADkAo//XADkApP/XADkApf/XADkApv/XADkAp//XADkAqP/XADkAqf/XADkAqv/XADkAq//XADkArP/XADkArf/XADkAtP/XADkAtf/XADkAtv/XADkAt//XADkAuP/XADkAuv/XADkAu//sADkAvP/sADkAvf/sADkAvv/sADkAwv+uADkAw//XADkAxP+uADkAxf/XADkAxv+uADkAx//XADkAyP/sADkAyf/XADkAyv/sADkAy//XADkAzP/sADkAzf/XADkAzv/sADkAz//XADkA0f/XADkA0//XADkA1f/XADkA1//XADkA2f/XADkA2//XADkA3f/XADkA3v/sADkA3//sADkA4P/sADkA4f/sADkA4v/sADkA4//sADkA5P/sADkA5f/sADkA+v/sADkBBv/sADkBCP/sADkBDf/sADkBDv/sADkBD//XADkBEP/sADkBEf/XADkBEv/sADkBE//XADkBFP/sADkBFf/XADkBF//sADkBGf/sADkBHf/sADkBIf/sADkBK//sADkBLf/sADkBL//sADkBMf/sADkBM//sADkBNf/sADkBQ/+uADkBRP/XADkBRv/XADkBR//sADkBSP/XADkBSv/sADkCCP+aADkCDP+aADkCV//sADkCWP+uADkCWf/XADkCX//sADkCYP/XADkCYv/sADkDHf+uADkDHv/XADkDH/+uADkDIP/XADkDIf+uADkDIv/XADkDI/+uADkDJf+uADkDJv/XADkDJ/+uADkDKP/XADkDKf+uADkDKv/XADkDK/+uADkDLP/XADkDLf+uADkDLv/XADkDL/+uADkDMP/XADkDMf+uADkDMv/XADkDM/+uADkDNP/XADkDNv/XADkDOP/XADkDOv/XADkDPP/XADkDQP/XADkDQv/XADkDRP/XADkDSf/sADkDSv/XADkDS//sADkDTP/XADkDTf/sADkDTv/XADkDT//sADkDUf/sADkDUv/XADkDU//sADkDVP/XADkDVf/sADkDVv/XADkDV//sADkDWP/XADkDWf/sADkDWv/XADkDW//sADkDXP/XADkDXf/sADkDXv/XADkDX//sADkDYP/XADkDYv/sADkDZP/sADkDZv/sADkDaP/sADkDav/sADkDbP/sADkDbv/sADoAD/+aADoAEf+aADoAIgApADoAJP+uADoAJv/sADoAKv/sADoAMv/sADoANP/sADoARP/XADoARv/XADoAR//XADoASP/XADoASv/sADoAUP/sADoAUf/sADoAUv/XADoAU//sADoAVP/XADoAVf/sADoAVv/sADoAWP/sADoAgv+uADoAg/+uADoAhP+uADoAhf+uADoAhv+uADoAh/+uADoAif/sADoAlP/sADoAlf/sADoAlv/sADoAl//sADoAmP/sADoAmv/sADoAov/XADoAo//XADoApP/XADoApf/XADoApv/XADoAp//XADoAqP/XADoAqf/XADoAqv/XADoAq//XADoArP/XADoArf/XADoAtP/XADoAtf/XADoAtv/XADoAt//XADoAuP/XADoAuv/XADoAu//sADoAvP/sADoAvf/sADoAvv/sADoAwv+uADoAw//XADoAxP+uADoAxf/XADoAxv+uADoAx//XADoAyP/sADoAyf/XADoAyv/sADoAy//XADoAzP/sADoAzf/XADoAzv/sADoAz//XADoA0f/XADoA0//XADoA1f/XADoA1//XADoA2f/XADoA2//XADoA3f/XADoA3v/sADoA3//sADoA4P/sADoA4f/sADoA4v/sADoA4//sADoA5P/sADoA5f/sADoA+v/sADoBBv/sADoBCP/sADoBDf/sADoBDv/sADoBD//XADoBEP/sADoBEf/XADoBEv/sADoBE//XADoBFP/sADoBFf/XADoBF//sADoBGf/sADoBHf/sADoBIf/sADoBK//sADoBLf/sADoBL//sADoBMf/sADoBM//sADoBNf/sADoBQ/+uADoBRP/XADoBRv/XADoBR//sADoBSP/XADoBSv/sADoCCP+aADoCDP+aADoCV//sADoCWP+uADoCWf/XADoCX//sADoCYP/XADoCYv/sADoDHf+uADoDHv/XADoDH/+uADoDIP/XADoDIf+uADoDIv/XADoDI/+uADoDJf+uADoDJv/XADoDJ/+uADoDKP/XADoDKf+uADoDKv/XADoDK/+uADoDLP/XADoDLf+uADoDLv/XADoDL/+uADoDMP/XADoDMf+uADoDMv/XADoDM/+uADoDNP/XADoDNv/XADoDOP/XADoDOv/XADoDPP/XADoDQP/XADoDQv/XADoDRP/XADoDSf/sADoDSv/XADoDS//sADoDTP/XADoDTf/sADoDTv/XADoDT//sADoDUf/sADoDUv/XADoDU//sADoDVP/XADoDVf/sADoDVv/XADoDV//sADoDWP/XADoDWf/sADoDWv/XADoDW//sADoDXP/XADoDXf/sADoDXv/XADoDX//sADoDYP/XADoDYv/sADoDZP/sADoDZv/sADoDaP/sADoDav/sADoDbP/sADoDbv/sADsAJv/XADsAKv/XADsAMv/XADsANP/XADsAif/XADsAlP/XADsAlf/XADsAlv/XADsAl//XADsAmP/XADsAmv/XADsAyP/XADsAyv/XADsAzP/XADsAzv/XADsA3v/XADsA4P/XADsA4v/XADsA5P/XADsBDv/XADsBEP/XADsBEv/XADsBFP/XADsBR//XADsCX//XADsDSf/XADsDS//XADsDTf/XADsDT//XADsDUf/XADsDU//XADsDVf/XADsDV//XADsDWf/XADsDW//XADsDXf/XADsDX//XADwAD/+FADwAEf+FADwAIgApADwAJP+FADwAJv/XADwAKv/XADwAMv/XADwANP/XADwARP+aADwARv+aADwAR/+aADwASP+aADwASv/XADwAUP/DADwAUf/DADwAUv+aADwAU//DADwAVP+aADwAVf/DADwAVv+uADwAWP/DADwAXf/XADwAgv+FADwAg/+FADwAhP+FADwAhf+FADwAhv+FADwAh/+FADwAif/XADwAlP/XADwAlf/XADwAlv/XADwAl//XADwAmP/XADwAmv/XADwAov+aADwAo/+aADwApP+aADwApf+aADwApv+aADwAp/+aADwAqP+aADwAqf+aADwAqv+aADwAq/+aADwArP+aADwArf+aADwAtP+aADwAtf+aADwAtv+aADwAt/+aADwAuP+aADwAuv+aADwAu//DADwAvP/DADwAvf/DADwAvv/DADwAwv+FADwAw/+aADwAxP+FADwAxf+aADwAxv+FADwAx/+aADwAyP/XADwAyf+aADwAyv/XADwAy/+aADwAzP/XADwAzf+aADwAzv/XADwAz/+aADwA0f+aADwA0/+aADwA1f+aADwA1/+aADwA2f+aADwA2/+aADwA3f+aADwA3v/XADwA3//XADwA4P/XADwA4f/XADwA4v/XADwA4//XADwA5P/XADwA5f/XADwA+v/DADwBBv/DADwBCP/DADwBDf/DADwBDv/XADwBD/+aADwBEP/XADwBEf+aADwBEv/XADwBE/+aADwBFP/XADwBFf+aADwBF//DADwBGf/DADwBHf+uADwBIf+uADwBK//DADwBLf/DADwBL//DADwBMf/DADwBM//DADwBNf/DADwBPP/XADwBPv/XADwBQP/XADwBQ/+FADwBRP+aADwBRv+aADwBR//XADwBSP+aADwBSv+uADwCCP+FADwCDP+FADwCV//DADwCWP+FADwCWf+aADwCX//XADwCYP+aADwCYv/DADwDHf+FADwDHv+aADwDH/+FADwDIP+aADwDIf+FADwDIv+aADwDI/+FADwDJf+FADwDJv+aADwDJ/+FADwDKP+aADwDKf+FADwDKv+aADwDK/+FADwDLP+aADwDLf+FADwDLv+aADwDL/+FADwDMP+aADwDMf+FADwDMv+aADwDM/+FADwDNP+aADwDNv+aADwDOP+aADwDOv+aADwDPP+aADwDQP+aADwDQv+aADwDRP+aADwDSf/XADwDSv+aADwDS//XADwDTP+aADwDTf/XADwDTv+aADwDT//XADwDUf/XADwDUv+aADwDU//XADwDVP+aADwDVf/XADwDVv+aADwDV//XADwDWP+aADwDWf/XADwDWv+aADwDW//XADwDXP+aADwDXf/XADwDXv+aADwDX//XADwDYP+aADwDYv/DADwDZP/DADwDZv/DADwDaP/DADwDav/DADwDbP/DADwDbv/DAD0AJv/sAD0AKv/sAD0AMv/sAD0ANP/sAD0Aif/sAD0AlP/sAD0Alf/sAD0Alv/sAD0Al//sAD0AmP/sAD0Amv/sAD0AyP/sAD0Ayv/sAD0AzP/sAD0Azv/sAD0A3v/sAD0A4P/sAD0A4v/sAD0A5P/sAD0BDv/sAD0BEP/sAD0BEv/sAD0BFP/sAD0BR//sAD0CX//sAD0DSf/sAD0DS//sAD0DTf/sAD0DT//sAD0DUf/sAD0DU//sAD0DVf/sAD0DV//sAD0DWf/sAD0DW//sAD0DXf/sAD0DX//sAD4ALQC4AEQABf/sAEQACv/sAEQCB//sAEQCC//sAEUABf/sAEUACv/sAEUAWf/XAEUAWv/XAEUAW//XAEUAXP/XAEUAXf/sAEUAv//XAEUBN//XAEUBPP/sAEUBPv/sAEUBQP/sAEUB+//XAEUB/f/XAEUCB//sAEUCC//sAEUDcP/XAEYABQApAEYACgApAEYCBwApAEYCCwApAEgABf/sAEgACv/sAEgAWf/XAEgAWv/XAEgAW//XAEgAXP/XAEgAXf/sAEgAv//XAEgBN//XAEgBPP/sAEgBPv/sAEgBQP/sAEgB+//XAEgB/f/XAEgCB//sAEgCC//sAEgDcP/XAEkABQB7AEkACgB7AEkCBwB7AEkCCwB7AEsABf/sAEsACv/sAEsCB//sAEsCC//sAE4ARv/XAE4AR//XAE4ASP/XAE4AUv/XAE4AVP/XAE4Aov/XAE4Aqf/XAE4Aqv/XAE4Aq//XAE4ArP/XAE4Arf/XAE4AtP/XAE4Atf/XAE4Atv/XAE4At//XAE4AuP/XAE4Auv/XAE4Ayf/XAE4Ay//XAE4Azf/XAE4Az//XAE4A0f/XAE4A0//XAE4A1f/XAE4A1//XAE4A2f/XAE4A2//XAE4A3f/XAE4BD//XAE4BEf/XAE4BE//XAE4BFf/XAE4BSP/XAE4CYP/XAE4DNv/XAE4DOP/XAE4DOv/XAE4DPP/XAE4DQP/XAE4DQv/XAE4DRP/XAE4DSv/XAE4DTP/XAE4DTv/XAE4DUv/XAE4DVP/XAE4DVv/XAE4DWP/XAE4DWv/XAE4DXP/XAE4DXv/XAE4DYP/XAFAABf/sAFAACv/sAFACB//sAFACC//sAFEABf/sAFEACv/sAFECB//sAFECC//sAFIABf/sAFIACv/sAFIAWf/XAFIAWv/XAFIAW//XAFIAXP/XAFIAXf/sAFIAv//XAFIBN//XAFIBPP/sAFIBPv/sAFIBQP/sAFIB+//XAFIB/f/XAFICB//sAFICC//sAFIDcP/XAFMABf/sAFMACv/sAFMAWf/XAFMAWv/XAFMAW//XAFMAXP/XAFMAXf/sAFMAv//XAFMBN//XAFMBPP/sAFMBPv/sAFMBQP/sAFMB+//XAFMB/f/XAFMCB//sAFMCC//sAFMDcP/XAFUABQBSAFUACgBSAFUARP/XAFUARv/XAFUAR//XAFUASP/XAFUASv/sAFUAUv/XAFUAVP/XAFUAov/XAFUAo//XAFUApP/XAFUApf/XAFUApv/XAFUAp//XAFUAqP/XAFUAqf/XAFUAqv/XAFUAq//XAFUArP/XAFUArf/XAFUAtP/XAFUAtf/XAFUAtv/XAFUAt//XAFUAuP/XAFUAuv/XAFUAw//XAFUAxf/XAFUAx//XAFUAyf/XAFUAy//XAFUAzf/XAFUAz//XAFUA0f/XAFUA0//XAFUA1f/XAFUA1//XAFUA2f/XAFUA2//XAFUA3f/XAFUA3//sAFUA4f/sAFUA4//sAFUA5f/sAFUBD//XAFUBEf/XAFUBE//XAFUBFf/XAFUBRP/XAFUBRv/XAFUBSP/XAFUCBwBSAFUCCwBSAFUCWf/XAFUCYP/XAFUDHv/XAFUDIP/XAFUDIv/XAFUDJv/XAFUDKP/XAFUDKv/XAFUDLP/XAFUDLv/XAFUDMP/XAFUDMv/XAFUDNP/XAFUDNv/XAFUDOP/XAFUDOv/XAFUDPP/XAFUDQP/XAFUDQv/XAFUDRP/XAFUDSv/XAFUDTP/XAFUDTv/XAFUDUv/XAFUDVP/XAFUDVv/XAFUDWP/XAFUDWv/XAFUDXP/XAFUDXv/XAFUDYP/XAFcABQApAFcACgApAFcCBwApAFcCCwApAFkABQBSAFkACgBSAFkAD/+uAFkAEf+uAFkAIgApAFkCBwBSAFkCCP+uAFkCCwBSAFkCDP+uAFoABQBSAFoACgBSAFoAD/+uAFoAEf+uAFoAIgApAFoCBwBSAFoCCP+uAFoCCwBSAFoCDP+uAFsARv/XAFsAR//XAFsASP/XAFsAUv/XAFsAVP/XAFsAov/XAFsAqf/XAFsAqv/XAFsAq//XAFsArP/XAFsArf/XAFsAtP/XAFsAtf/XAFsAtv/XAFsAt//XAFsAuP/XAFsAuv/XAFsAyf/XAFsAy//XAFsAzf/XAFsAz//XAFsA0f/XAFsA0//XAFsA1f/XAFsA1//XAFsA2f/XAFsA2//XAFsA3f/XAFsBD//XAFsBEf/XAFsBE//XAFsBFf/XAFsBSP/XAFsCYP/XAFsDNv/XAFsDOP/XAFsDOv/XAFsDPP/XAFsDQP/XAFsDQv/XAFsDRP/XAFsDSv/XAFsDTP/XAFsDTv/XAFsDUv/XAFsDVP/XAFsDVv/XAFsDWP/XAFsDWv/XAFsDXP/XAFsDXv/XAFsDYP/XAFwABQBSAFwACgBSAFwAD/+uAFwAEf+uAFwAIgApAFwCBwBSAFwCCP+uAFwCCwBSAFwCDP+uAF4ALQC4AIIABf9xAIIACv9xAIIAJv/XAIIAKv/XAIIALQEKAIIAMv/XAIIANP/XAIIAN/9xAIIAOf+uAIIAOv+uAIIAPP+FAIIAif/XAIIAlP/XAIIAlf/XAIIAlv/XAIIAl//XAIIAmP/XAIIAmv/XAIIAn/+FAIIAyP/XAIIAyv/XAIIAzP/XAIIAzv/XAIIA3v/XAIIA4P/XAIIA4v/XAIIA5P/XAIIBDv/XAIIBEP/XAIIBEv/XAIIBFP/XAIIBJP9xAIIBJv9xAIIBNv+uAIIBOP+FAIIBOv+FAIIBR//XAIIB+v+uAIIB/P+uAIIB/v+uAIICAP+FAIICB/9xAIICC/9xAIICX//XAIIDSf/XAIIDS//XAIIDTf/XAIIDT//XAIIDUf/XAIIDU//XAIIDVf/XAIIDV//XAIIDWf/XAIIDW//XAIIDXf/XAIIDX//XAIIDb/+FAIIDcf+FAIIDc/+FAIIDj/9xAIMABf9xAIMACv9xAIMAJv/XAIMAKv/XAIMALQEKAIMAMv/XAIMANP/XAIMAN/9xAIMAOf+uAIMAOv+uAIMAPP+FAIMAif/XAIMAlP/XAIMAlf/XAIMAlv/XAIMAl//XAIMAmP/XAIMAmv/XAIMAn/+FAIMAyP/XAIMAyv/XAIMAzP/XAIMAzv/XAIMA3v/XAIMA4P/XAIMA4v/XAIMA5P/XAIMBDv/XAIMBEP/XAIMBEv/XAIMBFP/XAIMBJP9xAIMBJv9xAIMBNv+uAIMBOP+FAIMBOv+FAIMBR//XAIMB+v+uAIMB/P+uAIMB/v+uAIMCAP+FAIMCB/9xAIMCC/9xAIMCX//XAIMDSf/XAIMDS//XAIMDTf/XAIMDT//XAIMDUf/XAIMDU//XAIMDVf/XAIMDV//XAIMDWf/XAIMDW//XAIMDXf/XAIMDX//XAIMDb/+FAIMDcf+FAIMDc/+FAIMDj/9xAIQABf9xAIQACv9xAIQAJv/XAIQAKv/XAIQALQEKAIQAMv/XAIQANP/XAIQAN/9xAIQAOf+uAIQAOv+uAIQAPP+FAIQAif/XAIQAlP/XAIQAlf/XAIQAlv/XAIQAl//XAIQAmP/XAIQAmv/XAIQAn/+FAIQAyP/XAIQAyv/XAIQAzP/XAIQAzv/XAIQA3v/XAIQA4P/XAIQA4v/XAIQA5P/XAIQBDv/XAIQBEP/XAIQBEv/XAIQBFP/XAIQBJP9xAIQBJv9xAIQBNv+uAIQBOP+FAIQBOv+FAIQBR//XAIQB+v+uAIQB/P+uAIQB/v+uAIQCAP+FAIQCB/9xAIQCC/9xAIQCX//XAIQDSf/XAIQDS//XAIQDTf/XAIQDT//XAIQDUf/XAIQDU//XAIQDVf/XAIQDV//XAIQDWf/XAIQDW//XAIQDXf/XAIQDX//XAIQDb/+FAIQDcf+FAIQDc/+FAIQDj/9xAIUABf9xAIUACv9xAIUAJv/XAIUAKv/XAIUALQEKAIUAMv/XAIUANP/XAIUAN/9xAIUAOf+uAIUAOv+uAIUAPP+FAIUAif/XAIUAlP/XAIUAlf/XAIUAlv/XAIUAl//XAIUAmP/XAIUAmv/XAIUAn/+FAIUAyP/XAIUAyv/XAIUAzP/XAIUAzv/XAIUA3v/XAIUA4P/XAIUA4v/XAIUA5P/XAIUBDv/XAIUBEP/XAIUBEv/XAIUBFP/XAIUBJP9xAIUBJv9xAIUBNv+uAIUBOP+FAIUBOv+FAIUBR//XAIUB+v+uAIUB/P+uAIUB/v+uAIUCAP+FAIUCB/9xAIUCC/9xAIUCX//XAIUDSf/XAIUDS//XAIUDTf/XAIUDT//XAIUDUf/XAIUDU//XAIUDVf/XAIUDV//XAIUDWf/XAIUDW//XAIUDXf/XAIUDX//XAIUDb/+FAIUDcf+FAIUDc/+FAIUDj/9xAIYABf9xAIYACv9xAIYAJv/XAIYAKv/XAIYALQEKAIYAMv/XAIYANP/XAIYAN/9xAIYAOf+uAIYAOv+uAIYAPP+FAIYAif/XAIYAlP/XAIYAlf/XAIYAlv/XAIYAl//XAIYAmP/XAIYAmv/XAIYAn/+FAIYAyP/XAIYAyv/XAIYAzP/XAIYAzv/XAIYA3v/XAIYA4P/XAIYA4v/XAIYA5P/XAIYBDv/XAIYBEP/XAIYBEv/XAIYBFP/XAIYBJP9xAIYBJv9xAIYBNv+uAIYBOP+FAIYBOv+FAIYBR//XAIYB+v+uAIYB/P+uAIYB/v+uAIYCAP+FAIYCB/9xAIYCC/9xAIYCX//XAIYDSf/XAIYDS//XAIYDTf/XAIYDT//XAIYDUf/XAIYDU//XAIYDVf/XAIYDV//XAIYDWf/XAIYDW//XAIYDXf/XAIYDX//XAIYDb/+FAIYDcf+FAIYDc/+FAIYDj/9xAIcABf9xAIcACv9xAIcAJv/XAIcAKv/XAIcALQEKAIcAMv/XAIcANP/XAIcAN/9xAIcAOf+uAIcAOv+uAIcAPP+FAIcAif/XAIcAlP/XAIcAlf/XAIcAlv/XAIcAl//XAIcAmP/XAIcAmv/XAIcAn/+FAIcAyP/XAIcAyv/XAIcAzP/XAIcAzv/XAIcA3v/XAIcA4P/XAIcA4v/XAIcA5P/XAIcBDv/XAIcBEP/XAIcBEv/XAIcBFP/XAIcBJP9xAIcBJv9xAIcBNv+uAIcBOP+FAIcBOv+FAIcBR//XAIcB+v+uAIcB/P+uAIcB/v+uAIcCAP+FAIcCB/9xAIcCC/9xAIcCX//XAIcDSf/XAIcDS//XAIcDTf/XAIcDT//XAIcDUf/XAIcDU//XAIcDVf/XAIcDV//XAIcDWf/XAIcDW//XAIcDXf/XAIcDX//XAIcDb/+FAIcDcf+FAIcDc/+FAIcDj/9xAIgALQB7AIkAJv/XAIkAKv/XAIkAMv/XAIkANP/XAIkAif/XAIkAlP/XAIkAlf/XAIkAlv/XAIkAl//XAIkAmP/XAIkAmv/XAIkAyP/XAIkAyv/XAIkAzP/XAIkAzv/XAIkA3v/XAIkA4P/XAIkA4v/XAIkA5P/XAIkBDv/XAIkBEP/XAIkBEv/XAIkBFP/XAIkBR//XAIkCX//XAIkDSf/XAIkDS//XAIkDTf/XAIkDT//XAIkDUf/XAIkDU//XAIkDVf/XAIkDV//XAIkDWf/XAIkDW//XAIkDXf/XAIkDX//XAIoALQB7AIsALQB7AIwALQB7AI0ALQB7AJIAD/+uAJIAEf+uAJIAJP/XAJIAN//DAJIAOf/sAJIAOv/sAJIAO//XAJIAPP/sAJIAPf/sAJIAgv/XAJIAg//XAJIAhP/XAJIAhf/XAJIAhv/XAJIAh//XAJIAn//sAJIAwv/XAJIAxP/XAJIAxv/XAJIBJP/DAJIBJv/DAJIBNv/sAJIBOP/sAJIBOv/sAJIBO//sAJIBPf/sAJIBP//sAJIBQ//XAJIBoP/sAJIB+v/sAJIB/P/sAJIB/v/sAJICAP/sAJICCP+uAJICDP+uAJICWP/XAJIDHf/XAJIDH//XAJIDIf/XAJIDI//XAJIDJf/XAJIDJ//XAJIDKf/XAJIDK//XAJIDLf/XAJIDL//XAJIDMf/XAJIDM//XAJIDb//sAJIDcf/sAJIDc//sAJIDj//DAJQAD/+uAJQAEf+uAJQAJP/XAJQAN//DAJQAOf/sAJQAOv/sAJQAO//XAJQAPP/sAJQAPf/sAJQAgv/XAJQAg//XAJQAhP/XAJQAhf/XAJQAhv/XAJQAh//XAJQAn//sAJQAwv/XAJQAxP/XAJQAxv/XAJQBJP/DAJQBJv/DAJQBNv/sAJQBOP/sAJQBOv/sAJQBO//sAJQBPf/sAJQBP//sAJQBQ//XAJQBoP/sAJQB+v/sAJQB/P/sAJQB/v/sAJQCAP/sAJQCCP+uAJQCDP+uAJQCWP/XAJQDHf/XAJQDH//XAJQDIf/XAJQDI//XAJQDJf/XAJQDJ//XAJQDKf/XAJQDK//XAJQDLf/XAJQDL//XAJQDMf/XAJQDM//XAJQDb//sAJQDcf/sAJQDc//sAJQDj//DAJUAD/+uAJUAEf+uAJUAJP/XAJUAN//DAJUAOf/sAJUAOv/sAJUAO//XAJUAPP/sAJUAPf/sAJUAgv/XAJUAg//XAJUAhP/XAJUAhf/XAJUAhv/XAJUAh//XAJUAn//sAJUAwv/XAJUAxP/XAJUAxv/XAJUBJP/DAJUBJv/DAJUBNv/sAJUBOP/sAJUBOv/sAJUBO//sAJUBPf/sAJUBP//sAJUBQ//XAJUBoP/sAJUB+v/sAJUB/P/sAJUB/v/sAJUCAP/sAJUCCP+uAJUCDP+uAJUCWP/XAJUDHf/XAJUDH//XAJUDIf/XAJUDI//XAJUDJf/XAJUDJ//XAJUDKf/XAJUDK//XAJUDLf/XAJUDL//XAJUDMf/XAJUDM//XAJUDb//sAJUDcf/sAJUDc//sAJUDj//DAJYAD/+uAJYAEf+uAJYAJP/XAJYAN//DAJYAOf/sAJYAOv/sAJYAO//XAJYAPP/sAJYAPf/sAJYAgv/XAJYAg//XAJYAhP/XAJYAhf/XAJYAhv/XAJYAh//XAJYAn//sAJYAwv/XAJYAxP/XAJYAxv/XAJYBJP/DAJYBJv/DAJYBNv/sAJYBOP/sAJYBOv/sAJYBO//sAJYBPf/sAJYBP//sAJYBQ//XAJYBoP/sAJYB+v/sAJYB/P/sAJYB/v/sAJYCAP/sAJYCCP+uAJYCDP+uAJYCWP/XAJYDHf/XAJYDH//XAJYDIf/XAJYDI//XAJYDJf/XAJYDJ//XAJYDKf/XAJYDK//XAJYDLf/XAJYDL//XAJYDMf/XAJYDM//XAJYDb//sAJYDcf/sAJYDc//sAJYDj//DAJcAD/+uAJcAEf+uAJcAJP/XAJcAN//DAJcAOf/sAJcAOv/sAJcAO//XAJcAPP/sAJcAPf/sAJcAgv/XAJcAg//XAJcAhP/XAJcAhf/XAJcAhv/XAJcAh//XAJcAn//sAJcAwv/XAJcAxP/XAJcAxv/XAJcBJP/DAJcBJv/DAJcBNv/sAJcBOP/sAJcBOv/sAJcBO//sAJcBPf/sAJcBP//sAJcBQ//XAJcBoP/sAJcB+v/sAJcB/P/sAJcB/v/sAJcCAP/sAJcCCP+uAJcCDP+uAJcCWP/XAJcDHf/XAJcDH//XAJcDIf/XAJcDI//XAJcDJf/XAJcDJ//XAJcDKf/XAJcDK//XAJcDLf/XAJcDL//XAJcDMf/XAJcDM//XAJcDb//sAJcDcf/sAJcDc//sAJcDj//DAJgAD/+uAJgAEf+uAJgAJP/XAJgAN//DAJgAOf/sAJgAOv/sAJgAO//XAJgAPP/sAJgAPf/sAJgAgv/XAJgAg//XAJgAhP/XAJgAhf/XAJgAhv/XAJgAh//XAJgAn//sAJgAwv/XAJgAxP/XAJgAxv/XAJgBJP/DAJgBJv/DAJgBNv/sAJgBOP/sAJgBOv/sAJgBO//sAJgBPf/sAJgBP//sAJgBQ//XAJgBoP/sAJgB+v/sAJgB/P/sAJgB/v/sAJgCAP/sAJgCCP+uAJgCDP+uAJgCWP/XAJgDHf/XAJgDH//XAJgDIf/XAJgDI//XAJgDJf/XAJgDJ//XAJgDKf/XAJgDK//XAJgDLf/XAJgDL//XAJgDMf/XAJgDM//XAJgDb//sAJgDcf/sAJgDc//sAJgDj//DAJoAD/+uAJoAEf+uAJoAJP/XAJoAN//DAJoAOf/sAJoAOv/sAJoAO//XAJoAPP/sAJoAPf/sAJoAgv/XAJoAg//XAJoAhP/XAJoAhf/XAJoAhv/XAJoAh//XAJoAn//sAJoAwv/XAJoAxP/XAJoAxv/XAJoBJP/DAJoBJv/DAJoBNv/sAJoBOP/sAJoBOv/sAJoBO//sAJoBPf/sAJoBP//sAJoBQ//XAJoBoP/sAJoB+v/sAJoB/P/sAJoB/v/sAJoCAP/sAJoCCP+uAJoCDP+uAJoCWP/XAJoDHf/XAJoDH//XAJoDIf/XAJoDI//XAJoDJf/XAJoDJ//XAJoDKf/XAJoDK//XAJoDLf/XAJoDL//XAJoDMf/XAJoDM//XAJoDb//sAJoDcf/sAJoDc//sAJoDj//DAJsAD//XAJsAEf/XAJsAJP/sAJsAgv/sAJsAg//sAJsAhP/sAJsAhf/sAJsAhv/sAJsAh//sAJsAwv/sAJsAxP/sAJsAxv/sAJsBQ//sAJsCCP/XAJsCDP/XAJsCWP/sAJsDHf/sAJsDH//sAJsDIf/sAJsDI//sAJsDJf/sAJsDJ//sAJsDKf/sAJsDK//sAJsDLf/sAJsDL//sAJsDMf/sAJsDM//sAJwAD//XAJwAEf/XAJwAJP/sAJwAgv/sAJwAg//sAJwAhP/sAJwAhf/sAJwAhv/sAJwAh//sAJwAwv/sAJwAxP/sAJwAxv/sAJwBQ//sAJwCCP/XAJwCDP/XAJwCWP/sAJwDHf/sAJwDH//sAJwDIf/sAJwDI//sAJwDJf/sAJwDJ//sAJwDKf/sAJwDK//sAJwDLf/sAJwDL//sAJwDMf/sAJwDM//sAJ0AD//XAJ0AEf/XAJ0AJP/sAJ0Agv/sAJ0Ag//sAJ0AhP/sAJ0Ahf/sAJ0Ahv/sAJ0Ah//sAJ0Awv/sAJ0AxP/sAJ0Axv/sAJ0BQ//sAJ0CCP/XAJ0CDP/XAJ0CWP/sAJ0DHf/sAJ0DH//sAJ0DIf/sAJ0DI//sAJ0DJf/sAJ0DJ//sAJ0DKf/sAJ0DK//sAJ0DLf/sAJ0DL//sAJ0DMf/sAJ0DM//sAJ4AD//XAJ4AEf/XAJ4AJP/sAJ4Agv/sAJ4Ag//sAJ4AhP/sAJ4Ahf/sAJ4Ahv/sAJ4Ah//sAJ4Awv/sAJ4AxP/sAJ4Axv/sAJ4BQ//sAJ4CCP/XAJ4CDP/XAJ4CWP/sAJ4DHf/sAJ4DH//sAJ4DIf/sAJ4DI//sAJ4DJf/sAJ4DJ//sAJ4DKf/sAJ4DK//sAJ4DLf/sAJ4DL//sAJ4DMf/sAJ4DM//sAJ8AD/+FAJ8AEf+FAJ8AIgApAJ8AJP+FAJ8AJv/XAJ8AKv/XAJ8AMv/XAJ8ANP/XAJ8ARP+aAJ8ARv+aAJ8AR/+aAJ8ASP+aAJ8ASv/XAJ8AUP/DAJ8AUf/DAJ8AUv+aAJ8AU//DAJ8AVP+aAJ8AVf/DAJ8AVv+uAJ8AWP/DAJ8AXf/XAJ8Agv+FAJ8Ag/+FAJ8AhP+FAJ8Ahf+FAJ8Ahv+FAJ8Ah/+FAJ8Aif/XAJ8AlP/XAJ8Alf/XAJ8Alv/XAJ8Al//XAJ8AmP/XAJ8Amv/XAJ8Aov+aAJ8Ao/+aAJ8ApP+aAJ8Apf+aAJ8Apv+aAJ8Ap/+aAJ8AqP+aAJ8Aqf+aAJ8Aqv+aAJ8Aq/+aAJ8ArP+aAJ8Arf+aAJ8AtP+aAJ8Atf+aAJ8Atv+aAJ8At/+aAJ8AuP+aAJ8Auv+aAJ8Au//DAJ8AvP/DAJ8Avf/DAJ8Avv/DAJ8Awv+FAJ8Aw/+aAJ8AxP+FAJ8Axf+aAJ8Axv+FAJ8Ax/+aAJ8AyP/XAJ8Ayf+aAJ8Ayv/XAJ8Ay/+aAJ8AzP/XAJ8Azf+aAJ8Azv/XAJ8Az/+aAJ8A0f+aAJ8A0/+aAJ8A1f+aAJ8A1/+aAJ8A2f+aAJ8A2/+aAJ8A3f+aAJ8A3v/XAJ8A3//XAJ8A4P/XAJ8A4f/XAJ8A4v/XAJ8A4//XAJ8A5P/XAJ8A5f/XAJ8A+v/DAJ8BBv/DAJ8BCP/DAJ8BDf/DAJ8BDv/XAJ8BD/+aAJ8BEP/XAJ8BEf+aAJ8BEv/XAJ8BE/+aAJ8BFP/XAJ8BFf+aAJ8BF//DAJ8BGf/DAJ8BHf+uAJ8BIf+uAJ8BK//DAJ8BLf/DAJ8BL//DAJ8BMf/DAJ8BM//DAJ8BNf/DAJ8BPP/XAJ8BPv/XAJ8BQP/XAJ8BQ/+FAJ8BRP+aAJ8BRv+aAJ8BR//XAJ8BSP+aAJ8BSv+uAJ8CCP+FAJ8CDP+FAJ8CV//DAJ8CWP+FAJ8CWf+aAJ8CX//XAJ8CYP+aAJ8CYv/DAJ8DHf+FAJ8DHv+aAJ8DH/+FAJ8DIP+aAJ8DIf+FAJ8DIv+aAJ8DI/+FAJ8DJf+FAJ8DJv+aAJ8DJ/+FAJ8DKP+aAJ8DKf+FAJ8DKv+aAJ8DK/+FAJ8DLP+aAJ8DLf+FAJ8DLv+aAJ8DL/+FAJ8DMP+aAJ8DMf+FAJ8DMv+aAJ8DM/+FAJ8DNP+aAJ8DNv+aAJ8DOP+aAJ8DOv+aAJ8DPP+aAJ8DQP+aAJ8DQv+aAJ8DRP+aAJ8DSf/XAJ8DSv+aAJ8DS//XAJ8DTP+aAJ8DTf/XAJ8DTv+aAJ8DT//XAJ8DUf/XAJ8DUv+aAJ8DU//XAJ8DVP+aAJ8DVf/XAJ8DVv+aAJ8DV//XAJ8DWP+aAJ8DWf/XAJ8DWv+aAJ8DW//XAJ8DXP+aAJ8DXf/XAJ8DXv+aAJ8DX//XAJ8DYP+aAJ8DYv/DAJ8DZP/DAJ8DZv/DAJ8DaP/DAJ8Dav/DAJ8DbP/DAJ8Dbv/DAKAAD/72AKAAEf72AKAAJP+aAKAAO//XAKAAPf/sAKAAgv+aAKAAg/+aAKAAhP+aAKAAhf+aAKAAhv+aAKAAh/+aAKAAwv+aAKAAxP+aAKAAxv+aAKABO//sAKABPf/sAKABP//sAKABQ/+aAKACCP72AKACDP72AKACWP+aAKADHf+aAKADH/+aAKADIf+aAKADI/+aAKADJf+aAKADJ/+aAKADKf+aAKADK/+aAKADLf+aAKADL/+aAKADMf+aAKADM/+aAKIABf/sAKIACv/sAKICB//sAKICC//sAKMABf/sAKMACv/sAKMCB//sAKMCC//sAKQABf/sAKQACv/sAKQCB//sAKQCC//sAKUABf/sAKUACv/sAKUCB//sAKUCC//sAKYABf/sAKYACv/sAKYCB//sAKYCC//sAKcABf/sAKcACv/sAKcCB//sAKcCC//sAKoABf/sAKoACv/sAKoAWf/XAKoAWv/XAKoAW//XAKoAXP/XAKoAXf/sAKoAv//XAKoBN//XAKoBPP/sAKoBPv/sAKoBQP/sAKoB+//XAKoB/f/XAKoCB//sAKoCC//sAKoDcP/XAKsABf/sAKsACv/sAKsAWf/XAKsAWv/XAKsAW//XAKsAXP/XAKsAXf/sAKsAv//XAKsBN//XAKsBPP/sAKsBPv/sAKsBQP/sAKsB+//XAKsB/f/XAKsCB//sAKsCC//sAKsDcP/XAKwABf/sAKwACv/sAKwAWf/XAKwAWv/XAKwAW//XAKwAXP/XAKwAXf/sAKwAv//XAKwBN//XAKwBPP/sAKwBPv/sAKwBQP/sAKwB+//XAKwB/f/XAKwCB//sAKwCC//sAKwDcP/XAK0ABf/sAK0ACv/sAK0AWf/XAK0AWv/XAK0AW//XAK0AXP/XAK0AXf/sAK0Av//XAK0BN//XAK0BPP/sAK0BPv/sAK0BQP/sAK0B+//XAK0B/f/XAK0CB//sAK0CC//sAK0DcP/XALIABf/sALIACv/sALIAWf/XALIAWv/XALIAW//XALIAXP/XALIAXf/sALIAv//XALIBN//XALIBPP/sALIBPv/sALIBQP/sALIB+//XALIB/f/XALICB//sALICC//sALIDcP/XALQABf/sALQACv/sALQAWf/XALQAWv/XALQAW//XALQAXP/XALQAXf/sALQAv//XALQBN//XALQBPP/sALQBPv/sALQBQP/sALQB+//XALQB/f/XALQCB//sALQCC//sALQDcP/XALUABf/sALUACv/sALUAWf/XALUAWv/XALUAW//XALUAXP/XALUAXf/sALUAv//XALUBN//XALUBPP/sALUBPv/sALUBQP/sALUB+//XALUB/f/XALUCB//sALUCC//sALUDcP/XALYABf/sALYACv/sALYAWf/XALYAWv/XALYAW//XALYAXP/XALYAXf/sALYAv//XALYBN//XALYBPP/sALYBPv/sALYBQP/sALYB+//XALYB/f/XALYCB//sALYCC//sALYDcP/XALgABf/XALgACv/XALgCB//XALgCC//XALoABf/sALoACv/sALoAWf/XALoAWv/XALoAW//XALoAXP/XALoAXf/sALoAv//XALoBN//XALoBPP/sALoBPv/sALoBQP/sALoB+//XALoB/f/XALoCB//sALoCC//sALoDcP/XAL8ABQBSAL8ACgBSAL8AD/+uAL8AEf+uAL8AIgApAL8CBwBSAL8CCP+uAL8CCwBSAL8CDP+uAMAABf/sAMAACv/sAMAAWf/XAMAAWv/XAMAAW//XAMAAXP/XAMAAXf/sAMAAv//XAMABN//XAMABPP/sAMABPv/sAMABQP/sAMAB+//XAMAB/f/XAMACB//sAMACC//sAMADcP/XAMEABQBSAMEACgBSAMEAD/+uAMEAEf+uAMEAIgApAMECBwBSAMECCP+uAMECCwBSAMECDP+uAMIABf9xAMIACv9xAMIAJv/XAMIAKv/XAMIALQEKAMIAMv/XAMIANP/XAMIAN/9xAMIAOf+uAMIAOv+uAMIAPP+FAMIAif/XAMIAlP/XAMIAlf/XAMIAlv/XAMIAl//XAMIAmP/XAMIAmv/XAMIAn/+FAMIAyP/XAMIAyv/XAMIAzP/XAMIAzv/XAMIA3v/XAMIA4P/XAMIA4v/XAMIA5P/XAMIBDv/XAMIBEP/XAMIBEv/XAMIBFP/XAMIBJP9xAMIBJv9xAMIBNv+uAMIBOP+FAMIBOv+FAMIBR//XAMIB+v+uAMIB/P+uAMIB/v+uAMICAP+FAMICB/9xAMICC/9xAMICX//XAMIDSf/XAMIDS//XAMIDTf/XAMIDT//XAMIDUf/XAMIDU//XAMIDVf/XAMIDV//XAMIDWf/XAMIDW//XAMIDXf/XAMIDX//XAMIDb/+FAMIDcf+FAMIDc/+FAMIDj/9xAMMABf/sAMMACv/sAMMCB//sAMMCC//sAMQABf9xAMQACv9xAMQAJv/XAMQAKv/XAMQALQEKAMQAMv/XAMQANP/XAMQAN/9xAMQAOf+uAMQAOv+uAMQAPP+FAMQAif/XAMQAlP/XAMQAlf/XAMQAlv/XAMQAl//XAMQAmP/XAMQAmv/XAMQAn/+FAMQAyP/XAMQAyv/XAMQAzP/XAMQAzv/XAMQA3v/XAMQA4P/XAMQA4v/XAMQA5P/XAMQBDv/XAMQBEP/XAMQBEv/XAMQBFP/XAMQBJP9xAMQBJv9xAMQBNv+uAMQBOP+FAMQBOv+FAMQBR//XAMQB+v+uAMQB/P+uAMQB/v+uAMQCAP+FAMQCB/9xAMQCC/9xAMQCX//XAMQDSf/XAMQDS//XAMQDTf/XAMQDT//XAMQDUf/XAMQDU//XAMQDVf/XAMQDV//XAMQDWf/XAMQDW//XAMQDXf/XAMQDX//XAMQDb/+FAMQDcf+FAMQDc/+FAMQDj/9xAMUABf/sAMUACv/sAMUCB//sAMUCC//sAMYABf9xAMYACv9xAMYAJv/XAMYAKv/XAMYALQEKAMYAMv/XAMYANP/XAMYAN/9xAMYAOf+uAMYAOv+uAMYAPP+FAMYAif/XAMYAlP/XAMYAlf/XAMYAlv/XAMYAl//XAMYAmP/XAMYAmv/XAMYAn/+FAMYAyP/XAMYAyv/XAMYAzP/XAMYAzv/XAMYA3v/XAMYA4P/XAMYA4v/XAMYA5P/XAMYBDv/XAMYBEP/XAMYBEv/XAMYBFP/XAMYBJP9xAMYBJv9xAMYBNv+uAMYBOP+FAMYBOv+FAMYBR//XAMYB+v+uAMYB/P+uAMYB/v+uAMYCAP+FAMYCB/9xAMYCC/9xAMYCX//XAMYDSf/XAMYDS//XAMYDTf/XAMYDT//XAMYDUf/XAMYDU//XAMYDVf/XAMYDV//XAMYDWf/XAMYDW//XAMYDXf/XAMYDX//XAMYDb/+FAMYDcf+FAMYDc/+FAMYDj/9xAMcABf/sAMcACv/sAMcCB//sAMcCC//sAMgAJv/XAMgAKv/XAMgAMv/XAMgANP/XAMgAif/XAMgAlP/XAMgAlf/XAMgAlv/XAMgAl//XAMgAmP/XAMgAmv/XAMgAyP/XAMgAyv/XAMgAzP/XAMgAzv/XAMgA3v/XAMgA4P/XAMgA4v/XAMgA5P/XAMgBDv/XAMgBEP/XAMgBEv/XAMgBFP/XAMgBR//XAMgCX//XAMgDSf/XAMgDS//XAMgDTf/XAMgDT//XAMgDUf/XAMgDU//XAMgDVf/XAMgDV//XAMgDWf/XAMgDW//XAMgDXf/XAMgDX//XAMoAJv/XAMoAKv/XAMoAMv/XAMoANP/XAMoAif/XAMoAlP/XAMoAlf/XAMoAlv/XAMoAl//XAMoAmP/XAMoAmv/XAMoAyP/XAMoAyv/XAMoAzP/XAMoAzv/XAMoA3v/XAMoA4P/XAMoA4v/XAMoA5P/XAMoBDv/XAMoBEP/XAMoBEv/XAMoBFP/XAMoBR//XAMoCX//XAMoDSf/XAMoDS//XAMoDTf/XAMoDT//XAMoDUf/XAMoDU//XAMoDVf/XAMoDV//XAMoDWf/XAMoDW//XAMoDXf/XAMoDX//XAMwAJv/XAMwAKv/XAMwAMv/XAMwANP/XAMwAif/XAMwAlP/XAMwAlf/XAMwAlv/XAMwAl//XAMwAmP/XAMwAmv/XAMwAyP/XAMwAyv/XAMwAzP/XAMwAzv/XAMwA3v/XAMwA4P/XAMwA4v/XAMwA5P/XAMwBDv/XAMwBEP/XAMwBEv/XAMwBFP/XAMwBR//XAMwCX//XAMwDSf/XAMwDS//XAMwDTf/XAMwDT//XAMwDUf/XAMwDU//XAMwDVf/XAMwDV//XAMwDWf/XAMwDW//XAMwDXf/XAMwDX//XAM4AJv/XAM4AKv/XAM4AMv/XAM4ANP/XAM4Aif/XAM4AlP/XAM4Alf/XAM4Alv/XAM4Al//XAM4AmP/XAM4Amv/XAM4AyP/XAM4Ayv/XAM4AzP/XAM4Azv/XAM4A3v/XAM4A4P/XAM4A4v/XAM4A5P/XAM4BDv/XAM4BEP/XAM4BEv/XAM4BFP/XAM4BR//XAM4CX//XAM4DSf/XAM4DS//XAM4DTf/XAM4DT//XAM4DUf/XAM4DU//XAM4DVf/XAM4DV//XAM4DWf/XAM4DW//XAM4DXf/XAM4DX//XANAAD/+uANAAEf+uANAAJP/XANAAN//DANAAOf/sANAAOv/sANAAO//XANAAPP/sANAAPf/sANAAgv/XANAAg//XANAAhP/XANAAhf/XANAAhv/XANAAh//XANAAn//sANAAwv/XANAAxP/XANAAxv/XANABJP/DANABJv/DANABNv/sANABOP/sANABOv/sANABO//sANABPf/sANABP//sANABQ//XANABoP/sANAB+v/sANAB/P/sANAB/v/sANACAP/sANACCP+uANACDP+uANACWP/XANADHf/XANADH//XANADIf/XANADI//XANADJf/XANADJ//XANADKf/XANADK//XANADLf/XANADL//XANADMf/XANADM//XANADb//sANADcf/sANADc//sANADj//DANEABQBSANEACgBSANEADACPANEAIgCkANEAQACPANEARQA9ANEASwA9ANEATgA9ANEATwA9ANEAYACPANEA5wA9ANEA6QB7ANECBwBSANECCwBSANIAD/+uANIAEf+uANIAJP/XANIAN//DANIAOf/sANIAOv/sANIAO//XANIAPP/sANIAPf/sANIAgv/XANIAg//XANIAhP/XANIAhf/XANIAhv/XANIAh//XANIAn//sANIAwv/XANIAxP/XANIAxv/XANIBJP/DANIBJv/DANIBNv/sANIBOP/sANIBOv/sANIBO//sANIBPf/sANIBP//sANIBQ//XANIBoP/sANIB+v/sANIB/P/sANIB/v/sANICAP/sANICCP+uANICDP+uANICWP/XANIDHf/XANIDH//XANIDIf/XANIDI//XANIDJf/XANIDJ//XANIDKf/XANIDK//XANIDLf/XANIDL//XANIDMf/XANIDM//XANIDb//sANIDcf/sANIDc//sANIDj//DANQALQB7ANUABf/sANUACv/sANUAWf/XANUAWv/XANUAW//XANUAXP/XANUAXf/sANUAv//XANUBN//XANUBPP/sANUBPv/sANUBQP/sANUB+//XANUB/f/XANUCB//sANUCC//sANUDcP/XANYALQB7ANcABf/sANcACv/sANcAWf/XANcAWv/XANcAW//XANcAXP/XANcAXf/sANcAv//XANcBN//XANcBPP/sANcBPv/sANcBQP/sANcB+//XANcB/f/XANcCB//sANcCC//sANcDcP/XANgALQB7ANkABf/sANkACv/sANkAWf/XANkAWv/XANkAW//XANkAXP/XANkAXf/sANkAv//XANkBN//XANkBPP/sANkBPv/sANkBQP/sANkB+//XANkB/f/XANkCB//sANkCC//sANkDcP/XANoALQB7ANsABf/sANsACv/sANsAWf/XANsAWv/XANsAW//XANsAXP/XANsAXf/sANsAv//XANsBN//XANsBPP/sANsBPv/sANsBQP/sANsB+//XANsB/f/XANsCB//sANsCC//sANsDcP/XANwALQB7AN0ABf/sAN0ACv/sAN0AWf/XAN0AWv/XAN0AW//XAN0AXP/XAN0AXf/sAN0Av//XAN0BN//XAN0BPP/sAN0BPv/sAN0BQP/sAN0B+//XAN0B/f/XAN0CB//sAN0CC//sAN0DcP/XAOcABf/sAOcACv/sAOcCB//sAOcCC//sAPgAJv/XAPgAKv/XAPgAMv/XAPgANP/XAPgAif/XAPgAlP/XAPgAlf/XAPgAlv/XAPgAl//XAPgAmP/XAPgAmv/XAPgAyP/XAPgAyv/XAPgAzP/XAPgAzv/XAPgA3v/XAPgA4P/XAPgA4v/XAPgA5P/XAPgBDv/XAPgBEP/XAPgBEv/XAPgBFP/XAPgBR//XAPgCX//XAPgDSf/XAPgDS//XAPgDTf/XAPgDT//XAPgDUf/XAPgDU//XAPgDVf/XAPgDV//XAPgDWf/XAPgDW//XAPgDXf/XAPgDX//XAPkARv/XAPkAR//XAPkASP/XAPkAUv/XAPkAVP/XAPkAov/XAPkAqf/XAPkAqv/XAPkAq//XAPkArP/XAPkArf/XAPkAtP/XAPkAtf/XAPkAtv/XAPkAt//XAPkAuP/XAPkAuv/XAPkAyf/XAPkAy//XAPkAzf/XAPkAz//XAPkA0f/XAPkA0//XAPkA1f/XAPkA1//XAPkA2f/XAPkA2//XAPkA3f/XAPkBD//XAPkBEf/XAPkBE//XAPkBFf/XAPkBSP/XAPkCYP/XAPkDNv/XAPkDOP/XAPkDOv/XAPkDPP/XAPkDQP/XAPkDQv/XAPkDRP/XAPkDSv/XAPkDTP/XAPkDTv/XAPkDUv/XAPkDVP/XAPkDVv/XAPkDWP/XAPkDWv/XAPkDXP/XAPkDXv/XAPkDYP/XAPoARv/XAPoAR//XAPoASP/XAPoAUv/XAPoAVP/XAPoAov/XAPoAqf/XAPoAqv/XAPoAq//XAPoArP/XAPoArf/XAPoAtP/XAPoAtf/XAPoAtv/XAPoAt//XAPoAuP/XAPoAuv/XAPoAyf/XAPoAy//XAPoAzf/XAPoAz//XAPoA0f/XAPoA0//XAPoA1f/XAPoA1//XAPoA2f/XAPoA2//XAPoA3f/XAPoBD//XAPoBEf/XAPoBE//XAPoBFf/XAPoBSP/XAPoCYP/XAPoDNv/XAPoDOP/XAPoDOv/XAPoDPP/XAPoDQP/XAPoDQv/XAPoDRP/XAPoDSv/XAPoDTP/XAPoDTv/XAPoDUv/XAPoDVP/XAPoDVv/XAPoDWP/XAPoDWv/XAPoDXP/XAPoDXv/XAPoDYP/XAPsABf9cAPsACv9cAPsAJv/XAPsAKv/XAPsAMv/XAPsANP/XAPsAN//XAPsAOP/sAPsAOf/XAPsAOv/XAPsAPP/DAPsAif/XAPsAlP/XAPsAlf/XAPsAlv/XAPsAl//XAPsAmP/XAPsAmv/XAPsAm//sAPsAnP/sAPsAnf/sAPsAnv/sAPsAn//DAPsAyP/XAPsAyv/XAPsAzP/XAPsAzv/XAPsA3v/XAPsA4P/XAPsA4v/XAPsA5P/XAPsBDv/XAPsBEP/XAPsBEv/XAPsBFP/XAPsBJP/XAPsBJv/XAPsBKv/sAPsBLP/sAPsBLv/sAPsBMP/sAPsBMv/sAPsBNP/sAPsBNv/XAPsBOP/DAPsBOv/DAPsBR//XAPsB+v/XAPsB/P/XAPsB/v/XAPsCAP/DAPsCB/9cAPsCC/9cAPsCX//XAPsCYf/sAPsDSf/XAPsDS//XAPsDTf/XAPsDT//XAPsDUf/XAPsDU//XAPsDVf/XAPsDV//XAPsDWf/XAPsDW//XAPsDXf/XAPsDX//XAPsDYf/sAPsDY//sAPsDZf/sAPsDZ//sAPsDaf/sAPsDa//sAPsDbf/sAPsDb//DAPsDcf/DAPsDc//DAPsDj//XAP0ABf9cAP0ACv9cAP0AJv/XAP0AKv/XAP0AMv/XAP0ANP/XAP0AN//XAP0AOP/sAP0AOf/XAP0AOv/XAP0APP/DAP0Aif/XAP0AlP/XAP0Alf/XAP0Alv/XAP0Al//XAP0AmP/XAP0Amv/XAP0Am//sAP0AnP/sAP0Anf/sAP0Anv/sAP0An//DAP0AyP/XAP0Ayv/XAP0AzP/XAP0Azv/XAP0A3v/XAP0A4P/XAP0A4v/XAP0A5P/XAP0BDv/XAP0BEP/XAP0BEv/XAP0BFP/XAP0BJP/XAP0BJv/XAP0BKv/sAP0BLP/sAP0BLv/sAP0BMP/sAP0BMv/sAP0BNP/sAP0BNv/XAP0BOP/DAP0BOv/DAP0BR//XAP0B+v/XAP0B/P/XAP0B/v/XAP0CAP/DAP0CB/9cAP0CC/9cAP0CX//XAP0CYf/sAP0DSf/XAP0DS//XAP0DTf/XAP0DT//XAP0DUf/XAP0DU//XAP0DVf/XAP0DV//XAP0DWf/XAP0DW//XAP0DXf/XAP0DX//XAP0DYf/sAP0DY//sAP0DZf/sAP0DZ//sAP0Daf/sAP0Da//sAP0Dbf/sAP0Db//DAP0Dcf/DAP0Dc//DAP0Dj//XAP8ABf9cAP8ACv9cAP8AJv/XAP8AKv/XAP8AMv/XAP8ANP/XAP8AN//XAP8AOP/sAP8AOf/XAP8AOv/XAP8APP/DAP8Aif/XAP8AlP/XAP8Alf/XAP8Alv/XAP8Al//XAP8AmP/XAP8Amv/XAP8Am//sAP8AnP/sAP8Anf/sAP8Anv/sAP8An//DAP8AyP/XAP8Ayv/XAP8AzP/XAP8Azv/XAP8A3v/XAP8A4P/XAP8A4v/XAP8A5P/XAP8BDv/XAP8BEP/XAP8BEv/XAP8BFP/XAP8BJP/XAP8BJv/XAP8BKv/sAP8BLP/sAP8BLv/sAP8BMP/sAP8BMv/sAP8BNP/sAP8BNv/XAP8BOP/DAP8BOv/DAP8BR//XAP8B+v/XAP8B/P/XAP8B/v/XAP8CAP/DAP8CB/9cAP8CC/9cAP8CX//XAP8CYf/sAP8DSf/XAP8DS//XAP8DTf/XAP8DT//XAP8DUf/XAP8DU//XAP8DVf/XAP8DV//XAP8DWf/XAP8DW//XAP8DXf/XAP8DX//XAP8DYf/sAP8DY//sAP8DZf/sAP8DZ//sAP8Daf/sAP8Da//sAP8Dbf/sAP8Db//DAP8Dcf/DAP8Dc//DAP8Dj//XAQAABQBSAQAACgBSAQAADACPAQAAIgCPAQAAQACPAQAARQA9AQAASwA9AQAATgA9AQAATwA9AQAAYACPAQAA5wA9AQAA6QCPAQACBwBSAQACCwBSAQEABf9cAQEACv9cAQEAJv/XAQEAKv/XAQEAMv/XAQEANP/XAQEAN//XAQEAOP/sAQEAOf/XAQEAOv/XAQEAPP/DAQEAif/XAQEAlP/XAQEAlf/XAQEAlv/XAQEAl//XAQEAmP/XAQEAmv/XAQEAm//sAQEAnP/sAQEAnf/sAQEAnv/sAQEAn//DAQEAyP/XAQEAyv/XAQEAzP/XAQEAzv/XAQEA3v/XAQEA4P/XAQEA4v/XAQEA5P/XAQEBDv/XAQEBEP/XAQEBEv/XAQEBFP/XAQEBJP/XAQEBJv/XAQEBKv/sAQEBLP/sAQEBLv/sAQEBMP/sAQEBMv/sAQEBNP/sAQEBNv/XAQEBOP/DAQEBOv/DAQEBR//XAQEB+v/XAQEB/P/XAQEB/v/XAQECAP/DAQECB/9cAQECC/9cAQECX//XAQECYf/sAQEDSf/XAQEDS//XAQEDTf/XAQEDT//XAQEDUf/XAQEDU//XAQEDVf/XAQEDV//XAQEDWf/XAQEDW//XAQEDXf/XAQEDX//XAQEDYf/sAQEDY//sAQEDZf/sAQEDZ//sAQEDaf/sAQEDa//sAQEDbf/sAQEDb//DAQEDcf/DAQEDc//DAQEDj//XAQMABf9cAQMACv9cAQMAJv/XAQMAKv/XAQMAMv/XAQMANP/XAQMAN//XAQMAOP/sAQMAOf/XAQMAOv/XAQMAPP/DAQMAif/XAQMAlP/XAQMAlf/XAQMAlv/XAQMAl//XAQMAmP/XAQMAmv/XAQMAm//sAQMAnP/sAQMAnf/sAQMAnv/sAQMAn//DAQMAyP/XAQMAyv/XAQMAzP/XAQMAzv/XAQMA3v/XAQMA4P/XAQMA4v/XAQMA5P/XAQMBDv/XAQMBEP/XAQMBEv/XAQMBFP/XAQMBJP/XAQMBJv/XAQMBKv/sAQMBLP/sAQMBLv/sAQMBMP/sAQMBMv/sAQMBNP/sAQMBNv/XAQMBOP/DAQMBOv/DAQMBR//XAQMB+v/XAQMB/P/XAQMB/v/XAQMCAP/DAQMCB/9cAQMCC/9cAQMCX//XAQMCYf/sAQMDSf/XAQMDS//XAQMDTf/XAQMDT//XAQMDUf/XAQMDU//XAQMDVf/XAQMDV//XAQMDWf/XAQMDW//XAQMDXf/XAQMDX//XAQMDYf/sAQMDY//sAQMDZf/sAQMDZ//sAQMDaf/sAQMDa//sAQMDbf/sAQMDb//DAQMDcf/DAQMDc//DAQMDj//XAQgABf/sAQgACv/sAQgCB//sAQgCC//sAQ4AD/+uAQ4AEf+uAQ4AJP/XAQ4AN//DAQ4AOf/sAQ4AOv/sAQ4AO//XAQ4APP/sAQ4APf/sAQ4Agv/XAQ4Ag//XAQ4AhP/XAQ4Ahf/XAQ4Ahv/XAQ4Ah//XAQ4An//sAQ4Awv/XAQ4AxP/XAQ4Axv/XAQ4BJP/DAQ4BJv/DAQ4BNv/sAQ4BOP/sAQ4BOv/sAQ4BO//sAQ4BPf/sAQ4BP//sAQ4BQ//XAQ4BoP/sAQ4B+v/sAQ4B/P/sAQ4B/v/sAQ4CAP/sAQ4CCP+uAQ4CDP+uAQ4CWP/XAQ4DHf/XAQ4DH//XAQ4DIf/XAQ4DI//XAQ4DJf/XAQ4DJ//XAQ4DKf/XAQ4DK//XAQ4DLf/XAQ4DL//XAQ4DMf/XAQ4DM//XAQ4Db//sAQ4Dcf/sAQ4Dc//sAQ4Dj//DARAAD/+uARAAEf+uARAAJP/XARAAN//DARAAOf/sARAAOv/sARAAO//XARAAPP/sARAAPf/sARAAgv/XARAAg//XARAAhP/XARAAhf/XARAAhv/XARAAh//XARAAn//sARAAwv/XARAAxP/XARAAxv/XARABJP/DARABJv/DARABNv/sARABOP/sARABOv/sARABO//sARABPf/sARABP//sARABQ//XARABoP/sARAB+v/sARAB/P/sARAB/v/sARACAP/sARACCP+uARACDP+uARACWP/XARADHf/XARADH//XARADIf/XARADI//XARADJf/XARADJ//XARADKf/XARADK//XARADLf/XARADL//XARADMf/XARADM//XARADb//sARADcf/sARADc//sARADj//DARIAD/+uARIAEf+uARIAJP/XARIAN//DARIAOf/sARIAOv/sARIAO//XARIAPP/sARIAPf/sARIAgv/XARIAg//XARIAhP/XARIAhf/XARIAhv/XARIAh//XARIAn//sARIAwv/XARIAxP/XARIAxv/XARIBJP/DARIBJv/DARIBNv/sARIBOP/sARIBOv/sARIBO//sARIBPf/sARIBP//sARIBQ//XARIBoP/sARIB+v/sARIB/P/sARIB/v/sARICAP/sARICCP+uARICDP+uARICWP/XARIDHf/XARIDH//XARIDIf/XARIDI//XARIDJf/XARIDJ//XARIDKf/XARIDK//XARIDLf/XARIDL//XARIDMf/XARIDM//XARIDb//sARIDcf/sARIDc//sARIDj//DARQALQB7ARcABQBSARcACgBSARcARP/XARcARv/XARcAR//XARcASP/XARcASv/sARcAUv/XARcAVP/XARcAov/XARcAo//XARcApP/XARcApf/XARcApv/XARcAp//XARcAqP/XARcAqf/XARcAqv/XARcAq//XARcArP/XARcArf/XARcAtP/XARcAtf/XARcAtv/XARcAt//XARcAuP/XARcAuv/XARcAw//XARcAxf/XARcAx//XARcAyf/XARcAy//XARcAzf/XARcAz//XARcA0f/XARcA0//XARcA1f/XARcA1//XARcA2f/XARcA2//XARcA3f/XARcA3//sARcA4f/sARcA4//sARcA5f/sARcBD//XARcBEf/XARcBE//XARcBFf/XARcBRP/XARcBRv/XARcBSP/XARcCBwBSARcCCwBSARcCWf/XARcCYP/XARcDHv/XARcDIP/XARcDIv/XARcDJv/XARcDKP/XARcDKv/XARcDLP/XARcDLv/XARcDMP/XARcDMv/XARcDNP/XARcDNv/XARcDOP/XARcDOv/XARcDPP/XARcDQP/XARcDQv/XARcDRP/XARcDSv/XARcDTP/XARcDTv/XARcDUv/XARcDVP/XARcDVv/XARcDWP/XARcDWv/XARcDXP/XARcDXv/XARcDYP/XARkABQBSARkACgBSARkARP/XARkARv/XARkAR//XARkASP/XARkASv/sARkAUv/XARkAVP/XARkAov/XARkAo//XARkApP/XARkApf/XARkApv/XARkAp//XARkAqP/XARkAqf/XARkAqv/XARkAq//XARkArP/XARkArf/XARkAtP/XARkAtf/XARkAtv/XARkAt//XARkAuP/XARkAuv/XARkAw//XARkAxf/XARkAx//XARkAyf/XARkAy//XARkAzf/XARkAz//XARkA0f/XARkA0//XARkA1f/XARkA1//XARkA2f/XARkA2//XARkA3f/XARkA3//sARkA4f/sARkA4//sARkA5f/sARkBD//XARkBEf/XARkBE//XARkBFf/XARkBRP/XARkBRv/XARkBSP/XARkCBwBSARkCCwBSARkCWf/XARkCYP/XARkDHv/XARkDIP/XARkDIv/XARkDJv/XARkDKP/XARkDKv/XARkDLP/XARkDLv/XARkDMP/XARkDMv/XARkDNP/XARkDNv/XARkDOP/XARkDOv/XARkDPP/XARkDQP/XARkDQv/XARkDRP/XARkDSv/XARkDTP/XARkDTv/XARkDUv/XARkDVP/XARkDVv/XARkDWP/XARkDWv/XARkDXP/XARkDXv/XARkDYP/XARsABQBSARsACgBSARsARP/XARsARv/XARsAR//XARsASP/XARsASv/sARsAUv/XARsAVP/XARsAov/XARsAo//XARsApP/XARsApf/XARsApv/XARsAp//XARsAqP/XARsAqf/XARsAqv/XARsAq//XARsArP/XARsArf/XARsAtP/XARsAtf/XARsAtv/XARsAt//XARsAuP/XARsAuv/XARsAw//XARsAxf/XARsAx//XARsAyf/XARsAy//XARsAzf/XARsAz//XARsA0f/XARsA0//XARsA1f/XARsA1//XARsA2f/XARsA2//XARsA3f/XARsA3//sARsA4f/sARsA4//sARsA5f/sARsBD//XARsBEf/XARsBE//XARsBFf/XARsBRP/XARsBRv/XARsBSP/XARsCBwBSARsCCwBSARsCWf/XARsCYP/XARsDHv/XARsDIP/XARsDIv/XARsDJv/XARsDKP/XARsDKv/XARsDLP/XARsDLv/XARsDMP/XARsDMv/XARsDNP/XARsDNv/XARsDOP/XARsDOv/XARsDPP/XARsDQP/XARsDQv/XARsDRP/XARsDSv/XARsDTP/XARsDTv/XARsDUv/XARsDVP/XARsDVv/XARsDWP/XARsDWv/XARsDXP/XARsDXv/XARsDYP/XASQAD/+FASQAEP+uASQAEf+FASQAIgApASQAJP9xASQAJv/XASQAKv/XASQAMv/XASQANP/XASQANwApASQARP9cASQARv9xASQAR/9xASQASP9xASQASv9xASQAUP+aASQAUf+aASQAUv9xASQAU/+aASQAVP9xASQAVf+aASQAVv+FASQAWP+aASQAWf/XASQAWv/XASQAW//XASQAXP/XASQAXf+uASQAgv9xASQAg/9xASQAhP9xASQAhf9xASQAhv9xASQAh/9xASQAif/XASQAlP/XASQAlf/XASQAlv/XASQAl//XASQAmP/XASQAmv/XASQAov9xASQAo/9cASQApP9cASQApf9cASQApv9cASQAp/9cASQAqP9cASQAqf9xASQAqv9xASQAq/9xASQArP9xASQArf9xASQAtP9xASQAtf9xASQAtv9xASQAt/9xASQAuP9xASQAuv9xASQAu/+aASQAvP+aASQAvf+aASQAvv+aASQAv//XASQAwv9xASQAw/9cASQAxP9xASQAxf9cASQAxv9xASQAx/9cASQAyP/XASQAyf9xASQAyv/XASQAy/9xASQAzP/XASQAzf9xASQAzv/XASQAz/9xASQA0f9xASQA0/9xASQA1f9xASQA1/9xASQA2f9xASQA2/9xASQA3f9xASQA3v/XASQA3/9xASQA4P/XASQA4f9xASQA4v/XASQA4/9xASQA5P/XASQA5f9xASQA+v+aASQBBv+aASQBCP+aASQBDf+aASQBDv/XASQBD/9xASQBEP/XASQBEf9xASQBEv/XASQBE/9xASQBFP/XASQBFf9xASQBF/+aASQBGf+aASQBHf+FASQBIf+FASQBJAApASQBJgApASQBK/+aASQBLf+aASQBL/+aASQBMf+aASQBM/+aASQBNf+aASQBN//XASQBPP+uASQBPv+uASQBQP+uASQBQ/9xASQBRP9cASQBRv9cASQBR//XASQBSP9xASQBSv+FASQB+//XASQB/f/XASQCAv+uASQCA/+uASQCBP+uASQCCP+FASQCDP+FASQCV/+aASQCWP9xASQCWf9cASQCX//XASQCYP9xASQCYv+aASQDHf9xASQDHv9cASQDH/9xASQDIP9cASQDIf9xASQDIv9cASQDI/9xASQDJf9xASQDJv9cASQDJ/9xASQDKP9cASQDKf9xASQDKv9cASQDK/9xASQDLP9cASQDLf9xASQDLv9cASQDL/9xASQDMP9cASQDMf9xASQDMv9cASQDM/9xASQDNP9cASQDNv9xASQDOP9xASQDOv9xASQDPP9xASQDQP9xASQDQv9xASQDRP9xASQDSf/XASQDSv9xASQDS//XASQDTP9xASQDTf/XASQDTv9xASQDT//XASQDUf/XASQDUv9xASQDU//XASQDVP9xASQDVf/XASQDVv9xASQDV//XASQDWP9xASQDWf/XASQDWv9xASQDW//XASQDXP9xASQDXf/XASQDXv9xASQDX//XASQDYP9xASQDYv+aASQDZP+aASQDZv+aASQDaP+aASQDav+aASQDbP+aASQDbv+aASQDcP/XASQDjwApASUABQApASUACgApASUCBwApASUCCwApASYAD/+FASYAEP+uASYAEf+FASYAIgApASYAJP9xASYAJv/XASYAKv/XASYAMv/XASYANP/XASYANwApASYARP9cASYARv9xASYAR/9xASYASP9xASYASv9xASYAUP+aASYAUf+aASYAUv9xASYAU/+aASYAVP9xASYAVf+aASYAVv+FASYAWP+aASYAWf/XASYAWv/XASYAW//XASYAXP/XASYAXf+uASYAgv9xASYAg/9xASYAhP9xASYAhf9xASYAhv9xASYAh/9xASYAif/XASYAlP/XASYAlf/XASYAlv/XASYAl//XASYAmP/XASYAmv/XASYAov9xASYAo/9cASYApP9cASYApf9cASYApv9cASYAp/9cASYAqP9cASYAqf9xASYAqv9xASYAq/9xASYArP9xASYArf9xASYAtP9xASYAtf9xASYAtv9xASYAt/9xASYAuP9xASYAuv9xASYAu/+aASYAvP+aASYAvf+aASYAvv+aASYAv//XASYAwv9xASYAw/9cASYAxP9xASYAxf9cASYAxv9xASYAx/9cASYAyP/XASYAyf9xASYAyv/XASYAy/9xASYAzP/XASYAzf9xASYAzv/XASYAz/9xASYA0f9xASYA0/9xASYA1f9xASYA1/9xASYA2f9xASYA2/9xASYA3f9xASYA3v/XASYA3/9xASYA4P/XASYA4f9xASYA4v/XASYA4/9xASYA5P/XASYA5f9xASYA+v+aASYBBv+aASYBCP+aASYBDf+aASYBDv/XASYBD/9xASYBEP/XASYBEf9xASYBEv/XASYBE/9xASYBFP/XASYBFf9xASYBF/+aASYBGf+aASYBHf+FASYBIf+FASYBJAApASYBJgApASYBK/+aASYBLf+aASYBL/+aASYBMf+aASYBM/+aASYBNf+aASYBN//XASYBPP+uASYBPv+uASYBQP+uASYBQ/9xASYBRP9cASYBRv9cASYBR//XASYBSP9xASYBSv+FASYB+//XASYB/f/XASYCAv+uASYCA/+uASYCBP+uASYCCP+FASYCDP+FASYCV/+aASYCWP9xASYCWf9cASYCX//XASYCYP9xASYCYv+aASYDHf9xASYDHv9cASYDH/9xASYDIP9cASYDIf9xASYDIv9cASYDI/9xASYDJf9xASYDJv9cASYDJ/9xASYDKP9cASYDKf9xASYDKv9cASYDK/9xASYDLP9cASYDLf9xASYDLv9cASYDL/9xASYDMP9cASYDMf9xASYDMv9cASYDM/9xASYDNP9cASYDNv9xASYDOP9xASYDOv9xASYDPP9xASYDQP9xASYDQv9xASYDRP9xASYDSf/XASYDSv9xASYDS//XASYDTP9xASYDTf/XASYDTv9xASYDT//XASYDUf/XASYDUv9xASYDU//XASYDVP9xASYDVf/XASYDVv9xASYDV//XASYDWP9xASYDWf/XASYDWv9xASYDW//XASYDXP9xASYDXf/XASYDXv9xASYDX//XASYDYP9xASYDYv+aASYDZP+aASYDZv+aASYDaP+aASYDav+aASYDbP+aASYDbv+aASYDcP/XASYDjwApAScABQApAScACgApAScCBwApAScCCwApASgAD/+FASgAEP+uASgAEf+FASgAIgApASgAJP9xASgAJv/XASgAKv/XASgAMv/XASgANP/XASgANwApASgARP9cASgARv9xASgAR/9xASgASP9xASgASv9xASgAUP+aASgAUf+aASgAUv9xASgAU/+aASgAVP9xASgAVf+aASgAVv+FASgAWP+aASgAWf/XASgAWv/XASgAW//XASgAXP/XASgAXf+uASgAgv9xASgAg/9xASgAhP9xASgAhf9xASgAhv9xASgAh/9xASgAif/XASgAlP/XASgAlf/XASgAlv/XASgAl//XASgAmP/XASgAmv/XASgAov9xASgAo/9cASgApP9cASgApf9cASgApv9cASgAp/9cASgAqP9cASgAqf9xASgAqv9xASgAq/9xASgArP9xASgArf9xASgAtP9xASgAtf9xASgAtv9xASgAt/9xASgAuP9xASgAuv9xASgAu/+aASgAvP+aASgAvf+aASgAvv+aASgAv//XASgAwv9xASgAw/9cASgAxP9xASgAxf9cASgAxv9xASgAx/9cASgAyP/XASgAyf9xASgAyv/XASgAy/9xASgAzP/XASgAzf9xASgAzv/XASgAz/9xASgA0f9xASgA0/9xASgA1f9xASgA1/9xASgA2f9xASgA2/9xASgA3f9xASgA3v/XASgA3/9xASgA4P/XASgA4f9xASgA4v/XASgA4/9xASgA5P/XASgA5f9xASgA+v+aASgBBv+aASgBCP+aASgBDf+aASgBDv/XASgBD/9xASgBEP/XASgBEf9xASgBEv/XASgBE/9xASgBFP/XASgBFf9xASgBF/+aASgBGf+aASgBHf+FASgBIf+FASgBJAApASgBJgApASgBK/+aASgBLf+aASgBL/+aASgBMf+aASgBM/+aASgBNf+aASgBN//XASgBPP+uASgBPv+uASgBQP+uASgBQ/9xASgBRP9cASgBRv9cASgBR//XASgBSP9xASgBSv+FASgB+//XASgB/f/XASgCAv+uASgCA/+uASgCBP+uASgCCP+FASgCDP+FASgCV/+aASgCWP9xASgCWf9cASgCX//XASgCYP9xASgCYv+aASgDHf9xASgDHv9cASgDH/9xASgDIP9cASgDIf9xASgDIv9cASgDI/9xASgDJf9xASgDJv9cASgDJ/9xASgDKP9cASgDKf9xASgDKv9cASgDK/9xASgDLP9cASgDLf9xASgDLv9cASgDL/9xASgDMP9cASgDMf9xASgDMv9cASgDM/9xASgDNP9cASgDNv9xASgDOP9xASgDOv9xASgDPP9xASgDQP9xASgDQv9xASgDRP9xASgDSf/XASgDSv9xASgDS//XASgDTP9xASgDTf/XASgDTv9xASgDT//XASgDUf/XASgDUv9xASgDU//XASgDVP9xASgDVf/XASgDVv9xASgDV//XASgDWP9xASgDWf/XASgDWv9xASgDW//XASgDXP9xASgDXf/XASgDXv9xASgDX//XASgDYP9xASgDYv+aASgDZP+aASgDZv+aASgDaP+aASgDav+aASgDbP+aASgDbv+aASgDcP/XASgDjwApASoAD//XASoAEf/XASoAJP/sASoAgv/sASoAg//sASoAhP/sASoAhf/sASoAhv/sASoAh//sASoAwv/sASoAxP/sASoAxv/sASoBQ//sASoCCP/XASoCDP/XASoCWP/sASoDHf/sASoDH//sASoDIf/sASoDI//sASoDJf/sASoDJ//sASoDKf/sASoDK//sASoDLf/sASoDL//sASoDMf/sASoDM//sASwAD//XASwAEf/XASwAJP/sASwAgv/sASwAg//sASwAhP/sASwAhf/sASwAhv/sASwAh//sASwAwv/sASwAxP/sASwAxv/sASwBQ//sASwCCP/XASwCDP/XASwCWP/sASwDHf/sASwDH//sASwDIf/sASwDI//sASwDJf/sASwDJ//sASwDKf/sASwDK//sASwDLf/sASwDL//sASwDMf/sASwDM//sAS4AD//XAS4AEf/XAS4AJP/sAS4Agv/sAS4Ag//sAS4AhP/sAS4Ahf/sAS4Ahv/sAS4Ah//sAS4Awv/sAS4AxP/sAS4Axv/sAS4BQ//sAS4CCP/XAS4CDP/XAS4CWP/sAS4DHf/sAS4DH//sAS4DIf/sAS4DI//sAS4DJf/sAS4DJ//sAS4DKf/sAS4DK//sAS4DLf/sAS4DL//sAS4DMf/sAS4DM//sATAAD//XATAAEf/XATAAJP/sATAAgv/sATAAg//sATAAhP/sATAAhf/sATAAhv/sATAAh//sATAAwv/sATAAxP/sATAAxv/sATABQ//sATACCP/XATACDP/XATACWP/sATADHf/sATADH//sATADIf/sATADI//sATADJf/sATADJ//sATADKf/sATADK//sATADLf/sATADL//sATADMf/sATADM//sATIAD//XATIAEf/XATIAJP/sATIAgv/sATIAg//sATIAhP/sATIAhf/sATIAhv/sATIAh//sATIAwv/sATIAxP/sATIAxv/sATIBQ//sATICCP/XATICDP/XATICWP/sATIDHf/sATIDH//sATIDIf/sATIDI//sATIDJf/sATIDJ//sATIDKf/sATIDK//sATIDLf/sATIDL//sATIDMf/sATIDM//sATQAD//XATQAEf/XATQAJP/sATQAgv/sATQAg//sATQAhP/sATQAhf/sATQAhv/sATQAh//sATQAwv/sATQAxP/sATQAxv/sATQBQ//sATQCCP/XATQCDP/XATQCWP/sATQDHf/sATQDH//sATQDIf/sATQDI//sATQDJf/sATQDJ//sATQDKf/sATQDK//sATQDLf/sATQDL//sATQDMf/sATQDM//sATYAD/+aATYAEf+aATYAIgApATYAJP+uATYAJv/sATYAKv/sATYAMv/sATYANP/sATYARP/XATYARv/XATYAR//XATYASP/XATYASv/sATYAUP/sATYAUf/sATYAUv/XATYAU//sATYAVP/XATYAVf/sATYAVv/sATYAWP/sATYAgv+uATYAg/+uATYAhP+uATYAhf+uATYAhv+uATYAh/+uATYAif/sATYAlP/sATYAlf/sATYAlv/sATYAl//sATYAmP/sATYAmv/sATYAov/XATYAo//XATYApP/XATYApf/XATYApv/XATYAp//XATYAqP/XATYAqf/XATYAqv/XATYAq//XATYArP/XATYArf/XATYAtP/XATYAtf/XATYAtv/XATYAt//XATYAuP/XATYAuv/XATYAu//sATYAvP/sATYAvf/sATYAvv/sATYAwv+uATYAw//XATYAxP+uATYAxf/XATYAxv+uATYAx//XATYAyP/sATYAyf/XATYAyv/sATYAy//XATYAzP/sATYAzf/XATYAzv/sATYAz//XATYA0f/XATYA0//XATYA1f/XATYA1//XATYA2f/XATYA2//XATYA3f/XATYA3v/sATYA3//sATYA4P/sATYA4f/sATYA4v/sATYA4//sATYA5P/sATYA5f/sATYA+v/sATYBBv/sATYBCP/sATYBDf/sATYBDv/sATYBD//XATYBEP/sATYBEf/XATYBEv/sATYBE//XATYBFP/sATYBFf/XATYBF//sATYBGf/sATYBHf/sATYBIf/sATYBK//sATYBLf/sATYBL//sATYBMf/sATYBM//sATYBNf/sATYBQ/+uATYBRP/XATYBRv/XATYBR//sATYBSP/XATYBSv/sATYCCP+aATYCDP+aATYCV//sATYCWP+uATYCWf/XATYCX//sATYCYP/XATYCYv/sATYDHf+uATYDHv/XATYDH/+uATYDIP/XATYDIf+uATYDIv/XATYDI/+uATYDJf+uATYDJv/XATYDJ/+uATYDKP/XATYDKf+uATYDKv/XATYDK/+uATYDLP/XATYDLf+uATYDLv/XATYDL/+uATYDMP/XATYDMf+uATYDMv/XATYDM/+uATYDNP/XATYDNv/XATYDOP/XATYDOv/XATYDPP/XATYDQP/XATYDQv/XATYDRP/XATYDSf/sATYDSv/XATYDS//sATYDTP/XATYDTf/sATYDTv/XATYDT//sATYDUf/sATYDUv/XATYDU//sATYDVP/XATYDVf/sATYDVv/XATYDV//sATYDWP/XATYDWf/sATYDWv/XATYDW//sATYDXP/XATYDXf/sATYDXv/XATYDX//sATYDYP/XATYDYv/sATYDZP/sATYDZv/sATYDaP/sATYDav/sATYDbP/sATYDbv/sATcABQBSATcACgBSATcAD/+uATcAEf+uATcAIgApATcCBwBSATcCCP+uATcCCwBSATcCDP+uATgAD/+FATgAEf+FATgAIgApATgAJP+FATgAJv/XATgAKv/XATgAMv/XATgANP/XATgARP+aATgARv+aATgAR/+aATgASP+aATgASv/XATgAUP/DATgAUf/DATgAUv+aATgAU//DATgAVP+aATgAVf/DATgAVv+uATgAWP/DATgAXf/XATgAgv+FATgAg/+FATgAhP+FATgAhf+FATgAhv+FATgAh/+FATgAif/XATgAlP/XATgAlf/XATgAlv/XATgAl//XATgAmP/XATgAmv/XATgAov+aATgAo/+aATgApP+aATgApf+aATgApv+aATgAp/+aATgAqP+aATgAqf+aATgAqv+aATgAq/+aATgArP+aATgArf+aATgAtP+aATgAtf+aATgAtv+aATgAt/+aATgAuP+aATgAuv+aATgAu//DATgAvP/DATgAvf/DATgAvv/DATgAwv+FATgAw/+aATgAxP+FATgAxf+aATgAxv+FATgAx/+aATgAyP/XATgAyf+aATgAyv/XATgAy/+aATgAzP/XATgAzf+aATgAzv/XATgAz/+aATgA0f+aATgA0/+aATgA1f+aATgA1/+aATgA2f+aATgA2/+aATgA3f+aATgA3v/XATgA3//XATgA4P/XATgA4f/XATgA4v/XATgA4//XATgA5P/XATgA5f/XATgA+v/DATgBBv/DATgBCP/DATgBDf/DATgBDv/XATgBD/+aATgBEP/XATgBEf+aATgBEv/XATgBE/+aATgBFP/XATgBFf+aATgBF//DATgBGf/DATgBHf+uATgBIf+uATgBK//DATgBLf/DATgBL//DATgBMf/DATgBM//DATgBNf/DATgBPP/XATgBPv/XATgBQP/XATgBQ/+FATgBRP+aATgBRv+aATgBR//XATgBSP+aATgBSv+uATgCCP+FATgCDP+FATgCV//DATgCWP+FATgCWf+aATgCX//XATgCYP+aATgCYv/DATgDHf+FATgDHv+aATgDH/+FATgDIP+aATgDIf+FATgDIv+aATgDI/+FATgDJf+FATgDJv+aATgDJ/+FATgDKP+aATgDKf+FATgDKv+aATgDK/+FATgDLP+aATgDLf+FATgDLv+aATgDL/+FATgDMP+aATgDMf+FATgDMv+aATgDM/+FATgDNP+aATgDNv+aATgDOP+aATgDOv+aATgDPP+aATgDQP+aATgDQv+aATgDRP+aATgDSf/XATgDSv+aATgDS//XATgDTP+aATgDTf/XATgDTv+aATgDT//XATgDUf/XATgDUv+aATgDU//XATgDVP+aATgDVf/XATgDVv+aATgDV//XATgDWP+aATgDWf/XATgDWv+aATgDW//XATgDXP+aATgDXf/XATgDXv+aATgDX//XATgDYP+aATgDYv/DATgDZP/DATgDZv/DATgDaP/DATgDav/DATgDbP/DATgDbv/DATkABQBSATkACgBSATkAD/+uATkAEf+uATkAIgApATkCBwBSATkCCP+uATkCCwBSATkCDP+uAToAD/+FAToAEf+FAToAIgApAToAJP+FAToAJv/XAToAKv/XAToAMv/XAToANP/XAToARP+aAToARv+aAToAR/+aAToASP+aAToASv/XAToAUP/DAToAUf/DAToAUv+aAToAU//DAToAVP+aAToAVf/DAToAVv+uAToAWP/DAToAXf/XAToAgv+FAToAg/+FAToAhP+FAToAhf+FAToAhv+FAToAh/+FAToAif/XAToAlP/XAToAlf/XAToAlv/XAToAl//XAToAmP/XAToAmv/XAToAov+aAToAo/+aAToApP+aAToApf+aAToApv+aAToAp/+aAToAqP+aAToAqf+aAToAqv+aAToAq/+aAToArP+aAToArf+aAToAtP+aAToAtf+aAToAtv+aAToAt/+aAToAuP+aAToAuv+aAToAu//DAToAvP/DAToAvf/DAToAvv/DAToAwv+FAToAw/+aAToAxP+FAToAxf+aAToAxv+FAToAx/+aAToAyP/XAToAyf+aAToAyv/XAToAy/+aAToAzP/XAToAzf+aAToAzv/XAToAz/+aAToA0f+aAToA0/+aAToA1f+aAToA1/+aAToA2f+aAToA2/+aAToA3f+aAToA3v/XAToA3//XAToA4P/XAToA4f/XAToA4v/XAToA4//XAToA5P/XAToA5f/XAToA+v/DAToBBv/DAToBCP/DAToBDf/DAToBDv/XAToBD/+aAToBEP/XAToBEf+aAToBEv/XAToBE/+aAToBFP/XAToBFf+aAToBF//DAToBGf/DAToBHf+uAToBIf+uAToBK//DAToBLf/DAToBL//DAToBMf/DAToBM//DAToBNf/DAToBPP/XAToBPv/XAToBQP/XAToBQ/+FAToBRP+aAToBRv+aAToBR//XAToBSP+aAToBSv+uAToCCP+FAToCDP+FAToCV//DAToCWP+FAToCWf+aAToCX//XAToCYP+aAToCYv/DAToDHf+FAToDHv+aAToDH/+FAToDIP+aAToDIf+FAToDIv+aAToDI/+FAToDJf+FAToDJv+aAToDJ/+FAToDKP+aAToDKf+FAToDKv+aAToDK/+FAToDLP+aAToDLf+FAToDLv+aAToDL/+FAToDMP+aAToDMf+FAToDMv+aAToDM/+FAToDNP+aAToDNv+aAToDOP+aAToDOv+aAToDPP+aAToDQP+aAToDQv+aAToDRP+aAToDSf/XAToDSv+aAToDS//XAToDTP+aAToDTf/XAToDTv+aAToDT//XAToDUf/XAToDUv+aAToDU//XAToDVP+aAToDVf/XAToDVv+aAToDV//XAToDWP+aAToDWf/XAToDWv+aAToDW//XAToDXP+aAToDXf/XAToDXv+aAToDX//XAToDYP+aAToDYv/DAToDZP/DAToDZv/DAToDaP/DAToDav/DAToDbP/DAToDbv/DATsAJv/sATsAKv/sATsAMv/sATsANP/sATsAif/sATsAlP/sATsAlf/sATsAlv/sATsAl//sATsAmP/sATsAmv/sATsAyP/sATsAyv/sATsAzP/sATsAzv/sATsA3v/sATsA4P/sATsA4v/sATsA5P/sATsBDv/sATsBEP/sATsBEv/sATsBFP/sATsBR//sATsCX//sATsDSf/sATsDS//sATsDTf/sATsDT//sATsDUf/sATsDU//sATsDVf/sATsDV//sATsDWf/sATsDW//sATsDXf/sATsDX//sAT0AJv/sAT0AKv/sAT0AMv/sAT0ANP/sAT0Aif/sAT0AlP/sAT0Alf/sAT0Alv/sAT0Al//sAT0AmP/sAT0Amv/sAT0AyP/sAT0Ayv/sAT0AzP/sAT0Azv/sAT0A3v/sAT0A4P/sAT0A4v/sAT0A5P/sAT0BDv/sAT0BEP/sAT0BEv/sAT0BFP/sAT0BR//sAT0CX//sAT0DSf/sAT0DS//sAT0DTf/sAT0DT//sAT0DUf/sAT0DU//sAT0DVf/sAT0DV//sAT0DWf/sAT0DW//sAT0DXf/sAT0DX//sAT8AJv/sAT8AKv/sAT8AMv/sAT8ANP/sAT8Aif/sAT8AlP/sAT8Alf/sAT8Alv/sAT8Al//sAT8AmP/sAT8Amv/sAT8AyP/sAT8Ayv/sAT8AzP/sAT8Azv/sAT8A3v/sAT8A4P/sAT8A4v/sAT8A5P/sAT8BDv/sAT8BEP/sAT8BEv/sAT8BFP/sAT8BR//sAT8CX//sAT8DSf/sAT8DS//sAT8DTf/sAT8DT//sAT8DUf/sAT8DU//sAT8DVf/sAT8DV//sAT8DWf/sAT8DW//sAT8DXf/sAT8DX//sAUMABf9xAUMACv9xAUMAJv/XAUMAKv/XAUMALQEKAUMAMv/XAUMANP/XAUMAN/9xAUMAOf+uAUMAOv+uAUMAPP+FAUMAif/XAUMAlP/XAUMAlf/XAUMAlv/XAUMAl//XAUMAmP/XAUMAmv/XAUMAn/+FAUMAyP/XAUMAyv/XAUMAzP/XAUMAzv/XAUMA3v/XAUMA4P/XAUMA4v/XAUMA5P/XAUMBDv/XAUMBEP/XAUMBEv/XAUMBFP/XAUMBJP9xAUMBJv9xAUMBNv+uAUMBOP+FAUMBOv+FAUMBR//XAUMB+v+uAUMB/P+uAUMB/v+uAUMCAP+FAUMCB/9xAUMCC/9xAUMCX//XAUMDSf/XAUMDS//XAUMDTf/XAUMDT//XAUMDUf/XAUMDU//XAUMDVf/XAUMDV//XAUMDWf/XAUMDW//XAUMDXf/XAUMDX//XAUMDb/+FAUMDcf+FAUMDc/+FAUMDj/9xAUQABf/sAUQACv/sAUQCB//sAUQCC//sAUUALQB7AUcAD/+uAUcAEf+uAUcAJP/XAUcAN//DAUcAOf/sAUcAOv/sAUcAO//XAUcAPP/sAUcAPf/sAUcAgv/XAUcAg//XAUcAhP/XAUcAhf/XAUcAhv/XAUcAh//XAUcAn//sAUcAwv/XAUcAxP/XAUcAxv/XAUcBJP/DAUcBJv/DAUcBNv/sAUcBOP/sAUcBOv/sAUcBO//sAUcBPf/sAUcBP//sAUcBQ//XAUcBoP/sAUcB+v/sAUcB/P/sAUcB/v/sAUcCAP/sAUcCCP+uAUcCDP+uAUcCWP/XAUcDHf/XAUcDH//XAUcDIf/XAUcDI//XAUcDJf/XAUcDJ//XAUcDKf/XAUcDK//XAUcDLf/XAUcDL//XAUcDMf/XAUcDM//XAUcDb//sAUcDcf/sAUcDc//sAUcDj//DAVYABf9xAVYACv9xAVYBZv/XAVYBbf/XAVYBcf9xAVYBcv+FAVYBc//XAVYBdf+uAVYBeP+FAVYCB/9xAVYCC/9xAVYCVP+FAVsAD/+uAVsAEf+uAVsBVv/XAVsBX//XAVsBYv/XAVsBZP/sAVsBaf/XAVsBcP/sAVsBcf/DAVsBcv/sAVsBdP/XAVsBdf/sAVsBeP/sAVsBiP/sAVsCCP+uAVsCDP+uAVsCVP/sAVwAD/+FAVwAEf+FAVwBVv+FAVwBX/+FAVwBYv+FAVwBZv/XAVwBaf+FAVwBbf/XAVwBc//DAVwBdv/sAVwBef+aAVwBev+uAVwBe//DAVwBfP/DAVwBff/DAVwBfv+aAVwBgf/DAVwBgv+uAVwBhP/DAVwBhv/DAVwBh//DAVwBif/DAVwBjP+aAVwBjv+aAVwBj/+aAVwBkP+aAVwBkv/DAVwBk/+aAVwBlf/DAVwBlv/DAVwBmP/DAVwBmf+aAVwBmv/DAVwBm//DAVwCCP+FAVwCDP+FAVwCIf/sAV0Bcf/XAV0Bcv/sAV0BeP/sAV0CVP/sAV4ABf/XAV4ACv/XAV4CB//XAV4CC//XAV8ABf9xAV8ACv9xAV8BZv/XAV8Bbf/XAV8Bcf9xAV8Bcv+FAV8Bc//XAV8Bdf+uAV8BeP+FAV8CB/9xAV8CC/9xAV8CVP+FAWAAD/+uAWAAEf+uAWABVv/XAWABX//XAWABYv/XAWABaf/XAWABdP/XAWACCP+uAWACDP+uAWEAD/+FAWEAEP+uAWEAEf+FAWEBVv9cAWEBX/9cAWEBYv9cAWEBZv/DAWEBaf9cAWEBbf/DAWEBc/+aAWEBdv/DAWEBef9xAWEBev+aAWEBe/+aAWEBfP+uAWEBff+aAWEBfv9xAWEBgP/XAWEBgf/DAWEBgv+aAWEBhP+aAWEBhv+uAWEBh/+aAWEBif+aAWEBiv/XAWEBjP9xAWEBjv+aAWEBj/9xAWEBkP9xAWEBkv+aAWEBk/9xAWEBlP/XAWEBlf+aAWEBlv+aAWEBmP+aAWEBmf9xAWEBmv+aAWEBm/+aAWECAv+uAWECA/+uAWECBP+uAWECCP+FAWECDP+FAWECIf/DAWECU//XAWIABf9xAWIACv9xAWIBZv/XAWIBbf/XAWIBcf9xAWIBcv+FAWIBc//XAWIBdf+uAWIBeP+FAWICB/9xAWICC/9xAWICVP+FAWQBZv/sAWQBbf/sAWQBc//DAWYAD/+uAWYAEf+uAWYBVv/XAWYBX//XAWYBYv/XAWYBZP/sAWYBaf/XAWYBcP/sAWYBcf/DAWYBcv/sAWYBdP/XAWYBdf/sAWYBeP/sAWYBiP/sAWYCCP+uAWYCDP+uAWYCVP/sAWgBZv/XAWgBbf/XAWgBc//DAWgBjf/sAWgBkf/sAWkABf9xAWkACv9xAWkBZv/XAWkBbf/XAWkBcf9xAWkBcv+FAWkBc//XAWkBdf+uAWkBeP+FAWkCB/9xAWkCC/9xAWkCVP+FAW0AD/+uAW0AEf+uAW0BVv/XAW0BX//XAW0BYv/XAW0BZP/sAW0Baf/XAW0BcP/sAW0Bcf/DAW0Bcv/sAW0BdP/XAW0Bdf/sAW0BeP/sAW0BiP/sAW0CCP+uAW0CDP+uAW0CVP/sAW8AD/72AW8AEf72AW8BVv+aAW8BX/+aAW8BYv+aAW8BZP/sAW8Baf+aAW8BdP/XAW8BiP/XAW8CCP72AW8CDP72AXEAD/+FAXEAEP+uAXEAEf+FAXEBVv9cAXEBX/9cAXEBYv9cAXEBZv/DAXEBaf9cAXEBbf/DAXEBc/+aAXEBdv/DAXEBef9xAXEBev+aAXEBe/+aAXEBfP+uAXEBff+aAXEBfv9xAXEBgP/XAXEBgf/DAXEBgv+aAXEBhP+aAXEBhv+uAXEBh/+aAXEBif+aAXEBiv/XAXEBjP9xAXEBjv+aAXEBj/9xAXEBkP9xAXEBkv+aAXEBk/9xAXEBlP/XAXEBlf+aAXEBlv+aAXEBmP+aAXEBmf9xAXEBmv+aAXEBm/+aAXECAv+uAXECA/+uAXECBP+uAXECCP+FAXECDP+FAXECIf/DAXECU//XAXIAD/+FAXIAEf+FAXIBVv+FAXIBX/+FAXIBYv+FAXIBZv/XAXIBaf+FAXIBbf/XAXIBc//DAXIBdv/sAXIBef+aAXIBev+uAXIBe//DAXIBfP/DAXIBff/DAXIBfv+aAXIBgf/DAXIBgv+uAXIBhP/DAXIBhv/DAXIBh//DAXIBif/DAXIBjP+aAXIBjv+aAXIBj/+aAXIBkP+aAXIBkv/DAXIBk/+aAXIBlf/DAXIBlv/DAXIBmP/DAXIBmf+aAXIBmv/DAXIBm//DAXICCP+FAXICDP+FAXICIf/sAXMAD/+aAXMAEf+aAXMBVv/XAXMBX//XAXMBYv/XAXMBZP/DAXMBaf/XAXMBcP/sAXMBcf+uAXMBcv/DAXMBdP/sAXMBeP/DAXMBiP/sAXMCCP+aAXMCDP+aAXMCVP/DAXQBZv/XAXQBbf/XAXQBc//DAXQBjf/sAXQBkf/sAXUAD/+FAXUAEf+FAXUBVv+uAXUBX/+uAXUBYv+uAXUBZv/sAXUBaf+uAXUBbf/sAXUCCP+FAXUCDP+FAXYBcf/XAXYBcv/sAXYBeP/sAXYCVP/sAXgAD/+FAXgAEf+FAXgBVv+FAXgBX/+FAXgBYv+FAXgBZv/XAXgBaf+FAXgBbf/XAXgBc//DAXgBdv/sAXgBef+aAXgBev+uAXgBe//DAXgBfP/DAXgBff/DAXgBfv+aAXgBgf/DAXgBgv+uAXgBhP/DAXgBhv/DAXgBh//DAXgBif/DAXgBjP+aAXgBjv+aAXgBj/+aAXgBkP+aAXgBkv/DAXgBk/+aAXgBlf/DAXgBlv/DAXgBmP/DAXgBmf+aAXgBmv/DAXgBm//DAXgCCP+FAXgCDP+FAXgCIf/sAXkBiAApAXsABf/sAXsACv/sAXsCB//sAXsCC//sAXwABf+uAXwACv+uAXwBjf/sAXwBkf/sAXwCB/+uAXwCC/+uAX4BiAApAYAAD/+uAYAAEf+uAYABiP/sAYACCP+uAYACDP+uAYMAEP+aAYMBef/XAYMBfv/XAYMBgf/XAYMBjP/XAYMBjf/XAYMBj//XAYMBkP/XAYMBkf/XAYMBk//XAYMBmf/XAYMCAv+aAYMCA/+aAYMCBP+aAYQABf/sAYQACv/sAYQCB//sAYQCC//sAYUAD//XAYUAEf/XAYUCCP/XAYUCDP/XAYYABf+uAYYACv+uAYYBjf/sAYYBkf/sAYYCB/+uAYYCC/+uAYcBef/XAYcBfv/XAYcBjP/XAYcBj//XAYcBkP/XAYcBk//XAYcBmf/XAYgABf+FAYgACv+FAYgBef/sAYgBfv/sAYgBgP/XAYgBiv/XAYgBjP/sAYgBjf/XAYgBj//sAYgBkP/sAYgBkf/XAYgBk//sAYgBmf/sAYgCB/+FAYgCC/+FAYoAD/+uAYoAEf+uAYoBiP/sAYoCCP+uAYoCDP+uAYwABf/sAYwACv/sAYwBgP/XAYwBiv/XAYwCB//sAYwCC//sAY4ABf/sAY4ACv/sAY4BgP/XAY4Biv/XAY4CB//sAY4CC//sAZAAD//sAZAAEf/sAZACCP/sAZACDP/sAZMABf/sAZMACv/sAZMBgP/XAZMBiv/XAZMCB//sAZMCC//sAZQAD//DAZQAEP/XAZQAEf/DAZQBef/XAZQBfv/XAZQBgf/XAZQBjP/XAZQBj//XAZQBkP/XAZQBk//XAZQBmf/XAZQCAv/XAZQCA//XAZQCBP/XAZQCCP/DAZQCDP/DAZcABf/XAZcACv/XAZcCB//XAZcCC//XAZkABf/sAZkACv/sAZkBgP/XAZkBiv/XAZkCB//sAZkCC//sAZ0ABf+uAZ0ACv+uAZ0Bnf+FAZ0Bpv+FAZ0BqP/XAZ0BvP+aAZ0Bvf/XAZ0Bwf+aAZ0BxP+FAZ0B3P/XAZ0B3f/XAZ0B4f/XAZ0B5P/XAZ0B9v/XAZ0CB/+uAZ0CC/+uAZ0Cbv+uAZ0CfP+aAZ0CgP+uAZ0Cgv+uAZ0Cl/+uAZ0Cm/+uAZ0Cp/+uAZ0Cqf+FAZ0Cqv/XAZ0Ctf+aAZ0Ctv/XAZ0Ct/+aAZ0CuP/XAZ0Cuf+aAZ0Cuv/XAZ0Cvf+FAZ0Cvv/XAZ0Cv/+aAZ0CwP/XAZ0Cwf+aAZ0Cwv/XAZ0C1P+aAZ0C1f/XAZ0C9//XAZ0C+P/XAZ0C+f/XAZ0C+v/XAZ0C+//XAZ0C/P/XAZ0C/f+aAZ0C/v/XAZ0DA/+uAZ0DDf+aAZ0DDv/DAZ0DD/+aAZ0DEP/DAZ0DF/+FAZ0DGP/XAZ4AD/+FAZ4AEP+uAZ4AEf+FAZ4Bn//XAZ4BpP+aAZ4Bqv9xAZ4Brv+aAZ4Btf+aAZ4BuP/XAZ4Bu//XAZ4BvAApAZ4Bvv+uAZ4BzP+aAZ4Bzf+aAZ4Bzv+FAZ4Bz/9xAZ4B0P/XAZ4B0f/XAZ4B0v+aAZ4B0/+aAZ4B1P+aAZ4B1f+FAZ4B1v+aAZ4B1/+aAZ4B2P9xAZ4B2f+aAZ4B2v+aAZ4B2/9xAZ4B3P+uAZ4B3f+uAZ4B3v9xAZ4B3//XAZ4B4P+aAZ4B4f+aAZ4B4v+aAZ4B4/+aAZ4B5P+uAZ4B5f+aAZ4B5v+aAZ4B5//XAZ4B6P+aAZ4B6f/DAZ4B6v9xAZ4B7P+aAZ4B7f9xAZ4B7v+FAZ4B8v+FAZ4B8/+aAZ4B9f+aAZ4B9v+uAZ4B9/+aAZ4B+f+aAZ4CAv+uAZ4CA/+uAZ4CBP+uAZ4CCP+FAZ4CDP+FAZ4Cav9xAZ4Ca/+aAZ4CbP/XAZ4Cbf/XAZ4Ccf+aAZ4Ccv9xAZ4Cc/+FAZ4Cdf+aAZ4Cd/+aAZ4Cef+aAZ4Cff+aAZ4Cfv/XAZ4Cf/9xAZ4Cgf/XAZ4Cg//XAZ4ChP/XAZ4Chf9xAZ4Chv/XAZ4Ch/9xAZ4CiP/XAZ4Cif9xAZ4Civ/XAZ4Ci//XAZ4CjP/XAZ4Cjf9xAZ4Clv+aAZ4Cmv+aAZ4Cnv+aAZ4CoP/XAZ4Cov/XAZ4CpP+aAZ4Cpv+aAZ4Cqv+uAZ4CrP+aAZ4Crv+aAZ4CsP+aAZ4Csf/XAZ4Csv9xAZ4Cs//XAZ4CtP9xAZ4CtQApAZ4Ctv+uAZ4CuP+uAZ4Cuv+uAZ4CvP/XAZ4Cvv+uAZ4CwP+aAZ4Cwv+aAZ4CxP+aAZ4Cxf+aAZ4Cxv9xAZ4Cx/+aAZ4CyP9xAZ4Cy//XAZ4Czf+aAZ4Czv+aAZ4Cz/+FAZ4C0f+aAZ4C0/+aAZ4C1f+aAZ4C1/+aAZ4C2f9xAZ4C2/9xAZ4C3f9xAZ4C4P9xAZ4C5v/XAZ4C6P/XAZ4C6v/DAZ4C7P+aAZ4C7v+aAZ4C7//XAZ4C8P9xAZ4C8f/XAZ4C8v9xAZ4C8//XAZ4C9P9xAZ4C9v/XAZ4C+P+uAZ4C+v+uAZ4C/P+uAZ4C/v+aAZ4DAP+aAZ4DAv+aAZ4DBv/XAZ4DCP/XAZ4DCf9xAZ4DCv9xAZ4DC/9xAZ4DDP9xAZ4DDv+aAZ4DEP+aAZ4DEf+aAZ4DEv+FAZ4DFP+aAZ4DFf/XAZ4DFv9xAZ4DGP+uAZ4DGv9xAZ4DG/+aAZ4DHP+FAZ8Bn//XAZ8BuP/XAZ8Bu//XAZ8Bvv/XAZ8B4f/XAZ8CbP/XAZ8Cfv/XAZ8ChP/XAZ8Chv/XAZ8CiP/XAZ8Civ/XAZ8CjP/XAZ8Csf/XAZ8Cs//XAZ8CwP/XAZ8Cwv/XAZ8Cxf/XAZ8Cx//XAZ8C1f/XAZ8C7//XAZ8C8f/XAZ8C8//XAZ8C/v/XAZ8DCf/XAZ8DC//XAZ8DDv/XAZ8DEP/XAZ8DFf/XAaADDv/XAaADEP/XAaQABf+uAaQACv+uAaQBnf+FAaQBpv+FAaQBqP/XAaQBvP+aAaQBvf/XAaQBwf+aAaQBxP+FAaQB3P/XAaQB3f/XAaQB4f/XAaQB5P/XAaQB9v/XAaQCB/+uAaQCC/+uAaQCbv+uAaQCfP+aAaQCgP+uAaQCgv+uAaQCl/+uAaQCm/+uAaQCp/+uAaQCqf+FAaQCqv/XAaQCtf+aAaQCtv/XAaQCt/+aAaQCuP/XAaQCuf+aAaQCuv/XAaQCvf+FAaQCvv/XAaQCv/+aAaQCwP/XAaQCwf+aAaQCwv/XAaQC1P+aAaQC1f/XAaQC9//XAaQC+P/XAaQC+f/XAaQC+v/XAaQC+//XAaQC/P/XAaQC/f+aAaQC/v/XAaQDA/+uAaQDDf+aAaQDDv/DAaQDD/+aAaQDEP/DAaQDF/+FAaQDGP/XAaUABf+uAaUACv+uAaUBnf+FAaUBpv+FAaUBqP/XAaUBvP+aAaUBvf/XAaUBwf+aAaUBxP+FAaUB3P/XAaUB3f/XAaUB4f/XAaUB5P/XAaUB9v/XAaUCB/+uAaUCC/+uAaUCbv+uAaUCfP+aAaUCgP+uAaUCgv+uAaUCl/+uAaUCm/+uAaUCp/+uAaUCqf+FAaUCqv/XAaUCtf+aAaUCtv/XAaUCt/+aAaUCuP/XAaUCuf+aAaUCuv/XAaUCvf+FAaUCvv/XAaUCv/+aAaUCwP/XAaUCwf+aAaUCwv/XAaUC1P+aAaUC1f/XAaUC9//XAaUC+P/XAaUC+f/XAaUC+v/XAaUC+//XAaUC/P/XAaUC/f+aAaUC/v/XAaUDA/+uAaUDDf+aAaUDDv/DAaUDD/+aAaUDEP/DAaUDF/+FAaUDGP/XAaYABf+uAaYACv+uAaYBnf+FAaYBpv+FAaYBqP/XAaYBvP+aAaYBvf/XAaYBwf+aAaYBxP+FAaYB3P/XAaYB3f/XAaYB4f/XAaYB5P/XAaYB9v/XAaYCB/+uAaYCC/+uAaYCbv+uAaYCfP+aAaYCgP+uAaYCgv+uAaYCl/+uAaYCm/+uAaYCp/+uAaYCqf+FAaYCqv/XAaYCtf+aAaYCtv/XAaYCt/+aAaYCuP/XAaYCuf+aAaYCuv/XAaYCvf+FAaYCvv/XAaYCv/+aAaYCwP/XAaYCwf+aAaYCwv/XAaYC1P+aAaYC1f/XAaYC9//XAaYC+P/XAaYC+f/XAaYC+v/XAaYC+//XAaYC/P/XAaYC/f+aAaYC/v/XAaYDA/+uAaYDDf+aAaYDDv/DAaYDD/+aAaYDEP/DAaYDF/+FAaYDGP/XAacBn//XAacBuP/XAacBu//XAacBvv/XAacBwf/XAacB4f/XAacCbP/XAacCfP/XAacCfv/XAacChP/XAacChv/XAacCiP/XAacCiv/XAacCjP/XAacCsf/XAacCs//XAacCv//XAacCwP/XAacCwf/XAacCwv/XAacCxf+aAacCx/+aAacC1P/XAacC1f/XAacC7//XAacC8f/XAacC8//XAacC/f/XAacC/v/XAacDCf/XAacDC//XAacDDv/XAacDEP/XAacDFf/XAacDGf/sAagAD/+FAagAEf+FAagBn//sAagBpP+aAagBqv9xAagBrv+aAagBtf+aAagBuP/sAagBu//sAagBvv/DAagByf/sAagBzv+uAagBz//XAagB1f+uAagB2P/XAagB2//XAagB3v/XAagB4f/XAagB6v/XAagB6wBmAagB7f/XAagB7v/sAagB8v+uAagB9ABmAagCCP+FAagCDP+FAagCav/XAagCbP/sAagCcv9xAagCc/+uAagCfv/sAagCf//XAagChP/sAagChf/XAagChv/sAagCh//XAagCiP/sAagCif/XAagCiv/sAagCjP/sAagCjf/XAagCmABmAagCqABmAagCsf/sAagCsv/XAagCs//sAagCtP/XAagCwP/XAagCwv/XAagCxf/XAagCxv/DAagCx//XAagCyP/DAagCzv+aAagCz/+uAagC1f/XAagC2f9xAagC2/9xAagC3f9xAagC4P/XAagC7//sAagC8P/XAagC8f/sAagC8v/XAagC8//sAagC9P/XAagC/v/XAagDCf9xAagDCv/XAagDC/9xAagDDP/XAagDEf+aAagDEv+uAagDFf/sAagDFv/XAagDGv/XAagDG/+aAagDHP+uAaoABf9xAaoACv9xAaoBnf+aAaoBpv+aAaoBvP9xAaoBvv/XAaoBwf+aAaoBxP+aAaoB3P/XAaoB4f/XAaoB5P/XAaoCB/9xAaoCC/9xAaoCbv/XAaoCfP+aAaoCgP+uAaoCgv+uAaoCl//XAaoCm//XAaoCp//XAaoCqf+aAaoCqv/XAaoCtf9xAaoCtv/XAaoCt/+FAaoCuf+FAaoCvf+aAaoCvv/XAaoCv/+aAaoCwP/XAaoCwf+aAaoCwv/XAaoCxf+aAaoCx/+aAaoC1P+aAaoC1f/XAaoC4f/XAaoC4//XAaoC/f+aAaoC/v/XAaoDA//XAaoDDf9xAaoDDv/XAaoDD/9xAaoDEP/XAaoDF/+aAaoDGP/XAasABf/XAasACv/XAasBqv/sAasBwf/XAasCB//XAasCC//XAasCcv/sAasCfP/XAasCv//XAasCwf/XAasCxf/XAasCx//XAasC1P/XAasC2f/sAasC2//sAasC3f/sAasC/f/XAawAD/+uAawAEf+uAawCCP+uAawCDP+uAawCgP/sAawCgv/sAawCt//sAawCuf/sAawDDf/XAawDD//XAa0AD/+FAa0AEP+uAa0AEf+FAa0Bn//XAa0BpP+aAa0Bqv9xAa0Brv+aAa0Btf+aAa0BuP/XAa0Bu//XAa0BvAApAa0Bvv+uAa0BzP+aAa0Bzf+aAa0Bzv+FAa0Bz/9xAa0B0P/XAa0B0f/XAa0B0v+aAa0B0/+aAa0B1P+aAa0B1f+FAa0B1v+aAa0B1/+aAa0B2P9xAa0B2f+aAa0B2v+aAa0B2/9xAa0B3P+uAa0B3f+uAa0B3v9xAa0B3//XAa0B4P+aAa0B4f+aAa0B4v+aAa0B4/+aAa0B5P+uAa0B5f+aAa0B5v+aAa0B5//XAa0B6P+aAa0B6f/DAa0B6v9xAa0B7P+aAa0B7f9xAa0B7v+FAa0B8v+FAa0B8/+aAa0B9f+aAa0B9v+uAa0B9/+aAa0B+f+aAa0CAv+uAa0CA/+uAa0CBP+uAa0CCP+FAa0CDP+FAa0Cav9xAa0Ca/+aAa0CbP/XAa0Cbf/XAa0Ccf+aAa0Ccv9xAa0Cc/+FAa0Cdf+aAa0Cd/+aAa0Cef+aAa0Cff+aAa0Cfv/XAa0Cf/9xAa0Cgf/XAa0Cg//XAa0ChP/XAa0Chf9xAa0Chv/XAa0Ch/9xAa0CiP/XAa0Cif9xAa0Civ/XAa0Ci//XAa0CjP/XAa0Cjf9xAa0Clv+aAa0Cmv+aAa0Cnv+aAa0CoP/XAa0Cov/XAa0CpP+aAa0Cpv+aAa0Cqv+uAa0CrP+aAa0Crv+aAa0CsP+aAa0Csf/XAa0Csv9xAa0Cs//XAa0CtP9xAa0CtQApAa0Ctv+uAa0CuP+uAa0Cuv+uAa0CvP/XAa0Cvv+uAa0CwP+aAa0Cwv+aAa0CxP+aAa0Cxf+aAa0Cxv9xAa0Cx/+aAa0CyP9xAa0Cy//XAa0Czf+aAa0Czv+aAa0Cz/+FAa0C0f+aAa0C0/+aAa0C1f+aAa0C1/+aAa0C2f9xAa0C2/9xAa0C3f9xAa0C4P9xAa0C5v/XAa0C6P/XAa0C6v/DAa0C7P+aAa0C7v+aAa0C7//XAa0C8P9xAa0C8f/XAa0C8v9xAa0C8//XAa0C9P9xAa0C9v/XAa0C+P+uAa0C+v+uAa0C/P+uAa0C/v+aAa0DAP+aAa0DAv+aAa0DBv/XAa0DCP/XAa0DCf9xAa0DCv9xAa0DC/9xAa0DDP9xAa0DDv+aAa0DEP+aAa0DEf+aAa0DEv+FAa0DFP+aAa0DFf/XAa0DFv9xAa0DGP+uAa0DGv9xAa0DG/+aAa0DHP+FAa4BowDhAa4C6gApAa4DDv/XAa4DEP/XAbABn//XAbABuP/XAbABu//XAbABvv/XAbABwf/XAbAB4f/XAbACbP/XAbACfP/XAbACfv/XAbAChP/XAbAChv/XAbACiP/XAbACiv/XAbACjP/XAbACsf/XAbACs//XAbACv//XAbACwP/XAbACwf/XAbACwv/XAbACxf+aAbACx/+aAbAC1P/XAbAC1f/XAbAC7//XAbAC8f/XAbAC8//XAbAC/f/XAbAC/v/XAbADCf/XAbADC//XAbADDv/XAbADEP/XAbADFf/XAbADGf/sAbEAD/+uAbEAEf+uAbECCP+uAbECDP+uAbECgP/sAbECgv/sAbECt//sAbECuf/sAbEDDf/XAbEDD//XAbQBn//XAbQBuP/XAbQBu//XAbQBvv/XAbQBwf/XAbQB4f/XAbQCbP/XAbQCfP/XAbQCfv/XAbQChP/XAbQChv/XAbQCiP/XAbQCiv/XAbQCjP/XAbQCsf/XAbQCs//XAbQCv//XAbQCwP/XAbQCwf/XAbQCwv/XAbQCxf+aAbQCx/+aAbQC1P/XAbQC1f/XAbQC7//XAbQC8f/XAbQC8//XAbQC/f/XAbQC/v/XAbQDCf/XAbQDC//XAbQDDv/XAbQDEP/XAbQDFf/XAbQDGf/sAbgAD/+uAbgAEf+uAbgBnf/sAbgBpP/XAbgBpv/sAbgBqP/XAbgBqv/XAbgBrv/XAbgBsP/XAbgBsf/sAbgBtf/XAbgBvP/DAbgBvf/XAbgBv//XAbgBwf/XAbgBxP/sAbgBx//sAbgBzv/sAbgB1f/sAbgB8v/sAbgCCP+uAbgCDP+uAbgCcv/XAbgCc//sAbgCev/sAbgCfP/XAbgCgP/sAbgCgv/sAbgCn//XAbgCof/sAbgCqf/sAbgCtf/DAbgCt//sAbgCuf/sAbgCu//XAbgCvf/sAbgCv//XAbgCwf/XAbgCyv/XAbgCzv/XAbgCz//sAbgC1P/XAbgC2f/XAbgC2//XAbgC3f/XAbgC5f/XAbgC5//sAbgC9f/sAbgC9//XAbgC+f/XAbgC+//XAbgC/f/XAbgDBf/XAbgDB//XAbgDDf/XAbgDD//XAbgDEf/XAbgDEv/sAbgDF//sAbgDG//XAbgDHP/sAboAD/72AboAEf72AboBpP+FAboBqv+aAboBrv+FAboBsP/XAboBtf+FAboBv//XAboBzv+aAboB1f+aAboB8v+aAboCCP72AboCDP72AboCcv+aAboCc/+aAboCdv/sAboCn//XAboCu//XAboCyv/XAboCzv+FAboCz/+aAboC2f+aAboC2/+aAboC3f+aAboC5f/XAboDBf/XAboDB//XAboDCf+uAboDC/+uAboDEf+FAboDEv+aAboDG/+FAboDHP+aAbsBn//XAbsBuP/XAbsBu//XAbsBvv/XAbsB4f/XAbsCbP/XAbsCfv/XAbsChP/XAbsChv/XAbsCiP/XAbsCiv/XAbsCjP/XAbsCsf/XAbsCs//XAbsCwP/XAbsCwv/XAbsCxf/XAbsCx//XAbsC1f/XAbsC7//XAbsC8f/XAbsC8//XAbsC/v/XAbsDCf/XAbsDC//XAbsDDv/XAbsDEP/XAbsDFf/XAbwAD/+FAbwAEP+uAbwAEf+FAbwBn//XAbwBpP+aAbwBqv9xAbwBrv+aAbwBtf+aAbwBuP/XAbwBu//XAbwBvAApAbwBvv+uAbwBzP+aAbwBzf+aAbwBzv+FAbwBz/9xAbwB0P/XAbwB0f/XAbwB0v+aAbwB0/+aAbwB1P+aAbwB1f+FAbwB1v+aAbwB1/+aAbwB2P9xAbwB2f+aAbwB2v+aAbwB2/9xAbwB3P+uAbwB3f+uAbwB3v9xAbwB3//XAbwB4P+aAbwB4f+aAbwB4v+aAbwB4/+aAbwB5P+uAbwB5f+aAbwB5v+aAbwB5//XAbwB6P+aAbwB6f/DAbwB6v9xAbwB7P+aAbwB7f9xAbwB7v+FAbwB8v+FAbwB8/+aAbwB9f+aAbwB9v+uAbwB9/+aAbwB+f+aAbwCAv+uAbwCA/+uAbwCBP+uAbwCCP+FAbwCDP+FAbwCav9xAbwCa/+aAbwCbP/XAbwCbf/XAbwCcf+aAbwCcv9xAbwCc/+FAbwCdf+aAbwCd/+aAbwCef+aAbwCff+aAbwCfv/XAbwCf/9xAbwCgf/XAbwCg//XAbwChP/XAbwChf9xAbwChv/XAbwCh/9xAbwCiP/XAbwCif9xAbwCiv/XAbwCi//XAbwCjP/XAbwCjf9xAbwClv+aAbwCmv+aAbwCnv+aAbwCoP/XAbwCov/XAbwCpP+aAbwCpv+aAbwCqv+uAbwCrP+aAbwCrv+aAbwCsP+aAbwCsf/XAbwCsv9xAbwCs//XAbwCtP9xAbwCtQApAbwCtv+uAbwCuP+uAbwCuv+uAbwCvP/XAbwCvv+uAbwCwP+aAbwCwv+aAbwCxP+aAbwCxf+aAbwCxv9xAbwCx/+aAbwCyP9xAbwCy//XAbwCzf+aAbwCzv+aAbwCz/+FAbwC0f+aAbwC0/+aAbwC1f+aAbwC1/+aAbwC2f9xAbwC2/9xAbwC3f9xAbwC4P9xAbwC5v/XAbwC6P/XAbwC6v/DAbwC7P+aAbwC7v+aAbwC7//XAbwC8P9xAbwC8f/XAbwC8v9xAbwC8//XAbwC9P9xAbwC9v/XAbwC+P+uAbwC+v+uAbwC/P+uAbwC/v+aAbwDAP+aAbwDAv+aAbwDBv/XAbwDCP/XAbwDCf9xAbwDCv9xAbwDC/9xAbwDDP9xAbwDDv+aAbwDEP+aAbwDEf+aAbwDEv+FAbwDFP+aAbwDFf/XAbwDFv9xAbwDGP+uAbwDGv9xAbwDG/+aAbwDHP+FAb0AD/+FAb0AEf+FAb0Bn//sAb0BpP+aAb0Bqv9xAb0Brv+aAb0Btf+aAb0BuP/sAb0Bu//sAb0Bvv/DAb0Byf/sAb0Bzv+uAb0Bz//XAb0B1f+uAb0B2P/XAb0B2//XAb0B3v/XAb0B4f/XAb0B6v/XAb0B6wBmAb0B7f/XAb0B7v/sAb0B8v+uAb0B9ABmAb0CCP+FAb0CDP+FAb0Cav/XAb0CbP/sAb0Ccv9xAb0Cc/+uAb0Cfv/sAb0Cf//XAb0ChP/sAb0Chf/XAb0Chv/sAb0Ch//XAb0CiP/sAb0Cif/XAb0Civ/sAb0CjP/sAb0Cjf/XAb0CmABmAb0CqABmAb0Csf/sAb0Csv/XAb0Cs//sAb0CtP/XAb0CwP/XAb0Cwv/XAb0Cxf/XAb0Cxv/DAb0Cx//XAb0CyP/DAb0Czv+aAb0Cz/+uAb0C1f/XAb0C2f9xAb0C2/9xAb0C3f9xAb0C4P/XAb0C7//sAb0C8P/XAb0C8f/sAb0C8v/XAb0C8//sAb0C9P/XAb0C/v/XAb0DCf9xAb0DCv/XAb0DC/9xAb0DDP/XAb0DEf+aAb0DEv+uAb0DFf/sAb0DFv/XAb0DGv/XAb0DG/+aAb0DHP+uAb4AD/+uAb4AEf+uAb4Bnf/XAb4BpP/XAb4Bpv/XAb4BqP/DAb4Bqv/XAb4Brv/XAb4BsP/XAb4Bsf/XAb4Btf/XAb4BvP/DAb4Bvf/DAb4Bv//XAb4BxP/XAb4Bx//XAb4Bzv/sAb4B1f/sAb4B8v/sAb4CCP+uAb4CDP+uAb4Ccv/XAb4Cc//sAb4Cev/XAb4CgP/sAb4Cgv/sAb4Cn//XAb4Cof/XAb4Cqf/XAb4Ctf/DAb4Ct//DAb4Cuf/DAb4Cu//XAb4Cvf/XAb4Cyv/XAb4Czv/XAb4Cz//sAb4C2f/XAb4C2//XAb4C3f/XAb4C5f/XAb4C5//XAb4C9f/XAb4C9//DAb4C+f/DAb4C+//DAb4DBf/XAb4DB//XAb4DDf/XAb4DD//XAb4DEf/XAb4DEv/sAb4DF//XAb4DG//XAb4DHP/sAb8Bn//XAb8BuP/XAb8Bu//XAb8Bvv/XAb8Bwf/XAb8B4f/XAb8CbP/XAb8CfP/XAb8Cfv/XAb8ChP/XAb8Chv/XAb8CiP/XAb8Civ/XAb8CjP/XAb8Csf/XAb8Cs//XAb8Cv//XAb8CwP/XAb8Cwf/XAb8Cwv/XAb8Cxf+aAb8Cx/+aAb8C1P/XAb8C1f/XAb8C7//XAb8C8f/XAb8C8//XAb8C/f/XAb8C/v/XAb8DCf/XAb8DC//XAb8DDv/XAb8DEP/XAb8DFf/XAb8DGf/sAcABowDhAcAC6gApAcADDv/XAcADEP/XAcMBowDhAcMC6gApAcMDDv/XAcMDEP/XAcQABf+uAcQACv+uAcQBnf+FAcQBpv+FAcQBqP/XAcQBvP+aAcQBvf/XAcQBwf+aAcQBxP+FAcQB3P/XAcQB3f/XAcQB4f/XAcQB5P/XAcQB9v/XAcQCB/+uAcQCC/+uAcQCbv+uAcQCfP+aAcQCgP+uAcQCgv+uAcQCl/+uAcQCm/+uAcQCp/+uAcQCqf+FAcQCqv/XAcQCtf+aAcQCtv/XAcQCt/+aAcQCuP/XAcQCuf+aAcQCuv/XAcQCvf+FAcQCvv/XAcQCv/+aAcQCwP/XAcQCwf+aAcQCwv/XAcQC1P+aAcQC1f/XAcQC9//XAcQC+P/XAcQC+f/XAcQC+v/XAcQC+//XAcQC/P/XAcQC/f+aAcQC/v/XAcQDA/+uAcQDDf+aAcQDDv/DAcQDD/+aAcQDEP/DAcQDF/+FAcQDGP/XAcYABf+uAcYACv+uAcYBnf+FAcYBpv+FAcYBqP/XAcYBvP+aAcYBvf/XAcYBwf+aAcYBxP+FAcYB3P/XAcYB3f/XAcYB4f/XAcYB5P/XAcYB9v/XAcYCB/+uAcYCC/+uAcYCbv+uAcYCfP+aAcYCgP+uAcYCgv+uAcYCl/+uAcYCm/+uAcYCp/+uAcYCqf+FAcYCqv/XAcYCtf+aAcYCtv/XAcYCt/+aAcYCuP/XAcYCuf+aAcYCuv/XAcYCvf+FAcYCvv/XAcYCv/+aAcYCwP/XAcYCwf+aAcYCwv/XAcYC1P+aAcYC1f/XAcYC9//XAcYC+P/XAcYC+f/XAcYC+v/XAcYC+//XAcYC/P/XAcYC/f+aAcYC/v/XAcYDA/+uAcYDDf+aAcYDDv/DAcYDD/+aAcYDEP/DAcYDF/+FAcYDGP/XAccAD/+uAccAEf+uAccBnf/sAccBpP/XAccBpv/sAccBqP/XAccBqv/XAccBrv/XAccBsP/XAccBsf/sAccBtf/XAccBvP/DAccBvf/XAccBv//XAccBwf/XAccBxP/sAccBx//sAccBzv/sAccB1f/sAccB8v/sAccCCP+uAccCDP+uAccCcv/XAccCc//sAccCev/sAccCfP/XAccCgP/sAccCgv/sAccCn//XAccCof/sAccCqf/sAccCtf/DAccCt//sAccCuf/sAccCu//XAccCvf/sAccCv//XAccCwf/XAccCyv/XAccCzv/XAccCz//sAccC1P/XAccC2f/XAccC2//XAccC3f/XAccC5f/XAccC5//sAccC9f/sAccC9//XAccC+f/XAccC+//XAccC/f/XAccDBf/XAccDB//XAccDDf/XAccDD//XAccDEf/XAccDEv/sAccDF//sAccDG//XAccDHP/sAcgAD/+uAcgAEf+uAcgBnf/sAcgBpP/XAcgBpv/sAcgBqP/XAcgBqv/XAcgBrv/XAcgBsP/XAcgBsf/sAcgBtf/XAcgBvP/DAcgBvf/XAcgBv//XAcgBwf/XAcgBxP/sAcgBx//sAcgBzv/sAcgB1f/sAcgB8v/sAcgCCP+uAcgCDP+uAcgCcv/XAcgCc//sAcgCev/sAcgCfP/XAcgCgP/sAcgCgv/sAcgCn//XAcgCof/sAcgCqf/sAcgCtf/DAcgCt//sAcgCuf/sAcgCu//XAcgCvf/sAcgCv//XAcgCwf/XAcgCyv/XAcgCzv/XAcgCz//sAcgC1P/XAcgC2f/XAcgC2//XAcgC3f/XAcgC5f/XAcgC5//sAcgC9f/sAcgC9//XAcgC+f/XAcgC+//XAcgC/f/XAcgDBf/XAcgDB//XAcgDDf/XAcgDD//XAcgDEf/XAcgDEv/sAcgDF//sAcgDG//XAcgDHP/sAcoABf/sAcoACv/sAcoCB//sAcoCC//sAcwB6QApAc0AD/+aAc0AEP/XAc0AEf+aAc0Bzv/DAc0Bz//sAc0B1f/DAc0B2P/sAc0B2//sAc0B3v/sAc0B6v/sAc0B7f/sAc0B8v/DAc0CAv/XAc0CA//XAc0CBP/XAc0CCP+aAc0CDP+aAc0Cav/sAc0Cc//DAc0Cf//sAc0Chf/sAc0Ch//sAc0Cif/sAc0Cjf/sAc0Csv/sAc0CtP/sAc0Cz//DAc0C4P/sAc0C8P/sAc0C8v/sAc0C9P/sAc0DCv/sAc0DDP/sAc0DEv/DAc0DFv/sAc0DGv/sAc0DHP/DAc4ABf/sAc4ACv/sAc4CB//sAc4CC//sAc8ABf/sAc8ACv/sAc8CB//sAc8CC//sAdABz//XAdAB2P/XAdAB2//XAdAB3v/XAdAB4f/XAdAB6v/XAdAB7f/XAdACav/XAdACf//XAdAChf/XAdACh//XAdACif/XAdACjf/XAdACsv/XAdACtP/XAdACwP/XAdACwv/XAdACxv/XAdACyP/XAdAC1f/XAdAC4P/XAdAC8P/XAdAC8v/XAdAC9P/XAdAC/v/XAdADCv/XAdADDP/XAdADFv/XAdADGv/XAdEB6QApAdQBz//XAdQB2P/XAdQB2//XAdQB3v/XAdQB4f/XAdQB6v/XAdQB7f/XAdQCav/XAdQCf//XAdQChf/XAdQCh//XAdQCif/XAdQCjf/XAdQCsv/XAdQCtP/XAdQCwP/XAdQCwv/XAdQCxv/XAdQCyP/XAdQC1f/XAdQC4P/XAdQC8P/XAdQC8v/XAdQC9P/XAdQC/v/XAdQDCv/XAdQDDP/XAdQDFv/XAdQDGv/XAdgABf/sAdgACv/sAdgB0P/XAdgB3P/sAdgB3f/sAdgB3//XAdgB4f/sAdgB5P/sAdgB9v/sAdgCB//sAdgCC//sAdgCoP/XAdgCqv/sAdgCtv/sAdgCvP/XAdgCvv/sAdgCwP/sAdgCwv/sAdgCy//XAdgC1f/sAdgC5v/XAdgC+P/sAdgC+v/sAdgC/P/sAdgC/v/sAdgDBv/XAdgDCP/XAdgDDv/sAdgDEP/sAdgDGP/sAdoABf/sAdoACv/sAdoB0P/XAdoB3P/sAdoB3f/sAdoB3//XAdoB4f/sAdoB5P/sAdoB9v/sAdoCB//sAdoCC//sAdoCoP/XAdoCqv/sAdoCtv/sAdoCvP/XAdoCvv/sAdoCwP/sAdoCwv/sAdoCy//XAdoC1f/sAdoC5v/XAdoC+P/sAdoC+v/sAdoC/P/sAdoC/v/sAdoDBv/XAdoDCP/XAdoDDv/sAdoDEP/sAdoDGP/sAdwAD/+aAdwAEP/XAdwAEf+aAdwBzv/DAdwBz//sAdwB1f/DAdwB2P/sAdwB2//sAdwB3v/sAdwB6v/sAdwB7f/sAdwB8v/DAdwCAv/XAdwCA//XAdwCBP/XAdwCCP+aAdwCDP+aAdwCav/sAdwCc//DAdwCf//sAdwChf/sAdwCh//sAdwCif/sAdwCjf/sAdwCsv/sAdwCtP/sAdwCz//DAdwC4P/sAdwC8P/sAdwC8v/sAdwC9P/sAdwDCv/sAdwDDP/sAdwDEv/DAdwDFv/sAdwDGv/sAdwDHP/DAd0AD/+uAd0AEf+uAd0Bzv/XAd0B1f/XAd0B8v/XAd0CCP+uAd0CDP+uAd0Cc//XAd0Cz//XAd0DEv/XAd0DHP/XAd4ABf/sAd4ACv/sAd4B0P/XAd4B3P/sAd4B3f/sAd4B3//XAd4B4f/sAd4B5P/sAd4B9v/sAd4CB//sAd4CC//sAd4CoP/XAd4Cqv/sAd4Ctv/sAd4CvP/XAd4Cvv/sAd4CwP/sAd4Cwv/sAd4Cy//XAd4C1f/sAd4C5v/XAd4C+P/sAd4C+v/sAd4C/P/sAd4C/v/sAd4DBv/XAd4DCP/XAd4DDv/sAd4DEP/sAd4DGP/sAd8Bz//XAd8B2P/XAd8B2//XAd8B3v/XAd8B4f/XAd8B6v/XAd8B7f/XAd8Cav/XAd8Cf//XAd8Chf/XAd8Ch//XAd8Cif/XAd8Cjf/XAd8Csv/XAd8CtP/XAd8CwP/XAd8Cwv/XAd8Cxv/XAd8CyP/XAd8C1f/XAd8C4P/XAd8C8P/XAd8C8v/XAd8C9P/XAd8C/v/XAd8DCv/XAd8DDP/XAd8DFv/XAd8DGv/XAeAABf/sAeAACv/sAeACB//sAeACC//sAeMABf/sAeMACv/sAeMCB//sAeMCC//sAeQABf+FAeQACv+FAeQB0P/XAeQB3P+aAeQB3f/DAeQB3//XAeQB4f+uAeQB5P+aAeQB9v/DAeQCB/+FAeQCC/+FAeQCbf/XAeQCgf/XAeQCg//XAeQCi//XAeQCoP/XAeQCqv+aAeQCtv+aAeQCuP/DAeQCuv/DAeQCvP/XAeQCvv+aAeQCwP+uAeQCwv+uAeQCxv/XAeQCyP/XAeQCy//XAeQC1f+uAeQC5v/XAeQC6v/XAeQC+P/DAeQC+v/DAeQC/P/DAeQC/v+uAeQDBv/XAeQDCP/XAeQDDv+aAeQDEP+aAeQDGP+aAeYABf+FAeYACv+FAeYB0P/XAeYB3P+aAeYB3f/DAeYB3//XAeYB4f+uAeYB5P+aAeYB9v/DAeYCB/+FAeYCC/+FAeYCbf/XAeYCgf/XAeYCg//XAeYCi//XAeYCoP/XAeYCqv+aAeYCtv+aAeYCuP/DAeYCuv/DAeYCvP/XAeYCvv+aAeYCwP+uAeYCwv+uAeYCxv/XAeYCyP/XAeYCy//XAeYC1f+uAeYC5v/XAeYC6v/XAeYC+P/DAeYC+v/DAeYC/P/DAeYC/v+uAeYDBv/XAeYDCP/XAeYDDv+aAeYDEP+aAeYDGP+aAecABf/sAecACv/sAecB0P/XAecB3P/sAecB3f/sAecB3//XAecB4f/sAecB5P/sAecB9v/sAecCB//sAecCC//sAecCoP/XAecCqv/sAecCtv/sAecCvP/XAecCvv/sAecCwP/sAecCwv/sAecCy//XAecC1f/sAecC5v/XAecC+P/sAecC+v/sAecC/P/sAecC/v/sAecDBv/XAecDCP/XAecDDv/sAecDEP/sAecDGP/sAegABf/sAegACv/sAegB0P/XAegB3P/sAegB3f/sAegB3//XAegB4f/sAegB5P/sAegB9v/sAegCB//sAegCC//sAegCoP/XAegCqv/sAegCtv/sAegCvP/XAegCvv/sAegCwP/sAegCwv/sAegCy//XAegC1f/sAegC5v/XAegC+P/sAegC+v/sAegC/P/sAegC/v/sAegDBv/XAegDCP/XAegDDv/sAegDEP/sAegDGP/sAeoABf/sAeoACv/sAeoCB//sAeoCC//sAesABf/sAesACv/sAesCB//sAesCC//sAesDDv/XAesDEP/XAewAD/+aAewAEP/XAewAEf+aAewBzv/DAewBz//sAewB1f/DAewB2P/sAewB2//sAewB3v/sAewB6v/sAewB7f/sAewB8v/DAewCAv/XAewCA//XAewCBP/XAewCCP+aAewCDP+aAewCav/sAewCc//DAewCf//sAewChf/sAewCh//sAewCif/sAewCjf/sAewCsv/sAewCtP/sAewCz//DAewC4P/sAewC8P/sAewC8v/sAewC9P/sAewDCv/sAewDDP/sAewDEv/DAewDFv/sAewDGv/sAewDHP/DAfIABf+FAfIACv+FAfIB0P/XAfIB3P+aAfIB3f/DAfIB3//XAfIB4f+uAfIB5P+aAfIB9v/DAfICB/+FAfICC/+FAfICbf/XAfICgf/XAfICg//XAfICi//XAfICoP/XAfICqv+aAfICtv+aAfICuP/DAfICuv/DAfICvP/XAfICvv+aAfICwP+uAfICwv+uAfICxv/XAfICyP/XAfICy//XAfIC1f+uAfIC5v/XAfIC6v/XAfIC+P/DAfIC+v/DAfIC/P/DAfIC/v+uAfIDBv/XAfIDCP/XAfIDDv+aAfIDEP+aAfIDGP+aAfMABf+FAfMACv+FAfMB0P/XAfMB3P+aAfMB3f/DAfMB3//XAfMB4f+uAfMB5P+aAfMB9v/DAfMCB/+FAfMCC/+FAfMCbf/XAfMCgf/XAfMCg//XAfMCi//XAfMCoP/XAfMCqv+aAfMCtv+aAfMCuP/DAfMCuv/DAfMCvP/XAfMCvv+aAfMCwP+uAfMCwv+uAfMCxv/XAfMCyP/XAfMCy//XAfMC1f+uAfMC5v/XAfMC6v/XAfMC+P/DAfMC+v/DAfMC/P/DAfMC/v+uAfMDBv/XAfMDCP/XAfMDDv+aAfMDEP+aAfMDGP+aAfQABf/sAfQACv/sAfQCB//sAfQCC//sAfQDDv/XAfQDEP/XAfUBz//XAfUB2P/XAfUB2//XAfUB3v/XAfUB4f/XAfUB6v/XAfUB7f/XAfUCav/XAfUCf//XAfUChf/XAfUCh//XAfUCif/XAfUCjf/XAfUCsv/XAfUCtP/XAfUCwP/XAfUCwv/XAfUCxv/XAfUCyP/XAfUC1f/XAfUC4P/XAfUC8P/XAfUC8v/XAfUC9P/XAfUC/v/XAfUDCv/XAfUDDP/XAfUDFv/XAfUDGv/XAfYAD/+uAfYAEf+uAfYBzv/XAfYB1f/XAfYB8v/XAfYCCP+uAfYCDP+uAfYCc//XAfYCz//XAfYDEv/XAfYDHP/XAfgAD/+FAfgAEP+uAfgAEf+FAfgBn//XAfgBpP+aAfgBqv9xAfgBrv+aAfgBtf+aAfgBuP/XAfgBu//XAfgBvAApAfgBvv+uAfgBzP+aAfgBzf+aAfgBzv+FAfgBz/9xAfgB0P/XAfgB0f/XAfgB0v+aAfgB0/+aAfgB1P+aAfgB1f+FAfgB1v+aAfgB1/+aAfgB2P9xAfgB2f+aAfgB2v+aAfgB2/9xAfgB3P+uAfgB3f+uAfgB3v9xAfgB3//XAfgB4P+aAfgB4f+aAfgB4v+aAfgB4/+aAfgB5P+uAfgB5f+aAfgB5v+aAfgB5//XAfgB6P+aAfgB6f/DAfgB6v9xAfgB7P+aAfgB7f9xAfgB7v+FAfgB8v+FAfgB8/+aAfgB9f+aAfgB9v+uAfgB9/+aAfgB+f+aAfgCAv+uAfgCA/+uAfgCBP+uAfgCCP+FAfgCDP+FAfgCav9xAfgCa/+aAfgCbP/XAfgCbf/XAfgCcf+aAfgCcv9xAfgCc/+FAfgCdf+aAfgCd/+aAfgCef+aAfgCff+aAfgCfv/XAfgCf/9xAfgCgf/XAfgCg//XAfgChP/XAfgChf9xAfgChv/XAfgCh/9xAfgCiP/XAfgCif9xAfgCiv/XAfgCi//XAfgCjP/XAfgCjf9xAfgClv+aAfgCmv+aAfgCnv+aAfgCoP/XAfgCov/XAfgCpP+aAfgCpv+aAfgCqv+uAfgCrP+aAfgCrv+aAfgCsP+aAfgCsf/XAfgCsv9xAfgCs//XAfgCtP9xAfgCtQApAfgCtv+uAfgCuP+uAfgCuv+uAfgCvP/XAfgCvv+uAfgCwP+aAfgCwv+aAfgCxP+aAfgCxf+aAfgCxv9xAfgCx/+aAfgCyP9xAfgCy//XAfgCzf+aAfgCzv+aAfgCz/+FAfgC0f+aAfgC0/+aAfgC1f+aAfgC1/+aAfgC2f9xAfgC2/9xAfgC3f9xAfgC4P9xAfgC5v/XAfgC6P/XAfgC6v/DAfgC7P+aAfgC7v+aAfgC7//XAfgC8P9xAfgC8f/XAfgC8v9xAfgC8//XAfgC9P9xAfgC9v/XAfgC+P+uAfgC+v+uAfgC/P+uAfgC/v+aAfgDAP+aAfgDAv+aAfgDBv/XAfgDCP/XAfgDCf9xAfgDCv9xAfgDC/9xAfgDDP9xAfgDDv+aAfgDEP+aAfgDEf+aAfgDEv+FAfgDFP+aAfgDFf/XAfgDFv9xAfgDGP+uAfgDGv9xAfgDG/+aAfgDHP+FAfkAD/+aAfkAEP/XAfkAEf+aAfkBzv/DAfkBz//sAfkB1f/DAfkB2P/sAfkB2//sAfkB3v/sAfkB6v/sAfkB7f/sAfkB8v/DAfkCAv/XAfkCA//XAfkCBP/XAfkCCP+aAfkCDP+aAfkCav/sAfkCc//DAfkCf//sAfkChf/sAfkCh//sAfkCif/sAfkCjf/sAfkCsv/sAfkCtP/sAfkCz//DAfkC4P/sAfkC8P/sAfkC8v/sAfkC9P/sAfkDCv/sAfkDDP/sAfkDEv/DAfkDFv/sAfkDGv/sAfkDHP/DAfoAD/+aAfoAEf+aAfoAIgApAfoAJP+uAfoAJv/sAfoAKv/sAfoAMv/sAfoANP/sAfoARP/XAfoARv/XAfoAR//XAfoASP/XAfoASv/sAfoAUP/sAfoAUf/sAfoAUv/XAfoAU//sAfoAVP/XAfoAVf/sAfoAVv/sAfoAWP/sAfoAgv+uAfoAg/+uAfoAhP+uAfoAhf+uAfoAhv+uAfoAh/+uAfoAif/sAfoAlP/sAfoAlf/sAfoAlv/sAfoAl//sAfoAmP/sAfoAmv/sAfoAov/XAfoAo//XAfoApP/XAfoApf/XAfoApv/XAfoAp//XAfoAqP/XAfoAqf/XAfoAqv/XAfoAq//XAfoArP/XAfoArf/XAfoAtP/XAfoAtf/XAfoAtv/XAfoAt//XAfoAuP/XAfoAuv/XAfoAu//sAfoAvP/sAfoAvf/sAfoAvv/sAfoAwv+uAfoAw//XAfoAxP+uAfoAxf/XAfoAxv+uAfoAx//XAfoAyP/sAfoAyf/XAfoAyv/sAfoAy//XAfoAzP/sAfoAzf/XAfoAzv/sAfoAz//XAfoA0f/XAfoA0//XAfoA1f/XAfoA1//XAfoA2f/XAfoA2//XAfoA3f/XAfoA3v/sAfoA3//sAfoA4P/sAfoA4f/sAfoA4v/sAfoA4//sAfoA5P/sAfoA5f/sAfoA+v/sAfoBBv/sAfoBCP/sAfoBDf/sAfoBDv/sAfoBD//XAfoBEP/sAfoBEf/XAfoBEv/sAfoBE//XAfoBFP/sAfoBFf/XAfoBF//sAfoBGf/sAfoBHf/sAfoBIf/sAfoBK//sAfoBLf/sAfoBL//sAfoBMf/sAfoBM//sAfoBNf/sAfoBQ/+uAfoBRP/XAfoBRv/XAfoBR//sAfoBSP/XAfoBSv/sAfoCCP+aAfoCDP+aAfoCV//sAfoCWP+uAfoCWf/XAfoCX//sAfoCYP/XAfoCYv/sAfoDHf+uAfoDHv/XAfoDH/+uAfoDIP/XAfoDIf+uAfoDIv/XAfoDI/+uAfoDJf+uAfoDJv/XAfoDJ/+uAfoDKP/XAfoDKf+uAfoDKv/XAfoDK/+uAfoDLP/XAfoDLf+uAfoDLv/XAfoDL/+uAfoDMP/XAfoDMf+uAfoDMv/XAfoDM/+uAfoDNP/XAfoDNv/XAfoDOP/XAfoDOv/XAfoDPP/XAfoDQP/XAfoDQv/XAfoDRP/XAfoDSf/sAfoDSv/XAfoDS//sAfoDTP/XAfoDTf/sAfoDTv/XAfoDT//sAfoDUf/sAfoDUv/XAfoDU//sAfoDVP/XAfoDVf/sAfoDVv/XAfoDV//sAfoDWP/XAfoDWf/sAfoDWv/XAfoDW//sAfoDXP/XAfoDXf/sAfoDXv/XAfoDX//sAfoDYP/XAfoDYv/sAfoDZP/sAfoDZv/sAfoDaP/sAfoDav/sAfoDbP/sAfoDbv/sAfsABQBSAfsACgBSAfsAD/+uAfsAEf+uAfsAIgApAfsCBwBSAfsCCP+uAfsCCwBSAfsCDP+uAfwAD/+aAfwAEf+aAfwAIgApAfwAJP+uAfwAJv/sAfwAKv/sAfwAMv/sAfwANP/sAfwARP/XAfwARv/XAfwAR//XAfwASP/XAfwASv/sAfwAUP/sAfwAUf/sAfwAUv/XAfwAU//sAfwAVP/XAfwAVf/sAfwAVv/sAfwAWP/sAfwAgv+uAfwAg/+uAfwAhP+uAfwAhf+uAfwAhv+uAfwAh/+uAfwAif/sAfwAlP/sAfwAlf/sAfwAlv/sAfwAl//sAfwAmP/sAfwAmv/sAfwAov/XAfwAo//XAfwApP/XAfwApf/XAfwApv/XAfwAp//XAfwAqP/XAfwAqf/XAfwAqv/XAfwAq//XAfwArP/XAfwArf/XAfwAtP/XAfwAtf/XAfwAtv/XAfwAt//XAfwAuP/XAfwAuv/XAfwAu//sAfwAvP/sAfwAvf/sAfwAvv/sAfwAwv+uAfwAw//XAfwAxP+uAfwAxf/XAfwAxv+uAfwAx//XAfwAyP/sAfwAyf/XAfwAyv/sAfwAy//XAfwAzP/sAfwAzf/XAfwAzv/sAfwAz//XAfwA0f/XAfwA0//XAfwA1f/XAfwA1//XAfwA2f/XAfwA2//XAfwA3f/XAfwA3v/sAfwA3//sAfwA4P/sAfwA4f/sAfwA4v/sAfwA4//sAfwA5P/sAfwA5f/sAfwA+v/sAfwBBv/sAfwBCP/sAfwBDf/sAfwBDv/sAfwBD//XAfwBEP/sAfwBEf/XAfwBEv/sAfwBE//XAfwBFP/sAfwBFf/XAfwBF//sAfwBGf/sAfwBHf/sAfwBIf/sAfwBK//sAfwBLf/sAfwBL//sAfwBMf/sAfwBM//sAfwBNf/sAfwBQ/+uAfwBRP/XAfwBRv/XAfwBR//sAfwBSP/XAfwBSv/sAfwCCP+aAfwCDP+aAfwCV//sAfwCWP+uAfwCWf/XAfwCX//sAfwCYP/XAfwCYv/sAfwDHf+uAfwDHv/XAfwDH/+uAfwDIP/XAfwDIf+uAfwDIv/XAfwDI/+uAfwDJf+uAfwDJv/XAfwDJ/+uAfwDKP/XAfwDKf+uAfwDKv/XAfwDK/+uAfwDLP/XAfwDLf+uAfwDLv/XAfwDL/+uAfwDMP/XAfwDMf+uAfwDMv/XAfwDM/+uAfwDNP/XAfwDNv/XAfwDOP/XAfwDOv/XAfwDPP/XAfwDQP/XAfwDQv/XAfwDRP/XAfwDSf/sAfwDSv/XAfwDS//sAfwDTP/XAfwDTf/sAfwDTv/XAfwDT//sAfwDUf/sAfwDUv/XAfwDU//sAfwDVP/XAfwDVf/sAfwDVv/XAfwDV//sAfwDWP/XAfwDWf/sAfwDWv/XAfwDW//sAfwDXP/XAfwDXf/sAfwDXv/XAfwDX//sAfwDYP/XAfwDYv/sAfwDZP/sAfwDZv/sAfwDaP/sAfwDav/sAfwDbP/sAfwDbv/sAf0ABQBSAf0ACgBSAf0AD/+uAf0AEf+uAf0AIgApAf0CBwBSAf0CCP+uAf0CCwBSAf0CDP+uAf4AD/+aAf4AEf+aAf4AIgApAf4AJP+uAf4AJv/sAf4AKv/sAf4AMv/sAf4ANP/sAf4ARP/XAf4ARv/XAf4AR//XAf4ASP/XAf4ASv/sAf4AUP/sAf4AUf/sAf4AUv/XAf4AU//sAf4AVP/XAf4AVf/sAf4AVv/sAf4AWP/sAf4Agv+uAf4Ag/+uAf4AhP+uAf4Ahf+uAf4Ahv+uAf4Ah/+uAf4Aif/sAf4AlP/sAf4Alf/sAf4Alv/sAf4Al//sAf4AmP/sAf4Amv/sAf4Aov/XAf4Ao//XAf4ApP/XAf4Apf/XAf4Apv/XAf4Ap//XAf4AqP/XAf4Aqf/XAf4Aqv/XAf4Aq//XAf4ArP/XAf4Arf/XAf4AtP/XAf4Atf/XAf4Atv/XAf4At//XAf4AuP/XAf4Auv/XAf4Au//sAf4AvP/sAf4Avf/sAf4Avv/sAf4Awv+uAf4Aw//XAf4AxP+uAf4Axf/XAf4Axv+uAf4Ax//XAf4AyP/sAf4Ayf/XAf4Ayv/sAf4Ay//XAf4AzP/sAf4Azf/XAf4Azv/sAf4Az//XAf4A0f/XAf4A0//XAf4A1f/XAf4A1//XAf4A2f/XAf4A2//XAf4A3f/XAf4A3v/sAf4A3//sAf4A4P/sAf4A4f/sAf4A4v/sAf4A4//sAf4A5P/sAf4A5f/sAf4A+v/sAf4BBv/sAf4BCP/sAf4BDf/sAf4BDv/sAf4BD//XAf4BEP/sAf4BEf/XAf4BEv/sAf4BE//XAf4BFP/sAf4BFf/XAf4BF//sAf4BGf/sAf4BHf/sAf4BIf/sAf4BK//sAf4BLf/sAf4BL//sAf4BMf/sAf4BM//sAf4BNf/sAf4BQ/+uAf4BRP/XAf4BRv/XAf4BR//sAf4BSP/XAf4BSv/sAf4CCP+aAf4CDP+aAf4CV//sAf4CWP+uAf4CWf/XAf4CX//sAf4CYP/XAf4CYv/sAf4DHf+uAf4DHv/XAf4DH/+uAf4DIP/XAf4DIf+uAf4DIv/XAf4DI/+uAf4DJf+uAf4DJv/XAf4DJ/+uAf4DKP/XAf4DKf+uAf4DKv/XAf4DK/+uAf4DLP/XAf4DLf+uAf4DLv/XAf4DL/+uAf4DMP/XAf4DMf+uAf4DMv/XAf4DM/+uAf4DNP/XAf4DNv/XAf4DOP/XAf4DOv/XAf4DPP/XAf4DQP/XAf4DQv/XAf4DRP/XAf4DSf/sAf4DSv/XAf4DS//sAf4DTP/XAf4DTf/sAf4DTv/XAf4DT//sAf4DUf/sAf4DUv/XAf4DU//sAf4DVP/XAf4DVf/sAf4DVv/XAf4DV//sAf4DWP/XAf4DWf/sAf4DWv/XAf4DW//sAf4DXP/XAf4DXf/sAf4DXv/XAf4DX//sAf4DYP/XAf4DYv/sAf4DZP/sAf4DZv/sAf4DaP/sAf4Dav/sAf4DbP/sAf4Dbv/sAf8ABQBSAf8ACgBSAf8AD/+uAf8AEf+uAf8AIgApAf8CBwBSAf8CCP+uAf8CCwBSAf8CDP+uAgAAD/+FAgAAEf+FAgAAIgApAgAAJP+FAgAAJv/XAgAAKv/XAgAAMv/XAgAANP/XAgAARP+aAgAARv+aAgAAR/+aAgAASP+aAgAASv/XAgAAUP/DAgAAUf/DAgAAUv+aAgAAU//DAgAAVP+aAgAAVf/DAgAAVv+uAgAAWP/DAgAAXf/XAgAAgv+FAgAAg/+FAgAAhP+FAgAAhf+FAgAAhv+FAgAAh/+FAgAAif/XAgAAlP/XAgAAlf/XAgAAlv/XAgAAl//XAgAAmP/XAgAAmv/XAgAAov+aAgAAo/+aAgAApP+aAgAApf+aAgAApv+aAgAAp/+aAgAAqP+aAgAAqf+aAgAAqv+aAgAAq/+aAgAArP+aAgAArf+aAgAAtP+aAgAAtf+aAgAAtv+aAgAAt/+aAgAAuP+aAgAAuv+aAgAAu//DAgAAvP/DAgAAvf/DAgAAvv/DAgAAwv+FAgAAw/+aAgAAxP+FAgAAxf+aAgAAxv+FAgAAx/+aAgAAyP/XAgAAyf+aAgAAyv/XAgAAy/+aAgAAzP/XAgAAzf+aAgAAzv/XAgAAz/+aAgAA0f+aAgAA0/+aAgAA1f+aAgAA1/+aAgAA2f+aAgAA2/+aAgAA3f+aAgAA3v/XAgAA3//XAgAA4P/XAgAA4f/XAgAA4v/XAgAA4//XAgAA5P/XAgAA5f/XAgAA+v/DAgABBv/DAgABCP/DAgABDf/DAgABDv/XAgABD/+aAgABEP/XAgABEf+aAgABEv/XAgABE/+aAgABFP/XAgABFf+aAgABF//DAgABGf/DAgABHf+uAgABIf+uAgABK//DAgABLf/DAgABL//DAgABMf/DAgABM//DAgABNf/DAgABPP/XAgABPv/XAgABQP/XAgABQ/+FAgABRP+aAgABRv+aAgABR//XAgABSP+aAgABSv+uAgACCP+FAgACDP+FAgACV//DAgACWP+FAgACWf+aAgACX//XAgACYP+aAgACYv/DAgADHf+FAgADHv+aAgADH/+FAgADIP+aAgADIf+FAgADIv+aAgADI/+FAgADJf+FAgADJv+aAgADJ/+FAgADKP+aAgADKf+FAgADKv+aAgADK/+FAgADLP+aAgADLf+FAgADLv+aAgADL/+FAgADMP+aAgADMf+FAgADMv+aAgADM/+FAgADNP+aAgADNv+aAgADOP+aAgADOv+aAgADPP+aAgADQP+aAgADQv+aAgADRP+aAgADSf/XAgADSv+aAgADS//XAgADTP+aAgADTf/XAgADTv+aAgADT//XAgADUf/XAgADUv+aAgADU//XAgADVP+aAgADVf/XAgADVv+aAgADV//XAgADWP+aAgADWf/XAgADWv+aAgADW//XAgADXP+aAgADXf/XAgADXv+aAgADX//XAgADYP+aAgADYv/DAgADZP/DAgADZv/DAgADaP/DAgADav/DAgADbP/DAgADbv/DAgEABQBSAgEACgBSAgEAD/+uAgEAEf+uAgEAIgApAgECBwBSAgECCP+uAgECCwBSAgECDP+uAgIAN/+uAgIBJP+uAgIBJv+uAgIBcf+uAgIBnf+uAgIBpv+uAgIBvP+uAgIBxP+uAgIB3P/XAgIB5P/XAgICqf+uAgICqv/XAgICtf+uAgICtv/XAgICvf+uAgICvv/XAgIDF/+uAgIDGP/XAgIDj/+uAgMAN/+uAgMBJP+uAgMBJv+uAgMBcf+uAgMBnf+uAgMBpv+uAgMBvP+uAgMBxP+uAgMB3P/XAgMB5P/XAgMCqf+uAgMCqv/XAgMCtf+uAgMCtv/XAgMCvf+uAgMCvv/XAgMDF/+uAgMDGP/XAgMDj/+uAgQAN/+uAgQBJP+uAgQBJv+uAgQBcf+uAgQBnf+uAgQBpv+uAgQBvP+uAgQBxP+uAgQB3P/XAgQB5P/XAgQCqf+uAgQCqv/XAgQCtf+uAgQCtv/XAgQCvf+uAgQCvv/XAgQDF/+uAgQDGP/XAgQDj/+uAgYAJP9xAgYANwApAgYAOQApAgYAOgApAgYAPAAUAgYARP+uAgYARv+FAgYAR/+FAgYASP+FAgYASv/DAgYAUP/DAgYAUf/DAgYAUv+FAgYAU//DAgYAVP+FAgYAVf/DAgYAVv/DAgYAWP/DAgYAgv9xAgYAg/9xAgYAhP9xAgYAhf9xAgYAhv9xAgYAh/9xAgYAnwAUAgYAov+FAgYAo/+uAgYApP+uAgYApf+uAgYApv+uAgYAp/+uAgYAqP+uAgYAqf+FAgYAqv+FAgYAq/+FAgYArP+FAgYArf+FAgYAtP+FAgYAtf+FAgYAtv+FAgYAt/+FAgYAuP+FAgYAuv+FAgYAu//DAgYAvP/DAgYAvf/DAgYAvv/DAgYAwv9xAgYAw/+uAgYAxP9xAgYAxf+uAgYAxv9xAgYAx/+uAgYAyf+FAgYAy/+FAgYAzf+FAgYAz/+FAgYA0f+FAgYA0/+FAgYA1f+FAgYA1/+FAgYA2f+FAgYA2/+FAgYA3f+FAgYA3//DAgYA4f/DAgYA4//DAgYA5f/DAgYA+v/DAgYBBv/DAgYBCP/DAgYBDf/DAgYBD/+FAgYBEf+FAgYBE/+FAgYBFf+FAgYBF//DAgYBGf/DAgYBHf/DAgYBIf/DAgYBJAApAgYBJgApAgYBK//DAgYBLf/DAgYBL//DAgYBMf/DAgYBM//DAgYBNf/DAgYBNgApAgYBOAAUAgYBOgAUAgYBQ/9xAgYBRP+uAgYBRv+uAgYBSP+FAgYBSv/DAgYBVv9xAgYBX/9xAgYBYv9xAgYBaf9xAgYBef+uAgYBev/XAgYBe//XAgYBfv+uAgYBgf/DAgYBgv/XAgYBg//XAgYBhP/XAgYBh//XAgYBif/XAgYBjP+uAgYBjv/DAgYBj/+uAgYBkP+uAgYBk/+uAgYBmf+uAgYBpP+FAgYBqv9xAgYBrv+FAgYBtf+FAgYByv/XAgYBzv9xAgYBz/+FAgYB1f9xAgYB2P+FAgYB2/+FAgYB3v+FAgYB6v+FAgYB7f+FAgYB7v/DAgYB8v9xAgYB+gApAgYB/AApAgYB/gApAgYCAAAUAgYCV//DAgYCWP9xAgYCWf+uAgYCYP+FAgYCYv/DAgYCav+FAgYCcv9xAgYCc/9xAgYCff/sAgYCf/+FAgYChf+FAgYCh/+FAgYCif+FAgYCjf+FAgYCsv+FAgYCtP+FAgYCzv+FAgYCz/9xAgYC2f9xAgYC2v/XAgYC2/9xAgYC3P/XAgYC3f9xAgYC3v/XAgYC4P+FAgYC4v/XAgYC5P/XAgYC8P+FAgYC8v+FAgYC9P+FAgYDCf9xAgYDCv+FAgYDC/9xAgYDDP+FAgYDEf+FAgYDEv9xAgYDFv+FAgYDGv+FAgYDG/+FAgYDHP9xAgYDHf9xAgYDHv+uAgYDH/9xAgYDIP+uAgYDIf9xAgYDIv+uAgYDI/9xAgYDJf9xAgYDJv+uAgYDJ/9xAgYDKP+uAgYDKf9xAgYDKv+uAgYDK/9xAgYDLP+uAgYDLf9xAgYDLv+uAgYDL/9xAgYDMP+uAgYDMf9xAgYDMv+uAgYDM/9xAgYDNP+uAgYDNv+FAgYDOP+FAgYDOv+FAgYDPP+FAgYDQP+FAgYDQv+FAgYDRP+FAgYDSv+FAgYDTP+FAgYDTv+FAgYDUv+FAgYDVP+FAgYDVv+FAgYDWP+FAgYDWv+FAgYDXP+FAgYDXv+FAgYDYP+FAgYDYv/DAgYDZP/DAgYDZv/DAgYDaP/DAgYDav/DAgYDbP/DAgYDbv/DAgYDbwAUAgYDcQAUAgYDcwAUAgYDjwApAgcAJP9xAgcANwApAgcAOQApAgcAOgApAgcAPAAUAgcARP+uAgcARv+FAgcAR/+FAgcASP+FAgcASv/DAgcAUP/DAgcAUf/DAgcAUv+FAgcAU//DAgcAVP+FAgcAVf/DAgcAVv/DAgcAWP/DAgcAgv9xAgcAg/9xAgcAhP9xAgcAhf9xAgcAhv9xAgcAh/9xAgcAnwAUAgcAov+FAgcAo/+uAgcApP+uAgcApf+uAgcApv+uAgcAp/+uAgcAqP+uAgcAqf+FAgcAqv+FAgcAq/+FAgcArP+FAgcArf+FAgcAtP+FAgcAtf+FAgcAtv+FAgcAt/+FAgcAuP+FAgcAuv+FAgcAu//DAgcAvP/DAgcAvf/DAgcAvv/DAgcAwv9xAgcAw/+uAgcAxP9xAgcAxf+uAgcAxv9xAgcAx/+uAgcAyf+FAgcAy/+FAgcAzf+FAgcAz/+FAgcA0f+FAgcA0/+FAgcA1f+FAgcA1/+FAgcA2f+FAgcA2/+FAgcA3f+FAgcA3//DAgcA4f/DAgcA4//DAgcA5f/DAgcA+v/DAgcBBv/DAgcBCP/DAgcBDf/DAgcBD/+FAgcBEf+FAgcBE/+FAgcBFf+FAgcBF//DAgcBGf/DAgcBHf/DAgcBIf/DAgcBJAApAgcBJgApAgcBK//DAgcBLf/DAgcBL//DAgcBMf/DAgcBM//DAgcBNf/DAgcBNgApAgcBOAAUAgcBOgAUAgcBQ/9xAgcBRP+uAgcBRv+uAgcBSP+FAgcBSv/DAgcBVv9xAgcBX/9xAgcBYv9xAgcBaf9xAgcBef+uAgcBev/XAgcBe//XAgcBfv+uAgcBgf/DAgcBgv/XAgcBg//XAgcBhP/XAgcBh//XAgcBif/XAgcBjP+uAgcBjv/DAgcBj/+uAgcBkP+uAgcBk/+uAgcBmf+uAgcBpP+FAgcBqv9xAgcBrv+FAgcBtf+FAgcByv/XAgcBzv9xAgcBz/+FAgcB1f9xAgcB2P+FAgcB2/+FAgcB3v+FAgcB6v+FAgcB7f+FAgcB7v/DAgcB8v9xAgcB+gApAgcB/AApAgcB/gApAgcCAAAUAgcCV//DAgcCWP9xAgcCWf+uAgcCYP+FAgcCYv/DAgcCav+FAgcCcv9xAgcCc/9xAgcCff/sAgcCf/+FAgcChf+FAgcCh/+FAgcCif+FAgcCjf+FAgcCsv+FAgcCtP+FAgcCzv+FAgcCz/9xAgcC2f9xAgcC2v/XAgcC2/9xAgcC3P/XAgcC3f9xAgcC3v/XAgcC4P+FAgcC4v/XAgcC5P/XAgcC8P+FAgcC8v+FAgcC9P+FAgcDCf9xAgcDCv+FAgcDC/9xAgcDDP+FAgcDEf+FAgcDEv9xAgcDFv+FAgcDGv+FAgcDG/+FAgcDHP9xAgcDHf9xAgcDHv+uAgcDH/9xAgcDIP+uAgcDIf9xAgcDIv+uAgcDI/9xAgcDJf9xAgcDJv+uAgcDJ/9xAgcDKP+uAgcDKf9xAgcDKv+uAgcDK/9xAgcDLP+uAgcDLf9xAgcDLv+uAgcDL/9xAgcDMP+uAgcDMf9xAgcDMv+uAgcDM/9xAgcDNP+uAgcDNv+FAgcDOP+FAgcDOv+FAgcDPP+FAgcDQP+FAgcDQv+FAgcDRP+FAgcDSv+FAgcDTP+FAgcDTv+FAgcDUv+FAgcDVP+FAgcDVv+FAgcDWP+FAgcDWv+FAgcDXP+FAgcDXv+FAgcDYP+FAgcDYv/DAgcDZP/DAgcDZv/DAgcDaP/DAgcDav/DAgcDbP/DAgcDbv/DAgcDbwAUAgcDcQAUAgcDcwAUAgcDjwApAggAJv+aAggAKv+aAggAMv+aAggANP+aAggAN/9xAggAOP/XAggAOf+FAggAOv+FAggAPP+FAggAif+aAggAlP+aAggAlf+aAggAlv+aAggAl/+aAggAmP+aAggAmv+aAggAm//XAggAnP/XAggAnf/XAggAnv/XAggAn/+FAggAyP+aAggAyv+aAggAzP+aAggAzv+aAggA3v+aAggA4P+aAggA4v+aAggA5P+aAggBDv+aAggBEP+aAggBEv+aAggBFP+aAggBJP9xAggBJv9xAggBKv/XAggBLP/XAggBLv/XAggBMP/XAggBMv/XAggBNP/XAggBNv+FAggBOP+FAggBOv+FAggBR/+aAggBZv+uAggBbf+uAggBcf9xAggBcv+FAggBc/+aAggBdf+FAggBeP+FAggBhf/XAggBnf9xAggBn/+aAggBpv9xAggBuP+aAggBu/+aAggBvP9xAggBvv+uAggBwf9cAggBxP9xAggB3P+aAggB4f+FAggB5P+aAggB+v+FAggB/P+FAggB/v+FAggCAP+FAggCVP+FAggCX/+aAggCYf/XAggCbP+aAggCfP9cAggCfv+aAggCgP+FAggCgv+FAggChP+aAggChv+aAggCiP+aAggCiv+aAggCjP+aAggCqf9xAggCqv+aAggCsf+aAggCs/+aAggCtf9xAggCtv+aAggCt/+FAggCuf+FAggCvf9xAggCvv+aAggCv/9cAggCwP+FAggCwf9cAggCwv+FAggCxf+FAggCx/+FAggC1P9cAggC1f+FAggC7/+aAggC8f+aAggC8/+aAggC/f9cAggC/v+FAggDDf+FAggDDv+aAggDD/+FAggDEP+aAggDFf+aAggDF/9xAggDGP+aAggDSf+aAggDS/+aAggDTf+aAggDT/+aAggDUf+aAggDU/+aAggDVf+aAggDV/+aAggDWf+aAggDW/+aAggDXf+aAggDX/+aAggDYf/XAggDY//XAggDZf/XAggDZ//XAggDaf/XAggDa//XAggDbf/XAggDb/+FAggDcf+FAggDc/+FAggDj/9xAgoAJP9xAgoANwApAgoAOQApAgoAOgApAgoAPAAUAgoARP+uAgoARv+FAgoAR/+FAgoASP+FAgoASv/DAgoAUP/DAgoAUf/DAgoAUv+FAgoAU//DAgoAVP+FAgoAVf/DAgoAVv/DAgoAWP/DAgoAgv9xAgoAg/9xAgoAhP9xAgoAhf9xAgoAhv9xAgoAh/9xAgoAnwAUAgoAov+FAgoAo/+uAgoApP+uAgoApf+uAgoApv+uAgoAp/+uAgoAqP+uAgoAqf+FAgoAqv+FAgoAq/+FAgoArP+FAgoArf+FAgoAtP+FAgoAtf+FAgoAtv+FAgoAt/+FAgoAuP+FAgoAuv+FAgoAu//DAgoAvP/DAgoAvf/DAgoAvv/DAgoAwv9xAgoAw/+uAgoAxP9xAgoAxf+uAgoAxv9xAgoAx/+uAgoAyf+FAgoAy/+FAgoAzf+FAgoAz/+FAgoA0f+FAgoA0/+FAgoA1f+FAgoA1/+FAgoA2f+FAgoA2/+FAgoA3f+FAgoA3//DAgoA4f/DAgoA4//DAgoA5f/DAgoA+v/DAgoBBv/DAgoBCP/DAgoBDf/DAgoBD/+FAgoBEf+FAgoBE/+FAgoBFf+FAgoBF//DAgoBGf/DAgoBHf/DAgoBIf/DAgoBJAApAgoBJgApAgoBK//DAgoBLf/DAgoBL//DAgoBMf/DAgoBM//DAgoBNf/DAgoBNgApAgoBOAAUAgoBOgAUAgoBQ/9xAgoBRP+uAgoBRv+uAgoBSP+FAgoBSv/DAgoBVv9xAgoBX/9xAgoBYv9xAgoBaf9xAgoBef+uAgoBev/XAgoBe//XAgoBfv+uAgoBgf/DAgoBgv/XAgoBg//XAgoBhP/XAgoBh//XAgoBif/XAgoBjP+uAgoBjv/DAgoBj/+uAgoBkP+uAgoBk/+uAgoBmf+uAgoBpP+FAgoBqv9xAgoBrv+FAgoBtf+FAgoByv/XAgoBzv9xAgoBz/+FAgoB1f9xAgoB2P+FAgoB2/+FAgoB3v+FAgoB6v+FAgoB7f+FAgoB7v/DAgoB8v9xAgoB+gApAgoB/AApAgoB/gApAgoCAAAUAgoCV//DAgoCWP9xAgoCWf+uAgoCYP+FAgoCYv/DAgoCav+FAgoCcv9xAgoCc/9xAgoCff/sAgoCf/+FAgoChf+FAgoCh/+FAgoCif+FAgoCjf+FAgoCsv+FAgoCtP+FAgoCzv+FAgoCz/9xAgoC2f9xAgoC2v/XAgoC2/9xAgoC3P/XAgoC3f9xAgoC3v/XAgoC4P+FAgoC4v/XAgoC5P/XAgoC8P+FAgoC8v+FAgoC9P+FAgoDCf9xAgoDCv+FAgoDC/9xAgoDDP+FAgoDEf+FAgoDEv9xAgoDFv+FAgoDGv+FAgoDG/+FAgoDHP9xAgoDHf9xAgoDHv+uAgoDH/9xAgoDIP+uAgoDIf9xAgoDIv+uAgoDI/9xAgoDJf9xAgoDJv+uAgoDJ/9xAgoDKP+uAgoDKf9xAgoDKv+uAgoDK/9xAgoDLP+uAgoDLf9xAgoDLv+uAgoDL/9xAgoDMP+uAgoDMf9xAgoDMv+uAgoDM/9xAgoDNP+uAgoDNv+FAgoDOP+FAgoDOv+FAgoDPP+FAgoDQP+FAgoDQv+FAgoDRP+FAgoDSv+FAgoDTP+FAgoDTv+FAgoDUv+FAgoDVP+FAgoDVv+FAgoDWP+FAgoDWv+FAgoDXP+FAgoDXv+FAgoDYP+FAgoDYv/DAgoDZP/DAgoDZv/DAgoDaP/DAgoDav/DAgoDbP/DAgoDbv/DAgoDbwAUAgoDcQAUAgoDcwAUAgoDjwApAgwAJv+aAgwAKv+aAgwAMv+aAgwANP+aAgwAN/9xAgwAOP/XAgwAOf+FAgwAOv+FAgwAPP+FAgwAif+aAgwAlP+aAgwAlf+aAgwAlv+aAgwAl/+aAgwAmP+aAgwAmv+aAgwAm//XAgwAnP/XAgwAnf/XAgwAnv/XAgwAn/+FAgwAyP+aAgwAyv+aAgwAzP+aAgwAzv+aAgwA3v+aAgwA4P+aAgwA4v+aAgwA5P+aAgwBDv+aAgwBEP+aAgwBEv+aAgwBFP+aAgwBJP9xAgwBJv9xAgwBKv/XAgwBLP/XAgwBLv/XAgwBMP/XAgwBMv/XAgwBNP/XAgwBNv+FAgwBOP+FAgwBOv+FAgwBR/+aAgwBZv+uAgwBbf+uAgwBcf9xAgwBcv+FAgwBc/+aAgwBdf+FAgwBeP+FAgwBhf/XAgwBnf9xAgwBn/+aAgwBpv9xAgwBuP+aAgwBu/+aAgwBvP9xAgwBvv+uAgwBwf9cAgwBxP9xAgwB3P+aAgwB4f+FAgwB5P+aAgwB+v+FAgwB/P+FAgwB/v+FAgwCAP+FAgwCVP+FAgwCX/+aAgwCYf/XAgwCbP+aAgwCfP9cAgwCfv+aAgwCgP+FAgwCgv+FAgwChP+aAgwChv+aAgwCiP+aAgwCiv+aAgwCjP+aAgwCqf9xAgwCqv+aAgwCsf+aAgwCs/+aAgwCtf9xAgwCtv+aAgwCt/+FAgwCuf+FAgwCvf9xAgwCvv+aAgwCv/9cAgwCwP+FAgwCwf9cAgwCwv+FAgwCxf+FAgwCx/+FAgwC1P9cAgwC1f+FAgwC7/+aAgwC8f+aAgwC8/+aAgwC/f9cAgwC/v+FAgwDDf+FAgwDDv+aAgwDD/+FAgwDEP+aAgwDFf+aAgwDF/9xAgwDGP+aAgwDSf+aAgwDS/+aAgwDTf+aAgwDT/+aAgwDUf+aAgwDU/+aAgwDVf+aAgwDV/+aAgwDWf+aAgwDW/+aAgwDXf+aAgwDX/+aAgwDYf/XAgwDY//XAgwDZf/XAgwDZ//XAgwDaf/XAgwDa//XAgwDbf/XAgwDb/+FAgwDcf+FAgwDc/+FAgwDj/9xAiEBcf/XAiEBcv/sAiEBeP/sAiECVP/sAlMAD//DAlMAEf/DAlMCCP/DAlMCDP/DAlQAD/+FAlQAEf+FAlQBVv+FAlQBX/+FAlQBYv+FAlQBZv/XAlQBaf+FAlQBbf/XAlQBc//DAlQBdv/sAlQBef+aAlQBev+uAlQBe//DAlQBfP/DAlQBff/DAlQBfv+aAlQBgf/DAlQBgv+uAlQBhP/DAlQBhv/DAlQBh//DAlQBif/DAlQBjP+aAlQBjv+aAlQBj/+aAlQBkP+aAlQBkv/DAlQBk/+aAlQBlf/DAlQBlv/DAlQBmP/DAlQBmf+aAlQBmv/DAlQBm//DAlQCCP+FAlQCDP+FAlQCIf/sAlgABf9xAlgACv9xAlgAJv/XAlgAKv/XAlgALQEKAlgAMv/XAlgANP/XAlgAN/9xAlgAOf+uAlgAOv+uAlgAPP+FAlgAif/XAlgAlP/XAlgAlf/XAlgAlv/XAlgAl//XAlgAmP/XAlgAmv/XAlgAn/+FAlgAyP/XAlgAyv/XAlgAzP/XAlgAzv/XAlgA3v/XAlgA4P/XAlgA4v/XAlgA5P/XAlgBDv/XAlgBEP/XAlgBEv/XAlgBFP/XAlgBJP9xAlgBJv9xAlgBNv+uAlgBOP+FAlgBOv+FAlgBR//XAlgB+v+uAlgB/P+uAlgB/v+uAlgCAP+FAlgCB/9xAlgCC/9xAlgCX//XAlgDSf/XAlgDS//XAlgDTf/XAlgDT//XAlgDUf/XAlgDU//XAlgDVf/XAlgDV//XAlgDWf/XAlgDW//XAlgDXf/XAlgDX//XAlgDb/+FAlgDcf+FAlgDc/+FAlgDj/9xAlkABf/sAlkACv/sAlkCB//sAlkCC//sAloAD/+uAloAEf+uAloBVv/XAloBX//XAloBYv/XAloBZP/sAloBaf/XAloBcP/sAloBcf/DAloBcv/sAloBdP/XAloBdf/sAloBeP/sAloBiP/sAloCCP+uAloCDP+uAloCVP/sAmAASQBSAmAAVwBSAmAAWQBmAmAAWgBmAmAAWwBmAmAAXABmAmAAvwBmAmABJQBSAmABJwBSAmABNwBmAmAB+wBmAmAB/QBmAmACNABSAmACNQBSAmACXQBSAmACXgBSAmADcABmAmADjQBSAmADkABSAmIASQBmAmIAVwBmAmIAWQBmAmIAWgBmAmIAWwBmAmIAXABmAmIAvwBmAmIBJQBmAmIBJwBmAmIBNwBmAmIB+wBmAmIB/QBmAmICNABmAmICNQBmAmICXQBmAmICXgBmAmIDcABmAmIDjQBmAmIDkABmAmoABf/sAmoACv/sAmoCB//sAmoCC//sAmwAD/+uAmwAEf+uAmwBnf/sAmwBpP/XAmwBpv/sAmwBqP/XAmwBqv/XAmwBrv/XAmwBsP/XAmwBsf/sAmwBtf/XAmwBvP/DAmwBvf/XAmwBv//XAmwBwf/XAmwBxP/sAmwBx//sAmwBzv/sAmwB1f/sAmwB8v/sAmwCCP+uAmwCDP+uAmwCcv/XAmwCc//sAmwCev/sAmwCfP/XAmwCgP/sAmwCgv/sAmwCn//XAmwCof/sAmwCqf/sAmwCtf/DAmwCt//sAmwCuf/sAmwCu//XAmwCvf/sAmwCv//XAmwCwf/XAmwCyv/XAmwCzv/XAmwCz//sAmwC1P/XAmwC2f/XAmwC2//XAmwC3f/XAmwC5f/XAmwC5//sAmwC9f/sAmwC9//XAmwC+f/XAmwC+//XAmwC/f/XAmwDBf/XAmwDB//XAmwDDf/XAmwDD//XAmwDEf/XAmwDEv/sAmwDF//sAmwDG//XAmwDHP/sAm0AD/+uAm0AEf+uAm0Bzv/XAm0B1f/XAm0B8v/XAm0CCP+uAm0CDP+uAm0Cc//XAm0Cz//XAm0DEv/XAm0DHP/XAm4ABf+uAm4ACv+uAm4Bnf/XAm4Bpv/XAm4BvP+uAm4Bwf+uAm4BxP/XAm4B3P/XAm4B5P/XAm4CB/+uAm4CC/+uAm4CfP+uAm4CgP/DAm4Cgv/DAm4Cqf/XAm4Cqv/XAm4Ctf+uAm4Ctv/XAm4Ct//DAm4Cuf/DAm4Cvf/XAm4Cvv/XAm4Cv/+uAm4Cwf+uAm4C1P+uAm4C/f+uAm4DDf+aAm4DD/+aAm4DF//XAm4DGP/XAm8ABf+FAm8ACv+FAm8B0P/XAm8B3P+aAm8B3f/DAm8B3//XAm8B4f+uAm8B5P+aAm8B9v/DAm8CB/+FAm8CC/+FAm8Cbf/XAm8Cgf/XAm8Cg//XAm8Ci//XAm8CoP/XAm8Cqv+aAm8Ctv+aAm8CuP/DAm8Cuv/DAm8CvP/XAm8Cvv+aAm8CwP+uAm8Cwv+uAm8Cxv/XAm8CyP/XAm8Cy//XAm8C1f+uAm8C5v/XAm8C6v/XAm8C+P/DAm8C+v/DAm8C/P/DAm8C/v+uAm8DBv/XAm8DCP/XAm8DDv+aAm8DEP+aAm8DGP+aAnABn//XAnABuP/XAnABu//XAnABvv/XAnAB4f/XAnACbP/XAnACfv/XAnAChP/XAnAChv/XAnACiP/XAnACiv/XAnACjP/XAnACsf/XAnACs//XAnACwP/XAnACwv/XAnACxf/XAnACx//XAnAC1f/XAnAC7//XAnAC8f/XAnAC8//XAnAC/v/XAnADCf/XAnADC//XAnADDv/XAnADEP/XAnADFf/XAnIABf9xAnIACv9xAnIBnf+aAnIBpv+aAnIBvP9xAnIBvv/XAnIBwf+aAnIBxP+aAnIB3P/XAnIB4f/XAnIB5P/XAnICB/9xAnICC/9xAnICbv/XAnICfP+aAnICgP+uAnICgv+uAnICl//XAnICm//XAnICp//XAnICqf+aAnICqv/XAnICtf9xAnICtv/XAnICt/+FAnICuf+FAnICvf+aAnICvv/XAnICv/+aAnICwP/XAnICwf+aAnICwv/XAnICxf+aAnICx/+aAnIC1P+aAnIC1f/XAnIC4f/XAnIC4//XAnIC/f+aAnIC/v/XAnIDA//XAnIDDf9xAnIDDv/XAnIDD/9xAnIDEP/XAnIDF/+aAnIDGP/XAnMABf9xAnMACv9xAnMBz//XAnMB2P/XAnMB2//XAnMB3P+aAnMB3f/DAnMB3v/XAnMB4f/DAnMB5P+aAnMB6v/XAnMB7f/XAnMB9v/DAnMCB/9xAnMCC/9xAnMCav/XAnMCbf/XAnMCff/sAnMCf//XAnMCgf/XAnMCg//XAnMChf/XAnMCh//XAnMCif/XAnMCi//XAnMCjf/XAnMCqv+aAnMCsv/XAnMCtP/XAnMCtv+aAnMCuP/XAnMCuv/XAnMCvv+aAnMCwP/DAnMCwv/DAnMCxv/XAnMCyP/XAnMC1f/DAnMC4P/XAnMC8P/XAnMC8v/XAnMC9P/XAnMC+P/DAnMC+v/DAnMC/P/DAnMC/v/DAnMDCv/XAnMDDP/XAnMDDv+FAnMDEP+FAnMDFv/XAnMDGP+aAnMDGv/XAnQABf9xAnQACv9xAnQBnf+aAnQBpv+aAnQBvP9xAnQBvv/XAnQBwf+aAnQBxP+aAnQB3P/XAnQB4f/XAnQB5P/XAnQCB/9xAnQCC/9xAnQCbv/XAnQCfP+aAnQCgP+uAnQCgv+uAnQCl//XAnQCm//XAnQCp//XAnQCqf+aAnQCqv/XAnQCtf9xAnQCtv/XAnQCt/+FAnQCuf+FAnQCvf+aAnQCvv/XAnQCv/+aAnQCwP/XAnQCwf+aAnQCwv/XAnQCxf+aAnQCx/+aAnQC1P+aAnQC1f/XAnQC4f/XAnQC4//XAnQC/f+aAnQC/v/XAnQDA//XAnQDDf9xAnQDDv/XAnQDD/9xAnQDEP/XAnQDF/+aAnQDGP/XAnUABf9xAnUACv9xAnUBz//XAnUB2P/XAnUB2//XAnUB3P+aAnUB3f/DAnUB3v/XAnUB4f/DAnUB5P+aAnUB6v/XAnUB7f/XAnUB9v/DAnUCB/9xAnUCC/9xAnUCav/XAnUCbf/XAnUCff/sAnUCf//XAnUCgf/XAnUCg//XAnUChf/XAnUCh//XAnUCif/XAnUCi//XAnUCjf/XAnUCqv+aAnUCsv/XAnUCtP/XAnUCtv+aAnUCuP/XAnUCuv/XAnUCvv+aAnUCwP/DAnUCwv/DAnUCxv/XAnUCyP/XAnUC1f/DAnUC4P/XAnUC8P/XAnUC8v/XAnUC9P/XAnUC+P/DAnUC+v/DAnUC/P/DAnUC/v/DAnUDCv/XAnUDDP/XAnUDDv+FAnUDEP+FAnUDFv/XAnUDGP+aAnUDGv/XAnYDDf/sAnYDD//sAngDDf/sAngDD//sAnoAD/+uAnoAEf+uAnoCCP+uAnoCDP+uAnoCgP/sAnoCgv/sAnoCt//sAnoCuf/sAnoDDf/XAnoDD//XAnwAD/9xAnwAEf9xAnwBpP/DAnwBqv+uAnwBrv/DAnwBtf/DAnwBzv/XAnwB1f/XAnwB8v/XAnwCCP9xAnwCDP9xAnwCcv+uAnwCc//XAnwCzv/DAnwCz//XAnwC2f+uAnwC2/+uAnwC3f+uAnwDCf+uAnwDC/+uAnwDEf/DAnwDEv/XAnwDG//DAnwDHP/XAn0ABf/sAn0ACv/sAn0B0P/XAn0B3P/sAn0B3f/sAn0B3//XAn0B4f/sAn0B5P/sAn0B9v/sAn0CB//sAn0CC//sAn0CoP/XAn0Cqv/sAn0Ctv/sAn0CvP/XAn0Cvv/sAn0CwP/sAn0Cwv/sAn0Cy//XAn0C1f/sAn0C5v/XAn0C+P/sAn0C+v/sAn0C/P/sAn0C/v/sAn0DBv/XAn0DCP/XAn0DDv/sAn0DEP/sAn0DGP/sAn4AD/+uAn4AEf+uAn4Bnf/sAn4BpP/XAn4Bpv/sAn4BqP/XAn4Bqv/XAn4Brv/XAn4BsP/XAn4Bsf/sAn4Btf/XAn4BvP/DAn4Bvf/XAn4Bv//XAn4Bwf/XAn4BxP/sAn4Bx//sAn4Bzv/sAn4B1f/sAn4B8v/sAn4CCP+uAn4CDP+uAn4Ccv/XAn4Cc//sAn4Cev/sAn4CfP/XAn4CgP/sAn4Cgv/sAn4Cn//XAn4Cof/sAn4Cqf/sAn4Ctf/DAn4Ct//sAn4Cuf/sAn4Cu//XAn4Cvf/sAn4Cv//XAn4Cwf/XAn4Cyv/XAn4Czv/XAn4Cz//sAn4C1P/XAn4C2f/XAn4C2//XAn4C3f/XAn4C5f/XAn4C5//sAn4C9f/sAn4C9//XAn4C+f/XAn4C+//XAn4C/f/XAn4DBf/XAn4DB//XAn4DDf/XAn4DD//XAn4DEf/XAn4DEv/sAn4DF//sAn4DG//XAn4DHP/sAn8ABf/sAn8ACv/sAn8B0P/XAn8B3P/sAn8B3f/sAn8B3//XAn8B4f/sAn8B5P/sAn8B9v/sAn8CB//sAn8CC//sAn8CoP/XAn8Cqv/sAn8Ctv/sAn8CvP/XAn8Cvv/sAn8CwP/sAn8Cwv/sAn8Cy//XAn8C1f/sAn8C5v/XAn8C+P/sAn8C+v/sAn8C/P/sAn8C/v/sAn8DBv/XAn8DCP/XAn8DDv/sAn8DEP/sAn8DGP/sAoAAD/+FAoAAEf+FAoABn//sAoABpP+aAoABqv9xAoABrv+aAoABtf+aAoABuP/sAoABu//sAoABvv/DAoAByf/sAoABzv+uAoABz//XAoAB1f+uAoAB2P/XAoAB2//XAoAB3v/XAoAB4f/XAoAB6v/XAoAB6wBmAoAB7f/XAoAB7v/sAoAB8v+uAoAB9ABmAoACCP+FAoACDP+FAoACav/XAoACbP/sAoACcv9xAoACc/+uAoACfv/sAoACf//XAoAChP/sAoAChf/XAoAChv/sAoACh//XAoACiP/sAoACif/XAoACiv/sAoACjP/sAoACjf/XAoACmABmAoACqABmAoACsf/sAoACsv/XAoACs//sAoACtP/XAoACwP/XAoACwv/XAoACxf/XAoACxv/DAoACx//XAoACyP/DAoACzv+aAoACz/+uAoAC1f/XAoAC2f9xAoAC2/9xAoAC3f9xAoAC4P/XAoAC7//sAoAC8P/XAoAC8f/sAoAC8v/XAoAC8//sAoAC9P/XAoAC/v/XAoADCf9xAoADCv/XAoADC/9xAoADDP/XAoADEf+aAoADEv+uAoADFf/sAoADFv/XAoADGv/XAoADG/+aAoADHP+uAoEAD/+uAoEAEf+uAoEBzv/XAoEB1f/XAoEB8v/XAoECCP+uAoECDP+uAoECc//XAoECz//XAoEDEv/XAoEDHP/XAoIAD/+FAoIAEf+FAoIBn//sAoIBpP+aAoIBqv9xAoIBrv+aAoIBtf+aAoIBuP/sAoIBu//sAoIBvv/DAoIByf/sAoIBzv+uAoIBz//XAoIB1f+uAoIB2P/XAoIB2//XAoIB3v/XAoIB4f/XAoIB6v/XAoIB6wBmAoIB7f/XAoIB7v/sAoIB8v+uAoIB9ABmAoICCP+FAoICDP+FAoICav/XAoICbP/sAoICcv9xAoICc/+uAoICfv/sAoICf//XAoIChP/sAoIChf/XAoIChv/sAoICh//XAoICiP/sAoICif/XAoICiv/sAoICjP/sAoICjf/XAoICmABmAoICqABmAoICsf/sAoICsv/XAoICs//sAoICtP/XAoICwP/XAoICwv/XAoICxf/XAoICxv/DAoICx//XAoICyP/DAoICzv+aAoICz/+uAoIC1f/XAoIC2f9xAoIC2/9xAoIC3f9xAoIC4P/XAoIC7//sAoIC8P/XAoIC8f/sAoIC8v/XAoIC8//sAoIC9P/XAoIC/v/XAoIDCf9xAoIDCv/XAoIDC/9xAoIDDP/XAoIDEf+aAoIDEv+uAoIDFf/sAoIDFv/XAoIDGv/XAoIDG/+aAoIDHP+uAoMAD/+uAoMAEf+uAoMBzv/XAoMB1f/XAoMB8v/XAoMCCP+uAoMCDP+uAoMCc//XAoMCz//XAoMDEv/XAoMDHP/XAoQAD/+uAoQAEf+uAoQBzv/XAoQB1f/XAoQB8v/XAoQCCP+uAoQCDP+uAoQCc//XAoQCz//XAoQDEv/XAoQDHP/XAoUAD/+uAoUAEf+uAoUBzv/XAoUB1f/XAoUB8v/XAoUCCP+uAoUCDP+uAoUCc//XAoUCz//XAoUDEv/XAoUDHP/XAoYAD/+uAoYAEf+uAoYBnf/sAoYBpP/XAoYBpv/sAoYBqP/XAoYBqv/XAoYBrv/XAoYBsP/XAoYBsf/sAoYBtf/XAoYBvP/DAoYBvf/XAoYBv//XAoYBwf/XAoYBxP/sAoYBx//sAoYBzv/sAoYB1f/sAoYB8v/sAoYCCP+uAoYCDP+uAoYCcv/XAoYCc//sAoYCev/sAoYCfP/XAoYCgP/sAoYCgv/sAoYCn//XAoYCof/sAoYCqf/sAoYCtf/DAoYCt//sAoYCuf/sAoYCu//XAoYCvf/sAoYCv//XAoYCwf/XAoYCyv/XAoYCzv/XAoYCz//sAoYC1P/XAoYC2f/XAoYC2//XAoYC3f/XAoYC5f/XAoYC5//sAoYC9f/sAoYC9//XAoYC+f/XAoYC+//XAoYC/f/XAoYDBf/XAoYDB//XAoYDDf/XAoYDD//XAoYDEf/XAoYDEv/sAoYDF//sAoYDG//XAoYDHP/sAocABf/sAocACv/sAocB0P/XAocB3P/sAocB3f/sAocB3//XAocB4f/sAocB5P/sAocB9v/sAocCB//sAocCC//sAocCoP/XAocCqv/sAocCtv/sAocCvP/XAocCvv/sAocCwP/sAocCwv/sAocCy//XAocC1f/sAocC5v/XAocC+P/sAocC+v/sAocC/P/sAocC/v/sAocDBv/XAocDCP/XAocDDv/sAocDEP/sAocDGP/sAogAD/+uAogAEf+uAogBnf/sAogBpP/XAogBpv/sAogBqP/XAogBqv/XAogBrv/XAogBsP/XAogBsf/sAogBtf/XAogBvP/DAogBvf/XAogBv//XAogBwf/XAogBxP/sAogBx//sAogBzv/sAogB1f/sAogB8v/sAogCCP+uAogCDP+uAogCcv/XAogCc//sAogCev/sAogCfP/XAogCgP/sAogCgv/sAogCn//XAogCof/sAogCqf/sAogCtf/DAogCt//sAogCuf/sAogCu//XAogCvf/sAogCv//XAogCwf/XAogCyv/XAogCzv/XAogCz//sAogC1P/XAogC2f/XAogC2//XAogC3f/XAogC5f/XAogC5//sAogC9f/sAogC9//XAogC+f/XAogC+//XAogC/f/XAogDBf/XAogDB//XAogDDf/XAogDD//XAogDEf/XAogDEv/sAogDF//sAogDG//XAogDHP/sAokABf/sAokACv/sAokB0P/XAokB3P/sAokB3f/sAokB3//XAokB4f/sAokB5P/sAokB9v/sAokCB//sAokCC//sAokCoP/XAokCqv/sAokCtv/sAokCvP/XAokCvv/sAokCwP/sAokCwv/sAokCy//XAokC1f/sAokC5v/XAokC+P/sAokC+v/sAokC/P/sAokC/v/sAokDBv/XAokDCP/XAokDDv/sAokDEP/sAokDGP/sAooAD/+uAooAEf+uAooBnf/sAooBpP/XAooBpv/sAooBqP/XAooBqv/XAooBrv/XAooBsP/XAooBsf/sAooBtf/XAooBvP/DAooBvf/XAooBv//XAooBwf/XAooBxP/sAooBx//sAooBzv/sAooB1f/sAooB8v/sAooCCP+uAooCDP+uAooCcv/XAooCc//sAooCev/sAooCfP/XAooCgP/sAooCgv/sAooCn//XAooCof/sAooCqf/sAooCtf/DAooCt//sAooCuf/sAooCu//XAooCvf/sAooCv//XAooCwf/XAooCyv/XAooCzv/XAooCz//sAooC1P/XAooC2f/XAooC2//XAooC3f/XAooC5f/XAooC5//sAooC9f/sAooC9//XAooC+f/XAooC+//XAooC/f/XAooDBf/XAooDB//XAooDDf/XAooDD//XAooDEf/XAooDEv/sAooDF//sAooDG//XAooDHP/sAosAD/+uAosAEf+uAosBzv/XAosB1f/XAosB8v/XAosCCP+uAosCDP+uAosCc//XAosCz//XAosDEv/XAosDHP/XAowBn//XAowBuP/XAowBu//XAowBvv/XAowB4f/XAowCbP/XAowCfv/XAowChP/XAowChv/XAowCiP/XAowCiv/XAowCjP/XAowCsf/XAowCs//XAowCwP/XAowCwv/XAowCxf/XAowCx//XAowC1f/XAowC7//XAowC8f/XAowC8//XAowC/v/XAowDCf/XAowDC//XAowDDv/XAowDEP/XAowDFf/XApUBowDhApUC6gApApUDDv/XApUDEP/XApYABf/sApYACv/sApYCB//sApYCC//sApcABf+uApcACv+uApcBnf/XApcBpv/XApcBvP+uApcBwf+uApcBxP/XApcB3P/XApcB5P/XApcCB/+uApcCC/+uApcCfP+uApcCgP/DApcCgv/DApcCqf/XApcCqv/XApcCtf+uApcCtv/XApcCt//DApcCuf/DApcCvf/XApcCvv/XApcCv/+uApcCwf+uApcC1P+uApcC/f+uApcDDf+aApcDD/+aApcDF//XApcDGP/XApgABf+FApgACv+FApgB0P/XApgB3P+aApgB3f/DApgB3//XApgB4f+uApgB5P+aApgB9v/DApgCB/+FApgCC/+FApgCbf/XApgCgf/XApgCg//XApgCi//XApgCoP/XApgCqv+aApgCtv+aApgCuP/DApgCuv/DApgCvP/XApgCvv+aApgCwP+uApgCwv+uApgCxv/XApgCyP/XApgCy//XApgC1f+uApgC5v/XApgC6v/XApgC+P/DApgC+v/DApgC/P/DApgC/v+uApgDBv/XApgDCP/XApgDDv+aApgDEP+aApgDGP+aApkAD/72ApkAEf72ApkBpP+FApkBqv+aApkBrv+FApkBsP/XApkBtf+FApkBv//XApkBzv+aApkB1f+aApkB8v+aApkCCP72ApkCDP72ApkCcv+aApkCc/+aApkCdv/sApkCn//XApkCu//XApkCyv/XApkCzv+FApkCz/+aApkC2f+aApkC2/+aApkC3f+aApkC5f/XApkDBf/XApkDB//XApkDCf+uApkDC/+uApkDEf+FApkDEv+aApkDG/+FApkDHP+aApoABf/sApoACv/sApoB0P/XApoB3P/sApoB3f/sApoB3//XApoB4f/sApoB5P/sApoB9v/sApoCB//sApoCC//sApoCoP/XApoCqv/sApoCtv/sApoCvP/XApoCvv/sApoCwP/sApoCwv/sApoCy//XApoC1f/sApoC5v/XApoC+P/sApoC+v/sApoC/P/sApoC/v/sApoDBv/XApoDCP/XApoDDv/sApoDEP/sApoDGP/sApsAD/+aApsAEP/XApsAEf+aApsBnQApApsBn//XApsBpP+uApsBpgApApsBqv+FApsBrv+uApsBtf+uApsBuP/XApsBu//XApsBvAApApsBvv/DApsBxAApApsBzP/DApsBzf/DApsBzv+aApsBz/+uApsB0P/XApsB0f/XApsB0v/DApsB0//DApsB1P/DApsB1f+aApsB1v/DApsB1//DApsB2P+uApsB2f/DApsB2v/DApsB2/+uApsB3v+uApsB3//XApsB4P/DApsB4f+aApsB4v/DApsB4//DApsB5f/DApsB5v/DApsB5//XApsB6P/DApsB6v+uApsB6wApApsB7P/DApsB7f+uApsB7v/DApsB8v+aApsB8//DApsB9AApApsB9f/DApsB9//DApsB+f/DApsCAv/XApsCA//XApsCBP/XApsCCP+aApsCDP+aApsCav+uApsCa//DApsCbP/XApsCcf/DApsCcv+FApsCc/+aApsCdf/DApsCd//XApsCef/DApsCff/DApsCfv/XApsCf/+uApsChP/XApsChf+uApsChv/XApsCh/+uApsCiP/XApsCif+uApsCiv/XApsCjP/XApsCjf+uApsClv/DApsCmAApApsCmv/DApsCnv/DApsCoP/XApsCov/XApsCpP/DApsCpv/DApsCqAApApsCqQApApsCrP/DApsCrv/DApsCsP/DApsCsf/XApsCsv+uApsCs//XApsCtP+uApsCtQApApsCvP/XApsCvQApApsCwP+aApsCwv+aApsCxP/DApsCxf/XApsCxv/DApsCx//XApsCyP/DApsCy//XApsCzf/DApsCzv+uApsCz/+aApsC0f/DApsC0//DApsC1f+aApsC1//DApsC2f+FApsC2/+FApsC3f+FApsC4P+uApsC5v/XApsC6P/XApsC7P/DApsC7v/DApsC7//XApsC8P+uApsC8f/XApsC8v+uApsC8//XApsC9P+uApsC9v/XApsC/v+aApsDAP/DApsDAv/DApsDBv/XApsDCP/XApsDCf+aApsDCv+uApsDC/+aApsDDP+uApsDDv/XApsDEP/XApsDEf+uApsDEv+aApsDFP/DApsDFf/XApsDFv+uApsDFwApApsDGv+uApsDG/+uApsDHP+aApwAD//DApwAEf/DApwBzv/DApwBz//XApwB1f/DApwB2P/XApwB2//XApwB3v/XApwB6v/XApwB7f/XApwB8v/DApwCCP/DApwCDP/DApwCav/XApwCc//DApwCf//XApwChf/XApwCh//XApwCif/XApwCjf/XApwCsv/XApwCtP/XApwCz//DApwC4P/XApwC8P/XApwC8v/XApwC9P/XApwDCv/XApwDDP/XApwDEv/DApwDFv/XApwDGv/XApwDHP/DAp0ABf/DAp0ACv/DAp0Bnf/DAp0BowBmAp0Bpv/DAp0BvP/DAp0Bwf+uAp0BxP/DAp0B3P/XAp0B4f/XAp0B5P/XAp0CB//DAp0CC//DAp0CfP+uAp0CgP/DAp0Cgv/DAp0Cqf/DAp0Cqv/XAp0Ctf/DAp0Ctv/XAp0Ct//XAp0Cuf/XAp0Cvf/DAp0Cvv/XAp0Cv/+uAp0CwP/XAp0Cwf+uAp0Cwv/XAp0C1P+uAp0C1f/XAp0C/f+uAp0C/v/XAp0DDf/XAp0DDv/DAp0DD//XAp0DEP/DAp0DF//DAp0DGP/XAp4ABf/DAp4ACv/DAp4CB//DAp4CC//DAp4DDv/XAp4DEP/XAp8Bn//XAp8BowDhAp8BuP/XAp8Bu//XAp8Bvv/DAp8B3P/XAp8B4f+uAp8B5P/XAp8CbP/XAp8CewA9Ap8Cff/sAp8Cfv/XAp8ChP/XAp8Chv/XAp8CiP/XAp8Civ/XAp8CjP/XAp8Cqv/XAp8Csf/XAp8Cs//XAp8Ctv/XAp8Cvv/XAp8CwP+uAp8Cwv+uAp8Cxf/DAp8Cxv/XAp8Cx//DAp8CyP/XAp8C1f+uAp8C7//XAp8C8f/XAp8C8//XAp8C/v+uAp8DDv/XAp8DEP/XAp8DFf/XAp8DGP/XAqABz//sAqAB2P/sAqAB2//sAqAB3v/sAqAB4f/sAqAB6v/sAqAB7f/sAqACav/sAqACf//sAqAChf/sAqACh//sAqACif/sAqACjf/sAqACsv/sAqACtP/sAqACwP/sAqACwv/sAqAC1f/sAqAC4P/sAqAC8P/sAqAC8v/sAqAC9P/sAqAC/v/sAqADCv/sAqADDP/sAqADDv/XAqADEP/XAqADFv/sAqADGv/sAqEAD/+uAqEAEf+uAqECCP+uAqECDP+uAqECgP/sAqECgv/sAqECt//sAqECuf/sAqEDDf/XAqEDD//XAqIB6QApAqMBn//XAqMBowDhAqMBuP/XAqMBu//XAqMBvv/DAqMB3P/XAqMB4f+uAqMB5P/XAqMCbP/XAqMCewA9AqMCff/sAqMCfv/XAqMChP/XAqMChv/XAqMCiP/XAqMCiv/XAqMCjP/XAqMCqv/XAqMCsf/XAqMCs//XAqMCtv/XAqMCvv/XAqMCwP+uAqMCwv+uAqMCxf/DAqMCxv/XAqMCx//DAqMCyP/XAqMC1f+uAqMC7//XAqMC8f/XAqMC8//XAqMC/v+uAqMDDv/XAqMDEP/XAqMDFf/XAqMDGP/XAqQBz//sAqQB2P/sAqQB2//sAqQB3v/sAqQB4f/sAqQB6v/sAqQB7f/sAqQCav/sAqQCf//sAqQChf/sAqQCh//sAqQCif/sAqQCjf/sAqQCsv/sAqQCtP/sAqQCwP/sAqQCwv/sAqQC1f/sAqQC4P/sAqQC8P/sAqQC8v/sAqQC9P/sAqQC/v/sAqQDCv/sAqQDDP/sAqQDDv/XAqQDEP/XAqQDFv/sAqQDGv/sAqUBn//XAqUBuP/XAqUBu//XAqUBvv/XAqUBwf/XAqUB4f/XAqUCbP/XAqUCfP/XAqUCfv/XAqUChP/XAqUChv/XAqUCiP/XAqUCiv/XAqUCjP/XAqUCsf/XAqUCs//XAqUCv//XAqUCwP/XAqUCwf/XAqUCwv/XAqUCxf+aAqUCx/+aAqUC1P/XAqUC1f/XAqUC7//XAqUC8f/XAqUC8//XAqUC/f/XAqUC/v/XAqUDCf/XAqUDC//XAqUDDv/XAqUDEP/XAqUDFf/XAqUDGf/sAqYBz//XAqYB2P/XAqYB2//XAqYB3v/XAqYB4f/XAqYB6v/XAqYB7f/XAqYCav/XAqYCf//XAqYChf/XAqYCh//XAqYCif/XAqYCjf/XAqYCsv/XAqYCtP/XAqYCwP/XAqYCwv/XAqYCxv/XAqYCyP/XAqYC1f/XAqYC4P/XAqYC8P/XAqYC8v/XAqYC9P/XAqYC/v/XAqYDCv/XAqYDDP/XAqYDFv/XAqYDGv/XAqcBn//XAqcBuP/XAqcBu//XAqcBvv/XAqcBwf/XAqcB4f/XAqcCbP/XAqcCfP/XAqcCfv/XAqcChP/XAqcChv/XAqcCiP/XAqcCiv/XAqcCjP/XAqcCsf/XAqcCs//XAqcCv//XAqcCwP/XAqcCwf/XAqcCwv/XAqcCxf+aAqcCx/+aAqcC1P/XAqcC1f/XAqcC7//XAqcC8f/XAqcC8//XAqcC/f/XAqcC/v/XAqcDCf/XAqcDC//XAqcDDv/XAqcDEP/XAqcDFf/XAqcDGf/sAqgBz//XAqgB2P/XAqgB2//XAqgB3v/XAqgB4f/XAqgB6v/XAqgB7f/XAqgCav/XAqgCf//XAqgChf/XAqgCh//XAqgCif/XAqgCjf/XAqgCsv/XAqgCtP/XAqgCwP/XAqgCwv/XAqgCxv/XAqgCyP/XAqgC1f/XAqgC4P/XAqgC8P/XAqgC8v/XAqgC9P/XAqgC/v/XAqgDCv/XAqgDDP/XAqgDFv/XAqgDGv/XAqkBn//XAqkBuP/XAqkBu//XAqkBvv/XAqkBwf/XAqkB4f/XAqkCbP/XAqkCfP/XAqkCfv/XAqkChP/XAqkChv/XAqkCiP/XAqkCiv/XAqkCjP/XAqkCsf/XAqkCs//XAqkCv//XAqkCwP/XAqkCwf/XAqkCwv/XAqkCxf+aAqkCx/+aAqkC1P/XAqkC1f/XAqkC7//XAqkC8f/XAqkC8//XAqkC/f/XAqkC/v/XAqkDCf/XAqkDC//XAqkDDv/XAqkDEP/XAqkDFf/XAqkDGf/sAqoBz//XAqoB2P/XAqoB2//XAqoB3v/XAqoB4f/XAqoB6v/XAqoB7f/XAqoCav/XAqoCf//XAqoChf/XAqoCh//XAqoCif/XAqoCjf/XAqoCsv/XAqoCtP/XAqoCwP/XAqoCwv/XAqoCxv/XAqoCyP/XAqoC1f/XAqoC4P/XAqoC8P/XAqoC8v/XAqoC9P/XAqoC/v/XAqoDCv/XAqoDDP/XAqoDFv/XAqoDGv/XAqsBowDhAqsC6gApAqsDDv/XAqsDEP/XAqwABf/sAqwACv/sAqwCB//sAqwCC//sAq0AD/+aAq0AEP/XAq0AEf+aAq0BnQApAq0Bn//XAq0BpP+uAq0BpgApAq0Bqv+FAq0Brv+uAq0Btf+uAq0BuP/XAq0Bu//XAq0BvAApAq0Bvv/DAq0BxAApAq0BzP/DAq0Bzf/DAq0Bzv+aAq0Bz/+uAq0B0P/XAq0B0f/XAq0B0v/DAq0B0//DAq0B1P/DAq0B1f+aAq0B1v/DAq0B1//DAq0B2P+uAq0B2f/DAq0B2v/DAq0B2/+uAq0B3v+uAq0B3//XAq0B4P/DAq0B4f+aAq0B4v/DAq0B4//DAq0B5f/DAq0B5v/DAq0B5//XAq0B6P/DAq0B6v+uAq0B6wApAq0B7P/DAq0B7f+uAq0B7v/DAq0B8v+aAq0B8//DAq0B9AApAq0B9f/DAq0B9//DAq0B+f/DAq0CAv/XAq0CA//XAq0CBP/XAq0CCP+aAq0CDP+aAq0Cav+uAq0Ca//DAq0CbP/XAq0Ccf/DAq0Ccv+FAq0Cc/+aAq0Cdf/DAq0Cd//XAq0Cef/DAq0Cff/DAq0Cfv/XAq0Cf/+uAq0ChP/XAq0Chf+uAq0Chv/XAq0Ch/+uAq0CiP/XAq0Cif+uAq0Civ/XAq0CjP/XAq0Cjf+uAq0Clv/DAq0CmAApAq0Cmv/DAq0Cnv/DAq0CoP/XAq0Cov/XAq0CpP/DAq0Cpv/DAq0CqAApAq0CqQApAq0CrP/DAq0Crv/DAq0CsP/DAq0Csf/XAq0Csv+uAq0Cs//XAq0CtP+uAq0CtQApAq0CvP/XAq0CvQApAq0CwP+aAq0Cwv+aAq0CxP/DAq0Cxf/XAq0Cxv/DAq0Cx//XAq0CyP/DAq0Cy//XAq0Czf/DAq0Czv+uAq0Cz/+aAq0C0f/DAq0C0//DAq0C1f+aAq0C1//DAq0C2f+FAq0C2/+FAq0C3f+FAq0C4P+uAq0C5v/XAq0C6P/XAq0C7P/DAq0C7v/DAq0C7//XAq0C8P+uAq0C8f/XAq0C8v+uAq0C8//XAq0C9P+uAq0C9v/XAq0C/v+aAq0DAP/DAq0DAv/DAq0DBv/XAq0DCP/XAq0DCf+aAq0DCv+uAq0DC/+aAq0DDP+uAq0DDv/XAq0DEP/XAq0DEf+uAq0DEv+aAq0DFP/DAq0DFf/XAq0DFv+uAq0DFwApAq0DGv+uAq0DG/+uAq0DHP+aAq4AD/+aAq4AEP/XAq4AEf+aAq4Bzv/DAq4Bz//sAq4B1f/DAq4B2P/sAq4B2//sAq4B3v/sAq4B6v/sAq4B7f/sAq4B8v/DAq4CAv/XAq4CA//XAq4CBP/XAq4CCP+aAq4CDP+aAq4Cav/sAq4Cc//DAq4Cf//sAq4Chf/sAq4Ch//sAq4Cif/sAq4Cjf/sAq4Csv/sAq4CtP/sAq4Cz//DAq4C4P/sAq4C8P/sAq4C8v/sAq4C9P/sAq4DCv/sAq4DDP/sAq4DEv/DAq4DFv/sAq4DGv/sAq4DHP/DAq8ABf9cAq8ACv9cAq8Bnf+aAq8BowBmAq8Bpv+aAq8BvP9IAq8Bwf+FAq8BxP+aAq8B3P+uAq8B4f/XAq8B5P+uAq8CB/9cAq8CC/9cAq8CfP+FAq8CgP9xAq8Cgv9xAq8Cqf+aAq8Cqv+uAq8Ctf9IAq8Ctv+uAq8Ct/+aAq8Cuf+aAq8Cvf+aAq8Cvv+uAq8Cv/+FAq8CwP/XAq8Cwf+FAq8Cwv/XAq8Cxf/DAq8Cxv/XAq8Cx//DAq8CyP/XAq8C1P+FAq8C1f/XAq8C/f+FAq8C/v/XAq8DDf9IAq8DDv+uAq8DD/9IAq8DEP+uAq8DF/+aAq8DGP+uArAABf9xArAACv9xArAB3P+aArAB4f/XArAB5P+aArACB/9xArACC/9xArACbf/XArACgf/XArACg//XArACi//XArACqv+aArACtv+aArACuP/XArACuv/XArACvv+aArACwP/XArACwv/XArACxv/XArACyP/XArAC1f/XArAC/v/XArADDv9xArADEP9xArADGP+aArEBnf/XArEBpv/XArEBvP/DArEBxP/XArECgP/sArECgv/sArECqf/XArECtf/DArECt//sArECuf/sArECvf/XArEDDf/XArEDD//XArEDF//XArIABf/sArIACv/sArIB0P/XArIB3P/sArIB3f/sArIB3//XArIB4f/sArIB5P/sArIB9v/sArICB//sArICC//sArICoP/XArICqv/sArICtv/sArICvP/XArICvv/sArICwP/sArICwv/sArICy//XArIC1f/sArIC5v/XArIC+P/sArIC+v/sArIC/P/sArIC/v/sArIDBv/XArIDCP/XArIDDv/sArIDEP/sArIDGP/sArMBn//XArMBuP/XArMBu//XArMBvv/XArMB4f/XArMCbP/XArMCfv/XArMChP/XArMChv/XArMCiP/XArMCiv/XArMCjP/XArMCsf/XArMCs//XArMCwP/XArMCwv/XArMCxf/XArMCx//XArMC1f/XArMC7//XArMC8f/XArMC8//XArMC/v/XArMDCf/XArMDC//XArMDDv/XArMDEP/XArMDFf/XArUAD/+FArUAEP+uArUAEf+FArUBn//XArUBpP+aArUBqv9xArUBrv+aArUBtf+aArUBuP/XArUBu//XArUBvAApArUBvv+uArUBzP+aArUBzf+aArUBzv+FArUBz/9xArUB0P/XArUB0f/XArUB0v+aArUB0/+aArUB1P+aArUB1f+FArUB1v+aArUB1/+aArUB2P9xArUB2f+aArUB2v+aArUB2/9xArUB3P+uArUB3f+uArUB3v9xArUB3//XArUB4P+aArUB4f+aArUB4v+aArUB4/+aArUB5P+uArUB5f+aArUB5v+aArUB5//XArUB6P+aArUB6f/DArUB6v9xArUB7P+aArUB7f9xArUB7v+FArUB8v+FArUB8/+aArUB9f+aArUB9v+uArUB9/+aArUB+f+aArUCAv+uArUCA/+uArUCBP+uArUCCP+FArUCDP+FArUCav9xArUCa/+aArUCbP/XArUCbf/XArUCcf+aArUCcv9xArUCc/+FArUCdf+aArUCd/+aArUCef+aArUCff+aArUCfv/XArUCf/9xArUCgf/XArUCg//XArUChP/XArUChf9xArUChv/XArUCh/9xArUCiP/XArUCif9xArUCiv/XArUCi//XArUCjP/XArUCjf9xArUClv+aArUCmv+aArUCnv+aArUCoP/XArUCov/XArUCpP+aArUCpv+aArUCqv+uArUCrP+aArUCrv+aArUCsP+aArUCsf/XArUCsv9xArUCs//XArUCtP9xArUCtQApArUCtv+uArUCuP+uArUCuv+uArUCvP/XArUCvv+uArUCwP+aArUCwv+aArUCxP+aArUCxf+aArUCxv9xArUCx/+aArUCyP9xArUCy//XArUCzf+aArUCzv+aArUCz/+FArUC0f+aArUC0/+aArUC1f+aArUC1/+aArUC2f9xArUC2/9xArUC3f9xArUC4P9xArUC5v/XArUC6P/XArUC6v/DArUC7P+aArUC7v+aArUC7//XArUC8P9xArUC8f/XArUC8v9xArUC8//XArUC9P9xArUC9v/XArUC+P+uArUC+v+uArUC/P+uArUC/v+aArUDAP+aArUDAv+aArUDBv/XArUDCP/XArUDCf9xArUDCv9xArUDC/9xArUDDP9xArUDDv+aArUDEP+aArUDEf+aArUDEv+FArUDFP+aArUDFf/XArUDFv9xArUDGP+uArUDGv9xArUDG/+aArUDHP+FArYAD/+aArYAEP/XArYAEf+aArYBzv/DArYBz//sArYB1f/DArYB2P/sArYB2//sArYB3v/sArYB6v/sArYB7f/sArYB8v/DArYCAv/XArYCA//XArYCBP/XArYCCP+aArYCDP+aArYCav/sArYCc//DArYCf//sArYChf/sArYCh//sArYCif/sArYCjf/sArYCsv/sArYCtP/sArYCz//DArYC4P/sArYC8P/sArYC8v/sArYC9P/sArYDCv/sArYDDP/sArYDEv/DArYDFv/sArYDGv/sArYDHP/DArcAD/+FArcAEf+FArcBn//XArcBpP+uArcBqv+FArcBrv+uArcBtf+uArcBuP/XArcBu//XArcBvv/DArcByv+uArcBzP/DArcBzf/DArcBzv+aArcBz/+aArcB0v/DArcB0//DArcB1P/DArcB1f+aArcB1v/DArcB1//DArcB2P+aArcB2f/DArcB2v/DArcB2/+aArcB3v+aArcB4P/DArcB4f+uArcB4v/DArcB4//DArcB5f/DArcB5v/DArcB6P/DArcB6f/XArcB6v+aArcB6wApArcB7P/DArcB7f+aArcB7v+uArcB8v+aArcB8//DArcB9AApArcB9f/DArcB9//DArcB+f/DArcCCP+FArcCDP+FArcCav+aArcCa//DArcCbP/XArcCcf/DArcCcv+FArcCc/+aArcCdf/DArcCd//XArcCef/DArcCff/XArcCfv/XArcCf/+aArcChP/XArcChf+aArcChv/XArcCh/+aArcCiP/XArcCif+aArcCiv/XArcCjP/XArcCjf+aArcClv/DArcCmAApArcCmv/DArcCnv/DArcCpP/DArcCpv/DArcCqAApArcCrP/DArcCrv/DArcCsP/DArcCsf/XArcCsv+aArcCs//XArcCtP+aArcCwP+uArcCwv+uArcCxP/DArcCxv+uArcCyP+uArcCzf/DArcCzv+uArcCz/+aArcC0f/DArcC0//DArcC1f+uArcC1//DArcC2f+FArcC2v+uArcC2/+FArcC3P+uArcC3f+FArcC3v+uArcC4P+aArcC4f/sArcC4v+uArcC4//sArcC5P+uArcC7P/DArcC7v/DArcC7//XArcC8P+aArcC8f/XArcC8v+aArcC8//XArcC9P+aArcC/v+uArcDAP/DArcDAv/DArcDCf+uArcDCv+aArcDC/+uArcDDP+aArcDDv/XArcDEP/XArcDEf+uArcDEv+aArcDFP/DArcDFf/XArcDFv+aArcDGf/sArcDGv+aArcDG/+uArcDHP+aArgAD/+uArgAEf+uArgBzv/sArgB1f/sArgB8v/sArgCCP+uArgCDP+uArgCc//sArgCz//sArgDEv/sArgDHP/sArkAD/+FArkAEf+FArkBn//XArkBpP+uArkBqv+FArkBrv+uArkBtf+uArkBuP/XArkBu//XArkBvv/DArkByv+uArkBzP/DArkBzf/DArkBzv+aArkBz/+aArkB0v/DArkB0//DArkB1P/DArkB1f+aArkB1v/DArkB1//DArkB2P+aArkB2f/DArkB2v/DArkB2/+aArkB3v+aArkB4P/DArkB4f+uArkB4v/DArkB4//DArkB5f/DArkB5v/DArkB6P/DArkB6f/XArkB6v+aArkB6wApArkB7P/DArkB7f+aArkB7v+uArkB8v+aArkB8//DArkB9AApArkB9f/DArkB9//DArkB+f/DArkCCP+FArkCDP+FArkCav+aArkCa//DArkCbP/XArkCcf/DArkCcv+FArkCc/+aArkCdf/DArkCd//XArkCef/DArkCff/XArkCfv/XArkCf/+aArkChP/XArkChf+aArkChv/XArkCh/+aArkCiP/XArkCif+aArkCiv/XArkCjP/XArkCjf+aArkClv/DArkCmAApArkCmv/DArkCnv/DArkCpP/DArkCpv/DArkCqAApArkCrP/DArkCrv/DArkCsP/DArkCsf/XArkCsv+aArkCs//XArkCtP+aArkCwP+uArkCwv+uArkCxP/DArkCxv+uArkCyP+uArkCzf/DArkCzv+uArkCz/+aArkC0f/DArkC0//DArkC1f+uArkC1//DArkC2f+FArkC2v+uArkC2/+FArkC3P+uArkC3f+FArkC3v+uArkC4P+aArkC4f/sArkC4v+uArkC4//sArkC5P+uArkC7P/DArkC7v/DArkC7//XArkC8P+aArkC8f/XArkC8v+aArkC8//XArkC9P+aArkC/v+uArkDAP/DArkDAv/DArkDCf+uArkDCv+aArkDC/+uArkDDP+aArkDDv/XArkDEP/XArkDEf+uArkDEv+aArkDFP/DArkDFf/XArkDFv+aArkDGf/sArkDGv+aArkDG/+uArkDHP+aAroAD/+uAroAEf+uAroBzv/sAroB1f/sAroB8v/sAroCCP+uAroCDP+uAroCc//sAroCz//sAroDEv/sAroDHP/sArsBn//XArsBowDhArsBuP/XArsBu//XArsBvv/DArsB3P/XArsB4f+uArsB5P/XArsCbP/XArsCewA9ArsCff/sArsCfv/XArsChP/XArsChv/XArsCiP/XArsCiv/XArsCjP/XArsCqv/XArsCsf/XArsCs//XArsCtv/XArsCvv/XArsCwP+uArsCwv+uArsCxf/DArsCxv/XArsCx//DArsCyP/XArsC1f+uArsC7//XArsC8f/XArsC8//XArsC/v+uArsDDv/XArsDEP/XArsDFf/XArsDGP/XArwBz//sArwB2P/sArwB2//sArwB3v/sArwB4f/sArwB6v/sArwB7f/sArwCav/sArwCf//sArwChf/sArwCh//sArwCif/sArwCjf/sArwCsv/sArwCtP/sArwCwP/sArwCwv/sArwC1f/sArwC4P/sArwC8P/sArwC8v/sArwC9P/sArwC/v/sArwDCv/sArwDDP/sArwDDv/XArwDEP/XArwDFv/sArwDGv/sAr0BowDhAr0C6gApAr0DDv/XAr0DEP/XAr4ABf/sAr4ACv/sAr4CB//sAr4CC//sAr8BowDhAr8C6gApAr8DDv/XAr8DEP/XAsAABf/sAsAACv/sAsACB//sAsACC//sAsMABf/DAsMACv/DAsMBnf/XAsMBpv/XAsMBvP+FAsMBwf+uAsMBxP/XAsMB3P/XAsMB3f/sAsMB4f/sAsMB5P/XAsMB9v/sAsMCB//DAsMCC//DAsMCfP+uAsMCgP/DAsMCgv/DAsMCqf/XAsMCqv/XAsMCtf+FAsMCtv/XAsMCt/+aAsMCuf+aAsMCvf/XAsMCvv/XAsMCv/+uAsMCwP/sAsMCwf+uAsMCwv/sAsMC1P+uAsMC1f/sAsMC+P/sAsMC+v/sAsMC/P/sAsMC/f+uAsMC/v/sAsMDDf+uAsMDDv/XAsMDD/+uAsMDEP/XAsMDF//XAsMDGP/XAsQABf+aAsQACv+aAsQB3P/XAsQB3f/XAsQB5P/XAsQB9v/XAsQCB/+aAsQCC/+aAsQCqv/XAsQCtv/XAsQCuP/XAsQCuv/XAsQCvv/XAsQC+P/XAsQC+v/XAsQC/P/XAsQDDv+uAsQDEP+uAsQDGP/XAsUBvP/XAsUCgP/sAsUCgv/sAsUCtf/XAsUCt//sAsUCuf/sAsUDDf/sAsUDD//sAsYABf/sAsYACv/sAsYCB//sAsYCC//sAscBvP/XAscCgP/sAscCgv/sAscCtf/XAscCt//sAscCuf/sAscDDf/sAscDD//sAsgABf/sAsgACv/sAsgCB//sAsgCC//sAsoBn//XAsoBuP/XAsoBu//XAsoBvv/XAsoBwf/XAsoB4f/XAsoCbP/XAsoCfP/XAsoCfv/XAsoChP/XAsoChv/XAsoCiP/XAsoCiv/XAsoCjP/XAsoCsf/XAsoCs//XAsoCv//XAsoCwP/XAsoCwf/XAsoCwv/XAsoCxf+aAsoCx/+aAsoC1P/XAsoC1f/XAsoC7//XAsoC8f/XAsoC8//XAsoC/f/XAsoC/v/XAsoDCf/XAsoDC//XAsoDDv/XAsoDEP/XAsoDFf/XAsoDGf/sAssBz//XAssB2P/XAssB2//XAssB3v/XAssB4f/XAssB6v/XAssB7f/XAssCav/XAssCf//XAssChf/XAssCh//XAssCif/XAssCjf/XAssCsv/XAssCtP/XAssCwP/XAssCwv/XAssCxv/XAssCyP/XAssC1f/XAssC4P/XAssC8P/XAssC8v/XAssC9P/XAssC/v/XAssDCv/XAssDDP/XAssDFv/XAssDGv/XAswABf/DAswACv/DAswBowBmAswBvP/XAswBvv/XAswBwf+uAswB3P/DAswB4f/XAswB5P/DAswCB//DAswCC//DAswCbf/sAswCfP+uAswCgP/XAswCgf/sAswCgv/XAswCg//sAswCi//sAswCqv/DAswCtf/XAswCtv/DAswCt//XAswCuP/sAswCuf/XAswCuv/sAswCvv/DAswCv/+uAswCwP/XAswCwf+uAswCwv/XAswCxf/DAswCxv/XAswCx//DAswCyP/XAswC1P+uAswC1f/XAswC/f+uAswC/v/XAswDDf/XAswDDv/DAswDD//XAswDEP/DAswDGP/DAs0B4f/XAs0CwP/XAs0Cwv/XAs0C1f/XAs0C/v/XAs4BowDhAs4C6gApAs4DDv/XAs4DEP/XAs8ABf/sAs8ACv/sAs8CB//sAs8CC//sAtIBowDhAtIC6gApAtIDDv/XAtIDEP/XAtMABf/sAtMACv/sAtMCB//sAtMCC//sAtYBowDhAtYC6gApAtYDDv/XAtYDEP/XAtcABf/sAtcACv/sAtcCB//sAtcCC//sAtkABf9xAtkACv9xAtkBnf+aAtkBpv+aAtkBvP9xAtkBvv/XAtkBwf+aAtkBxP+aAtkB3P/XAtkB4f/XAtkB5P/XAtkCB/9xAtkCC/9xAtkCbv/XAtkCfP+aAtkCgP+uAtkCgv+uAtkCl//XAtkCm//XAtkCp//XAtkCqf+aAtkCqv/XAtkCtf9xAtkCtv/XAtkCt/+FAtkCuf+FAtkCvf+aAtkCvv/XAtkCv/+aAtkCwP/XAtkCwf+aAtkCwv/XAtkCxf+aAtkCx/+aAtkC1P+aAtkC1f/XAtkC4f/XAtkC4//XAtkC/f+aAtkC/v/XAtkDA//XAtkDDf9xAtkDDv/XAtkDD/9xAtkDEP/XAtkDF/+aAtkDGP/XAtoABf/sAtoACv/sAtoCB//sAtoCC//sAtsABf9xAtsACv9xAtsBnf+aAtsBpv+aAtsBvP9xAtsBvv/XAtsBwf+aAtsBxP+aAtsB3P/XAtsB4f/XAtsB5P/XAtsCB/9xAtsCC/9xAtsCbv/XAtsCfP+aAtsCgP+uAtsCgv+uAtsCl//XAtsCm//XAtsCp//XAtsCqf+aAtsCqv/XAtsCtf9xAtsCtv/XAtsCt/+FAtsCuf+FAtsCvf+aAtsCvv/XAtsCv/+aAtsCwP/XAtsCwf+aAtsCwv/XAtsCxf+aAtsCx/+aAtsC1P+aAtsC1f/XAtsC4f/XAtsC4//XAtsC/f+aAtsC/v/XAtsDA//XAtsDDf9xAtsDDv/XAtsDD/9xAtsDEP/XAtsDF/+aAtsDGP/XAtwABf/sAtwACv/sAtwCB//sAtwCC//sAt4ABf/sAt4ACv/sAt4CB//sAt4CC//sAuAABf/sAuAACv/sAuACB//sAuACC//sAuEAD/+uAuEAEf+uAuEBnf/sAuEBpP/XAuEBpv/sAuEBqP/XAuEBqv/XAuEBrv/XAuEBsP/XAuEBsf/sAuEBtf/XAuEBvP/DAuEBvf/XAuEBv//XAuEBwf/XAuEBxP/sAuEBx//sAuEBzv/sAuEB1f/sAuEB8v/sAuECCP+uAuECDP+uAuECcv/XAuECc//sAuECev/sAuECfP/XAuECgP/sAuECgv/sAuECn//XAuECof/sAuECqf/sAuECtf/DAuECt//sAuECuf/sAuECu//XAuECvf/sAuECv//XAuECwf/XAuECyv/XAuECzv/XAuECz//sAuEC1P/XAuEC2f/XAuEC2//XAuEC3f/XAuEC5f/XAuEC5//sAuEC9f/sAuEC9//XAuEC+f/XAuEC+//XAuEC/f/XAuEDBf/XAuEDB//XAuEDDf/XAuEDD//XAuEDEf/XAuEDEv/sAuEDF//sAuEDG//XAuEDHP/sAuIABf/sAuIACv/sAuIB0P/XAuIB3P/sAuIB3f/sAuIB3//XAuIB4f/sAuIB5P/sAuIB9v/sAuICB//sAuICC//sAuICoP/XAuICqv/sAuICtv/sAuICvP/XAuICvv/sAuICwP/sAuICwv/sAuICy//XAuIC1f/sAuIC5v/XAuIC+P/sAuIC+v/sAuIC/P/sAuIC/v/sAuIDBv/XAuIDCP/XAuIDDv/sAuIDEP/sAuIDGP/sAuMAD/+uAuMAEf+uAuMBnf/sAuMBpP/XAuMBpv/sAuMBqP/XAuMBqv/XAuMBrv/XAuMBsP/XAuMBsf/sAuMBtf/XAuMBvP/DAuMBvf/XAuMBv//XAuMBwf/XAuMBxP/sAuMBx//sAuMBzv/sAuMB1f/sAuMB8v/sAuMCCP+uAuMCDP+uAuMCcv/XAuMCc//sAuMCev/sAuMCfP/XAuMCgP/sAuMCgv/sAuMCn//XAuMCof/sAuMCqf/sAuMCtf/DAuMCt//sAuMCuf/sAuMCu//XAuMCvf/sAuMCv//XAuMCwf/XAuMCyv/XAuMCzv/XAuMCz//sAuMC1P/XAuMC2f/XAuMC2//XAuMC3f/XAuMC5f/XAuMC5//sAuMC9f/sAuMC9//XAuMC+f/XAuMC+//XAuMC/f/XAuMDBf/XAuMDB//XAuMDDf/XAuMDD//XAuMDEf/XAuMDEv/sAuMDF//sAuMDG//XAuMDHP/sAuQABf/sAuQACv/sAuQB0P/XAuQB3P/sAuQB3f/sAuQB3//XAuQB4f/sAuQB5P/sAuQB9v/sAuQCB//sAuQCC//sAuQCoP/XAuQCqv/sAuQCtv/sAuQCvP/XAuQCvv/sAuQCwP/sAuQCwv/sAuQCy//XAuQC1f/sAuQC5v/XAuQC+P/sAuQC+v/sAuQC/P/sAuQC/v/sAuQDBv/XAuQDCP/XAuQDDv/sAuQDEP/sAuQDGP/sAuUBn//XAuUBuP/XAuUBu//XAuUBvv/XAuUBwf/XAuUB4f/XAuUCbP/XAuUCfP/XAuUCfv/XAuUChP/XAuUChv/XAuUCiP/XAuUCiv/XAuUCjP/XAuUCsf/XAuUCs//XAuUCv//XAuUCwP/XAuUCwf/XAuUCwv/XAuUCxf+aAuUCx/+aAuUC1P/XAuUC1f/XAuUC7//XAuUC8f/XAuUC8//XAuUC/f/XAuUC/v/XAuUDCf/XAuUDC//XAuUDDv/XAuUDEP/XAuUDFf/XAuUDGf/sAuYBz//XAuYB2P/XAuYB2//XAuYB3v/XAuYB4f/XAuYB6v/XAuYB7f/XAuYCav/XAuYCf//XAuYChf/XAuYCh//XAuYCif/XAuYCjf/XAuYCsv/XAuYCtP/XAuYCwP/XAuYCwv/XAuYCxv/XAuYCyP/XAuYC1f/XAuYC4P/XAuYC8P/XAuYC8v/XAuYC9P/XAuYC/v/XAuYDCv/XAuYDDP/XAuYDFv/XAuYDGv/XAucAD/+uAucAEf+uAucCCP+uAucCDP+uAucCgP/sAucCgv/sAucCt//sAucCuf/sAucDDf/XAucDD//XAugB6QApAukABf/sAukACv/sAukCB//sAukCC//sAukDDv/XAukDEP/XAu8AD/+uAu8AEf+uAu8Bnf/sAu8BpP/XAu8Bpv/sAu8BqP/XAu8Bqv/XAu8Brv/XAu8BsP/XAu8Bsf/sAu8Btf/XAu8BvP/DAu8Bvf/XAu8Bv//XAu8Bwf/XAu8BxP/sAu8Bx//sAu8Bzv/sAu8B1f/sAu8B8v/sAu8CCP+uAu8CDP+uAu8Ccv/XAu8Cc//sAu8Cev/sAu8CfP/XAu8CgP/sAu8Cgv/sAu8Cn//XAu8Cof/sAu8Cqf/sAu8Ctf/DAu8Ct//sAu8Cuf/sAu8Cu//XAu8Cvf/sAu8Cv//XAu8Cwf/XAu8Cyv/XAu8Czv/XAu8Cz//sAu8C1P/XAu8C2f/XAu8C2//XAu8C3f/XAu8C5f/XAu8C5//sAu8C9f/sAu8C9//XAu8C+f/XAu8C+//XAu8C/f/XAu8DBf/XAu8DB//XAu8DDf/XAu8DD//XAu8DEf/XAu8DEv/sAu8DF//sAu8DG//XAu8DHP/sAvAABf/sAvAACv/sAvAB0P/XAvAB3P/sAvAB3f/sAvAB3//XAvAB4f/sAvAB5P/sAvAB9v/sAvACB//sAvACC//sAvACoP/XAvACqv/sAvACtv/sAvACvP/XAvACvv/sAvACwP/sAvACwv/sAvACy//XAvAC1f/sAvAC5v/XAvAC+P/sAvAC+v/sAvAC/P/sAvAC/v/sAvADBv/XAvADCP/XAvADDv/sAvADEP/sAvADGP/sAvEAD/+uAvEAEf+uAvEBnf/sAvEBpP/XAvEBpv/sAvEBqP/XAvEBqv/XAvEBrv/XAvEBsP/XAvEBsf/sAvEBtf/XAvEBvP/DAvEBvf/XAvEBv//XAvEBwf/XAvEBxP/sAvEBx//sAvEBzv/sAvEB1f/sAvEB8v/sAvECCP+uAvECDP+uAvECcv/XAvECc//sAvECev/sAvECfP/XAvECgP/sAvECgv/sAvECn//XAvECof/sAvECqf/sAvECtf/DAvECt//sAvECuf/sAvECu//XAvECvf/sAvECv//XAvECwf/XAvECyv/XAvECzv/XAvECz//sAvEC1P/XAvEC2f/XAvEC2//XAvEC3f/XAvEC5f/XAvEC5//sAvEC9f/sAvEC9//XAvEC+f/XAvEC+//XAvEC/f/XAvEDBf/XAvEDB//XAvEDDf/XAvEDD//XAvEDEf/XAvEDEv/sAvEDF//sAvEDG//XAvEDHP/sAvIABf/sAvIACv/sAvIB0P/XAvIB3P/sAvIB3f/sAvIB3//XAvIB4f/sAvIB5P/sAvIB9v/sAvICB//sAvICC//sAvICoP/XAvICqv/sAvICtv/sAvICvP/XAvICvv/sAvICwP/sAvICwv/sAvICy//XAvIC1f/sAvIC5v/XAvIC+P/sAvIC+v/sAvIC/P/sAvIC/v/sAvIDBv/XAvIDCP/XAvIDDv/sAvIDEP/sAvIDGP/sAvMAD/+uAvMAEf+uAvMBnf/sAvMBpP/XAvMBpv/sAvMBqP/XAvMBqv/XAvMBrv/XAvMBsP/XAvMBsf/sAvMBtf/XAvMBvP/DAvMBvf/XAvMBv//XAvMBwf/XAvMBxP/sAvMBx//sAvMBzv/sAvMB1f/sAvMB8v/sAvMCCP+uAvMCDP+uAvMCcv/XAvMCc//sAvMCev/sAvMCfP/XAvMCgP/sAvMCgv/sAvMCn//XAvMCof/sAvMCqf/sAvMCtf/DAvMCt//sAvMCuf/sAvMCu//XAvMCvf/sAvMCv//XAvMCwf/XAvMCyv/XAvMCzv/XAvMCz//sAvMC1P/XAvMC2f/XAvMC2//XAvMC3f/XAvMC5f/XAvMC5//sAvMC9f/sAvMC9//XAvMC+f/XAvMC+//XAvMC/f/XAvMDBf/XAvMDB//XAvMDDf/XAvMDD//XAvMDEf/XAvMDEv/sAvMDF//sAvMDG//XAvMDHP/sAvQABf/sAvQACv/sAvQB0P/XAvQB3P/sAvQB3f/sAvQB3//XAvQB4f/sAvQB5P/sAvQB9v/sAvQCB//sAvQCC//sAvQCoP/XAvQCqv/sAvQCtv/sAvQCvP/XAvQCvv/sAvQCwP/sAvQCwv/sAvQCy//XAvQC1f/sAvQC5v/XAvQC+P/sAvQC+v/sAvQC/P/sAvQC/v/sAvQDBv/XAvQDCP/XAvQDDv/sAvQDEP/sAvQDGP/sAvUAD/+uAvUAEf+uAvUBnf/sAvUBpP/XAvUBpv/sAvUBqP/XAvUBqv/XAvUBrv/XAvUBsP/XAvUBsf/sAvUBtf/XAvUBvP/DAvUBvf/XAvUBv//XAvUBwf/XAvUBxP/sAvUBx//sAvUBzv/sAvUB1f/sAvUB8v/sAvUCCP+uAvUCDP+uAvUCcv/XAvUCc//sAvUCev/sAvUCfP/XAvUCgP/sAvUCgv/sAvUCn//XAvUCof/sAvUCqf/sAvUCtf/DAvUCt//sAvUCuf/sAvUCu//XAvUCvf/sAvUCv//XAvUCwf/XAvUCyv/XAvUCzv/XAvUCz//sAvUC1P/XAvUC2f/XAvUC2//XAvUC3f/XAvUC5f/XAvUC5//sAvUC9f/sAvUC9//XAvUC+f/XAvUC+//XAvUC/f/XAvUDBf/XAvUDB//XAvUDDf/XAvUDD//XAvUDEf/XAvUDEv/sAvUDF//sAvUDG//XAvUDHP/sAvYABf/sAvYACv/sAvYB0P/XAvYB3P/sAvYB3f/sAvYB3//XAvYB4f/sAvYB5P/sAvYB9v/sAvYCB//sAvYCC//sAvYCoP/XAvYCqv/sAvYCtv/sAvYCvP/XAvYCvv/sAvYCwP/sAvYCwv/sAvYCy//XAvYC1f/sAvYC5v/XAvYC+P/sAvYC+v/sAvYC/P/sAvYC/v/sAvYDBv/XAvYDCP/XAvYDDv/sAvYDEP/sAvYDGP/sAvcAD/+FAvcAEf+FAvcBn//sAvcBpP+aAvcBqv9xAvcBrv+aAvcBtf+aAvcBuP/sAvcBu//sAvcBvv/DAvcByf/sAvcBzv+uAvcBz//XAvcB1f+uAvcB2P/XAvcB2//XAvcB3v/XAvcB4f/XAvcB6v/XAvcB6wBmAvcB7f/XAvcB7v/sAvcB8v+uAvcB9ABmAvcCCP+FAvcCDP+FAvcCav/XAvcCbP/sAvcCcv9xAvcCc/+uAvcCfv/sAvcCf//XAvcChP/sAvcChf/XAvcChv/sAvcCh//XAvcCiP/sAvcCif/XAvcCiv/sAvcCjP/sAvcCjf/XAvcCmABmAvcCqABmAvcCsf/sAvcCsv/XAvcCs//sAvcCtP/XAvcCwP/XAvcCwv/XAvcCxf/XAvcCxv/DAvcCx//XAvcCyP/DAvcCzv+aAvcCz/+uAvcC1f/XAvcC2f9xAvcC2/9xAvcC3f9xAvcC4P/XAvcC7//sAvcC8P/XAvcC8f/sAvcC8v/XAvcC8//sAvcC9P/XAvcC/v/XAvcDCf9xAvcDCv/XAvcDC/9xAvcDDP/XAvcDEf+aAvcDEv+uAvcDFf/sAvcDFv/XAvcDGv/XAvcDG/+aAvcDHP+uAvgAD/+uAvgAEf+uAvgBzv/XAvgB1f/XAvgB8v/XAvgCCP+uAvgCDP+uAvgCc//XAvgCz//XAvgDEv/XAvgDHP/XAvkAD/+FAvkAEf+FAvkBn//sAvkBpP+aAvkBqv9xAvkBrv+aAvkBtf+aAvkBuP/sAvkBu//sAvkBvv/DAvkByf/sAvkBzv+uAvkBz//XAvkB1f+uAvkB2P/XAvkB2//XAvkB3v/XAvkB4f/XAvkB6v/XAvkB6wBmAvkB7f/XAvkB7v/sAvkB8v+uAvkB9ABmAvkCCP+FAvkCDP+FAvkCav/XAvkCbP/sAvkCcv9xAvkCc/+uAvkCfv/sAvkCf//XAvkChP/sAvkChf/XAvkChv/sAvkCh//XAvkCiP/sAvkCif/XAvkCiv/sAvkCjP/sAvkCjf/XAvkCmABmAvkCqABmAvkCsf/sAvkCsv/XAvkCs//sAvkCtP/XAvkCwP/XAvkCwv/XAvkCxf/XAvkCxv/DAvkCx//XAvkCyP/DAvkCzv+aAvkCz/+uAvkC1f/XAvkC2f9xAvkC2/9xAvkC3f9xAvkC4P/XAvkC7//sAvkC8P/XAvkC8f/sAvkC8v/XAvkC8//sAvkC9P/XAvkC/v/XAvkDCf9xAvkDCv/XAvkDC/9xAvkDDP/XAvkDEf+aAvkDEv+uAvkDFf/sAvkDFv/XAvkDGv/XAvkDG/+aAvkDHP+uAvoAD/+uAvoAEf+uAvoBzv/XAvoB1f/XAvoB8v/XAvoCCP+uAvoCDP+uAvoCc//XAvoCz//XAvoDEv/XAvoDHP/XAvsAD/+FAvsAEf+FAvsBn//sAvsBpP+aAvsBqv9xAvsBrv+aAvsBtf+aAvsBuP/sAvsBu//sAvsBvv/DAvsByf/sAvsBzv+uAvsBz//XAvsB1f+uAvsB2P/XAvsB2//XAvsB3v/XAvsB4f/XAvsB6v/XAvsB6wBmAvsB7f/XAvsB7v/sAvsB8v+uAvsB9ABmAvsCCP+FAvsCDP+FAvsCav/XAvsCbP/sAvsCcv9xAvsCc/+uAvsCfv/sAvsCf//XAvsChP/sAvsChf/XAvsChv/sAvsCh//XAvsCiP/sAvsCif/XAvsCiv/sAvsCjP/sAvsCjf/XAvsCmABmAvsCqABmAvsCsf/sAvsCsv/XAvsCs//sAvsCtP/XAvsCwP/XAvsCwv/XAvsCxf/XAvsCxv/DAvsCx//XAvsCyP/DAvsCzv+aAvsCz/+uAvsC1f/XAvsC2f9xAvsC2/9xAvsC3f9xAvsC4P/XAvsC7//sAvsC8P/XAvsC8f/sAvsC8v/XAvsC8//sAvsC9P/XAvsC/v/XAvsDCf9xAvsDCv/XAvsDC/9xAvsDDP/XAvsDEf+aAvsDEv+uAvsDFf/sAvsDFv/XAvsDGv/XAvsDG/+aAvsDHP+uAvwAD/+uAvwAEf+uAvwBzv/XAvwB1f/XAvwB8v/XAvwCCP+uAvwCDP+uAvwCc//XAvwCz//XAvwDEv/XAvwDHP/XAv8AD/+FAv8AEP+uAv8AEf+FAv8Bn//XAv8BpP+aAv8Bqv9xAv8Brv+aAv8Btf+aAv8BuP/XAv8Bu//XAv8BvAApAv8Bvv+uAv8BzP+aAv8Bzf+aAv8Bzv+FAv8Bz/9xAv8B0P/XAv8B0f/XAv8B0v+aAv8B0/+aAv8B1P+aAv8B1f+FAv8B1v+aAv8B1/+aAv8B2P9xAv8B2f+aAv8B2v+aAv8B2/9xAv8B3P+uAv8B3f+uAv8B3v9xAv8B3//XAv8B4P+aAv8B4f+aAv8B4v+aAv8B4/+aAv8B5P+uAv8B5f+aAv8B5v+aAv8B5//XAv8B6P+aAv8B6f/DAv8B6v9xAv8B7P+aAv8B7f9xAv8B7v+FAv8B8v+FAv8B8/+aAv8B9f+aAv8B9v+uAv8B9/+aAv8B+f+aAv8CAv+uAv8CA/+uAv8CBP+uAv8CCP+FAv8CDP+FAv8Cav9xAv8Ca/+aAv8CbP/XAv8Cbf/XAv8Ccf+aAv8Ccv9xAv8Cc/+FAv8Cdf+aAv8Cd/+aAv8Cef+aAv8Cff+aAv8Cfv/XAv8Cf/9xAv8Cgf/XAv8Cg//XAv8ChP/XAv8Chf9xAv8Chv/XAv8Ch/9xAv8CiP/XAv8Cif9xAv8Civ/XAv8Ci//XAv8CjP/XAv8Cjf9xAv8Clv+aAv8Cmv+aAv8Cnv+aAv8CoP/XAv8Cov/XAv8CpP+aAv8Cpv+aAv8Cqv+uAv8CrP+aAv8Crv+aAv8CsP+aAv8Csf/XAv8Csv9xAv8Cs//XAv8CtP9xAv8CtQApAv8Ctv+uAv8CuP+uAv8Cuv+uAv8CvP/XAv8Cvv+uAv8CwP+aAv8Cwv+aAv8CxP+aAv8Cxf+aAv8Cxv9xAv8Cx/+aAv8CyP9xAv8Cy//XAv8Czf+aAv8Czv+aAv8Cz/+FAv8C0f+aAv8C0/+aAv8C1f+aAv8C1/+aAv8C2f9xAv8C2/9xAv8C3f9xAv8C4P9xAv8C5v/XAv8C6P/XAv8C6v/DAv8C7P+aAv8C7v+aAv8C7//XAv8C8P9xAv8C8f/XAv8C8v9xAv8C8//XAv8C9P9xAv8C9v/XAv8C+P+uAv8C+v+uAv8C/P+uAv8C/v+aAv8DAP+aAv8DAv+aAv8DBv/XAv8DCP/XAv8DCf9xAv8DCv9xAv8DC/9xAv8DDP9xAv8DDv+aAv8DEP+aAv8DEf+aAv8DEv+FAv8DFP+aAv8DFf/XAv8DFv9xAv8DGP+uAv8DGv9xAv8DG/+aAv8DHP+FAwAAD/+aAwAAEP/XAwAAEf+aAwABzv/DAwABz//sAwAB1f/DAwAB2P/sAwAB2//sAwAB3v/sAwAB6v/sAwAB7f/sAwAB8v/DAwACAv/XAwACA//XAwACBP/XAwACCP+aAwACDP+aAwACav/sAwACc//DAwACf//sAwAChf/sAwACh//sAwACif/sAwACjf/sAwACsv/sAwACtP/sAwACz//DAwAC4P/sAwAC8P/sAwAC8v/sAwAC9P/sAwADCv/sAwADDP/sAwADEv/DAwADFv/sAwADGv/sAwADHP/DAwMAD/+aAwMAEP/XAwMAEf+aAwMBnQApAwMBn//XAwMBpP+uAwMBpgApAwMBqv+FAwMBrv+uAwMBtf+uAwMBuP/XAwMBu//XAwMBvAApAwMBvv/DAwMBxAApAwMBzP/DAwMBzf/DAwMBzv+aAwMBz/+uAwMB0P/XAwMB0f/XAwMB0v/DAwMB0//DAwMB1P/DAwMB1f+aAwMB1v/DAwMB1//DAwMB2P+uAwMB2f/DAwMB2v/DAwMB2/+uAwMB3v+uAwMB3//XAwMB4P/DAwMB4f+aAwMB4v/DAwMB4//DAwMB5f/DAwMB5v/DAwMB5//XAwMB6P/DAwMB6v+uAwMB6wApAwMB7P/DAwMB7f+uAwMB7v/DAwMB8v+aAwMB8//DAwMB9AApAwMB9f/DAwMB9//DAwMB+f/DAwMCAv/XAwMCA//XAwMCBP/XAwMCCP+aAwMCDP+aAwMCav+uAwMCa//DAwMCbP/XAwMCcf/DAwMCcv+FAwMCc/+aAwMCdf/DAwMCd//XAwMCef/DAwMCff/DAwMCfv/XAwMCf/+uAwMChP/XAwMChf+uAwMChv/XAwMCh/+uAwMCiP/XAwMCif+uAwMCiv/XAwMCjP/XAwMCjf+uAwMClv/DAwMCmAApAwMCmv/DAwMCnv/DAwMCoP/XAwMCov/XAwMCpP/DAwMCpv/DAwMCqAApAwMCqQApAwMCrP/DAwMCrv/DAwMCsP/DAwMCsf/XAwMCsv+uAwMCs//XAwMCtP+uAwMCtQApAwMCvP/XAwMCvQApAwMCwP+aAwMCwv+aAwMCxP/DAwMCxf/XAwMCxv/DAwMCx//XAwMCyP/DAwMCy//XAwMCzf/DAwMCzv+uAwMCz/+aAwMC0f/DAwMC0//DAwMC1f+aAwMC1//DAwMC2f+FAwMC2/+FAwMC3f+FAwMC4P+uAwMC5v/XAwMC6P/XAwMC7P/DAwMC7v/DAwMC7//XAwMC8P+uAwMC8f/XAwMC8v+uAwMC8//XAwMC9P+uAwMC9v/XAwMC/v+aAwMDAP/DAwMDAv/DAwMDBv/XAwMDCP/XAwMDCf+aAwMDCv+uAwMDC/+aAwMDDP+uAwMDDv/XAwMDEP/XAwMDEf+uAwMDEv+aAwMDFP/DAwMDFf/XAwMDFv+uAwMDFwApAwMDGv+uAwMDG/+uAwMDHP+aAwQAD//DAwQAEf/DAwQBzv/DAwQBz//XAwQB1f/DAwQB2P/XAwQB2//XAwQB3v/XAwQB6v/XAwQB7f/XAwQB8v/DAwQCCP/DAwQCDP/DAwQCav/XAwQCc//DAwQCf//XAwQChf/XAwQCh//XAwQCif/XAwQCjf/XAwQCsv/XAwQCtP/XAwQCz//DAwQC4P/XAwQC8P/XAwQC8v/XAwQC9P/XAwQDCv/XAwQDDP/XAwQDEv/DAwQDFv/XAwQDGv/XAwQDHP/DAwUBn//XAwUBowDhAwUBuP/XAwUBu//XAwUBvv/DAwUB3P/XAwUB4f+uAwUB5P/XAwUCbP/XAwUCewA9AwUCff/sAwUCfv/XAwUChP/XAwUChv/XAwUCiP/XAwUCiv/XAwUCjP/XAwUCqv/XAwUCsf/XAwUCs//XAwUCtv/XAwUCvv/XAwUCwP+uAwUCwv+uAwUCxf/DAwUCxv/XAwUCx//DAwUCyP/XAwUC1f+uAwUC7//XAwUC8f/XAwUC8//XAwUC/v+uAwUDDv/XAwUDEP/XAwUDFf/XAwUDGP/XAwYBz//sAwYB2P/sAwYB2//sAwYB3v/sAwYB4f/sAwYB6v/sAwYB7f/sAwYCav/sAwYCf//sAwYChf/sAwYCh//sAwYCif/sAwYCjf/sAwYCsv/sAwYCtP/sAwYCwP/sAwYCwv/sAwYC1f/sAwYC4P/sAwYC8P/sAwYC8v/sAwYC9P/sAwYC/v/sAwYDCv/sAwYDDP/sAwYDDv/XAwYDEP/XAwYDFv/sAwYDGv/sAwcBn//XAwcBuP/XAwcBu//XAwcBvv/XAwcBwf/XAwcB4f/XAwcCbP/XAwcCfP/XAwcCfv/XAwcChP/XAwcChv/XAwcCiP/XAwcCiv/XAwcCjP/XAwcCsf/XAwcCs//XAwcCv//XAwcCwP/XAwcCwf/XAwcCwv/XAwcCxf+aAwcCx/+aAwcC1P/XAwcC1f/XAwcC7//XAwcC8f/XAwcC8//XAwcC/f/XAwcC/v/XAwcDCf/XAwcDC//XAwcDDv/XAwcDEP/XAwcDFf/XAwcDGf/sAwgBz//sAwgB2P/sAwgB2//sAwgB3v/sAwgB4f/sAwgB6v/sAwgB7f/sAwgCav/sAwgCf//sAwgChf/sAwgCh//sAwgCif/sAwgCjf/sAwgCsv/sAwgCtP/sAwgCwP/sAwgCwv/sAwgC1f/sAwgC4P/sAwgC8P/sAwgC8v/sAwgC9P/sAwgC/v/sAwgDCv/sAwgDDP/sAwgDDv/XAwgDEP/XAwgDFv/sAwgDGv/sAwsABf+aAwsACv+aAwsBnf+uAwsBpv+uAwsBqP/DAwsBqv/DAwsBsP/DAwsBvP9xAwsBvf/DAwsBv//DAwsBwf/DAwsBxP+uAwsB0P/XAwsB3P/DAwsB3//XAwsB4f/XAwsB5P/DAwsCB/+aAwsCC/+aAwsCcv/DAwsCdv/XAwsCfP/DAwsCgP/DAwsCgv/DAwsCn//DAwsCoP/XAwsCqf+uAwsCqv/DAwsCtf9xAwsCtv/DAwsCt//DAwsCuf/DAwsCu//DAwsCvP/XAwsCvf+uAwsCvv/DAwsCv//DAwsCwP/XAwsCwf/DAwsCwv/XAwsCyv/DAwsCy//XAwsC1P/DAwsC1f/XAwsC2f/DAwsC2//DAwsC3f/DAwsC5f/DAwsC5v/XAwsC9//DAwsC+f/DAwsC+//DAwsC/f/DAwsC/v/XAwsDBf/DAwsDBv/XAwsDB//DAwsDCP/XAwsDDf/XAwsDDv/XAwsDD//XAwsDEP/XAwsDF/+uAwsDGP/DAwwABf+aAwwACv+aAwwB0P/XAwwB3P/DAwwB3f/XAwwB3//XAwwB4f/XAwwB5P/DAwwB9v/XAwwCB/+aAwwCC/+aAwwCoP/XAwwCqv/DAwwCtv/DAwwCvP/XAwwCvv/DAwwCwP/XAwwCwv/XAwwCy//XAwwC1f/XAwwC5v/XAwwC+P/XAwwC+v/XAwwC/P/XAwwC/v/XAwwDBv/XAwwDCP/XAwwDDv+aAwwDEP+aAwwDGP/DAw0ABf+aAw0ACv+aAw0Bnf+uAw0Bpv+uAw0BqP/DAw0Bqv/DAw0BsP/DAw0BvP9xAw0Bvf/DAw0Bv//DAw0Bwf/DAw0BxP+uAw0B0P/XAw0B3P/DAw0B3//XAw0B4f/XAw0B5P/DAw0CB/+aAw0CC/+aAw0Ccv/DAw0Cdv/XAw0CfP/DAw0CgP/DAw0Cgv/DAw0Cn//DAw0CoP/XAw0Cqf+uAw0Cqv/DAw0Ctf9xAw0Ctv/DAw0Ct//DAw0Cuf/DAw0Cu//DAw0CvP/XAw0Cvf+uAw0Cvv/DAw0Cv//DAw0CwP/XAw0Cwf/DAw0Cwv/XAw0Cyv/DAw0Cy//XAw0C1P/DAw0C1f/XAw0C2f/DAw0C2//DAw0C3f/DAw0C5f/DAw0C5v/XAw0C9//DAw0C+f/DAw0C+//DAw0C/f/DAw0C/v/XAw0DBf/DAw0DBv/XAw0DB//DAw0DCP/XAw0DDf/XAw0DDv/XAw0DD//XAw0DEP/XAw0DF/+uAw0DGP/DAw4ABf+aAw4ACv+aAw4B0P/XAw4B3P/DAw4B3f/XAw4B3//XAw4B4f/XAw4B5P/DAw4B9v/XAw4CB/+aAw4CC/+aAw4CoP/XAw4Cqv/DAw4Ctv/DAw4CvP/XAw4Cvv/DAw4CwP/XAw4Cwv/XAw4Cy//XAw4C1f/XAw4C5v/XAw4C+P/XAw4C+v/XAw4C/P/XAw4C/v/XAw4DBv/XAw4DCP/XAw4DDv+aAw4DEP+aAw4DGP/DAw8BowDhAw8C6gApAw8DDv/XAw8DEP/XAxAABf/sAxAACv/sAxACB//sAxACC//sAxEABf+aAxEACv+aAxEBnf+uAxEBpv+uAxEBqP/DAxEBqv/DAxEBsP/DAxEBvP9xAxEBvf/DAxEBv//DAxEBwf/DAxEBxP+uAxEB0P/XAxEB3P/DAxEB3//XAxEB4f/XAxEB5P/DAxECB/+aAxECC/+aAxECcv/DAxECdv/XAxECfP/DAxECgP/DAxECgv/DAxECn//DAxECoP/XAxECqf+uAxECqv/DAxECtf9xAxECtv/DAxECt//DAxECuf/DAxECu//DAxECvP/XAxECvf+uAxECvv/DAxECv//DAxECwP/XAxECwf/DAxECwv/XAxECyv/DAxECy//XAxEC1P/DAxEC1f/XAxEC2f/DAxEC2//DAxEC3f/DAxEC5f/DAxEC5v/XAxEC9//DAxEC+f/DAxEC+//DAxEC/f/DAxEC/v/XAxEDBf/DAxEDBv/XAxEDB//DAxEDCP/XAxEDDf/XAxEDDv/XAxEDD//XAxEDEP/XAxEDF/+uAxEDGP/DAxIABf+aAxIACv+aAxIB0P/XAxIB3P/DAxIB3f/XAxIB3//XAxIB4f/XAxIB5P/DAxIB9v/XAxICB/+aAxICC/+aAxICoP/XAxICqv/DAxICtv/DAxICvP/XAxICvv/DAxICwP/XAxICwv/XAxICy//XAxIC1f/XAxIC5v/XAxIC+P/XAxIC+v/XAxIC/P/XAxIC/v/XAxIDBv/XAxIDCP/XAxIDDv+aAxIDEP+aAxIDGP/DAxMABf+aAxMACv+aAxMBnf+uAxMBpv+uAxMBqP/DAxMBqv/DAxMBsP/DAxMBvP9xAxMBvf/DAxMBv//DAxMBwf/DAxMBxP+uAxMB0P/XAxMB3P/DAxMB3//XAxMB4f/XAxMB5P/DAxMCB/+aAxMCC/+aAxMCcv/DAxMCdv/XAxMCfP/DAxMCgP/DAxMCgv/DAxMCn//DAxMCoP/XAxMCqf+uAxMCqv/DAxMCtf9xAxMCtv/DAxMCt//DAxMCuf/DAxMCu//DAxMCvP/XAxMCvf+uAxMCvv/DAxMCv//DAxMCwP/XAxMCwf/DAxMCwv/XAxMCyv/DAxMCy//XAxMC1P/DAxMC1f/XAxMC2f/DAxMC2//DAxMC3f/DAxMC5f/DAxMC5v/XAxMC9//DAxMC+f/DAxMC+//DAxMC/f/DAxMC/v/XAxMDBf/DAxMDBv/XAxMDB//DAxMDCP/XAxMDDf/XAxMDDv/XAxMDD//XAxMDEP/XAxMDF/+uAxMDGP/DAxQABf+aAxQACv+aAxQB0P/XAxQB3P/DAxQB3f/XAxQB3//XAxQB4f/XAxQB5P/DAxQB9v/XAxQCB/+aAxQCC/+aAxQCoP/XAxQCqv/DAxQCtv/DAxQCvP/XAxQCvv/DAxQCwP/XAxQCwv/XAxQCy//XAxQC1f/XAxQC5v/XAxQC+P/XAxQC+v/XAxQC/P/XAxQC/v/XAxQDBv/XAxQDCP/XAxQDDv+aAxQDEP+aAxQDGP/DAxUAD/+uAxUAEf+uAxUBqv/sAxUBsP/XAxUBvP/XAxUBv//XAxUCCP+uAxUCDP+uAxUCcv/sAxUCgP/sAxUCgv/sAxUCn//XAxUCtf/XAxUCt//sAxUCuf/sAxUCu//XAxUCyv/XAxUC2f/sAxUC2//sAxUC3f/sAxUC5f/XAxUDBf/XAxUDB//XAxYABf/XAxYACv/XAxYB0P/sAxYB3f/sAxYB3//sAxYB9v/sAxYCB//XAxYCC//XAxYCoP/sAxYCvP/sAxYCy//sAxYC5v/sAxYC+P/sAxYC+v/sAxYC/P/sAxYDBv/sAxYDCP/sAxYDDv/XAxYDEP/XAxcABf+uAxcACv+uAxcBnf/DAxcBpv/DAxcBqv/XAxcBsP/XAxcBvP/DAxcBv//XAxcBwf/XAxcBxP/DAxcB3P/XAxcB5P/XAxcCB/+uAxcCC/+uAxcCcv/XAxcCfP/XAxcCgP/XAxcCgv/XAxcCn//XAxcCqf/DAxcCqv/XAxcCtf/DAxcCtv/XAxcCt//XAxcCuf/XAxcCu//XAxcCvf/DAxcCvv/XAxcCv//XAxcCwf/XAxcCyv/XAxcC1P/XAxcC2f/XAxcC2//XAxcC3f/XAxcC5f/XAxcC/f/XAxcDBf/XAxcDB//XAxcDDf/XAxcDD//XAxcDF//DAxcDGP/XAxgABf+aAxgACv+aAxgB0P/XAxgB3P/DAxgB3f/XAxgB3//XAxgB4f/XAxgB5P/DAxgB9v/XAxgCB/+aAxgCC/+aAxgCoP/XAxgCqv/DAxgCtv/DAxgCvP/XAxgCvv/DAxgCwP/XAxgCwv/XAxgCy//XAxgC1f/XAxgC5v/XAxgC+P/XAxgC+v/XAxgC/P/XAxgC/v/XAxgDBv/XAxgDCP/XAxgDDv+aAxgDEP+aAxgDGP/DAxkB4f/XAxkCwP/XAxkCwv/XAxkC1f/XAxkC/v/XAxsBowDhAxsC6gApAxsDDv/XAxsDEP/XAxwABf/sAxwACv/sAxwCB//sAxwCC//sAx0ABf9xAx0ACv9xAx0AJv/XAx0AKv/XAx0ALQEKAx0AMv/XAx0ANP/XAx0AN/9xAx0AOf+uAx0AOv+uAx0APP+FAx0Aif/XAx0AlP/XAx0Alf/XAx0Alv/XAx0Al//XAx0AmP/XAx0Amv/XAx0An/+FAx0AyP/XAx0Ayv/XAx0AzP/XAx0Azv/XAx0A3v/XAx0A4P/XAx0A4v/XAx0A5P/XAx0BDv/XAx0BEP/XAx0BEv/XAx0BFP/XAx0BJP9xAx0BJv9xAx0BNv+uAx0BOP+FAx0BOv+FAx0BR//XAx0B+v+uAx0B/P+uAx0B/v+uAx0CAP+FAx0CB/9xAx0CC/9xAx0CX//XAx0DSf/XAx0DS//XAx0DTf/XAx0DT//XAx0DUf/XAx0DU//XAx0DVf/XAx0DV//XAx0DWf/XAx0DW//XAx0DXf/XAx0DX//XAx0Db/+FAx0Dcf+FAx0Dc/+FAx0Dj/9xAx4ABf/sAx4ACv/sAx4CB//sAx4CC//sAx8ABf9xAx8ACv9xAx8AJv/XAx8AKv/XAx8ALQEKAx8AMv/XAx8ANP/XAx8AN/9xAx8AOf+uAx8AOv+uAx8APP+FAx8Aif/XAx8AlP/XAx8Alf/XAx8Alv/XAx8Al//XAx8AmP/XAx8Amv/XAx8An/+FAx8AyP/XAx8Ayv/XAx8AzP/XAx8Azv/XAx8A3v/XAx8A4P/XAx8A4v/XAx8A5P/XAx8BDv/XAx8BEP/XAx8BEv/XAx8BFP/XAx8BJP9xAx8BJv9xAx8BNv+uAx8BOP+FAx8BOv+FAx8BR//XAx8B+v+uAx8B/P+uAx8B/v+uAx8CAP+FAx8CB/9xAx8CC/9xAx8CX//XAx8DSf/XAx8DS//XAx8DTf/XAx8DT//XAx8DUf/XAx8DU//XAx8DVf/XAx8DV//XAx8DWf/XAx8DW//XAx8DXf/XAx8DX//XAx8Db/+FAx8Dcf+FAx8Dc/+FAx8Dj/9xAyAABf/sAyAACv/sAyACB//sAyACC//sAyEABf9xAyEACv9xAyEAJv/XAyEAKv/XAyEALQEKAyEAMv/XAyEANP/XAyEAN/9xAyEAOf+uAyEAOv+uAyEAPP+FAyEAif/XAyEAlP/XAyEAlf/XAyEAlv/XAyEAl//XAyEAmP/XAyEAmv/XAyEAn/+FAyEAyP/XAyEAyv/XAyEAzP/XAyEAzv/XAyEA3v/XAyEA4P/XAyEA4v/XAyEA5P/XAyEBDv/XAyEBEP/XAyEBEv/XAyEBFP/XAyEBJP9xAyEBJv9xAyEBNv+uAyEBOP+FAyEBOv+FAyEBR//XAyEB+v+uAyEB/P+uAyEB/v+uAyECAP+FAyECB/9xAyECC/9xAyECX//XAyEDSf/XAyEDS//XAyEDTf/XAyEDT//XAyEDUf/XAyEDU//XAyEDVf/XAyEDV//XAyEDWf/XAyEDW//XAyEDXf/XAyEDX//XAyEDb/+FAyEDcf+FAyEDc/+FAyEDj/9xAyIABf/sAyIACv/sAyICB//sAyICC//sAyMABf9xAyMACv9xAyMAJv/XAyMAKv/XAyMALQEKAyMAMv/XAyMANP/XAyMAN/9xAyMAOf+uAyMAOv+uAyMAPP+FAyMAif/XAyMAlP/XAyMAlf/XAyMAlv/XAyMAl//XAyMAmP/XAyMAmv/XAyMAn/+FAyMAyP/XAyMAyv/XAyMAzP/XAyMAzv/XAyMA3v/XAyMA4P/XAyMA4v/XAyMA5P/XAyMBDv/XAyMBEP/XAyMBEv/XAyMBFP/XAyMBJP9xAyMBJv9xAyMBNv+uAyMBOP+FAyMBOv+FAyMBR//XAyMB+v+uAyMB/P+uAyMB/v+uAyMCAP+FAyMCB/9xAyMCC/9xAyMCX//XAyMDSf/XAyMDS//XAyMDTf/XAyMDT//XAyMDUf/XAyMDU//XAyMDVf/XAyMDV//XAyMDWf/XAyMDW//XAyMDXf/XAyMDX//XAyMDb/+FAyMDcf+FAyMDc/+FAyMDj/9xAyQABf/sAyQACv/sAyQCB//sAyQCC//sAyUABf9xAyUACv9xAyUAJv/XAyUAKv/XAyUALQEKAyUAMv/XAyUANP/XAyUAN/9xAyUAOf+uAyUAOv+uAyUAPP+FAyUAif/XAyUAlP/XAyUAlf/XAyUAlv/XAyUAl//XAyUAmP/XAyUAmv/XAyUAn/+FAyUAyP/XAyUAyv/XAyUAzP/XAyUAzv/XAyUA3v/XAyUA4P/XAyUA4v/XAyUA5P/XAyUBDv/XAyUBEP/XAyUBEv/XAyUBFP/XAyUBJP9xAyUBJv9xAyUBNv+uAyUBOP+FAyUBOv+FAyUBR//XAyUB+v+uAyUB/P+uAyUB/v+uAyUCAP+FAyUCB/9xAyUCC/9xAyUCX//XAyUDSf/XAyUDS//XAyUDTf/XAyUDT//XAyUDUf/XAyUDU//XAyUDVf/XAyUDV//XAyUDWf/XAyUDW//XAyUDXf/XAyUDX//XAyUDb/+FAyUDcf+FAyUDc/+FAyUDj/9xAyYABf/sAyYACv/sAyYCB//sAyYCC//sAycABf9xAycACv9xAycAJv/XAycAKv/XAycALQEKAycAMv/XAycANP/XAycAN/9xAycAOf+uAycAOv+uAycAPP+FAycAif/XAycAlP/XAycAlf/XAycAlv/XAycAl//XAycAmP/XAycAmv/XAycAn/+FAycAyP/XAycAyv/XAycAzP/XAycAzv/XAycA3v/XAycA4P/XAycA4v/XAycA5P/XAycBDv/XAycBEP/XAycBEv/XAycBFP/XAycBJP9xAycBJv9xAycBNv+uAycBOP+FAycBOv+FAycBR//XAycB+v+uAycB/P+uAycB/v+uAycCAP+FAycCB/9xAycCC/9xAycCX//XAycDSf/XAycDS//XAycDTf/XAycDT//XAycDUf/XAycDU//XAycDVf/XAycDV//XAycDWf/XAycDW//XAycDXf/XAycDX//XAycDb/+FAycDcf+FAycDc/+FAycDj/9xAygABf/sAygACv/sAygCB//sAygCC//sAykABf9xAykACv9xAykAJv/XAykAKv/XAykALQEKAykAMv/XAykANP/XAykAN/9xAykAOf+uAykAOv+uAykAPP+FAykAif/XAykAlP/XAykAlf/XAykAlv/XAykAl//XAykAmP/XAykAmv/XAykAn/+FAykAyP/XAykAyv/XAykAzP/XAykAzv/XAykA3v/XAykA4P/XAykA4v/XAykA5P/XAykBDv/XAykBEP/XAykBEv/XAykBFP/XAykBJP9xAykBJv9xAykBNv+uAykBOP+FAykBOv+FAykBR//XAykB+v+uAykB/P+uAykB/v+uAykCAP+FAykCB/9xAykCC/9xAykCX//XAykDSf/XAykDS//XAykDTf/XAykDT//XAykDUf/XAykDU//XAykDVf/XAykDV//XAykDWf/XAykDW//XAykDXf/XAykDX//XAykDb/+FAykDcf+FAykDc/+FAykDj/9xAyoABf/sAyoACv/sAyoCB//sAyoCC//sAysABf9xAysACv9xAysAJv/XAysAKv/XAysALQEKAysAMv/XAysANP/XAysAN/9xAysAOf+uAysAOv+uAysAPP+FAysAif/XAysAlP/XAysAlf/XAysAlv/XAysAl//XAysAmP/XAysAmv/XAysAn/+FAysAyP/XAysAyv/XAysAzP/XAysAzv/XAysA3v/XAysA4P/XAysA4v/XAysA5P/XAysBDv/XAysBEP/XAysBEv/XAysBFP/XAysBJP9xAysBJv9xAysBNv+uAysBOP+FAysBOv+FAysBR//XAysB+v+uAysB/P+uAysB/v+uAysCAP+FAysCB/9xAysCC/9xAysCX//XAysDSf/XAysDS//XAysDTf/XAysDT//XAysDUf/XAysDU//XAysDVf/XAysDV//XAysDWf/XAysDW//XAysDXf/XAysDX//XAysDb/+FAysDcf+FAysDc/+FAysDj/9xAywABf/sAywACv/sAywCB//sAywCC//sAy0ABf9xAy0ACv9xAy0AJv/XAy0AKv/XAy0ALQEKAy0AMv/XAy0ANP/XAy0AN/9xAy0AOf+uAy0AOv+uAy0APP+FAy0Aif/XAy0AlP/XAy0Alf/XAy0Alv/XAy0Al//XAy0AmP/XAy0Amv/XAy0An/+FAy0AyP/XAy0Ayv/XAy0AzP/XAy0Azv/XAy0A3v/XAy0A4P/XAy0A4v/XAy0A5P/XAy0BDv/XAy0BEP/XAy0BEv/XAy0BFP/XAy0BJP9xAy0BJv9xAy0BNv+uAy0BOP+FAy0BOv+FAy0BR//XAy0B+v+uAy0B/P+uAy0B/v+uAy0CAP+FAy0CB/9xAy0CC/9xAy0CX//XAy0DSf/XAy0DS//XAy0DTf/XAy0DT//XAy0DUf/XAy0DU//XAy0DVf/XAy0DV//XAy0DWf/XAy0DW//XAy0DXf/XAy0DX//XAy0Db/+FAy0Dcf+FAy0Dc/+FAy0Dj/9xAy4ABf/sAy4ACv/sAy4CB//sAy4CC//sAy8ABf9xAy8ACv9xAy8AJv/XAy8AKv/XAy8ALQEKAy8AMv/XAy8ANP/XAy8AN/9xAy8AOf+uAy8AOv+uAy8APP+FAy8Aif/XAy8AlP/XAy8Alf/XAy8Alv/XAy8Al//XAy8AmP/XAy8Amv/XAy8An/+FAy8AyP/XAy8Ayv/XAy8AzP/XAy8Azv/XAy8A3v/XAy8A4P/XAy8A4v/XAy8A5P/XAy8BDv/XAy8BEP/XAy8BEv/XAy8BFP/XAy8BJP9xAy8BJv9xAy8BNv+uAy8BOP+FAy8BOv+FAy8BR//XAy8B+v+uAy8B/P+uAy8B/v+uAy8CAP+FAy8CB/9xAy8CC/9xAy8CX//XAy8DSf/XAy8DS//XAy8DTf/XAy8DT//XAy8DUf/XAy8DU//XAy8DVf/XAy8DV//XAy8DWf/XAy8DW//XAy8DXf/XAy8DX//XAy8Db/+FAy8Dcf+FAy8Dc/+FAy8Dj/9xAzAABf/sAzAACv/sAzACB//sAzACC//sAzEABf9xAzEACv9xAzEAJv/XAzEAKv/XAzEALQEKAzEAMv/XAzEANP/XAzEAN/9xAzEAOf+uAzEAOv+uAzEAPP+FAzEAif/XAzEAlP/XAzEAlf/XAzEAlv/XAzEAl//XAzEAmP/XAzEAmv/XAzEAn/+FAzEAyP/XAzEAyv/XAzEAzP/XAzEAzv/XAzEA3v/XAzEA4P/XAzEA4v/XAzEA5P/XAzEBDv/XAzEBEP/XAzEBEv/XAzEBFP/XAzEBJP9xAzEBJv9xAzEBNv+uAzEBOP+FAzEBOv+FAzEBR//XAzEB+v+uAzEB/P+uAzEB/v+uAzECAP+FAzECB/9xAzECC/9xAzECX//XAzEDSf/XAzEDS//XAzEDTf/XAzEDT//XAzEDUf/XAzEDU//XAzEDVf/XAzEDV//XAzEDWf/XAzEDW//XAzEDXf/XAzEDX//XAzEDb/+FAzEDcf+FAzEDc/+FAzEDj/9xAzIABf/sAzIACv/sAzICB//sAzICC//sAzMABf9xAzMACv9xAzMAJv/XAzMAKv/XAzMALQEKAzMAMv/XAzMANP/XAzMAN/9xAzMAOf+uAzMAOv+uAzMAPP+FAzMAif/XAzMAlP/XAzMAlf/XAzMAlv/XAzMAl//XAzMAmP/XAzMAmv/XAzMAn/+FAzMAyP/XAzMAyv/XAzMAzP/XAzMAzv/XAzMA3v/XAzMA4P/XAzMA4v/XAzMA5P/XAzMBDv/XAzMBEP/XAzMBEv/XAzMBFP/XAzMBJP9xAzMBJv9xAzMBNv+uAzMBOP+FAzMBOv+FAzMBR//XAzMB+v+uAzMB/P+uAzMB/v+uAzMCAP+FAzMCB/9xAzMCC/9xAzMCX//XAzMDSf/XAzMDS//XAzMDTf/XAzMDT//XAzMDUf/XAzMDU//XAzMDVf/XAzMDV//XAzMDWf/XAzMDW//XAzMDXf/XAzMDX//XAzMDb/+FAzMDcf+FAzMDc/+FAzMDj/9xAzQABf/sAzQACv/sAzQCB//sAzQCC//sAzUALQB7AzYABf/sAzYACv/sAzYAWf/XAzYAWv/XAzYAW//XAzYAXP/XAzYAXf/sAzYAv//XAzYBN//XAzYBPP/sAzYBPv/sAzYBQP/sAzYB+//XAzYB/f/XAzYCB//sAzYCC//sAzYDcP/XAzcALQB7AzgABf/sAzgACv/sAzgAWf/XAzgAWv/XAzgAW//XAzgAXP/XAzgAXf/sAzgAv//XAzgBN//XAzgBPP/sAzgBPv/sAzgBQP/sAzgB+//XAzgB/f/XAzgCB//sAzgCC//sAzgDcP/XAzkALQB7AzoABf/sAzoACv/sAzoAWf/XAzoAWv/XAzoAW//XAzoAXP/XAzoAXf/sAzoAv//XAzoBN//XAzoBPP/sAzoBPv/sAzoBQP/sAzoB+//XAzoB/f/XAzoCB//sAzoCC//sAzoDcP/XAzsALQB7AzwABf/sAzwACv/sAzwAWf/XAzwAWv/XAzwAW//XAzwAXP/XAzwAXf/sAzwAv//XAzwBN//XAzwBPP/sAzwBPv/sAzwBQP/sAzwB+//XAzwB/f/XAzwCB//sAzwCC//sAzwDcP/XAz0ALQB7Az4ABf/sAz4ACv/sAz4AWf/XAz4AWv/XAz4AW//XAz4AXP/XAz4AXf/sAz4Av//XAz4BN//XAz4BPP/sAz4BPv/sAz4BQP/sAz4B+//XAz4B/f/XAz4CB//sAz4CC//sAz4DcP/XAz8ALQB7A0AABf/sA0AACv/sA0AAWf/XA0AAWv/XA0AAW//XA0AAXP/XA0AAXf/sA0AAv//XA0ABN//XA0ABPP/sA0ABPv/sA0ABQP/sA0AB+//XA0AB/f/XA0ACB//sA0ACC//sA0ADcP/XA0EALQB7A0IABf/sA0IACv/sA0IAWf/XA0IAWv/XA0IAW//XA0IAXP/XA0IAXf/sA0IAv//XA0IBN//XA0IBPP/sA0IBPv/sA0IBQP/sA0IB+//XA0IB/f/XA0ICB//sA0ICC//sA0IDcP/XA0MALQB7A0QABf/sA0QACv/sA0QAWf/XA0QAWv/XA0QAW//XA0QAXP/XA0QAXf/sA0QAv//XA0QBN//XA0QBPP/sA0QBPv/sA0QBQP/sA0QB+//XA0QB/f/XA0QCB//sA0QCC//sA0QDcP/XA0kAD/+uA0kAEf+uA0kAJP/XA0kAN//DA0kAOf/sA0kAOv/sA0kAO//XA0kAPP/sA0kAPf/sA0kAgv/XA0kAg//XA0kAhP/XA0kAhf/XA0kAhv/XA0kAh//XA0kAn//sA0kAwv/XA0kAxP/XA0kAxv/XA0kBJP/DA0kBJv/DA0kBNv/sA0kBOP/sA0kBOv/sA0kBO//sA0kBPf/sA0kBP//sA0kBQ//XA0kBoP/sA0kB+v/sA0kB/P/sA0kB/v/sA0kCAP/sA0kCCP+uA0kCDP+uA0kCWP/XA0kDHf/XA0kDH//XA0kDIf/XA0kDI//XA0kDJf/XA0kDJ//XA0kDKf/XA0kDK//XA0kDLf/XA0kDL//XA0kDMf/XA0kDM//XA0kDb//sA0kDcf/sA0kDc//sA0kDj//DA0oABf/sA0oACv/sA0oAWf/XA0oAWv/XA0oAW//XA0oAXP/XA0oAXf/sA0oAv//XA0oBN//XA0oBPP/sA0oBPv/sA0oBQP/sA0oB+//XA0oB/f/XA0oCB//sA0oCC//sA0oDcP/XA0sAD/+uA0sAEf+uA0sAJP/XA0sAN//DA0sAOf/sA0sAOv/sA0sAO//XA0sAPP/sA0sAPf/sA0sAgv/XA0sAg//XA0sAhP/XA0sAhf/XA0sAhv/XA0sAh//XA0sAn//sA0sAwv/XA0sAxP/XA0sAxv/XA0sBJP/DA0sBJv/DA0sBNv/sA0sBOP/sA0sBOv/sA0sBO//sA0sBPf/sA0sBP//sA0sBQ//XA0sBoP/sA0sB+v/sA0sB/P/sA0sB/v/sA0sCAP/sA0sCCP+uA0sCDP+uA0sCWP/XA0sDHf/XA0sDH//XA0sDIf/XA0sDI//XA0sDJf/XA0sDJ//XA0sDKf/XA0sDK//XA0sDLf/XA0sDL//XA0sDMf/XA0sDM//XA0sDb//sA0sDcf/sA0sDc//sA0sDj//DA0wABf/sA0wACv/sA0wAWf/XA0wAWv/XA0wAW//XA0wAXP/XA0wAXf/sA0wAv//XA0wBN//XA0wBPP/sA0wBPv/sA0wBQP/sA0wB+//XA0wB/f/XA0wCB//sA0wCC//sA0wDcP/XA00AD/+uA00AEf+uA00AJP/XA00AN//DA00AOf/sA00AOv/sA00AO//XA00APP/sA00APf/sA00Agv/XA00Ag//XA00AhP/XA00Ahf/XA00Ahv/XA00Ah//XA00An//sA00Awv/XA00AxP/XA00Axv/XA00BJP/DA00BJv/DA00BNv/sA00BOP/sA00BOv/sA00BO//sA00BPf/sA00BP//sA00BQ//XA00BoP/sA00B+v/sA00B/P/sA00B/v/sA00CAP/sA00CCP+uA00CDP+uA00CWP/XA00DHf/XA00DH//XA00DIf/XA00DI//XA00DJf/XA00DJ//XA00DKf/XA00DK//XA00DLf/XA00DL//XA00DMf/XA00DM//XA00Db//sA00Dcf/sA00Dc//sA00Dj//DA08AD/+uA08AEf+uA08AJP/XA08AN//DA08AOf/sA08AOv/sA08AO//XA08APP/sA08APf/sA08Agv/XA08Ag//XA08AhP/XA08Ahf/XA08Ahv/XA08Ah//XA08An//sA08Awv/XA08AxP/XA08Axv/XA08BJP/DA08BJv/DA08BNv/sA08BOP/sA08BOv/sA08BO//sA08BPf/sA08BP//sA08BQ//XA08BoP/sA08B+v/sA08B/P/sA08B/v/sA08CAP/sA08CCP+uA08CDP+uA08CWP/XA08DHf/XA08DH//XA08DIf/XA08DI//XA08DJf/XA08DJ//XA08DKf/XA08DK//XA08DLf/XA08DL//XA08DMf/XA08DM//XA08Db//sA08Dcf/sA08Dc//sA08Dj//DA1EAD/+uA1EAEf+uA1EAJP/XA1EAN//DA1EAOf/sA1EAOv/sA1EAO//XA1EAPP/sA1EAPf/sA1EAgv/XA1EAg//XA1EAhP/XA1EAhf/XA1EAhv/XA1EAh//XA1EAn//sA1EAwv/XA1EAxP/XA1EAxv/XA1EBJP/DA1EBJv/DA1EBNv/sA1EBOP/sA1EBOv/sA1EBO//sA1EBPf/sA1EBP//sA1EBQ//XA1EBoP/sA1EB+v/sA1EB/P/sA1EB/v/sA1ECAP/sA1ECCP+uA1ECDP+uA1ECWP/XA1EDHf/XA1EDH//XA1EDIf/XA1EDI//XA1EDJf/XA1EDJ//XA1EDKf/XA1EDK//XA1EDLf/XA1EDL//XA1EDMf/XA1EDM//XA1EDb//sA1EDcf/sA1EDc//sA1EDj//DA1MAD/+uA1MAEf+uA1MAJP/XA1MAN//DA1MAOf/sA1MAOv/sA1MAO//XA1MAPP/sA1MAPf/sA1MAgv/XA1MAg//XA1MAhP/XA1MAhf/XA1MAhv/XA1MAh//XA1MAn//sA1MAwv/XA1MAxP/XA1MAxv/XA1MBJP/DA1MBJv/DA1MBNv/sA1MBOP/sA1MBOv/sA1MBO//sA1MBPf/sA1MBP//sA1MBQ//XA1MBoP/sA1MB+v/sA1MB/P/sA1MB/v/sA1MCAP/sA1MCCP+uA1MCDP+uA1MCWP/XA1MDHf/XA1MDH//XA1MDIf/XA1MDI//XA1MDJf/XA1MDJ//XA1MDKf/XA1MDK//XA1MDLf/XA1MDL//XA1MDMf/XA1MDM//XA1MDb//sA1MDcf/sA1MDc//sA1MDj//DA1UAD/+uA1UAEf+uA1UAJP/XA1UAN//DA1UAOf/sA1UAOv/sA1UAO//XA1UAPP/sA1UAPf/sA1UAgv/XA1UAg//XA1UAhP/XA1UAhf/XA1UAhv/XA1UAh//XA1UAn//sA1UAwv/XA1UAxP/XA1UAxv/XA1UBJP/DA1UBJv/DA1UBNv/sA1UBOP/sA1UBOv/sA1UBO//sA1UBPf/sA1UBP//sA1UBQ//XA1UBoP/sA1UB+v/sA1UB/P/sA1UB/v/sA1UCAP/sA1UCCP+uA1UCDP+uA1UCWP/XA1UDHf/XA1UDH//XA1UDIf/XA1UDI//XA1UDJf/XA1UDJ//XA1UDKf/XA1UDK//XA1UDLf/XA1UDL//XA1UDMf/XA1UDM//XA1UDb//sA1UDcf/sA1UDc//sA1UDj//DA1gASQBSA1gAVwBSA1gAWQBmA1gAWgBmA1gAWwBmA1gAXABmA1gAvwBmA1gBJQBSA1gBJwBSA1gBNwBmA1gB+wBmA1gB/QBmA1gCNABSA1gCNQBSA1gCXQBSA1gCXgBSA1gDcABmA1gDjQBSA1gDkABSA1oASQBSA1oAVwBSA1oAWQBmA1oAWgBmA1oAWwBmA1oAXABmA1oAvwBmA1oBJQBSA1oBJwBSA1oBNwBmA1oB+wBmA1oB/QBmA1oCNABSA1oCNQBSA1oCXQBSA1oCXgBSA1oDcABmA1oDjQBSA1oDkABSA1wASQBSA1wAVwBSA1wAWQBmA1wAWgBmA1wAWwBmA1wAXABmA1wAvwBmA1wBJQBSA1wBJwBSA1wBNwBmA1wB+wBmA1wB/QBmA1wCNABSA1wCNQBSA1wCXQBSA1wCXgBSA1wDcABmA1wDjQBSA1wDkABSA14ASQBSA14AVwBSA14AWQBmA14AWgBmA14AWwBmA14AXABmA14AvwBmA14BJQBSA14BJwBSA14BNwBmA14B+wBmA14B/QBmA14CNABSA14CNQBSA14CXQBSA14CXgBSA14DcABmA14DjQBSA14DkABSA2AASQBSA2AAVwBSA2AAWQBmA2AAWgBmA2AAWwBmA2AAXABmA2AAvwBmA2ABJQBSA2ABJwBSA2ABNwBmA2AB+wBmA2AB/QBmA2ACNABSA2ACNQBSA2ACXQBSA2ACXgBSA2ADcABmA2ADjQBSA2ADkABSA2EAD//XA2EAEf/XA2EAJP/sA2EAgv/sA2EAg//sA2EAhP/sA2EAhf/sA2EAhv/sA2EAh//sA2EAwv/sA2EAxP/sA2EAxv/sA2EBQ//sA2ECCP/XA2ECDP/XA2ECWP/sA2EDHf/sA2EDH//sA2EDIf/sA2EDI//sA2EDJf/sA2EDJ//sA2EDKf/sA2EDK//sA2EDLf/sA2EDL//sA2EDMf/sA2EDM//sA2YASQBmA2YAVwBmA2YAWQBmA2YAWgBmA2YAWwBmA2YAXABmA2YAvwBmA2YBJQBmA2YBJwBmA2YBNwBmA2YB+wBmA2YB/QBmA2YCNABmA2YCNQBmA2YCXQBmA2YCXgBmA2YDcABmA2YDjQBmA2YDkABmA2gASQBmA2gAVwBmA2gAWQBmA2gAWgBmA2gAWwBmA2gAXABmA2gAvwBmA2gBJQBmA2gBJwBmA2gBNwBmA2gB+wBmA2gB/QBmA2gCNABmA2gCNQBmA2gCXQBmA2gCXgBmA2gDcABmA2gDjQBmA2gDkABmA2oASQBmA2oAVwBmA2oAWQBmA2oAWgBmA2oAWwBmA2oAXABmA2oAvwBmA2oBJQBmA2oBJwBmA2oBNwBmA2oB+wBmA2oB/QBmA2oCNABmA2oCNQBmA2oCXQBmA2oCXgBmA2oDcABmA2oDjQBmA2oDkABmA2wASQBmA2wAVwBmA2wAWQBmA2wAWgBmA2wAWwBmA2wAXABmA2wAvwBmA2wBJQBmA2wBJwBmA2wBNwBmA2wB+wBmA2wB/QBmA2wCNABmA2wCNQBmA2wCXQBmA2wCXgBmA2wDcABmA2wDjQBmA2wDkABmA24ASQBmA24AVwBmA24AWQBmA24AWgBmA24AWwBmA24AXABmA24AvwBmA24BJQBmA24BJwBmA24BNwBmA24B+wBmA24B/QBmA24CNABmA24CNQBmA24CXQBmA24CXgBmA24DcABmA24DjQBmA24DkABmA28AD/+FA28AEf+FA28AIgApA28AJP+FA28AJv/XA28AKv/XA28AMv/XA28ANP/XA28ARP+aA28ARv+aA28AR/+aA28ASP+aA28ASv/XA28AUP/DA28AUf/DA28AUv+aA28AU//DA28AVP+aA28AVf/DA28AVv+uA28AWP/DA28AXf/XA28Agv+FA28Ag/+FA28AhP+FA28Ahf+FA28Ahv+FA28Ah/+FA28Aif/XA28AlP/XA28Alf/XA28Alv/XA28Al//XA28AmP/XA28Amv/XA28Aov+aA28Ao/+aA28ApP+aA28Apf+aA28Apv+aA28Ap/+aA28AqP+aA28Aqf+aA28Aqv+aA28Aq/+aA28ArP+aA28Arf+aA28AtP+aA28Atf+aA28Atv+aA28At/+aA28AuP+aA28Auv+aA28Au//DA28AvP/DA28Avf/DA28Avv/DA28Awv+FA28Aw/+aA28AxP+FA28Axf+aA28Axv+FA28Ax/+aA28AyP/XA28Ayf+aA28Ayv/XA28Ay/+aA28AzP/XA28Azf+aA28Azv/XA28Az/+aA28A0f+aA28A0/+aA28A1f+aA28A1/+aA28A2f+aA28A2/+aA28A3f+aA28A3v/XA28A3//XA28A4P/XA28A4f/XA28A4v/XA28A4//XA28A5P/XA28A5f/XA28A+v/DA28BBv/DA28BCP/DA28BDf/DA28BDv/XA28BD/+aA28BEP/XA28BEf+aA28BEv/XA28BE/+aA28BFP/XA28BFf+aA28BF//DA28BGf/DA28BHf+uA28BIf+uA28BK//DA28BLf/DA28BL//DA28BMf/DA28BM//DA28BNf/DA28BPP/XA28BPv/XA28BQP/XA28BQ/+FA28BRP+aA28BRv+aA28BR//XA28BSP+aA28BSv+uA28CCP+FA28CDP+FA28CV//DA28CWP+FA28CWf+aA28CX//XA28CYP+aA28CYv/DA28DHf+FA28DHv+aA28DH/+FA28DIP+aA28DIf+FA28DIv+aA28DI/+FA28DJf+FA28DJv+aA28DJ/+FA28DKP+aA28DKf+FA28DKv+aA28DK/+FA28DLP+aA28DLf+FA28DLv+aA28DL/+FA28DMP+aA28DMf+FA28DMv+aA28DM/+FA28DNP+aA28DNv+aA28DOP+aA28DOv+aA28DPP+aA28DQP+aA28DQv+aA28DRP+aA28DSf/XA28DSv+aA28DS//XA28DTP+aA28DTf/XA28DTv+aA28DT//XA28DUf/XA28DUv+aA28DU//XA28DVP+aA28DVf/XA28DVv+aA28DV//XA28DWP+aA28DWf/XA28DWv+aA28DW//XA28DXP+aA28DXf/XA28DXv+aA28DX//XA28DYP+aA28DYv/DA28DZP/DA28DZv/DA28DaP/DA28Dav/DA28DbP/DA28Dbv/DA3AABQBSA3AACgBSA3AAD/+uA3AAEf+uA3AAIgApA3ACBwBSA3ACCP+uA3ACCwBSA3ACDP+uA3EAD/+FA3EAEf+FA3EAIgApA3EAJP+FA3EAJv/XA3EAKv/XA3EAMv/XA3EANP/XA3EARP+aA3EARv+aA3EAR/+aA3EASP+aA3EASv/XA3EAUP/DA3EAUf/DA3EAUv+aA3EAU//DA3EAVP+aA3EAVf/DA3EAVv+uA3EAWP/DA3EAXf/XA3EAgv+FA3EAg/+FA3EAhP+FA3EAhf+FA3EAhv+FA3EAh/+FA3EAif/XA3EAlP/XA3EAlf/XA3EAlv/XA3EAl//XA3EAmP/XA3EAmv/XA3EAov+aA3EAo/+aA3EApP+aA3EApf+aA3EApv+aA3EAp/+aA3EAqP+aA3EAqf+aA3EAqv+aA3EAq/+aA3EArP+aA3EArf+aA3EAtP+aA3EAtf+aA3EAtv+aA3EAt/+aA3EAuP+aA3EAuv+aA3EAu//DA3EAvP/DA3EAvf/DA3EAvv/DA3EAwv+FA3EAw/+aA3EAxP+FA3EAxf+aA3EAxv+FA3EAx/+aA3EAyP/XA3EAyf+aA3EAyv/XA3EAy/+aA3EAzP/XA3EAzf+aA3EAzv/XA3EAz/+aA3EA0f+aA3EA0/+aA3EA1f+aA3EA1/+aA3EA2f+aA3EA2/+aA3EA3f+aA3EA3v/XA3EA3//XA3EA4P/XA3EA4f/XA3EA4v/XA3EA4//XA3EA5P/XA3EA5f/XA3EA+v/DA3EBBv/DA3EBCP/DA3EBDf/DA3EBDv/XA3EBD/+aA3EBEP/XA3EBEf+aA3EBEv/XA3EBE/+aA3EBFP/XA3EBFf+aA3EBF//DA3EBGf/DA3EBHf+uA3EBIf+uA3EBK//DA3EBLf/DA3EBL//DA3EBMf/DA3EBM//DA3EBNf/DA3EBPP/XA3EBPv/XA3EBQP/XA3EBQ/+FA3EBRP+aA3EBRv+aA3EBR//XA3EBSP+aA3EBSv+uA3ECCP+FA3ECDP+FA3ECV//DA3ECWP+FA3ECWf+aA3ECX//XA3ECYP+aA3ECYv/DA3EDHf+FA3EDHv+aA3EDH/+FA3EDIP+aA3EDIf+FA3EDIv+aA3EDI/+FA3EDJf+FA3EDJv+aA3EDJ/+FA3EDKP+aA3EDKf+FA3EDKv+aA3EDK/+FA3EDLP+aA3EDLf+FA3EDLv+aA3EDL/+FA3EDMP+aA3EDMf+FA3EDMv+aA3EDM/+FA3EDNP+aA3EDNv+aA3EDOP+aA3EDOv+aA3EDPP+aA3EDQP+aA3EDQv+aA3EDRP+aA3EDSf/XA3EDSv+aA3EDS//XA3EDTP+aA3EDTf/XA3EDTv+aA3EDT//XA3EDUf/XA3EDUv+aA3EDU//XA3EDVP+aA3EDVf/XA3EDVv+aA3EDV//XA3EDWP+aA3EDWf/XA3EDWv+aA3EDW//XA3EDXP+aA3EDXf/XA3EDXv+aA3EDX//XA3EDYP+aA3EDYv/DA3EDZP/DA3EDZv/DA3EDaP/DA3EDav/DA3EDbP/DA3EDbv/DA3IABQBSA3IACgBSA3IAD/+uA3IAEf+uA3IAIgApA3ICBwBSA3ICCP+uA3ICCwBSA3ICDP+uA3MAD/+FA3MAEf+FA3MAIgApA3MAJP+FA3MAJv/XA3MAKv/XA3MAMv/XA3MANP/XA3MARP+aA3MARv+aA3MAR/+aA3MASP+aA3MASv/XA3MAUP/DA3MAUf/DA3MAUv+aA3MAU//DA3MAVP+aA3MAVf/DA3MAVv+uA3MAWP/DA3MAXf/XA3MAgv+FA3MAg/+FA3MAhP+FA3MAhf+FA3MAhv+FA3MAh/+FA3MAif/XA3MAlP/XA3MAlf/XA3MAlv/XA3MAl//XA3MAmP/XA3MAmv/XA3MAov+aA3MAo/+aA3MApP+aA3MApf+aA3MApv+aA3MAp/+aA3MAqP+aA3MAqf+aA3MAqv+aA3MAq/+aA3MArP+aA3MArf+aA3MAtP+aA3MAtf+aA3MAtv+aA3MAt/+aA3MAuP+aA3MAuv+aA3MAu//DA3MAvP/DA3MAvf/DA3MAvv/DA3MAwv+FA3MAw/+aA3MAxP+FA3MAxf+aA3MAxv+FA3MAx/+aA3MAyP/XA3MAyf+aA3MAyv/XA3MAy/+aA3MAzP/XA3MAzf+aA3MAzv/XA3MAz/+aA3MA0f+aA3MA0/+aA3MA1f+aA3MA1/+aA3MA2f+aA3MA2/+aA3MA3f+aA3MA3v/XA3MA3//XA3MA4P/XA3MA4f/XA3MA4v/XA3MA4//XA3MA5P/XA3MA5f/XA3MA+v/DA3MBBv/DA3MBCP/DA3MBDf/DA3MBDv/XA3MBD/+aA3MBEP/XA3MBEf+aA3MBEv/XA3MBE/+aA3MBFP/XA3MBFf+aA3MBF//DA3MBGf/DA3MBHf+uA3MBIf+uA3MBK//DA3MBLf/DA3MBL//DA3MBMf/DA3MBM//DA3MBNf/DA3MBPP/XA3MBPv/XA3MBQP/XA3MBQ/+FA3MBRP+aA3MBRv+aA3MBR//XA3MBSP+aA3MBSv+uA3MCCP+FA3MCDP+FA3MCV//DA3MCWP+FA3MCWf+aA3MCX//XA3MCYP+aA3MCYv/DA3MDHf+FA3MDHv+aA3MDH/+FA3MDIP+aA3MDIf+FA3MDIv+aA3MDI/+FA3MDJf+FA3MDJv+aA3MDJ/+FA3MDKP+aA3MDKf+FA3MDKv+aA3MDK/+FA3MDLP+aA3MDLf+FA3MDLv+aA3MDL/+FA3MDMP+aA3MDMf+FA3MDMv+aA3MDM/+FA3MDNP+aA3MDNv+aA3MDOP+aA3MDOv+aA3MDPP+aA3MDQP+aA3MDQv+aA3MDRP+aA3MDSf/XA3MDSv+aA3MDS//XA3MDTP+aA3MDTf/XA3MDTv+aA3MDT//XA3MDUf/XA3MDUv+aA3MDU//XA3MDVP+aA3MDVf/XA3MDVv+aA3MDV//XA3MDWP+aA3MDWf/XA3MDWv+aA3MDW//XA3MDXP+aA3MDXf/XA3MDXv+aA3MDX//XA3MDYP+aA3MDYv/DA3MDZP/DA3MDZv/DA3MDaP/DA3MDav/DA3MDbP/DA3MDbv/DA3QABQBSA3QACgBSA3QAD/+uA3QAEf+uA3QAIgApA3QCBwBSA3QCCP+uA3QCCwBSA3QCDP+uA40ABQB7A40ACgB7A40CBwB7A40CCwB7A48AD/+FA48AEP+uA48AEf+FA48AIgApA48AJP9xA48AJv/XA48AKv/XA48AMv/XA48ANP/XA48ANwApA48ARP9cA48ARv9xA48AR/9xA48ASP9xA48ASv9xA48AUP+aA48AUf+aA48AUv9xA48AU/+aA48AVP9xA48AVf+aA48AVv+FA48AWP+aA48AWf/XA48AWv/XA48AW//XA48AXP/XA48AXf+uA48Agv9xA48Ag/9xA48AhP9xA48Ahf9xA48Ahv9xA48Ah/9xA48Aif/XA48AlP/XA48Alf/XA48Alv/XA48Al//XA48AmP/XA48Amv/XA48Aov9xA48Ao/9cA48ApP9cA48Apf9cA48Apv9cA48Ap/9cA48AqP9cA48Aqf9xA48Aqv9xA48Aq/9xA48ArP9xA48Arf9xA48AtP9xA48Atf9xA48Atv9xA48At/9xA48AuP9xA48Auv9xA48Au/+aA48AvP+aA48Avf+aA48Avv+aA48Av//XA48Awv9xA48Aw/9cA48AxP9xA48Axf9cA48Axv9xA48Ax/9cA48AyP/XA48Ayf9xA48Ayv/XA48Ay/9xA48AzP/XA48Azf9xA48Azv/XA48Az/9xA48A0f9xA48A0/9xA48A1f9xA48A1/9xA48A2f9xA48A2/9xA48A3f9xA48A3v/XA48A3/9xA48A4P/XA48A4f9xA48A4v/XA48A4/9xA48A5P/XA48A5f9xA48A+v+aA48BBv+aA48BCP+aA48BDf+aA48BDv/XA48BD/9xA48BEP/XA48BEf9xA48BEv/XA48BE/9xA48BFP/XA48BFf9xA48BF/+aA48BGf+aA48BHf+FA48BIf+FA48BJAApA48BJgApA48BK/+aA48BLf+aA48BL/+aA48BMf+aA48BM/+aA48BNf+aA48BN//XA48BPP+uA48BPv+uA48BQP+uA48BQ/9xA48BRP9cA48BRv9cA48BR//XA48BSP9xA48BSv+FA48B+//XA48B/f/XA48CAv+uA48CA/+uA48CBP+uA48CCP+FA48CDP+FA48CV/+aA48CWP9xA48CWf9cA48CX//XA48CYP9xA48CYv+aA48DHf9xA48DHv9cA48DH/9xA48DIP9cA48DIf9xA48DIv9cA48DI/9xA48DJf9xA48DJv9cA48DJ/9xA48DKP9cA48DKf9xA48DKv9cA48DK/9xA48DLP9cA48DLf9xA48DLv9cA48DL/9xA48DMP9cA48DMf9xA48DMv9cA48DM/9xA48DNP9cA48DNv9xA48DOP9xA48DOv9xA48DPP9xA48DQP9xA48DQv9xA48DRP9xA48DSf/XA48DSv9xA48DS//XA48DTP9xA48DTf/XA48DTv9xA48DT//XA48DUf/XA48DUv9xA48DU//XA48DVP9xA48DVf/XA48DVv9xA48DV//XA48DWP9xA48DWf/XA48DWv9xA48DW//XA48DXP9xA48DXf/XA48DXv9xA48DX//XA48DYP9xA48DYv+aA48DZP+aA48DZv+aA48DaP+aA48Dav+aA48DbP+aA48Dbv+aA48DcP/XA48DjwApA5AABQApA5AACgApA5ACBwApA5ACCwApAAAAAAAaAT4AAQAAAAAAAAA5AAAAAQAAAAAAAQAJADkAAQAAAAAAAgAHAEIAAQAAAAAAAwAeAEkAAQAAAAAABAAJADkAAQAAAAAABQAMAGcAAQAAAAAABgAIAHMAAQAAAAAABwBSAHsAAQAAAAAACAAUAM0AAQAAAAAACwAcAOEAAQAAAAAADAAuAP0AAQAAAAAADQAuASsAAQAAAAAADgAqAVkAAwABBAkAAAByAYMAAwABBAkAAQASAfUAAwABBAkAAgAOAgcAAwABBAkAAwA8AhUAAwABBAkABAASAfUAAwABBAkABQAYAlEAAwABBAkABgAQAmkAAwABBAkABwCkAnkAAwABBAkACAAoAx0AAwABBAkACwA4A0UAAwABBAkADABcA30AAwABBAkADQBcA9kAAwABBAkADgBUBDVEaWdpdGl6ZWQgZGF0YSBjb3B5cmlnaHQgqSAyMDEwLTIwMTEsIEdvb2dsZSBDb3Jwb3JhdGlvbi5PcGVuIFNhbnNSZWd1bGFyQXNjZW5kZXIgLSBPcGVuIFNhbnMgQnVpbGQgMTAwVmVyc2lvbiAxLjEwT3BlblNhbnNPcGVuIFNhbnMgaXMgYSB0cmFkZW1hcmsgb2YgR29vZ2xlIGFuZCBtYXkgYmUgcmVnaXN0ZXJlZCBpbiBjZXJ0YWluIGp1cmlzZGljdGlvbnMuQXNjZW5kZXIgQ29ycG9yYXRpb25odHRwOi8vd3d3LmFzY2VuZGVyY29ycC5jb20vaHR0cDovL3d3dy5hc2NlbmRlcmNvcnAuY29tL3R5cGVkZXNpZ25lcnMuaHRtbExpY2Vuc2VkIHVuZGVyIHRoZSBBcGFjaGUgTGljZW5zZSwgVmVyc2lvbiAyLjBodHRwOi8vd3d3LmFwYWNoZS5vcmcvbGljZW5zZXMvTElDRU5TRS0yLjAARABpAGcAaQB0AGkAegBlAGQAIABkAGEAdABhACAAYwBvAHAAeQByAGkAZwBoAHQAIACpACAAMgAwADEAMAAtADIAMAAxADEALAAgAEcAbwBvAGcAbABlACAAQwBvAHIAcABvAHIAYQB0AGkAbwBuAC4ATwBwAGUAbgAgAFMAYQBuAHMAUgBlAGcAdQBsAGEAcgBBAHMAYwBlAG4AZABlAHIAIAAtACAATwBwAGUAbgAgAFMAYQBuAHMAIABCAHUAaQBsAGQAIAAxADAAMABWAGUAcgBzAGkAbwBuACAAMQAuADEAMABPAHAAZQBuAFMAYQBuAHMATwBwAGUAbgAgAFMAYQBuAHMAIABpAHMAIABhACAAdAByAGEAZABlAG0AYQByAGsAIABvAGYAIABHAG8AbwBnAGwAZQAgAGEAbgBkACAAbQBhAHkAIABiAGUAIAByAGUAZwBpAHMAdABlAHIAZQBkACAAaQBuACAAYwBlAHIAdABhAGkAbgAgAGoAdQByAGkAcwBkAGkAYwB0AGkAbwBuAHMALgBBAHMAYwBlAG4AZABlAHIAIABDAG8AcgBwAG8AcgBhAHQAaQBvAG4AaAB0AHQAcAA6AC8ALwB3AHcAdwAuAGEAcwBjAGUAbgBkAGUAcgBjAG8AcgBwAC4AYwBvAG0ALwBoAHQAdABwADoALwAvAHcAdwB3AC4AYQBzAGMAZQBuAGQAZQByAGMAbwByAHAALgBjAG8AbQAvAHQAeQBwAGUAZABlAHMAaQBnAG4AZQByAHMALgBoAHQAbQBsAEwAaQBjAGUAbgBzAGUAZAAgAHUAbgBkAGUAcgAgAHQAaABlACAAQQBwAGEAYwBoAGUAIABMAGkAYwBlAG4AcwBlACwAIABWAGUAcgBzAGkAbwBuACAAMgAuADAAaAB0AHQAcAA6AC8ALwB3AHcAdwAuAGEAcABhAGMAaABlAC4AbwByAGcALwBsAGkAYwBlAG4AcwBlAHMALwBMAEkAQwBFAE4AUwBFAC0AMgAuADAAAAIAAAAAAAD/ZgBmAAAAAAAAAAAAAAAAAAAAAAAAAAADqgECAQMBBAEFAQYBBwEIAQkBCgELAQwBDQEOAQ8BEAERARIBEwEUARUBFgEXARgBGQEaARsBHAEdAR4BHwEgASEBIgEjASQBJQEmAScBKAEpASoBKwEsAS0BLgEvATABMQEyATMBNAE1ATYBNwE4ATkBOgE7ATwBPQE+AT8BQAFBAUIBQwFEAUUBRgFHAUgBSQFKAUsBTAFNAU4BTwFQAVEBUgFTAVQBVQFWAVcBWAFZAVoBWwFcAV0BXgFfAWABYQFiAWMBZAFlAWYBZwFoAWkBagFrAWwBbQFuAW8BcAFxAXIBcwF0AXUBdgF3AXgBeQF6AXsBfAF9AX4BfwGAAYEBggGDAYQBhQGGAYcBiAGJAYoBiwGMAY0BjgGPAZABkQGSAZMBlAGVAZYBlwGYAZkBmgGbAZwBnQGeAZ8BoAGhAaIBowGkAaUBpgGnAagBqQGqAasBrAGtAa4BrwGwAbEBsgGzAbQBtQG2AbcBuAG5AboBuwG8Ab0BvgG/AcABwQHCAcMBxAHFAcYBxwHIAckBygHLAcwBzQHOAc8B0AHRAdIB0wHUAdUB1gHXAdgB2QHaAdsB3AHdAd4B3wHgAeEB4gHjAeQB5QHmAecB6AHpAeoB6wHsAe0B7gHvAfAB8QHyAfMB9AH1AfYB9wH4AfkB+gH7AfwB/QH+Af8CAAIBAgICAwIEAgUCBgIHAggCCQIKAgsCDAINAg4CDwIQAhECEgITAhQCFQIWAhcCGAIZAhoCGwIcAh0CHgIfAiACIQIiAiMCJAIlAiYCJwIoAikCKgIrAiwCLQIuAi8CMAIxAjICMwI0AjUCNgI3AjgCOQI6AjsCPAI9Aj4CPwJAAkECQgJDAkQCRQJGAkcCSAJJAkoCSwJMAk0CTgJPAlACUQJSAlMCVAJVAlYCVwJYAlkCWgJbAlwCXQJeAl8CYAJhAmICYwJkAmUCZgJnAmgCaQJqAmsCbAJtAm4CbwJwAnECcgJzAnQCdQJ2AncCeAJ5AnoCewJ8An0CfgJ/AoACgQKCAoMChAKFAoYChwKIAokCigKLAowCjQKOAo8CkAKRApICkwKUApUClgKXApgCmQKaApsCnAKdAp4CnwKgAqECogKjAqQCpQKmAqcCqAKpAqoCqwKsAq0CrgKvArACsQKyArMCtAK1ArYCtwK4ArkCugK7ArwCvQK+Ar8CwALBAsICwwLEAsUCxgLHAsgCyQLKAssCzALNAs4CzwLQAtEC0gLTAtQC1QLWAtcC2ALZAtoC2wLcAt0C3gLfAuAC4QLiAuMC5ALlAuYC5wLoAukC6gLrAuwC7QLuAu8C8ALxAvIC8wL0AvUC9gL3AvgC+QL6AvsC/AL9Av4C/wMAAwEDAgMDAwQDBQMGAwcDCAMJAwoDCwMMAw0DDgMPAxADEQMSAxMDFAMVAxYDFwMYAxkDGgMbAxwDHQMeAx8DIAMhAyIDIwMkAyUDJgMnAygDKQMqAysDLAMtAy4DLwMwAzEDMgMzAzQDNQM2AzcDOAM5AzoDOwM8Az0DPgM/A0ADQQNCA0MDRANFA0YDRwNIA0kDSgNLA0wDTQNOA08DUANRA1IDUwNUA1UDVgNXA1gDWQNaA1sDXANdA14DXwNgA2EDYgNjA2QDZQNmA2cDaANpA2oDawNsA20DbgNvA3ADcQNyA3MDdAN1A3YDdwN4A3kDegN7A3wDfQN+A38DgAOBA4IDgwOEA4UDhgOHA4gDiQOKA4sDjAONA44DjwOQA5EDkgOTA5QDlQOWA5cDmAOZA5oDmwOcA50DngOfA6ADoQOiA6MDpAOlA6YDpwOoA6kDqgOrA6wDrQOuA68DsAOxA7IDswO0A7UDtgO3A7gDuQO6A7sDvAO9A74DvwPAA8EDwgPDA8QDxQPGA8cDyAPJA8oDywPMA80DzgPPA9AD0QPSA9MD1APVA9YD1wPYA9kD2gPbA9wD3QPeA98D4APhA+ID4wPkA+UD5gPnA+gD6QPqA+sD7APtA+4D7wPwA/ED8gPzA/QD9QP2A/cD+AP5A/oD+wP8A/0D/gP/BAAEAQQCBAMEBAQFBAYEBwQIBAkECgQLBAwEDQQOBA8EEAQRBBIEEwQUBBUEFgQXBBgEGQQaBBsEHAQdBB4EHwQgBCEEIgQjBCQEJQQmBCcEKAQpBCoEKwQsBC0ELgQvBDAEMQQyBDMENAQ1BDYENwQ4BDkEOgQ7BDwEPQQ+BD8EQARBBEIEQwREBEUERgRHBEgESQRKBEsETARNBE4ETwRQBFEEUgRTBFQEVQRWBFcEWARZBFoEWwRcBF0EXgRfBGAEYQRiBGMEZARlBGYEZwRoBGkEagRrBGwEbQRuBG8EcARxBHIEcwR0BHUEdgR3BHgEeQR6BHsEfAR9BH4EfwSABIEEggSDBIQEhQSGBIcEiASJBIoEiwSMBI0EjgSPBJAEkQSSBJMElASVBJYElwSYBJkEmgSbBJwEnQSeBJ8EoAShBKIEowSkBKUEpgSnBKgEqQSqBKsHLm5vdGRlZgRudWxsEG5vbm1hcmtpbmdyZXR1cm4Fc3BhY2UGZXhjbGFtCHF1b3RlZGJsCm51bWJlcnNpZ24GZG9sbGFyB3BlcmNlbnQJYW1wZXJzYW5kC3F1b3Rlc2luZ2xlCXBhcmVubGVmdApwYXJlbnJpZ2h0CGFzdGVyaXNrBHBsdXMFY29tbWEGaHlwaGVuBnBlcmlvZAVzbGFzaAR6ZXJvA29uZQN0d28FdGhyZWUEZm91cgRmaXZlA3NpeAVzZXZlbgVlaWdodARuaW5lBWNvbG9uCXNlbWljb2xvbgRsZXNzBWVxdWFsB2dyZWF0ZXIIcXVlc3Rpb24CYXQBQQFCAUMBRAFFAUYBRwFIBUkuYWx0AUoBSwFMAU0BTgFPAVABUQFSAVMBVAFVAVYBVwFYAVkBWgticmFja2V0bGVmdAliYWNrc2xhc2gMYnJhY2tldHJpZ2h0C2FzY2lpY2lyY3VtCnVuZGVyc2NvcmUFZ3JhdmUBYQFiAWMBZAFlAWYBZwFoAWkBagFrAWwBbQFuAW8BcAFxAXIBcwF0AXUBdgF3AXgBeQF6CWJyYWNlbGVmdANiYXIKYnJhY2VyaWdodAphc2NpaXRpbGRlEG5vbmJyZWFraW5nc3BhY2UKZXhjbGFtZG93bgRjZW50CHN0ZXJsaW5nCGN1cnJlbmN5A3llbglicm9rZW5iYXIHc2VjdGlvbghkaWVyZXNpcwljb3B5cmlnaHQLb3JkZmVtaW5pbmUNZ3VpbGxlbW90bGVmdApsb2dpY2Fsbm90B3VuaTAwQUQKcmVnaXN0ZXJlZAlvdmVyc2NvcmUGZGVncmVlCXBsdXNtaW51cwt0d29zdXBlcmlvcg10aHJlZXN1cGVyaW9yBWFjdXRlAm11CXBhcmFncmFwaA5wZXJpb2RjZW50ZXJlZAdjZWRpbGxhC29uZXN1cGVyaW9yDG9yZG1hc2N1bGluZQ5ndWlsbGVtb3RyaWdodApvbmVxdWFydGVyB29uZWhhbGYNdGhyZWVxdWFydGVycwxxdWVzdGlvbmRvd24GQWdyYXZlBkFhY3V0ZQtBY2lyY3VtZmxleAZBdGlsZGUJQWRpZXJlc2lzBUFyaW5nAkFFCENjZWRpbGxhBkVncmF2ZQZFYWN1dGULRWNpcmN1bWZsZXgJRWRpZXJlc2lzCklncmF2ZS5hbHQKSWFjdXRlLmFsdA9JY2lyY3VtZmxleC5hbHQNSWRpZXJlc2lzLmFsdANFdGgGTnRpbGRlBk9ncmF2ZQZPYWN1dGULT2NpcmN1bWZsZXgGT3RpbGRlCU9kaWVyZXNpcwhtdWx0aXBseQZPc2xhc2gGVWdyYXZlBlVhY3V0ZQtVY2lyY3VtZmxleAlVZGllcmVzaXMGWWFjdXRlBVRob3JuCmdlcm1hbmRibHMGYWdyYXZlBmFhY3V0ZQthY2lyY3VtZmxleAZhdGlsZGUJYWRpZXJlc2lzBWFyaW5nAmFlCGNjZWRpbGxhBmVncmF2ZQZlYWN1dGULZWNpcmN1bWZsZXgJZWRpZXJlc2lzBmlncmF2ZQZpYWN1dGULaWNpcmN1bWZsZXgJaWRpZXJlc2lzA2V0aAZudGlsZGUGb2dyYXZlBm9hY3V0ZQtvY2lyY3VtZmxleAZvdGlsZGUJb2RpZXJlc2lzBmRpdmlkZQZvc2xhc2gGdWdyYXZlBnVhY3V0ZQt1Y2lyY3VtZmxleAl1ZGllcmVzaXMGeWFjdXRlBXRob3JuCXlkaWVyZXNpcwdBbWFjcm9uB2FtYWNyb24GQWJyZXZlBmFicmV2ZQdBb2dvbmVrB2FvZ29uZWsGQ2FjdXRlBmNhY3V0ZQtDY2lyY3VtZmxleAtjY2lyY3VtZmxleARDZG90BGNkb3QGQ2Nhcm9uBmNjYXJvbgZEY2Fyb24GZGNhcm9uBkRjcm9hdAZkY3JvYXQHRW1hY3JvbgdlbWFjcm9uBkVicmV2ZQZlYnJldmUKRWRvdGFjY2VudAplZG90YWNjZW50B0VvZ29uZWsHZW9nb25lawZFY2Fyb24GZWNhcm9uC0djaXJjdW1mbGV4C2djaXJjdW1mbGV4BkdicmV2ZQZnYnJldmUER2RvdARnZG90DEdjb21tYWFjY2VudAxnY29tbWFhY2NlbnQLSGNpcmN1bWZsZXgLaGNpcmN1bWZsZXgESGJhcgRoYmFyCkl0aWxkZS5hbHQGaXRpbGRlC0ltYWNyb24uYWx0B2ltYWNyb24KSWJyZXZlLmFsdAZpYnJldmULSW9nb25lay5hbHQHaW9nb25law5JZG90YWNjZW50LmFsdAhkb3RsZXNzaQZJSi5hbHQCaWoLSmNpcmN1bWZsZXgLamNpcmN1bWZsZXgMS2NvbW1hYWNjZW50DGtjb21tYWFjY2VudAxrZ3JlZW5sYW5kaWMGTGFjdXRlBmxhY3V0ZQxMY29tbWFhY2NlbnQMbGNvbW1hYWNjZW50BkxjYXJvbgZsY2Fyb24ETGRvdARsZG90BkxzbGFzaAZsc2xhc2gGTmFjdXRlBm5hY3V0ZQxOY29tbWFhY2NlbnQMbmNvbW1hYWNjZW50Bk5jYXJvbgZuY2Fyb24LbmFwb3N0cm9waGUDRW5nA2VuZwdPbWFjcm9uB29tYWNyb24GT2JyZXZlBm9icmV2ZQ1PaHVuZ2FydW1sYXV0DW9odW5nYXJ1bWxhdXQCT0UCb2UGUmFjdXRlBnJhY3V0ZQxSY29tbWFhY2NlbnQMcmNvbW1hYWNjZW50BlJjYXJvbgZyY2Fyb24GU2FjdXRlBnNhY3V0ZQtTY2lyY3VtZmxleAtzY2lyY3VtZmxleAhTY2VkaWxsYQhzY2VkaWxsYQZTY2Fyb24Gc2Nhcm9uDFRjb21tYWFjY2VudAx0Y29tbWFhY2NlbnQGVGNhcm9uBnRjYXJvbgRUYmFyBHRiYXIGVXRpbGRlBnV0aWxkZQdVbWFjcm9uB3VtYWNyb24GVWJyZXZlBnVicmV2ZQVVcmluZwV1cmluZw1VaHVuZ2FydW1sYXV0DXVodW5nYXJ1bWxhdXQHVW9nb25lawd1b2dvbmVrC1djaXJjdW1mbGV4C3djaXJjdW1mbGV4C1ljaXJjdW1mbGV4C3ljaXJjdW1mbGV4CVlkaWVyZXNpcwZaYWN1dGUGemFjdXRlClpkb3RhY2NlbnQKemRvdGFjY2VudAZaY2Fyb24GemNhcm9uBWxvbmdzBmZsb3JpbgpBcmluZ2FjdXRlCmFyaW5nYWN1dGUHQUVhY3V0ZQdhZWFjdXRlC09zbGFzaGFjdXRlC29zbGFzaGFjdXRlDFNjb21tYWFjY2VudAxzY29tbWFhY2NlbnQKY2lyY3VtZmxleAVjYXJvbgZtYWNyb24FYnJldmUJZG90YWNjZW50BHJpbmcGb2dvbmVrBXRpbGRlDGh1bmdhcnVtbGF1dAV0b25vcw1kaWVyZXNpc3Rvbm9zCkFscGhhdG9ub3MJYW5vdGVsZWlhDEVwc2lsb250b25vcwhFdGF0b25vcw1Jb3RhdG9ub3MuYWx0DE9taWNyb250b25vcwxVcHNpbG9udG9ub3MKT21lZ2F0b25vcxFpb3RhZGllcmVzaXN0b25vcwVBbHBoYQRCZXRhBUdhbW1hB3VuaTAzOTQHRXBzaWxvbgRaZXRhA0V0YQVUaGV0YQhJb3RhLmFsdAVLYXBwYQZMYW1iZGECTXUCTnUCWGkHT21pY3JvbgJQaQNSaG8FU2lnbWEDVGF1B1Vwc2lsb24DUGhpA0NoaQNQc2kHdW5pMDNBORBJb3RhZGllcmVzaXMuYWx0D1Vwc2lsb25kaWVyZXNpcwphbHBoYXRvbm9zDGVwc2lsb250b25vcwhldGF0b25vcwlpb3RhdG9ub3MUdXBzaWxvbmRpZXJlc2lzdG9ub3MFYWxwaGEEYmV0YQVnYW1tYQVkZWx0YQdlcHNpbG9uBHpldGEDZXRhBXRoZXRhBGlvdGEFa2FwcGEGbGFtYmRhB3VuaTAzQkMCbnUCeGkHb21pY3JvbgJwaQNyaG8Gc2lnbWExBXNpZ21hA3RhdQd1cHNpbG9uA3BoaQNjaGkDcHNpBW9tZWdhDGlvdGFkaWVyZXNpcw91cHNpbG9uZGllcmVzaXMMb21pY3JvbnRvbm9zDHVwc2lsb250b25vcwpvbWVnYXRvbm9zCWFmaWkxMDAyMwlhZmlpMTAwNTEJYWZpaTEwMDUyCWFmaWkxMDA1MwlhZmlpMTAwNTQNYWZpaTEwMDU1LmFsdA1hZmlpMTAwNTYuYWx0CWFmaWkxMDA1NwlhZmlpMTAwNTgJYWZpaTEwMDU5CWFmaWkxMDA2MAlhZmlpMTAwNjEJYWZpaTEwMDYyCWFmaWkxMDE0NQlhZmlpMTAwMTcJYWZpaTEwMDE4CWFmaWkxMDAxOQlhZmlpMTAwMjAJYWZpaTEwMDIxCWFmaWkxMDAyMglhZmlpMTAwMjQJYWZpaTEwMDI1CWFmaWkxMDAyNglhZmlpMTAwMjcJYWZpaTEwMDI4CWFmaWkxMDAyOQlhZmlpMTAwMzAJYWZpaTEwMDMxCWFmaWkxMDAzMglhZmlpMTAwMzMJYWZpaTEwMDM0CWFmaWkxMDAzNQlhZmlpMTAwMzYJYWZpaTEwMDM3CWFmaWkxMDAzOAlhZmlpMTAwMzkJYWZpaTEwMDQwCWFmaWkxMDA0MQlhZmlpMTAwNDIJYWZpaTEwMDQzCWFmaWkxMDA0NAlhZmlpMTAwNDUJYWZpaTEwMDQ2CWFmaWkxMDA0NwlhZmlpMTAwNDgJYWZpaTEwMDQ5CWFmaWkxMDA2NQlhZmlpMTAwNjYJYWZpaTEwMDY3CWFmaWkxMDA2OAlhZmlpMTAwNjkJYWZpaTEwMDcwCWFmaWkxMDA3MglhZmlpMTAwNzMJYWZpaTEwMDc0CWFmaWkxMDA3NQlhZmlpMTAwNzYJYWZpaTEwMDc3CWFmaWkxMDA3OAlhZmlpMTAwNzkJYWZpaTEwMDgwCWFmaWkxMDA4MQlhZmlpMTAwODIJYWZpaTEwMDgzCWFmaWkxMDA4NAlhZmlpMTAwODUJYWZpaTEwMDg2CWFmaWkxMDA4NwlhZmlpMTAwODgJYWZpaTEwMDg5CWFmaWkxMDA5MAlhZmlpMTAwOTEJYWZpaTEwMDkyCWFmaWkxMDA5MwlhZmlpMTAwOTQJYWZpaTEwMDk1CWFmaWkxMDA5NglhZmlpMTAwOTcJYWZpaTEwMDcxCWFmaWkxMDA5OQlhZmlpMTAxMDAJYWZpaTEwMTAxCWFmaWkxMDEwMglhZmlpMTAxMDMJYWZpaTEwMTA0CWFmaWkxMDEwNQlhZmlpMTAxMDYJYWZpaTEwMTA3CWFmaWkxMDEwOAlhZmlpMTAxMDkJYWZpaTEwMTEwCWFmaWkxMDE5MwlhZmlpMTAwNTAJYWZpaTEwMDk4BldncmF2ZQZ3Z3JhdmUGV2FjdXRlBndhY3V0ZQlXZGllcmVzaXMJd2RpZXJlc2lzBllncmF2ZQZ5Z3JhdmUGZW5kYXNoBmVtZGFzaAlhZmlpMDAyMDgNdW5kZXJzY29yZWRibAlxdW90ZWxlZnQKcXVvdGVyaWdodA5xdW90ZXNpbmdsYmFzZQ1xdW90ZXJldmVyc2VkDHF1b3RlZGJsbGVmdA1xdW90ZWRibHJpZ2h0DHF1b3RlZGJsYmFzZQZkYWdnZXIJZGFnZ2VyZGJsBmJ1bGxldAhlbGxpcHNpcwtwZXJ0aG91c2FuZAZtaW51dGUGc2Vjb25kDWd1aWxzaW5nbGxlZnQOZ3VpbHNpbmdscmlnaHQJZXhjbGFtZGJsCGZyYWN0aW9uCW5zdXBlcmlvcgVmcmFuYwlhZmlpMDg5NDEGcGVzZXRhBEV1cm8JYWZpaTYxMjQ4CWFmaWk2MTI4OQlhZmlpNjEzNTIJdHJhZGVtYXJrBU9tZWdhCWVzdGltYXRlZAlvbmVlaWdodGgMdGhyZWVlaWdodGhzC2ZpdmVlaWdodGhzDHNldmVuZWlnaHRocwtwYXJ0aWFsZGlmZgVEZWx0YQdwcm9kdWN0CXN1bW1hdGlvbgVtaW51cwdyYWRpY2FsCGluZmluaXR5CGludGVncmFsC2FwcHJveGVxdWFsCG5vdGVxdWFsCWxlc3NlcXVhbAxncmVhdGVyZXF1YWwHbG96ZW5nZQd1bmlGQjAxB3VuaUZCMDINY3lyaWxsaWNicmV2ZQhkb3RsZXNzahBjYXJvbmNvbW1hYWNjZW50C2NvbW1hYWNjZW50EWNvbW1hYWNjZW50cm90YXRlDHplcm9zdXBlcmlvcgxmb3Vyc3VwZXJpb3IMZml2ZXN1cGVyaW9yC3NpeHN1cGVyaW9yDXNldmVuc3VwZXJpb3INZWlnaHRzdXBlcmlvcgxuaW5lc3VwZXJpb3IHdW5pMjAwMAd1bmkyMDAxB3VuaTIwMDIHdW5pMjAwMwd1bmkyMDA0B3VuaTIwMDUHdW5pMjAwNgd1bmkyMDA3B3VuaTIwMDgHdW5pMjAwOQd1bmkyMDBBB3VuaTIwMEIHdW5pRkVGRgd1bmlGRkZDB3VuaUZGRkQHdW5pMDFGMAd1bmkwMkJDB3VuaTAzRDEHdW5pMDNEMgd1bmkwM0Q2B3VuaTFFM0UHdW5pMUUzRgd1bmkxRTAwB3VuaTFFMDEHdW5pMUY0RAd1bmkwMkYzCWRhc2lhb3hpYQd1bmlGQjAzB3VuaUZCMDQFT2hvcm4Fb2hvcm4FVWhvcm4FdWhvcm4HdW5pMDMwMAd1bmkwMzAxB3VuaTAzMDMEaG9vawhkb3RiZWxvdwd1bmkwNDAwB3VuaTA0MEQHdW5pMDQ1MAd1bmkwNDVEB3VuaTA0NjAHdW5pMDQ2MQd1bmkwNDYyB3VuaTA0NjMHdW5pMDQ2NAd1bmkwNDY1B3VuaTA0NjYHdW5pMDQ2Nwd1bmkwNDY4B3VuaTA0NjkHdW5pMDQ2QQd1bmkwNDZCB3VuaTA0NkMHdW5pMDQ2RAd1bmkwNDZFB3VuaTA0NkYHdW5pMDQ3MAd1bmkwNDcxB3VuaTA0NzIHdW5pMDQ3Mwd1bmkwNDc0B3VuaTA0NzUHdW5pMDQ3Ngd1bmkwNDc3B3VuaTA0NzgHdW5pMDQ3OQd1bmkwNDdBB3VuaTA0N0IHdW5pMDQ3Qwd1bmkwNDdEB3VuaTA0N0UHdW5pMDQ3Rgd1bmkwNDgwB3VuaTA0ODEHdW5pMDQ4Mgd1bmkwNDgzB3VuaTA0ODQHdW5pMDQ4NQd1bmkwNDg2B3VuaTA0ODgHdW5pMDQ4OQd1bmkwNDhBB3VuaTA0OEIHdW5pMDQ4Qwd1bmkwNDhEB3VuaTA0OEUHdW5pMDQ4Rgd1bmkwNDkyB3VuaTA0OTMHdW5pMDQ5NAd1bmkwNDk1B3VuaTA0OTYHdW5pMDQ5Nwd1bmkwNDk4B3VuaTA0OTkHdW5pMDQ5QQd1bmkwNDlCB3VuaTA0OUMHdW5pMDQ5RAd1bmkwNDlFB3VuaTA0OUYHdW5pMDRBMAd1bmkwNEExB3VuaTA0QTIHdW5pMDRBMwd1bmkwNEE0B3VuaTA0QTUHdW5pMDRBNgd1bmkwNEE3B3VuaTA0QTgHdW5pMDRBOQd1bmkwNEFBB3VuaTA0QUIHdW5pMDRBQwd1bmkwNEFEB3VuaTA0QUUHdW5pMDRBRgd1bmkwNEIwB3VuaTA0QjEHdW5pMDRCMgd1bmkwNEIzB3VuaTA0QjQHdW5pMDRCNQd1bmkwNEI2B3VuaTA0QjcHdW5pMDRCOAd1bmkwNEI5B3VuaTA0QkEHdW5pMDRCQgd1bmkwNEJDB3VuaTA0QkQHdW5pMDRCRQd1bmkwNEJGC3VuaTA0QzAuYWx0B3VuaTA0QzEHdW5pMDRDMgd1bmkwNEMzB3VuaTA0QzQHdW5pMDRDNQd1bmkwNEM2B3VuaTA0QzcHdW5pMDRDOAd1bmkwNEM5B3VuaTA0Q0EHdW5pMDRDQgd1bmkwNENDB3VuaTA0Q0QHdW5pMDRDRQt1bmkwNENGLmFsdAd1bmkwNEQwB3VuaTA0RDEHdW5pMDREMgd1bmkwNEQzB3VuaTA0RDQHdW5pMDRENQd1bmkwNEQ2B3VuaTA0RDcHdW5pMDREOAd1bmkwNEQ5B3VuaTA0REEHdW5pMDREQgd1bmkwNERDB3VuaTA0REQHdW5pMDRERQd1bmkwNERGB3VuaTA0RTAHdW5pMDRFMQd1bmkwNEUyB3VuaTA0RTMHdW5pMDRFNAd1bmkwNEU1B3VuaTA0RTYHdW5pMDRFNwd1bmkwNEU4B3VuaTA0RTkHdW5pMDRFQQd1bmkwNEVCB3VuaTA0RUMHdW5pMDRFRAd1bmkwNEVFB3VuaTA0RUYHdW5pMDRGMAd1bmkwNEYxB3VuaTA0RjIHdW5pMDRGMwd1bmkwNEY0B3VuaTA0RjUHdW5pMDRGNgd1bmkwNEY3B3VuaTA0RjgHdW5pMDRGOQd1bmkwNEZBB3VuaTA0RkIHdW5pMDRGQwd1bmkwNEZEB3VuaTA0RkUHdW5pMDRGRgd1bmkwNTAwB3VuaTA1MDEHdW5pMDUwMgd1bmkwNTAzB3VuaTA1MDQHdW5pMDUwNQd1bmkwNTA2B3VuaTA1MDcHdW5pMDUwOAd1bmkwNTA5B3VuaTA1MEEHdW5pMDUwQgd1bmkwNTBDB3VuaTA1MEQHdW5pMDUwRQd1bmkwNTBGB3VuaTA1MTAHdW5pMDUxMQd1bmkwNTEyB3VuaTA1MTMHdW5pMUVBMAd1bmkxRUExB3VuaTFFQTIHdW5pMUVBMwd1bmkxRUE0B3VuaTFFQTUHdW5pMUVBNgd1bmkxRUE3B3VuaTFFQTgHdW5pMUVBOQd1bmkxRUFBB3VuaTFFQUIHdW5pMUVBQwd1bmkxRUFEB3VuaTFFQUUHdW5pMUVBRgd1bmkxRUIwB3VuaTFFQjEHdW5pMUVCMgd1bmkxRUIzB3VuaTFFQjQHdW5pMUVCNQd1bmkxRUI2B3VuaTFFQjcHdW5pMUVCOAd1bmkxRUI5B3VuaTFFQkEHdW5pMUVCQgd1bmkxRUJDB3VuaTFFQkQHdW5pMUVCRQd1bmkxRUJGB3VuaTFFQzAHdW5pMUVDMQd1bmkxRUMyB3VuaTFFQzMHdW5pMUVDNAd1bmkxRUM1B3VuaTFFQzYHdW5pMUVDNwt1bmkxRUM4LmFsdAd1bmkxRUM5C3VuaTFFQ0EuYWx0B3VuaTFFQ0IHdW5pMUVDQwd1bmkxRUNEB3VuaTFFQ0UHdW5pMUVDRgd1bmkxRUQwB3VuaTFFRDEHdW5pMUVEMgd1bmkxRUQzB3VuaTFFRDQHdW5pMUVENQd1bmkxRUQ2B3VuaTFFRDcHdW5pMUVEOAd1bmkxRUQ5B3VuaTFFREEHdW5pMUVEQgd1bmkxRURDB3VuaTFFREQHdW5pMUVERQd1bmkxRURGB3VuaTFFRTAHdW5pMUVFMQd1bmkxRUUyB3VuaTFFRTMHdW5pMUVFNAd1bmkxRUU1B3VuaTFFRTYHdW5pMUVFNwd1bmkxRUU4B3VuaTFFRTkHdW5pMUVFQQd1bmkxRUVCB3VuaTFFRUMHdW5pMUVFRAd1bmkxRUVFB3VuaTFFRUYHdW5pMUVGMAd1bmkxRUYxB3VuaTFFRjQHdW5pMUVGNQd1bmkxRUY2B3VuaTFFRjcHdW5pMUVGOAd1bmkxRUY5B3VuaTIwQUIHdW5pMDMwRhNjaXJjdW1mbGV4YWN1dGVjb21iE2NpcmN1bWZsZXhncmF2ZWNvbWISY2lyY3VtZmxleGhvb2tjb21iE2NpcmN1bWZsZXh0aWxkZWNvbWIOYnJldmVhY3V0ZWNvbWIOYnJldmVncmF2ZWNvbWINYnJldmVob29rY29tYg5icmV2ZXRpbGRlY29tYhBjeXJpbGxpY2hvb2tsZWZ0EWN5cmlsbGljYmlnaG9va1VDEWN5cmlsbGljYmlnaG9va0xDCG9uZS5wbnVtB3plcm8ub3MGb25lLm9zBnR3by5vcwh0aHJlZS5vcwdmb3VyLm9zB2ZpdmUub3MGc2l4Lm9zCHNldmVuLm9zCGVpZ2h0Lm9zB25pbmUub3MCZmYHdW5pMjEyMAhUY2VkaWxsYQh0Y2VkaWxsYQVnLmFsdA9nY2lyY3VtZmxleC5hbHQKZ2JyZXZlLmFsdAhnZG90LmFsdBBnY29tbWFhY2NlbnQuYWx0AUkGSWdyYXZlBklhY3V0ZQtJY2lyY3VtZmxleAlJZGllcmVzaXMGSXRpbGRlB0ltYWNyb24GSWJyZXZlB0lvZ29uZWsKSWRvdGFjY2VudAJJSglJb3RhdG9ub3MESW90YQxJb3RhZGllcmVzaXMJYWZpaTEwMDU1CWFmaWkxMDA1Ngd1bmkwNEMwB3VuaTA0Q0YHdW5pMUVDOAd1bmkxRUNBAAABAAMACAAKAA0AB///AA8AAQAAAAwAAAAWAAAAAgABAAADqQABAAQAAAABAAAAAAABAAAACgA0ADYAAWxhdG4ACAAQAAJNT0wgABZST00gABwAAP//AAAAAP//AAAAAP//AAAAAAAAAAEAAAAKAG4B5AABbGF0bgAIABAAAk1PTCAAKFJPTSAAQgAA//8ACQADAAgACwAAAA4AEQAUABcAGgAA//8ACgAEAAYACQAMAAEADwASABUAGAAbAAD//wAKAAUABwAKAA0AAgAQABMAFgAZABwAHWxpZ2EAsGxpZ2EAtmxpZ2EAvGxudW0AwmxudW0AyGxudW0AzmxvY2wA1GxvY2wA2m9udW0A4G9udW0A6G9udW0A8HBudW0A+HBudW0A/nBudW0BBHNhbHQBCnNhbHQBEnNhbHQBGnNzMDEBInNzMDEBKnNzMDEBMnNzMDIBOnNzMDIBQHNzMDIBRnNzMDMBTHNzMDMBUnNzMDMBWHRudW0BXnRudW0BZnRudW0BbgAAAAEACQAAAAEACQAAAAEACQAAAAEABwAAAAEABwAAAAEABwAAAAEACAAAAAEACAAAAAIAAgADAAAAAgACAAMAAAACAAIAAwAAAAEABAAAAAEABAAAAAEABAAAAAIAAAABAAAAAgAAAAEAAAACAAAAAQAAAAIAAAABAAAAAgAAAAEAAAACAAAAAQAAAAEAAAAAAAEAAAAAAAEAAAAAAAEAAQAAAAEAAQAAAAEAAQAAAAIABQAGAAAAAgAFAAYAAAACAAUABgAKABYAHgAmAC4ANgA+AEYATgBWAF4AAQAAAAEAUAABAAAAAQB6AAEAAAABAKoAAQAAAAEAxgABAAAAAQDuAAEAAAABAPQAAQAAAAEBEAABAAAAAQEWAAEAAAABATIABAAAAAEBSAACABAABQORA5IDkwOUA5UAAgAFAEoASgAAAN8A3wABAOEA4QACAOMA4wADAOUA5QAEAAIALgAUACwAjgCPAJAAkQDqAOwA7gDwAPIA9AFaAWcBdwGhAaICyQLYA0UDRwACAAEDlgOpAAAAAgAaAAoDgwOEA4UDhgOHA4gDiQOKA4sDjAACAAEAEwAcAAAAAgAaAAoDgwOFA4YDhwOIA4kDigOLA4wDhAACAAMAEwATAAAAFQAcAAEDggOCAAkAAgAIAAEDggABAAEAFAACABoACgATABQAFQAWABcAGAAZABoAGwAcAAIAAQODA4wAAAACAAgAAQAUAAEAAQOCAAIAGgAKABMDggAVABYAFwAYABkAGgAbABwAAgABA4MDjAAAAAIADgAEA48DkAEgASEAAgACASQBJQAAAUkBSgACAAEANgABAAgABQAMABQAHAAiACgCXgADAEkATwJdAAMASQBMA40AAgBJAjUAAgBPAjQAAgBMAAEAAQBJAAAAAAABAAEAAQAAAAEAABVeAAAAFAAAAAAAABVWMIIVUgYJKoZIhvcNAQcCoIIVQzCCFT8CAQExCzAJBgUrDgMCGgUAMGEGCisGAQQBgjcCAQSgUzBRMCwGCisGAQQBgjcCARyiHoAcADwAPAA8AE8AYgBzAG8AbABlAHQAZQA+AD4APjAhMAkGBSsOAwIaBQAEFIK4uYCP2fVApm1usxVUQTaZ3tN9oIIRXTCCA3owggJioAMCAQICEDgl1/r4Ya+e9JDnJrXWWtUwDQYJKoZIhvcNAQEFBQAwUzELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMSswKQYDVQQDEyJWZXJpU2lnbiBUaW1lIFN0YW1waW5nIFNlcnZpY2VzIENBMB4XDTA3MDYxNTAwMDAwMFoXDTEyMDYxNDIzNTk1OVowXDELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMTQwMgYDVQQDEytWZXJpU2lnbiBUaW1lIFN0YW1waW5nIFNlcnZpY2VzIFNpZ25lciAtIEcyMIGfMA0GCSqGSIb3DQEBAQUAA4GNADCBiQKBgQDEtfJSFbyIhmApFkpbL0uRa4eR8zVUWDXq0TZeYk1SUTRxwntmHYnI3SrEagr2N9mYdJH2kq6wtXaW8alKY0VHLmsLkk5LK4zuWEqL1AfkGiz4gqpY2c1C8y3Add6Nq8eOHZpsTAiVHt7b72fhcsJJwp5gPOHivhajY3hpFHutLQIDAQABo4HEMIHBMDQGCCsGAQUFBwEBBCgwJjAkBggrBgEFBQcwAYYYaHR0cDovL29jc3AudmVyaXNpZ24uY29tMAwGA1UdEwEB/wQCMAAwMwYDVR0fBCwwKjAooCagJIYiaHR0cDovL2NybC52ZXJpc2lnbi5jb20vdHNzLWNhLmNybDAWBgNVHSUBAf8EDDAKBggrBgEFBQcDCDAOBgNVHQ8BAf8EBAMCBsAwHgYDVR0RBBcwFaQTMBExDzANBgNVBAMTBlRTQTEtMjANBgkqhkiG9w0BAQUFAAOCAQEAUMVLyCSA3+QNJMLeGrGhAqGmgi0MgxWBNwqCDiywWhdhtdgF/ojb8ZGRs1YaQKbrkr44ObB1NnQ6mE/kN7qZicqVQh2wucegjVfg+tVkBEI1TgHRM6IXyE2qJ8fy4YZMAjhNg3jG/FPg6+AGh92klp5eDJjipb6/goXDYOHfrSjYx6VLZNrHG1u9rDkI1TgioTOLL4qa67wHIT9EQQkHtWUcJLxI00SA66HPyQK0FM9UxxajgFz5eT5dcn2IF54sQ6LKU859PfYqOrhPlAClbQqDXfleU/QYs1cPcMP79a2VoA4X3sQWgGDJDytuhgTx6/R4J9EFxe40W165STLyMzCCA8QwggMtoAMCAQICEEe/GZXfjVJGQ/fbbUgNMaQwDQYJKoZIhvcNAQEFBQAwgYsxCzAJBgNVBAYTAlpBMRUwEwYDVQQIEwxXZXN0ZXJuIENhcGUxFDASBgNVBAcTC0R1cmJhbnZpbGxlMQ8wDQYDVQQKEwZUaGF3dGUxHTAbBgNVBAsTFFRoYXd0ZSBDZXJ0aWZpY2F0aW9uMR8wHQYDVQQDExZUaGF3dGUgVGltZXN0YW1waW5nIENBMB4XDTAzMTIwNDAwMDAwMFoXDTEzMTIwMzIzNTk1OVowUzELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMSswKQYDVQQDEyJWZXJpU2lnbiBUaW1lIFN0YW1waW5nIFNlcnZpY2VzIENBMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAqcqypMzNIK8KfYmsh3XwtE7x38EPv2dhvaNkHNq7+cozq4QwiVh+jNtr3TaeD7/R7Hjyd6Z+bzy/k68Numj0bJTKvVItq0g99bbVXV8bAp/6L2sepPejmqYayALhf0xS4w5g7EAcfrkN3j/HtN+HvV96ajEuA5mBE6hHIM4xcw1XLc14NDOVEpkSud5oL6rm48KKjCrDiyGHZr2DWFdvdb88qiaHXcoQFTyfhOpUwQpuxP7FSt25BxGXInzbPifRHnjsnzHJ8eYiGdvEs0dDmhpfoB6Q5F717nzxfatiAY/1TQve0CJWqJXNroh2ru66DfPkTdmg+2igrhQ7s4fBuwIDAQABo4HbMIHYMDQGCCsGAQUFBwEBBCgwJjAkBggrBgEFBQcwAYYYaHR0cDovL29jc3AudmVyaXNpZ24uY29tMBIGA1UdEwEB/wQIMAYBAf8CAQAwQQYDVR0fBDowODA2oDSgMoYwaHR0cDovL2NybC52ZXJpc2lnbi5jb20vVGhhd3RlVGltZXN0YW1waW5nQ0EuY3JsMBMGA1UdJQQMMAoGCCsGAQUFBwMIMA4GA1UdDwEB/wQEAwIBBjAkBgNVHREEHTAbpBkwFzEVMBMGA1UEAxMMVFNBMjA0OC0xLTUzMA0GCSqGSIb3DQEBBQUAA4GBAEpr+epYwkQcMYl5mSuWv4KsAdYcTM2wilhu3wgpo17IypMT5wRSDe9HJy8AOLDkyZNOmtQiYhX3PzchT3AxgPGLOIez6OiXAP7PVZZOJNKpJ056rrdhQfMqzufJ2V7duyuFPrWdtdnhV/++tMV+9c8MnvCX/ivTO1IbGzgn9z9KMIIE/DCCBGWgAwIBAgIQZVIm4bIuGOFZDymFrCLnXDANBgkqhkiG9w0BAQUFADBfMQswCQYDVQQGEwJVUzEXMBUGA1UEChMOVmVyaVNpZ24sIEluYy4xNzA1BgNVBAsTLkNsYXNzIDMgUHVibGljIFByaW1hcnkgQ2VydGlmaWNhdGlvbiBBdXRob3JpdHkwHhcNMDkwNTIxMDAwMDAwWhcNMTkwNTIwMjM1OTU5WjCBtjELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMR8wHQYDVQQLExZWZXJpU2lnbiBUcnVzdCBOZXR3b3JrMTswOQYDVQQLEzJUZXJtcyBvZiB1c2UgYXQgaHR0cHM6Ly93d3cudmVyaXNpZ24uY29tL3JwYSAoYykwOTEwMC4GA1UEAxMnVmVyaVNpZ24gQ2xhc3MgMyBDb2RlIFNpZ25pbmcgMjAwOS0yIENBMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAvmcdtGCqEElvVhd8Zslehg3V8ayncYOOi4n4iASJFQa6LYQhleTRnFBM+9IivdrysjU7Ho/DCfv8Ey5av4l8PTslHvbzWHuc9AG1xgq4gM6+J3RhZydNauXsgWFYeaPgFxASFSew4U00fytHIES53mYkZorNT7ofxTjIVJDhcvYZZnVquUlozzh5DaowqNssYEie16oUAamD1ziRMDkTlgM6fEBUtq3gLxuD3KgRUj4Cs9cr/SG2p1yjDwupphBQDjQuTafOyV4l1Iy88258KbwBXfwxh1rVjIVnWIgZoL818OoroyHnkPaD5ajtYHhee2CD/VcLXUENY1Rg1kMh7wIDAQABo4IB2zCCAdcwEgYDVR0TAQH/BAgwBgEB/wIBADBwBgNVHSAEaTBnMGUGC2CGSAGG+EUBBxcDMFYwKAYIKwYBBQUHAgEWHGh0dHBzOi8vd3d3LnZlcmlzaWduLmNvbS9jcHMwKgYIKwYBBQUHAgIwHhocaHR0cHM6Ly93d3cudmVyaXNpZ24uY29tL3JwYTAOBgNVHQ8BAf8EBAMCAQYwbQYIKwYBBQUHAQwEYTBfoV2gWzBZMFcwVRYJaW1hZ2UvZ2lmMCEwHzAHBgUrDgMCGgQUj+XTGoasjY5rw8+AatRIGCx7GS4wJRYjaHR0cDovL2xvZ28udmVyaXNpZ24uY29tL3ZzbG9nby5naWYwHQYDVR0lBBYwFAYIKwYBBQUHAwIGCCsGAQUFBwMDMDQGCCsGAQUFBwEBBCgwJjAkBggrBgEFBQcwAYYYaHR0cDovL29jc3AudmVyaXNpZ24uY29tMDEGA1UdHwQqMCgwJqAkoCKGIGh0dHA6Ly9jcmwudmVyaXNpZ24uY29tL3BjYTMuY3JsMCkGA1UdEQQiMCCkHjAcMRowGAYDVQQDExFDbGFzczNDQTIwNDgtMS01NTAdBgNVHQ4EFgQUl9BrqCZwyKE/lB8ILcQ1m6ShHvIwDQYJKoZIhvcNAQEFBQADgYEAiwPA3ZTYQaJhabAVqHjHMMaQPH5C9yS25INzFwR/BBCcoeL6gS/rwMpE53LgULZVECCDbpaS5JpRarQ3MdylLeuMAMcdT+dNMrqF+E6++mdVZfBqvnrKZDgaEBB4RXYx84Z6Aw9gwrNdnfaLZnaCG1nhg+W9SaU4VuXeQXcOWA8wggUTMIID+6ADAgECAhBm4/BnecoVFm1QU2+IGRqDMA0GCSqGSIb3DQEBBQUAMIG2MQswCQYDVQQGEwJVUzEXMBUGA1UEChMOVmVyaVNpZ24sIEluYy4xHzAdBgNVBAsTFlZlcmlTaWduIFRydXN0IE5ldHdvcmsxOzA5BgNVBAsTMlRlcm1zIG9mIHVzZSBhdCBodHRwczovL3d3dy52ZXJpc2lnbi5jb20vcnBhIChjKTA5MTAwLgYDVQQDEydWZXJpU2lnbiBDbGFzcyAzIENvZGUgU2lnbmluZyAyMDA5LTIgQ0EwHhcNMTAwNzI5MDAwMDAwWhcNMTIwODA4MjM1OTU5WjCB0DELMAkGA1UEBhMCVVMxFjAUBgNVBAgTDU1hc3NhY2h1c2V0dHMxDzANBgNVBAcTBldvYnVybjEeMBwGA1UEChQVTW9ub3R5cGUgSW1hZ2luZyBJbmMuMT4wPAYDVQQLEzVEaWdpdGFsIElEIENsYXNzIDMgLSBNaWNyb3NvZnQgU29mdHdhcmUgVmFsaWRhdGlvbiB2MjEYMBYGA1UECxQPVHlwZSBPcGVyYXRpb25zMR4wHAYDVQQDFBVNb25vdHlwZSBJbWFnaW5nIEluYy4wgZ8wDQYJKoZIhvcNAQEBBQADgY0AMIGJAoGBAJREoJVpfFUN0NsWjTI1ikwzq14goUzXKoc415ilQPAZSQsiHlNPwkOmyoupVu9uSAaoBRU5HmM7JBKQuZjPygg1fXLjR1f9ecuKSudAcC01Y3+ugM/Er9j798n8idjXpKDbCfKi8nvvzXXB92VQZCKdvX28rbhLzFhFDk3RWUxNAgMBAAGjggGDMIIBfzAJBgNVHRMEAjAAMA4GA1UdDwEB/wQEAwIHgDBEBgNVHR8EPTA7MDmgN6A1hjNodHRwOi8vY3NjMy0yMDA5LTItY3JsLnZlcmlzaWduLmNvbS9DU0MzLTIwMDktMi5jcmwwRAYDVR0gBD0wOzA5BgtghkgBhvhFAQcXAzAqMCgGCCsGAQUFBwIBFhxodHRwczovL3d3dy52ZXJpc2lnbi5jb20vcnBhMBMGA1UdJQQMMAoGCCsGAQUFBwMDMHUGCCsGAQUFBwEBBGkwZzAkBggrBgEFBQcwAYYYaHR0cDovL29jc3AudmVyaXNpZ24uY29tMD8GCCsGAQUFBzAChjNodHRwOi8vY3NjMy0yMDA5LTItYWlhLnZlcmlzaWduLmNvbS9DU0MzLTIwMDktMi5jZXIwHwYDVR0jBBgwFoAUl9BrqCZwyKE/lB8ILcQ1m6ShHvIwEQYJYIZIAYb4QgEBBAQDAgQQMBYGCisGAQQBgjcCARsECDAGAQEAAQH/MA0GCSqGSIb3DQEBBQUAA4IBAQBO5iKH32dBFRfi0u5+Ds7CmdZjvfC1k+VqcmLh9dI8OO6oPQhfukeBgl9bS0n0HSD6D5MJ0B0ZVkQXoojz+42drvcNNd48DKxElGBFKpv+m29MO7E0Z3AQhv9aOVxa42yCqzV8ZUv9mG21FZRJnIhwEL49sWKVtNu01NronUGQfv59uaSS627yIorGdzZNiloLUwUx0ysor1LhjXprtXdEvQyt9F0lLOPNijA+SwOcecqmTq4LwswkBwvBlIL2EPG6kLabmthcPBPx6gIGGCdNPIlvM4rTht7pWDN1PeuTaeJEb04AbM/VhdpWppqmP8tMIWiQ8mC64egGXTkhEzLtMYIDZzCCA2MCAQEwgcswgbYxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5WZXJpU2lnbiwgSW5jLjEfMB0GA1UECxMWVmVyaVNpZ24gVHJ1c3QgTmV0d29yazE7MDkGA1UECxMyVGVybXMgb2YgdXNlIGF0IGh0dHBzOi8vd3d3LnZlcmlzaWduLmNvbS9ycGEgKGMpMDkxMDAuBgNVBAMTJ1ZlcmlTaWduIENsYXNzIDMgQ29kZSBTaWduaW5nIDIwMDktMiBDQQIQZuPwZ3nKFRZtUFNviBkagzAJBgUrDgMCGgUAoHAwEAYKKwYBBAGCNwIBDDECMAAwGQYJKoZIhvcNAQkDMQwGCisGAQQBgjcCAQQwHAYKKwYBBAGCNwIBCzEOMAwGCisGAQQBgjcCARUwIwYJKoZIhvcNAQkEMRYEFEjj6tsXY4/GsRVXJyC3ZfQZU5UYMA0GCSqGSIb3DQEBAQUABIGARTu81Lrv2hu/YjveEuxKBoRFcUHJAv4uDpXzibFS9EHrbTIsSL8pkbyyL11kJDQuupa0tkpzl+D2n0H392i29YAGeEG+U5DAfnhSWxyqDiFC3L4JnDPTRlCQOwWZECtZaeyF2GPRLcMGljTtFKOc8vFUQNVHF6ALAB+MZu/ePhuhggF/MIIBewYJKoZIhvcNAQkGMYIBbDCCAWgCAQEwZzBTMQswCQYDVQQGEwJVUzEXMBUGA1UEChMOVmVyaVNpZ24sIEluYy4xKzApBgNVBAMTIlZlcmlTaWduIFRpbWUgU3RhbXBpbmcgU2VydmljZXMgQ0ECEDgl1/r4Ya+e9JDnJrXWWtUwCQYFKw4DAhoFAKBdMBgGCSqGSIb3DQEJAzELBgkqhkiG9w0BBwEwHAYJKoZIhvcNAQkFMQ8XDTExMDUwNTE2NTUxMFowIwYJKoZIhvcNAQkEMRYEFFQXCCsLve4aJw4fjfxTk/Q4VhAPMA0GCSqGSIb3DQEBAQUABIGAHcF3ia6bbyLja+VF2k6RQPCf7zsfJ0pWrDr9qJRqfPecwX97k2BOG8QrV5WUyxbhmmcz0SspE8jsvrxZsQOkKZnsHYiZJId3D5vKFPvU1ElMdA7IPS5vIMkDzejlD9AhObNWGdX7rL2sqTi9sNUMo9ljrbCVtGhYw+LXKf+RpMcAAA==', false);

            break;
        case 'css':
            header('Content-Type: text/css');
            echo getFileContent('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', false);

            break;
        case 'jquery':
            header('Content-Type: text/javascript');
            echo getFileContent('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', false);

            break;
        case 'gif':
            header('Content-Type: image/gif');
            echo getFileContent('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', false);

            break;
        case 'png-installer':
            header('Content-Type: image/png');
            echo getFileContent('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', false);

            break;
        case 'js-runner':
            header('Content-Type: application/javascript');
            echo getFileContent('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', false);

            break;
    }
    exit;
}

?>
<!DOCTYPE html>
<html>
  <head>
      <meta charset="UTF-8">
      <title>PrestaShop installation</title>
      <link rel="stylesheet" type="text/css" href="<?php echo $selfUri; ?>?element=css">
  </head>
  <body>
    <div id="content-install-in-progress"
       data-extract-url="<?php echo $selfUri; ?>"
       data-check-version-url="<?php echo $selfUri; ?>?run=check-version"
       data-download-latest-url="<?php echo $selfUri; ?>">
        <div>
            <img id="spinner" src="<?php echo $selfUri; ?>?element=gif"/>
            <div id="versionPanel" style="display: none;">Installing Prestashop <?php echo _PS_VERSION_; ?></div>
            <div id="initializationMessage">Initialization ...</div>
            <div id="progressContainer">
                <div class="progressNumber">0 %</div>
                <div class="progress">
                    <div class="current">
                    </div>
                </div>
            </div>
            <div id="error-install-in-progress">
            </div>
        </div>
    </div>
    <div id="content-install-form" style="display: none">
      <div>
        <img id="puffin" src="<?php echo $selfUri; ?>?element=png-installer"/>
        <div id="header">
          The version you’re about to install is not
          the latest version of PrestaShop
        </div>
        <div id="question">
          Do you want to install the latest version instead? (recommended)
        </div>
        <div id="form-panel">
          <div id="form">
            <a id="skip-button" class="button button-no" href="#">No thanks</a>
            <a id="latest-button" class="button button-yes" href="#">Yes please!</a>
          </div>

          <div id="waiting" class="error-container"></div>
          <div id="error-install-form" class="error-container"></div>
          <div id="fallback-after-error" style="display:none;" class="error-container">Cannot download latest Prestashop version.<br/>
            Please click on 'No thanks' to resume standard installation.
          </div>
        </div>
      </div>
    </div>
    <script type="text/javascript" src="<?php echo $selfUri; ?>?element=jquery"></script>
    <script type="text/javascript" src="<?php echo $selfUri; ?>?element=js-runner"></script>
  </body>
</html>
